<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Filter
 *
 * @package    local_kicaquizgrading
 * @copyright  2020 Michael Gardener <mgardener@cissq.com>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace local_kicaquizgrading\output;

defined('MOODLE_INTERNAL') || die();

use local_kicaquizgrading\helper;
use local_kicaquizgrading\shared_lib as NED;

/**
 * Class filter
 * @package local_kicaquizgrading\output
 */
class filter implements \renderable, \templatable {
    /**
     * @var int
     */
    protected $courseid;
    /**
     * @var int
     */
    protected $quizid;
    /**
     * @var int
     */
    protected $attemptid;
    /**
     * @var int|null
     */
    protected $showquestiontext;
    /**
     * @var int|null
     */
    protected $showmanualquestions;
    /**
     * @var int|null
     */
    protected $showautoquestions;
    /**
     * @var int|null
     */
    protected $showquestionswithoutkica;
    /**
     * @var int|null
     */
    protected $hidegradedresponses;
    /**
     * @var int|null
     */
    protected $group;
    /**
     * Quiz Atttempt status graded, requiresgrading or all
     *
     * @var string|null
     */
    protected $status;
    /**
     * @var string|null
     */
    protected $show;
    /**
     * @var int|null
     */
    protected $slotnum;


    /**
     * filter constructor.
     * @param int $courseid
     * @param int $quizid
     * @param int $attemptid
     * @param $showquestiontext
     * @param $showmanualquestions
     * @param $showautoquestions
     * @param $status
     * @param $showquestionswithoutkica
     * @param $hidegradedresponses
     * @param $group
     * @param $show
     * @param $slotnum
     */
    public function __construct(int $courseid, int $quizid, int $attemptid, $showquestiontext, $showmanualquestions,
                                $showautoquestions, $status, $showquestionswithoutkica, $hidegradedresponses, $group, $show, $slotnum) {
        $this->courseid = $courseid;
        $this->quizid = $quizid;
        $this->attemptid = $attemptid;
        $this->showquestiontext = $showquestiontext;
        $this->showmanualquestions = $showmanualquestions;
        $this->showautoquestions = $showautoquestions;
        $this->showquestionswithoutkica = $showquestionswithoutkica;
        $this->hidegradedresponses = $hidegradedresponses;
        $this->group = $group;
        $this->status = $status;
        $this->show = $show;
        $this->slotnum = $slotnum;
    }

    /**
     * Export page contents for template
     * @param \renderer_base $output
     * @return \stdClass
     */
    public function export_for_template(\renderer_base $output) {

        $data = new \stdClass();
        $data->showquestiontext = (is_null($this->showquestiontext)) ? 1 : $this->showquestiontext;
        $data->showmanualquestions = (is_null($this->showmanualquestions)) ? 1 : $this->showmanualquestions;
        $data->showautoquestions = $this->showautoquestions;
        $data->showquestionswithoutkica = $this->showquestionswithoutkica;
        $data->hidegradedresponses = $this->hidegradedresponses;
        $data->group = $this->group;
        $data->{$this->show} = $this->show;
        $data->formurl = (new \moodle_url('/local/kicaquizgrading/index.php'))->out();

        $cm = null;
        if (!empty($this->quizid)) {
            $cm = get_coursemodule_from_instance('quiz', $this->quizid, $this->courseid);
        }

        // Statuses
        $data->statuses = [];
        foreach (['graded', 'requiresgrading'] as $item) {
            $rec = new \stdClass();
            $rec->id = $item;
            $rec->name = get_string($item, 'local_kicaquizgrading');
            if ($this->status == $item) {
                $rec->selected = true;
            }
            $data->statuses[] = $rec;
        }

        $courses = helper::get_courses();

        if (isset($courses[$this->courseid])) {
            $courses[$this->courseid]->selected = true;
        }

        foreach ($courses as $index => $course) {
            if (!$quizzes = helper::get_quizzes($course->id, $this->status)) {
                unset($courses[$index]);
            }
            if ($this->courseid == $course->id) {
                if (isset($quizzes[$this->quizid])) {
                    $quizzes[$this->quizid]->selected = true;
                }
                $data->quizzes = (is_array($quizzes)) ? array_values($quizzes) : [];
            }
        }

        $groups = null;
        if ($this->courseid && $this->quizid && $groups = \local_kicaquizgrading\helper::get_groups($this->courseid, $this->quizid, $this->status)) {
            if (isset($groups[$this->group])) {
                $groups[$this->group]->selected = true;
            }
            $data->groups = array_values($groups);
        }
        $data->hasgroup = !empty($groups);

        $attempts = null;

        $users = [];
        if ($this->courseid) {
            if ($this->group) {
                $users = \local_kicaquizgrading\helper::get_gradebook_users($this->courseid, $this->group);
            } else {
                $users = \local_kicaquizgrading\helper::get_gradebook_users($this->courseid);
            }
        }

        if ($cm && $this->show === 'question') {
            if ($this->quizid && $slots = helper::get_quiz_slots($this->quizid)) {

                $attempts = helper::get_quiz_attempts($this->quizid, $this->status, $users);

                foreach ($slots as $index => $_slot) {
                    foreach ($attempts as $attempt) {
                        $attemptobj = quiz_create_attempt_handling_errors($attempt->id, $cm->id);

                        /** @var \question_usage_by_activity $quba */
                        $quba = \question_engine::load_questions_usage_by_activity($attemptobj->get_uniqueid());

                        /** @var \question_attempt $questionattempt */
                        $questionattempt = $quba->get_question_attempt($_slot->slot);
                        $hidegrades = $this->hidegradedresponses || NED::grade_is_hidden_now_before_midn($cm, $attemptobj->get_userid());

                        $display = helper::display_slot($_slot->questionid, $questionattempt, $this->showquestionswithoutkica,
                            $this->showmanualquestions, $this->showautoquestions, $hidegrades);
                        if ($display) {
                            continue 2;
                        }
                    }
                    unset($slots[$index]);
                }

                $attempts = null;

                foreach ($slots as $_slot) {
                    if ($_slot->slot == $this->slotnum) {
                        $_slot->selected = true;
                    }
                }
                $data->slots = array_values($slots);
            }
        } else {
            if ($this->quizid && $attempts = helper::get_quiz_attempts($this->quizid, $this->status, $users)) {
                if (isset($attempts[$this->attemptid])) {
                    $attempts[$this->attemptid]->selected = true;
                }
                $data->attempts = array_values($attempts);
            }
        }

        $data->hasquiz = !empty($data->quizzes);
        $data->hascourse = !empty($courses);
        $data->hasattempt = !empty($attempts);
        $data->hasslot = !empty($slots);

        $data->courses = array_values($courses);

        if (!empty($this->quizid)) {
            if ($cm) {
                $data->quizlink = (new \moodle_url('/mod/quiz/view.php', ['id' => $cm->id]))->out();
            }
        }

        if (!empty($this->attemptid)) {
            $data->attemptlink = (new \moodle_url('/mod/quiz/review.php', ['attempt' => $this->attemptid]))->out();
        }



        return $data;
    }
}