<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Quiz grader.
 *
 * @package   local_kicaquizgrading
 * @copyright 2021 Michael Gardener <mgardener@cissq.com>
 * @license    http://www.gnu.org/copyleft/gpl.html gnu gpl v3 or later
 */

use local_kicaquizgrading\helper;
use local_kicaquizgrading\shared_lib as NED;

require_once(dirname(__FILE__, 3) . '/config.php');
require_once($CFG->dirroot . '/mod/quiz/locallib.php');
require_once($CFG->dirroot . '/mod/quiz/report/reportlib.php');
require_once($CFG->dirroot . '/question/engine/questionattempt.php');

$attemptid = required_param('attempt', PARAM_INT);

$showquestionswithoutkica = 1;
$showmanualquestions = 1;
$showautoquestions = 1;
//$hidegradedresponses = 0;
$showquestiontext = 1;
$group = null;
$status = 'requiresgrading';
$show = 'student';
$slotnum = 0;

$thispageurl = new moodle_url('/local/kicaquizgrading/view.php');
$attemptrec = $DB->get_record('quiz_attempts', ['id' => $attemptid]);
$quizrec = $DB->get_record('quiz', ['id' => $attemptrec->quiz]);

$cm = get_coursemodule_from_instance('quiz', $attemptrec->quiz, $quizrec->course);
$context = context_module::instance($cm->id);

require_login($quizrec->course, false, $cm);

if (!has_any_capability(['mod/quiz:attempt', 'mod/quiz:viewreports'], $context)) {
    NED::print_module_error('nopermissions', 'error');
}

$PAGE->set_url($thispageurl);
$PAGE->set_pagelayout('secure');

$title = get_string('pluginname', 'local_kicaquizgrading');

// Breadcrumb.
$PAGE->navbar->add($title, $thispageurl);

$output = $PAGE->get_renderer('mod_quiz');

/** @var \local_kicaquizgrading\output\renderer $renderer */
$renderer = $PAGE->get_renderer('local_kicaquizgrading');

try {
    $attemptobj = quiz_create_attempt_handling_errors($attemptid, $cm->id);
} catch (moodle_exception $e) {
    die;
}

// Create an object to manage all the other (non-roles) access rules.
$accessmanager = $attemptobj->get_access_manager(time());
$accessmanager->setup_attempt_page($PAGE);
$PAGE->blocks->show_only_fake_blocks();

echo $OUTPUT->header();
echo html_writer::tag('h1', $title, array('class' => 'page-title'));

if (!get_config('local_kicaquizgrading', 'enable')) {
    echo html_writer::div(get_string('disabledpluginwarning', 'local_kicaquizgrading'), 'alert alert-warning');
    echo $OUTPUT->footer();
    die;
}

// Can only grade finished attempts.
if (!$attemptobj->is_finished()) {
    NED::print_module_error('attemptclosed', 'quiz');
}

$student = $DB->get_record('user', array('id' => $attemptobj->get_userid()));
$hidegradedresponses = NED::grade_is_hidden_now_before_midn($cm, $student->id);

$slots = $attemptobj->get_slots();

/** @var question_usage_by_activity $quba */
$quba = \question_engine::load_questions_usage_by_activity($attemptobj->get_uniqueid());

foreach ($slots as $slot) {
    /** @var question_attempt $questionattempt */
    $questionattempt = $quba->get_question_attempt($slot);

    $slotrec = helper::get_slot_data($attemptrec->quiz, $slot);

    $display = helper::display_slot($slotrec->questionid, $questionattempt, $showquestionswithoutkica, $showmanualquestions,
        $showautoquestions, $hidegradedresponses);

    if (!$display) {
        continue;
    }
    // Prepare summary information about this question attempt.
    $summarydata = array();

    // Student name.
    $userpicture = new user_picture($student);
    $userpicture->courseid = $attemptobj->get_courseid();
    $summarydata['user'] = array(
        'title' => $userpicture,
        'content' => new action_link(new moodle_url('/user/view.php', array(
            'id' => $student->id, 'course' => $attemptobj->get_courseid())),
            fullname($student, true)),
    );

    // Quiz name.
    $summarydata['quizname'] = array(
        'title' => get_string('modulename', 'quiz'),
        'content' => format_string($attemptobj->get_quiz_name()),
    );

    // Question name.
    $summarydata['questionname'] = array(
        'title' => get_string('question', 'quiz'),
        'content' => $attemptobj->get_question_name($slot),
    );

    // Print quiz information.
    echo $output->review_summary_table($summarydata, 0);

    // Print the comment form.
    $data = new stdClass();
    $data->attemptid = $attemptobj->get_attemptid();
    $data->question = $renderer->render_question_for_commenting($attemptobj, $slot, $context);
    $data->slot = $slot;
    $data->course = $quizrec->course;
    $data->quiz = $attemptrec->quiz;
    $data->sesskey = sesskey();
    $data->showquestiontext = $showquestiontext;
    $data->showmanualquestions = $showmanualquestions;
    $data->showautoquestions = $showautoquestions;
    $data->showquestionswithoutkica = $showquestionswithoutkica;
    $data->hidegradedresponses = $hidegradedresponses;
    $data->group = $group;
    $data->status = $status;
    $data->show = $show;
    $data->slotnum = $slotnum;

    echo $OUTPUT->render_from_template('local_kicaquizgrading/gradingform', $data);
}

echo $OUTPUT->footer();