<?php
/**
 * Class for storing assign info (about assign, submissions and grade data)
 *
 * @package    local_ned_controller
 * @subpackage NED
 * @copyright  2022 NED {@link http://ned.ca}
 * @author     NED {@link http://ned.ca}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace local_ned_controller;
use local_ned_controller\shared_lib as NED;

defined('MOODLE_INTERNAL') || die();

/**
 * Base assign_submission class for file submission plugin extending submission plugin base class
 *
 * @package local_ned_controller
 */
abstract class assign_proxy_submission_base extends \assign_submission_plugin {
    /**
     * @return string
     */
    abstract static public function get_table();

    /**
     * @return array [config_key => form_data_key]
     */
    abstract static public function get_config_settings();

    /**
     * Keys from @see get_config_settings() which influence on the deadline
     *
     * @return string[]
     */
    abstract static public function get_deadline_configs();

    /**
     * @return string
     */
    abstract static public function get_file_area();

    /**
     * @return int
     */
    abstract static public function get_maxsummaryfiles();

    /**
     * Save 'was_inactive' is plugin disabled now
     */
    protected function _set_was_inactive_if_disabled(){
        if (!$this->is_enabled()){
            $this->set_was_inactive(true);
        }
    }

    /**
     * Get file submission information from the database
     *
     * @param int $submissionid
     *
     * @return mixed
     */
    protected function _get_file_submission($submissionid){
        return NED::db()->get_record(static::get_table(), ['submission' => $submissionid]);
    }

    /**
     * File format options
     *
     * @return array
     */
    protected function _get_file_options(){
        $fileoptions = array('subdirs' => 1,
            'maxbytes' => $this->get_config('maxsubmissionsizebytes'),
            'maxfiles' => $this->get_config('maxfilesubmissions'),
            'accepted_types' => $this->_get_configured_typesets(),
            'return_types' => (FILE_INTERNAL | FILE_CONTROLLED_LINK));
        if ($fileoptions['maxbytes'] == 0){
            // Use module default.
            $fileoptions['maxbytes'] = get_config($this->get_plugin(), 'maxbytes');
        }
        return $fileoptions;
    }

    /**
     * Get the type sets configured for this assignment.
     *
     * @return array('groupname', 'mime/type', ...)
     */
    protected function _get_configured_typesets(){
        $typeslist = (string)$this->get_config('filetypeslist');

        $util = new \core_form\filetypes_util();
        $sets = $util->normalize_file_types($typeslist);

        return $sets;
    }

    /**
     * Get possible activity list for the proxy point activity selector
     *
     * @param string|string[] $filter
     *
     * @return array
     */
    protected function _get_proxy_filter_activities($filter=[]){
        $cms = NED::dm_get_proxy_activities_choice_list($this->assignment->get_course_module(), $this->assignment->get_course());
        $activities = NED::cms2menu($cms, $filter);

        return ['' => NED::str('selectactivity')] + $activities;
    }

    /**
     * Return full plugin name ie 'assignsubmission_proxy', or short name ie 'proxy'
     *
     * @param bool $short - if true, return short plugin name
     *
     * @return string
     */
    public function get_plugin($short=false){
        if ($short){
            return $this->get_type();
        } else {
            return $this->get_subtype() . '_' . $this->get_type();
        }
    }

    /**
     * Translate string
     *
     * @param string $identifier The key identifier for the localized string
     * @param string|object|array $a An object, string or number that can be used within translation strings
     *
     * @return string The localized string.
     */
    public function str($identifier, $a=null){
        return get_string($identifier, $this->get_plugin(), $a);
    }

    /**
     * Get the name of the file submission plugin
     *
     * @return string
     */
    public function get_name(){
        return $this->str($this->get_plugin(true));
    }

    /**
     * @return bool
     */
    public function was_inactive(){
        return (bool)$this->get_config('was_inactive');
    }

    /**
     * @param bool $value
     *
     * @return void
     */
    public function set_was_inactive($value){
        $this->set_config('was_inactive', (int)((bool)$value));
    }

    /**
     * Disable assign deadlines if plugin is enable
     *
     * @param \MoodleQuickForm $mform
     */
    public function disable_assign_deadlines($mform){
        $plugin = $this->get_plugin();
        $mform->disabledIf('duedate', $plugin.'_enabled', 'checked');
        $mform->disabledIf('cutoffdate', $plugin.'_enabled', 'checked');
    }

    /**
     * Save the settings for file submission plugin
     *
     * @param \stdClass $formdata
     *
     * @return bool
     */
    public function save_settings(\stdClass $formdata){
        if (empty($formdata->assignsubmission_proxy_filetypes)){
            $formdata->assignsubmission_proxy_filetypes = '';
        }

        $new = empty($formdata->instance);
        $settings = static::get_config_settings();
        $deadline_keys = static::get_deadline_configs();
        $deadline_changes = $new || $this->was_inactive();
        $plugin = $this->get_plugin(true);

        foreach ($settings as $config => $form_key){
            $new_value = $formdata->$form_key;
            if (!$deadline_changes && in_array($config, $deadline_keys)){
                $old_value = $this->get_config($config);
                $deadline_changes = $old_value != $new_value;
            }

            $this->set_config($config, $new_value);
        }

        if ($new){
            $this->set_was_inactive(true);
        } else {
            $active_plugin = NED::dm_proxy_updated($this->assignment->get_course_module(), $plugin, $deadline_changes);
            $this->set_was_inactive($active_plugin != $plugin);
        }

        return true;
    }

    /**
     * Add elements to submission form
     *
     * @param mixed            $submissionorgrade stdClass|null
     * @param \MoodleQuickForm $mform
     * @param \stdClass        $data
     *
     * @return bool
     */
    public function get_form_elements($submissionorgrade, \MoodleQuickForm $mform, \stdClass $data){
        if ($this->get_config('maxfilesubmissions') <= 0){
            return false;
        }

        $fileoptions = $this->_get_file_options();
        $submissionid = $submissionorgrade ? $submissionorgrade->id : 0;

        file_prepare_standard_filemanager($data,
            'files',
            $fileoptions,
            $this->assignment->get_context(),
            $this->get_plugin(),
            static::get_file_area(),
            $submissionid);
        $mform->addElement('filemanager', 'files_filemanager', $this->get_name(), null, $fileoptions);

        return true;
    }

    /**
     * Count the number of files
     *
     * @param int $submissionid
     * @param string $area
     *
     * @return int
     */
    public function count_files($submissionid, $area){
        $fs = get_file_storage();
        $files = $fs->get_area_files($this->assignment->get_context()->id,
            $this->get_plugin(),
            $area,
            $submissionid,
            'id',
            false);

        return count($files);
    }

    /**
     * Remove files from this submission.
     *
     * @param \stdClass $submission The submission
     *
     * @return boolean
     */
    public function remove(\stdClass $submission){
        $fs = get_file_storage();

        $fs->delete_area_files($this->assignment->get_context()->id,
            $this->get_plugin(),
            static::get_file_area(),
            $submission->id);

        $currentsubmission = $this->_get_file_submission($submission->id);
        if ($currentsubmission){
            $currentsubmission->numfiles = 0;
            NED::db()->update_record(static::get_table(), $currentsubmission);
        }

        return true;
    }

    /**
     * Produce a list of files suitable for export that represent this feedback or submission
     *
     * @param \stdClass $submissionorgrade The submission
     * @param \stdClass $user              The user record - unused
     *
     * @return array - return an array of files indexed by filename
     */
    public function get_files(\stdClass $submissionorgrade, \stdClass $user){
        $result = array();
        $fs = get_file_storage();

        $files = $fs->get_area_files($this->assignment->get_context()->id,
            $this->get_plugin(),
            static::get_file_area(), $submissionorgrade->id,
            'timemodified',
            false);

        foreach ($files as $file){
            // Do we return the full folder path or just the file name?
            if (isset($submissionorgrade->exportfullpath) && !$submissionorgrade->exportfullpath){
                $result[$file->get_filename()] = $file;
            } else {
                $result[$file->get_filepath().$file->get_filename()] = $file;
            }
        }
        return $result;
    }

    /**
     * Display the list of files  in the submission status table
     *
     * @param \stdClass $submissionorgrade
     * @param bool      $showviewlink Set this to true if the list of files is long
     *
     * @return string
     */
    public function view_summary(\stdClass $submissionorgrade, &$showviewlink){
        $plugin = $this->get_plugin();
        $max = static::get_maxsummaryfiles();
        $file_area = static::get_file_area();
        $count = $this->count_files($submissionorgrade->id, $file_area);

        // Show we show a link to view all files for this plugin?
        $showviewlink = $count > $max;
        if ($count <= $max){
            return $this->assignment->render_area_files($plugin, $file_area, $submissionorgrade->id);
        } else {
            return $this->str('countfiles', $count);
        }
    }

    /**
     * No full submission view - the summary contains the list of files and that is the whole submission
     *
     * @param \stdClass $submissionorgrade
     *
     * @return string
     */
    public function view(\stdClass $submissionorgrade){
        return $this->assignment->render_area_files($this->get_plugin(),
            static::get_file_area(),
            $submissionorgrade->id);
    }

    /**
     * Return true if this plugin can upgrade an old Moodle 2.2 assignment of this type
     * and version.
     *
     * @param string $type
     * @param int $version
     *
     * @return bool True if upgrade is possible
     */
    public function can_upgrade($type, $version){

        $uploadsingletype ='uploadsingle';
        $uploadtype ='upload';

        if (($type == $uploadsingletype || $type == $uploadtype) && $version >= 2011112900){
            return true;
        }
        return false;
    }

    /**
     * Upgrade the settings from the old assignment
     * to the new plugin based one
     *
     * @param \context $oldcontext - the old assignment context
     * @param \stdClass $oldassignment - the old assignment data record
     * @param string $log record log events here
     *
     * @return bool Was it a success? (false will trigger rollback)
     */
    public function upgrade_settings(\context $oldcontext, \stdClass $oldassignment, & $log){
        global $DB;

        if ($oldassignment->assignmenttype == 'uploadsingle'){
            $this->set_config('maxfilesubmissions', 1);
            $this->set_config('maxsubmissionsizebytes', $oldassignment->maxbytes);
            return true;
        } else if ($oldassignment->assignmenttype == 'upload'){
            $this->set_config('maxfilesubmissions', $oldassignment->var1);
            $this->set_config('maxsubmissionsizebytes', $oldassignment->maxbytes);

            // Advanced file upload uses a different setting to do the same thing.
            $DB->set_field('assign',
                'submissiondrafts',
                $oldassignment->var4,
                array('id'=>$this->assignment->get_instance()->id));

            // Convert advanced file upload "hide description before due date" setting.
            $alwaysshow = 0;
            if (!$oldassignment->var3){
                $alwaysshow = 1;
            }
            $DB->set_field('assign',
                'alwaysshowdescription',
                $alwaysshow,
                array('id'=>$this->assignment->get_instance()->id));
            return true;
        }

        return false;
    }

    /**
     * Upgrade the submission from the old assignment to the new one
     *
     * @param \context  $oldcontext           The context of the old assignment
     * @param \stdClass $oldassignment        The data record for the old oldassignment
     * @param \stdClass $oldsubmissionorgrade The data record for the old submission
     * @param \stdClass $submissionorgrade    The data record for the new submission
     * @param string    $log                  Record upgrade messages in the log
     *
     * @return bool true or false - false will trigger a rollback
     */
    public function upgrade(\context $oldcontext, \stdClass $oldassignment, \stdClass $oldsubmissionorgrade, \stdClass $submissionorgrade, & $log){
        $filesubmission = new \stdClass();

        $filesubmission->numfiles = $oldsubmissionorgrade->numfiles;
        $filesubmission->submission = $submissionorgrade->id;
        $filesubmission->assignment = $this->assignment->get_instance()->id;

        if (!NED::db()->insert_record(static::get_table(), $filesubmission) > 0){
            $log .= get_string('couldnotconvertsubmission', 'mod_assign', $submissionorgrade->userid);
            return false;
        }

        // Now copy the area files.
        $this->assignment->copy_area_files_for_upgrade($oldcontext->id,
            'mod_assignment',
            'submission',
            $oldsubmissionorgrade->id,
            $this->assignment->get_context()->id,
            $this->get_plugin(),
            static::get_file_area(),
            $submissionorgrade->id);

        return true;
    }

    /**
     * The assignment has been deleted - cleanup
     *
     * @return bool
     */
    public function delete_instance(){
        // Will throw exception on failure.
        return NED::db()->delete_records(static::get_table(), ['assignment' => $this->assignment->get_instance()->id]);
    }

    /**
     * Formatting for log info
     *
     * @param \stdClass $submissionorgrade The submission
     *
     * @return string
     */
    public function format_for_log(\stdClass $submissionorgrade){
        // Format the info for each submission plugin (will be added to log).
        $filecount = $this->count_files($submissionorgrade->id, static::get_file_area());

        return $this->str('numfilesforlog', $filecount);
    }

    /**
     * Return true if there are no submission files
     *
     * @param \stdClass $submissionorgrade
     *
     * @return bool
     */
    public function is_empty(\stdClass $submissionorgrade){
        return $this->count_files($submissionorgrade->id, static::get_file_area()) == 0;
    }

    /**
     * Determine if a submission is empty
     *
     * This is distinct from is_empty in that it is intended to be used to
     * determine if a submission made before saving is empty.
     *
     * @param \stdClass $data The submission data
     *
     * @return bool
     */
    public function submission_is_empty(\stdClass $data){
        global $USER;
        $fs = get_file_storage();
        // Get a count of all the draft files, excluding any directories.
        $files = $fs->get_area_files(\context_user::instance($USER->id)->id,
            'user',
            'draft',
            $data->files_filemanager,
            'id',
            false);
        return count($files) == 0;
    }

    /**
     * Get file areas returns a list of areas this plugin stores files
     *
     * @return array - An array of fileareas (keys) and descriptions (values)
     */
    public function get_file_areas(){
        return [static::get_file_area() => $this->get_name()];
    }

    /**
     * Copy the student's submission from a previous submission. Used when a student opts to base their resubmission
     * on the last submission.
     *
     * @param \stdClass $oldsubmission
     * @param \stdClass $submission
     *
     * @return bool
     */
    public function copy_submission(\stdClass $oldsubmission, \stdClass $submission){
        // Copy the files across.
        $contextid = $this->assignment->get_context()->id;
        $fs = get_file_storage();
        $files = $fs->get_area_files($contextid,
            $this->get_plugin(),
            static::get_file_area(), $oldsubmission->id,
            'id',
            false);
        foreach ($files as $file){
            $fieldupdates = array('itemid' => $submission->id);
            $fs->create_file_from_storedfile($fieldupdates, $file);
        }

        // Copy the record.
        if ($filesubmission = $this->_get_file_submission($oldsubmission->id)){
            unset($filesubmission->id);
            $filesubmission->submission = $submission->id;
            NED::db()->insert_record(static::get_table(), $filesubmission);
        }

        return true;
    }

    /**
     * Return a description of external params suitable for uploading a file submission from a webservice.
     *
     * @return \external_description|array|null
     */
    public function get_external_parameters(){
        return array(
            'files_filemanager' => new \external_value(
                PARAM_INT,
                'The id of a draft area containing files for this submission.',
                VALUE_OPTIONAL
            )
        );
    }

    /**
     * Return the plugin configs for external functions.
     *
     * @return array the list of settings
     * @since Moodle 3.2
     */
    public function get_config_for_external(){
        global $CFG;

        $configs = $this->get_config();

        // Get a size in bytes.
        if ($configs->maxsubmissionsizebytes == 0){
            $configs->maxsubmissionsizebytes = get_max_upload_file_size($CFG->maxbytes, $this->assignment->get_course()->maxbytes,
                get_config($this->get_plugin(), 'maxbytes'));
        }
        return (array) $configs;
    }

    /**
     * Determine if the plugin allows image file conversion
     *
     * @return bool
     */
    public function allow_image_conversion(){
        return true;
    }
}
