<?php
/**
 * @package    local_ned_controller
 * @category   NED
 * @copyright  2021 NED {@link http://ned.ca}
 * @author     NED {@link http://ned.ca}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace local_ned_controller;
use \local_ned_controller\shared_lib as NED;

defined('MOODLE_INTERNAL') || die();

/**
 * Class core_course_renderer
 *
 * @package local_ned_controller
 */
class core_course_renderer extends \core_course_renderer {
    /**
     * Rewrite this constant in child class
     */
    const PLUGIN = 'ned_controller';

    /**
     * @var \stdClass @static - copy this property to the child class
     */
    protected static $themesettings = null;
    /**
     * @var \local_ned_controller\toolbox @static
     */
    protected static $nctoolbox = null;

    /**
     * @var bool
     */
    protected $editingoff;

    /**
     * core_course_renderer constructor.
     *
     * @param \moodle_page $page
     * @param              $target
     *
     * @throws \coding_exception
     */
    public function __construct(\moodle_page $page, $target) {
        parent::__construct($page, $target);
        if (empty(static::$nctoolbox)) {
            static::$nctoolbox = \local_ned_controller\toolbox::get_instance();
        }
        if (empty(static::$themesettings)) {
            static::$themesettings = \theme_config::load(static::PLUGIN)->settings;
        }
        $this->editingoff = !$page->user_is_editing();
    }

    /**
     * Output frontpage summary text and frontpage modules (stored as section 1 in site course)
     *
     * This may be disabled in settings
     *
     * @return string
     */
     public function frontpage_section1() {
         if (!NED::is_desktop()) {
            return '';
         }
         return parent::frontpage_section1();
    }

    /**
     * Renders html to display a course search form
     *
     * @param string $value default value to populate the search field
     * @param string $format display format - 'plain' (default), 'short' or 'navbar'
     * @return string
     */
    public function course_search_form($value = '', $format = 'plain') {
        if (!NED::is_plugin_exists(NED::REPORT_GHS)) {
            return NED::div(parent::course_search_form($value), 'nedcoursesearchbox');
        }

        static $count = 0;
        $formid = 'coursesearch';
        if ((++$count) > 1) {
            $formid .= $count;
        }

        switch ($format) {
            case 'navbar' :
                $formid = 'coursesearchnavbar';
                $inputid = 'navsearchbox';
                $inputsize = 20;
                break;
            case 'short' :
                $inputid = 'shortsearchbox';
                $inputsize = 12;
                break;
            default :
                $inputid = 'coursesearchbox';
                $inputsize = 30;
        }

        $data = new \stdClass();
        $data->searchurl = \core_search\manager::get_course_search_url()->out(false);
        $data->id = $formid;
        $data->inputid = $inputid;
        $data->inputsize = $inputsize;
        $data->value = $value;
        $data->areaids = 'core_course-course';

        if ($format != 'navbar') {
            $helpicon = new \help_icon('coursesearch', 'core');
            $data->helpicon = $helpicon->export_for_template($this);
        }

        if (class_exists('\report_ghs\helper')){
            $contextsystem = NED::ctx();
            $data->searchurl = (new \moodle_url('/report/ghs/ghs_group_enrollment.php'))->out();
            $data->viewgroupenrollment = has_capability('report/ghs:viewgroupenrollment', $contextsystem) ||
                \report_ghs\helper::has_capability_in_any_course('report/ghs:viewgroupenrollment');
        }

        $output =  $this->render_from_template('core_course/course_search_form', $data);
        $output = \html_writer::div($output, 'nedcoursesearchbox');
        return $output;
    }

    /**
     * Renders html to display a name with the link to the course module on a course page
     *
     * If module is unavailable for user but still needs to be displayed
     * in the list, just the name is returned without a link
     *
     * Note, that for course modules that never have separate pages (i.e. labels)
     * this function return an empty string
     *
     * @param \cm_info $mod
     * @param array $displayoptions
     * @return string
     */
    public function course_section_cm_name_title(\cm_info $mod, $displayoptions = array()) {
        $output = '';
        $url = $mod->url;
        if (!$mod->is_visible_on_course_page() || !$url) {
            // Nothing to be displayed to the user.
            return $output;
        }

        if ($mod->modname === 'assign') {
            if (class_exists('\assignsubmission_ctsubmission\helper')) {
                if ($ctassigns = \assignsubmission_ctsubmission\helper::get_ctsubmission_assignments($mod->course)) {
                    foreach ($ctassigns as $ctassign) {
                        if ($ctassign->assignment === $mod->instance) {
                            $url = new \moodle_url('/mod/assign/submission/ctsubmission/overview.php', ['courseid' => $mod->course, 'cmid' => $mod->id]);
                            break;
                        }
                    }
                }
            }
        }

        // Accessibility: for files get description via icon, this is very ugly hack!
        $instancename = $mod->get_formatted_name();
        $altname = $mod->modfullname;
        /* Avoid unnecessary duplication: if e.g. a forum name already
           includes the word forum (or Forum, etc) then it is unhelpful
           to include that in the accessible description that is added. */
        if (str_contains(\core_text::strtolower($instancename), \core_text::strtolower($altname))){
            $altname = '';
        }
        // File type after name, for alphabetic lists (screen reader).
        if ($altname) {
            $altname = get_accesshide(' '.$altname);
        }

        $format = course_get_format($mod->course);

        $cmclass = $format->get_output_classname('content\\cm');
        $cmoutput = new $cmclass($format, $mod->get_section_info(), $mod);

        $linkclasses = $cmoutput->get_link_classes();
        $textclasses = $cmoutput->get_text_classes();

        /* Get on-click attribute value if specified and decode the onclick - it
           has already been encoded for display. */
        $onclick = htmlspecialchars_decode($mod->onclick, ENT_QUOTES);

        // Display link itself.

        // Start of NED specific changes.
        // We should search only for reference to FontAwesome icons.  From the FontAwesome filter: https://moodle.org/plugins/pluginversions.php?plugin=filter_fontawesome.
        $fasearch = "(\[(fa-.*?)\])is";
        $fa_callback = function(array $matches) {
            if (!isset($matches[1])) return '';
            return NED::fa($matches[1]);
        };
        $instancename = preg_replace_callback($fasearch, $fa_callback, $instancename);
        // End of NED specific changes.

        $activitylink =
            \html_writer::img($mod->get_icon_url(), ' ', ['class' => 'iconlarge activityicon', 'role' => 'presentation']) .
            \html_writer::span($instancename . $altname,'instancename');
        if ($mod->uservisible) {
            $output .= \html_writer::link($url, $activitylink, ['class' => $linkclasses, 'onclick' => $onclick]);
        } else {
            /* We may be displaying this just in order to show information
               about visibility, without the actual link ($mod->is_visible_on_course_page()). */
            $output .= \html_writer::div($activitylink, $textclasses);
        }
        return $output;
    }

    /**
     * Returns HTML to print list of courses user is enrolled to for the frontpage
     *
     * Also lists remote courses or remote hosts if MNET authorisation is used
     *
     * @return string
     */
    public function frontpage_my_courses() {
        global $USER, $CFG, $DB;

        if (!isloggedin() or isguestuser()) {
            return '';
        }

        $output = '';
        $courses  = enrol_get_my_courses('summary, summaryformat');
        $rhosts   = array();
        $rcourses = array();
        if (!empty($CFG->mnet_dispatcher_mode) && $CFG->mnet_dispatcher_mode==='strict') {
            $rcourses = get_my_remotecourses($USER->id);
            $rhosts   = get_my_remotehosts();
        }

        // NED.
        foreach ($courses as $index => $course) {
            $format = course_get_format($course->id);
            if ($format->get_format() == 'ned'){
                /** @noinspection PhpUndefinedMethodInspection */
                if ($format->get_setting('donotshowinfrontpage')){
                    unset($courses[$index]);
                }
            }
        }

        if (!empty($courses) || !empty($rcourses) || !empty($rhosts)){

            $chelper = new \coursecat_helper();
            $totalcount = count($courses);
            if (count($courses) > $CFG->frontpagecourselimit){
                // There are more enrolled courses than we can display, display link to 'My courses'.
                $courses = array_slice($courses, 0, $CFG->frontpagecourselimit, true);
                $chelper->set_courses_display_options(array(
                    'viewmoreurl' => new \moodle_url('/my/'),
                    'viewmoretext' => new \lang_string('mycourses')
                ));
            } else if (\core_course_category::top()->is_uservisible()){
                // All enrolled courses are displayed, display link to 'All courses' if there are more courses in system.
                $chelper->set_courses_display_options(array(
                    'viewmoreurl' => new \moodle_url('/course/index.php'),
                    'viewmoretext' => new \lang_string('fulllistofcourses')
                ));
                $totalcount = $DB->count_records('course') - 1;
            }
            $chelper->set_show_courses(self::COURSECAT_SHOW_COURSES_EXPANDED)->
            set_attributes(array('class' => 'frontpage-course-list-enrolled'));
            $output .= $this->coursecat_courses($chelper, $courses, $totalcount);

            // MNET
            if (!empty($rcourses)) {
                // at the IDP, we know of all the remote courses
                $output .= \html_writer::start_tag('div', array('class' => 'courses'));
                foreach ($rcourses as $course) {
                    $output .= $this->frontpage_remote_course($course);
                }
                $output .= \html_writer::end_tag('div'); // .courses
            } elseif (!empty($rhosts)) {
                // non-IDP, we know of all the remote servers, but not courses
                $output .= \html_writer::start_tag('div', array('class' => 'courses'));
                foreach ($rhosts as $host) {
                    $output .= $this->frontpage_remote_host($host);
                }
                $output .= \html_writer::end_tag('div'); // .courses
            }
        }
        return $output;
    }

    /**
     * Displays one course in the list of courses.
     *
     * This is an internal function, to display an information about just one course
     * please use {@link core_course_renderer::course_info_box()}
     *
     * @param \coursecat_helper $chelper various display options
     * @param \core_course_list_element|\stdClass $course
     * @param string $additionalclasses additional classes to add to the main <div> tag (usually
     *    depend on the course position in list - first/last/even/odd)
     * @return string
     */
    protected function coursecat_coursebox(\coursecat_helper $chelper, $course, $additionalclasses = '') {
        global $OUTPUT;
        if (!isset($this->strings->summary)) {
            $this->strings->summary = get_string('summary');
        }
        if ($chelper->get_show_courses() <= static::COURSECAT_SHOW_COURSES_COUNT) {
            return '';
        }
        if ($course instanceof \stdClass) {
            $course = new \core_course_list_element($course);
        }
        $content = '';
        $classes = trim('coursebox clearfix '. $additionalclasses);
        if ($chelper->get_show_courses() >= static::COURSECAT_SHOW_COURSES_EXPANDED) {
            $nametag = 'h3';
        } else {
            $classes .= ' collapsed';
            $nametag = 'div';
        }

        // .coursebox
        $content .= \html_writer::start_tag('div', [
            'class' => $classes,
            'data-courseid' => $course->id,
            'data-type' => static::COURSECAT_TYPE_COURSE,
        ]);

        $content .= \html_writer::start_tag('div', ['class' => 'info']);

        // Course name.
        $coursename = $chelper->get_course_formatted_name($course);
        $coursenamelink = \html_writer::link(new \moodle_url('/course/view.php', ['id' => $course->id]),
            $coursename, ['class' => $course->visible ? '' : 'dimmed']);
        $content .= \html_writer::tag($nametag, $coursenamelink, ['class' => 'coursename']);

        // Begin NED specific changes.
        $show_ned_menu = false;
        $menu = new \local_ned_controller\output\ned_action_menu();
        $triggerextraclasses = [];
        if (!$course->visible){
            $triggerextraclasses[] = 'dimmed';
        }
        $menu->attributessecondary['class'] .= ' nedcoursename';

        $coursecontext = \context_course::instance($course->id);
        $editsettingsquicklink = static::$nctoolbox->get_editsettings_quicklink($coursecontext, $course->id, 'coursename');
        if ($editsettingsquicklink) {
            $menuitemstring = get_string('editsettings');
            $menu->add_secondary_action(new \action_link($editsettingsquicklink, $menuitemstring, null, null,
                new \pix_icon('t/edit', $menuitemstring)));
            $show_ned_menu = true;
        }

        $courseparticipantsquicklink = static::$nctoolbox->get_courseparticipants_quicklink($coursecontext, $course->id, 'coursename');
        if ($courseparticipantsquicklink) {
            $menuitemstring = NED::str('quicklinkscourseparticpants');
            $menu->add_secondary_action(new \action_link($courseparticipantsquicklink, $menuitemstring, null, null,
                new \pix_icon('t/groups', $menuitemstring)));
            $show_ned_menu = true;
        }

        $manualenrolmentquicklink = static::$nctoolbox->get_manualenrollment_quicklink($coursecontext, $course->id, 'coursename');
        if ($manualenrolmentquicklink) {
            $menuitemstring = NED::str('quicklinksmanualenrollments');
            $menu->add_secondary_action(new \action_link($manualenrolmentquicklink, $menuitemstring, null, null,
                new \pix_icon('t/enrolusers', $menuitemstring)));
            $show_ned_menu = true;
        }

        $nedprogressreportquicklink = static::$nctoolbox->get_nedprogressreport_quicklink($coursecontext, $course->id, 'coursename');
        if ($nedprogressreportquicklink) {
            $menuitemstring = NED::str('quicklinksnedprogressreport');
            $menu->add_secondary_action(new \action_link($nedprogressreportquicklink, $menuitemstring, null, null,
                new \pix_icon('i/grades', $menuitemstring)));
            $show_ned_menu = true;
        }

        $nedmarkingmanagerquicklink = static::$nctoolbox->get_nedmarkingmanager_quicklink($coursecontext, $course->id, 'coursename');
        if ($nedmarkingmanagerquicklink) {
            $menuitemstring = NED::str('quicklinksnedmarkingmanager');
            $menu->add_secondary_action(new \action_link($nedmarkingmanagerquicklink, $menuitemstring, null, null,
                new \pix_icon('i/competencies', $menuitemstring)));
            $show_ned_menu = true;
        }

        $nedstudentmenuquicklink = static::$nctoolbox->get_nedstudentmenu_quicklink($coursecontext, $course->id, 'coursename');
        if ($nedstudentmenuquicklink) {
            $menuitemstring = NED::str('quicklinksnedstudentmenu');
            $menu->add_secondary_action(new \action_link($nedstudentmenuquicklink, $menuitemstring, null, null,
                new \pix_icon('i/competencies', $menuitemstring)));
            $show_ned_menu = true;
        }

        $gradebookquicklink = static::$nctoolbox->get_gradebook_quicklink($coursecontext, $course->id, 'coursename');
        if ($gradebookquicklink) {
            $menuitemstring = NED::str('quicklinksgradebook');
            $menu->add_secondary_action(new \action_link($gradebookquicklink, $menuitemstring, null, null,
                new \pix_icon('i/report', $menuitemstring)));
            $show_ned_menu = true;
        }

        if ($show_ned_menu) {
            $menu->set_menu_trigger(' ', join(' ', $triggerextraclasses));
            $content .= $OUTPUT->render($menu);
        }

        if (NED::is_desktop()) {
            $content .= \html_writer::div($course->get_formatted_shortname(), 'course-shortname');
            // End NED specific changes.

            // If we display course in collapsed form but the course has summary or course contacts, display the link to the info page.
            $content .= \html_writer::start_tag('div', ['class' => 'moreinfo']);
            if ($chelper->get_show_courses() < static::COURSECAT_SHOW_COURSES_EXPANDED) {
                if ($course->has_summary() || $course->has_course_contacts() || $course->has_course_overviewfiles()) {
                    $url = new \moodle_url('/course/info.php', ['id' => $course->id]);
                    $image = $this->output->pix_icon('i/info', $this->strings->summary);
                    $content .= \html_writer::link($url, $image, ['title' => $this->strings->summary]);
                    // Make sure JS file to expand course content is included.
                    $this->coursecat_include_js();
                }
            }
            $content .= \html_writer::end_tag('div'); // .moreinfo
        }


        // Print enrolmenticons.
        if ($icons = enrol_get_course_info_icons($course)) {
            $content .= \html_writer::start_tag('div', ['class' => 'enrolmenticons']);
            foreach ($icons as $pixicon) {
                $content .= $this->render($pixicon);
            }
            $content .= \html_writer::end_tag('div'); // .enrolmenticons
        }

        $content .= \html_writer::end_tag('div'); // .info

        if (NED::is_desktop()) {
            $content .= \html_writer::div($this->coursecat_coursebox_content($chelper, $course), 'content');
        }

        $content .= \html_writer::end_tag('div'); // .coursebox
        return $content;
    }

    /**
     * Returns HTML to display course content (summary, course contacts and optionally category name)
     *
     * This method is called from coursecat_coursebox() and may be re-used in AJAX
     *
     * @param \coursecat_helper $chelper various display options
     * @param \stdClass|\core_course_list_element $course
     * @return string
     */
    protected function coursecat_coursebox_content(\coursecat_helper $chelper, $course) {
        global $CFG;
        if ($chelper->get_show_courses() < static::COURSECAT_SHOW_COURSES_EXPANDED) {
            return '';
        }
        if ($course instanceof \stdClass) {
            $course = new \core_course_list_element($course);
        }
        $content = '';

        // Display course summary.
        if ($course->has_summary()) {
            $content .= \html_writer::div(
                $chelper->get_course_formatted_summary($course, ['overflowdiv' => true, 'noclean' => true, 'para' => false]), 'summary'
            );
        }

        // Display course overview files.
        $contentimages = $contentfiles = '';
        foreach ($course->get_course_overviewfiles() as $file) {
            $isimage = $file->is_valid_image();
            $url = \moodle_url::make_file_url("$CFG->wwwroot/pluginfile.php",
                    '/'. $file->get_contextid(). '/'. $file->get_component(). '/'.
                    $file->get_filearea(). $file->get_filepath(). $file->get_filename(), !$isimage);
            if ($isimage) { // Begin NED specific changes.
                $courseimagelink = \html_writer::link(new \moodle_url('/course/view.php', ['id' => $course->id]),
                    \html_writer::img($url, ' '), ['class' => $course->visible ? '' : 'dimmed']);
                $contentimages .= \html_writer::div($courseimagelink,'courseimage');
            } else { // End NED specific changes.
                $image = $this->output->pix_icon(file_file_icon($file, 24), $file->get_filename(), 'moodle');
                $filename = \html_writer::span($image,'fp-icon').\html_writer::span($file->get_filename(), 'fp-filename');
                $contentfiles .= \html_writer::span(\html_writer::link($url, $filename), 'coursefile fp-filename-icon');
            }
        }
        $content .= $contentimages. $contentfiles;

        // Display course contacts. See course_in_list::get_course_contacts().
        if ($course->has_course_contacts()) {
            $content .= \html_writer::start_tag('ul', ['class' => 'teachers']);
            foreach ($course->get_course_contacts() as $userid => $coursecontact) {
                $name = $coursecontact['rolename'].': '.
                        \html_writer::link(new \moodle_url('/user/view.php',
                                ['id' => $userid, 'course' => SITEID]),
                            $coursecontact['username']);
                $content .= \html_writer::tag('li', $name);
            }
            $content .= \html_writer::end_tag('ul'); // .teachers
        }

        // Display course category if necessary (for example in search results).
        if ($chelper->get_show_courses() == static::COURSECAT_SHOW_COURSES_EXPANDED_WITH_CAT) {
            if ($cat = \core_course_category::get($course->category, IGNORE_MISSING)) {
                $link = \html_writer::link(new \moodle_url('/course/index.php',
                    ['categoryid' => $cat->id]),
                    $cat->get_formatted_name(),
                    ['class' => $cat->visible ? '' : 'dimmed']
                );
                $content .= \html_writer::div(get_string('category').': '.$link, 'coursecat');
            }
        }

        return $content;
    }
}