<?php
/**
 * Form class base_form
 *
 * @package    local_ned_controller
 * @subpackage NED
 * @copyright  2021 NED {@link http://ned.ca}
 * @author     NED {@link http://ned.ca}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace local_ned_controller\form;

defined('MOODLE_INTERNAL') || die();

use local_ned_controller\shared_lib as NED;

/** @var \stdClass $CFG */
require_once($CFG->libdir.'/formslib.php');

/**
 * Form base_form
 *
 * Uses form_element, so it can be generic from by customdata
 *
 * @see form_element
 * @see base_form::_create_custom_form()
 */
class base_form extends \moodleform {
    // data keys for the custom form
    const ELEMENTS = 'elements';
    const HIDDENS = 'hiddens';
    const ERRORS = 'errors';
    const DESCRIPTION = 'description';
    const RULES = 'rules';
    const GROUP_RULES = 'group_rules';
    const DISABLEDIFS = 'disabledifs';
    const HIDEIFS = 'hideifs';
    const ADD_ACTION_BUTTONS = 'add_action_buttons';
    const SUBMIT = 'submit';
    const CANCEL = 'cancel';
    const DISABLE_FORM_CHANGE_CHECKER = 'disable_form_change_checker';
    const ALL_REQUIRED = 'all_required';

    /** @var \local_ned_controller\form\form_element $FE */
    static $FE = '\local_ned_controller\form\form_element';

    protected $_data = null;

    public $class = [];

    /**
     * Form definition. For overriding, please, use _main_definition method
     * @see _main_definition
     */
    protected function definition() {
        $this->_before_definition();
        $this->_main_definition();
        $this->_after_definition();
    }

    /**
     * Return submitted data if properly submitted or returns FALSE if validation fails or
     * if there is no submitted data.
     *
     * @return object|false - submitted data; FALSE if not valid or not submitted or cancelled
     */
    public function get_uncached_data(){
        return parent::get_data() ?? false;
    }

    /**
     * Get and save result submitted data.
     * If you wish to rewrite getting data method - rewrite get_uncached_data() instead
     * @see get_uncached_data()
     *
     * @return object|false - submitted data; false if not valid or not submitted or cancelled
     */
    public function get_data(){
        if (is_null($this->_data)){
            $this->_data = $this->get_uncached_data();
        }
        return $this->_data;
    }

    /**
     * Before definition hook.
     */
    protected function _before_definition(){
        $mform = $this->_form;
        if (!empty($mform->_attributes['class'])){
            $this->class = NED::str2arr($mform->_attributes['class'], ['ned-form']);
        }

        $cl_name_list = explode('\\', get_called_class());
        $this->class[] = str_replace('_', '-', end($cl_name_list));
    }

    /**
     * Main form definition
     * Override it in the child class instead of public definition method
     */
    protected function _main_definition(){
        $this->_create_custom_form();
    }

    /**
     * After definition hook.
     * You can also use parent method, after_definition
     *
     * @see after_definition
     */
    protected function _after_definition(){
        $mform = $this->_form;
        $mform->_attributes['class'] = NED::arr2str($this->class);
    }

    /**
     * Create form by the custom data
     */
    protected function _create_custom_form(){
        $data  = $this->_customdata;

        $this->addHiddens($data[static::HIDDENS] ?? []);
        $this->addDescription($data[static::DESCRIPTION] ?? '');
        $this->addErrors($data[static::ERRORS] ?? []);
        $this->loadElements($data[static::ELEMENTS] ?? []);
        $this->addRules($data[static::RULES] ?? []);
        $this->addGroupRules($data[static::GROUP_RULES] ?? []);
        $this->addDisabledIfs($data[static::DISABLEDIFS] ?? []);
        $this->addHideIfs($data[static::HIDEIFS] ?? []);

        if ($data[static::ADD_ACTION_BUTTONS] ?? false){
            $this->base_add_action_buttons($data[static::SUBMIT] ?? get_string('yes'), $data[static::CANCEL] ?? true);
        }

        if ($data[static::DISABLE_FORM_CHANGE_CHECKER] ?? false){
            $this->disable_form_change_checker();
        }

        if ($data[static::ALL_REQUIRED] ?? false){
            $this->all_required();
        }
    }

    /**
     * Load $elements to the form
     *
     * @see \local_ned_controller\form\form_element::addElements()
     *
     * @param array|array[]|\HTML_QuickForm_element[]|\local_ned_controller\form\form_element[] $elements
     */
    public function loadElements($elements=[]){
        if (empty($elements)){
            return;
        }
        static::$FE::addElements($this->_form, $elements);
    }

    /**
     * @see \MoodleQuickForm::addGroupRule
     *
     * @param array $rules - list of args for the addGroupRule() function
     */
    public function addGroupRules($rules=[]){
        $mform = $this->_form;
        foreach ($rules as $rule){
            $mform->addGroupRule(...$rule);
        }
    }

    /**
     * @see \MoodleQuickForm::addRule()
     *
     * @param array $rules - list of args for the addRule() function
     */
    public function addRules($rules=[]){
        $mform = $this->_form;
        foreach ($rules as $rule){
            $mform->addRule(...$rule);
        }
    }

    /**
     * @see \MoodleQuickForm::disabledIf()
     *
     * @param array $disabledIfs - list of args for the disabledIf() function
     */
    public function addDisabledIfs($disabledIfs=[]){
        $mform = $this->_form;
        foreach ($disabledIfs as $disabledIf){
            $mform->disabledIf(...$disabledIf);
        }
    }

    /**
     * @see \MoodleQuickForm::hideIf()
     *
     * @param array $hideIfs - list of args for the hideIf() function
     */
    public function addHideIfs($hideIfs=[]){
        $mform = $this->_form;
        foreach ($hideIfs as $hideIf){
            $mform->hideIf(...$hideIf);
        }
    }

    /**
     * @see \local_ned_controller\form\form_element::addElement()
     *
     * @param array|\HTML_QuickForm_element|\local_ned_controller\form\form_element $element
     * @param string                                                                $group_name - group name it elements for group
     *
     * @return \HTML_QuickForm_element|\HTML_QuickForm_select|object|null
     */
    public function &addNedElement($element, $group_name=''){
        return static::$FE::addElement($this->_form, $element, $group_name);
    }

    /**
     * alias for the addNedElement()
     *
     * @see addNedElement()
     * @see \local_ned_controller\form\form_element::addElement()
     *
     * @param array|\HTML_QuickForm_element|\local_ned_controller\form\form_element $element
     *
     * @return \HTML_QuickForm_element|\HTML_QuickForm_select|object|null
     */
    public function &add($element){
        return $this->addNedElement($element);
    }

    /**
     * @see \local_ned_controller\form\form_element::createElement()
     *
     * @param array|\HTML_QuickForm_element|\local_ned_controller\form\form_element $element
     * @param string                                                                $group_name - group name it elements for group
     *
     * @return \HTML_QuickForm_element|\HTML_QuickForm_select|object|null
     */
    public function &createNedElement($element, $group_name=''){
        return static::$FE::createElement($this->_form, $element, false, $group_name);
    }

    /**
     * Add hidden values from the list
     *
     * @param array $hiddens
     */
    public function addHiddens($hiddens=[]){
        if (empty($hiddens)){
            return;
        }

        $mform = $this->_form;
        foreach ($hiddens as $key => $item){
            if (is_array($item)){
                $name = $item['name'] ?? $key;
                $type = $item['type'] ?? PARAM_TEXT;
                $value = $item['value'] ?? null;
            } else {
                $name = $key;
                $type = PARAM_TEXT;
                $value = $item;
            }

            $mform->addElement('hidden', $name, $value);
            $mform->setType($name, $type);
        }
    }

    /**
     * Add errors from the list
     *
     * @param array $errors
     */
    public function addErrors($errors=[]){
        if (empty($errors)){
            return;
        }

        foreach ($errors as $error){
            $this->error($error);
        }
    }

    /**
     * Add description as html element to the form
     *
     * @param string $text
     */
    public function addDescription($text=''){
        if (empty($text)) return;

        $this->addNedElement(static::$FE::div($text, 'description'));
    }

    /**
     * Alternative add_action_buttons (add cancel button, and after that - submit button)
     * @see add_action_buttons()
     *
     * Use this method to a cancel and submit button to the end of your form. Pass a param of false
     * if you don't want a cancel button in your form. If you have a cancel button make sure you
     * check for it being pressed using is_cancelled() and redirecting if it is true before trying to
     * get data with get_data().
     *
     * @param bool $cancel whether to show cancel button, default true
     * @param string $submitlabel label for submit button, defaults to get_string('savechanges')
     */
    function add_action_buttons_alt($cancel=true, $submitlabel=null){
        $mform =& $this->_form;
        $submitlabel = $submitlabel ?? get_string('savechanges');
        if ($cancel){
            //when two elements we need a group
            $buttonarray = array();
            $buttonarray[] = &$mform->createElement('cancel');
            $buttonarray[] = &$mform->createElement('submit', 'submitbutton', $submitlabel);
            $mform->addGroup($buttonarray, 'buttonar', '', array(' '), false);
            $mform->closeHeaderBefore('buttonar');
        } else {
            //no group needed
            $mform->addElement('submit', 'submitbutton', $submitlabel);
            $mform->closeHeaderBefore('submitbutton');
        }
    }

    /**
     * Some default example for the add_action_buttons
     *
     * @param string $submit_text
     * @param bool   $cancel
     */
    public function base_add_action_buttons($submit_text='', $cancel=true){
        $this->add_action_buttons_alt($cancel, $submit_text ?: get_string('ok'));
    }

    /**
     * Call this method if you don't want the formchangechecker JavaScript to be
     * automatically initialised for this form.
     */
    public function disable_form_change_checker(){
        $this->_form->disable_form_change_checker();
    }

    /**
     * Add rule "required" to all elements, except buttons
     */
    public function all_required(){
        $mform = $this->_form;
        foreach ($mform->_elementIndex as $element_name => $element_value){
            if (empty($element_name)){
                continue;
            }

            $type = $mform->getElementType($element_name);
            switch ($type){
                case 'button':
                case 'submit':
                case 'cancel':
                case 'group':
                    continue 2;
            }
            $mform->addRule(...static::$FE::CreateRequired($element_name));
        }
    }

    /**
     * Renders the html form (same as display, but returns the result).
     *
     * Note that you can only output this rendered result once per page, as
     * it contains IDs which must be unique.
     *
     * Alias for the render method
     *
     * @see render
     *
     * @return string HTML code for the form
     */
    public function draw() {
        return $this->render();
    }

    /**
     * Returns a 'safe' element's value
     *
     * This method first tries to find a cleaned-up submitted value,
     * it will return a value set by setValue()/setDefaults()/setConstants()
     * if submitted value does not exist for the given element.
     *
     * @param  string $element - Name of an element
     *
     * @return mixed
     */
    public function exportValue($element){
        $mform = $this->_form;
        if (!isset($mform->_elementIndex[$element])) {
            return null;
        }

        return $this->_form->exportValue($element);
    }

    /**
     * export submitted values
     *
     * @param string $elementList list of elements in form
     *
     * @return array
     */
    public function exportValues($elementList=null){
        return $this->_form->exportValues($elementList);
    }

    /**
     * Return true if a cancel button has been pressed resulting in the form being submitted.
     *
     * @param string $button_name - (Optional) special button name, if not specified - check all of them
     *
     * @return bool true if a cancel button has been pressed
     */
    function is_cancelled($button_name=null){
        if (!$button_name){
            return parent::is_cancelled();
        }

        $mform =& $this->_form;
        if (!$mform->isSubmitted()){
            return false;
        }

        if (!in_array($button_name, $mform->_cancelButtons)){
            return false;
        }

        return $this->optional_param($button_name, 0, PARAM_RAW);
    }

    /**
     * Add error text to the form
     *
     * @param $text
     */
    public function error($text){
        $this->addNedElement(static::$FE::div(
            \html_writer::span(get_string('error').': ', 'error-label').
            \html_writer::span($text, 'error-text'),
            'error'));
    }

    /**
     * The constructor function calls the abstract function definition() and it will then
     * process and clean and attempt to validate incoming data.
     *
     * It will call your custom validate method to validate data and will also check any rules
     * you have specified in definition using addRule
     *
     * The name of the form (id attribute of the form) is automatically generated depending on
     * the name you gave the class extending moodleform. You should call your class something
     * like
     *
     * @param mixed $action the action attribute for the form. If empty defaults to auto-detect the
     *              current url. If a moodle_url object then outputs params as hidden variables.
     * @param mixed $customdata if your form defintion method needs access to data such as $course
     *              $cm, etc. to construct the form definition then pass it in this array. You can
     *              use globals for something.
     * @param string $method if you set this to anything other than 'post' then _GET and _POST will
     *               be merged and used as incoming data to the form.
     * @param string $target target frame for form submission. You will rarely use this. Don't use
     *               it if you don't need to as the target attribute is deprecated in xhtml strict.
     * @param mixed $attributes you can pass a string of html attributes here or an array.
     *               Special attribute 'data-random-ids' will randomise generated elements ids. This
     *               is necessary when there are several forms on the same page.
     *               Special attribute 'data-double-submit-protection' set to 'off' will turn off
     *               double-submit protection JavaScript - this may be necessary if your form sends
     *               downloadable files in response to a submit button, and can't call
     *               \core_form\util::form_download_complete();
     * @param bool $editable
     * @param array $ajaxformdata Forms submitted via ajax, must pass their data here, instead of relying on _GET and _POST.
     *
     * @return static
     */
    public static function create($action=null, $customdata=null, $method='post', $target='', $attributes=null, $editable=true,
        $ajaxformdata=null){
        if (!empty($action) && is_a($action, 'moodle_url') && !empty($customdata) && is_array($customdata)){
            /**
             * If some elements name in the action URL -> remove them from the URL,
             *  so that duplicate elements are not created
             */
            $remove_params = [];
            // $data_to_check is list or dictionary with elements here
            $data_to_check = array_merge($customdata[static::ELEMENTS] ?? [], $customdata[static::HIDDENS] ?? []);
            foreach ($data_to_check as $name => $item){
                if (!is_numeric($name)){
                    $remove_params[] = $name;
                } elseif (is_a($item, '\local_ned_controller\form\form_element')){
                    $remove_params[] = $item->name;
                } elseif (is_array($item)){
                    $elem_name = reset($item);
                    if (!empty($elem_name)){
                        $remove_params[] = $elem_name;
                    }
                }
            }

            $action->remove_params($remove_params);
        }
        return new static($action, $customdata, $method, $target, $attributes, $editable, $ajaxformdata);
    }

    /**
     * @return string|null
     */
    public function get_action(){
        $mform = $this->_form;
        return $mform->_attributes['action'] ?? null;
    }
}
