<?php
/**
 * Plugins overview
 *
 * @package     local_ned_controller
 * @license     http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace local_ned_controller\form;
use local_ned_controller\ned_notifications as NN;
use local_ned_controller\shared_lib as NED;

defined('MOODLE_INTERNAL') || die();
/** @var \stdClass $CFG */
require_once(__DIR__ . '/../../lib.php');
require_once($CFG->libdir . '/formslib.php');


/**
 * Class notifications_edit_form
 *
 * @package local_ned_controller
 */
class notifications_edit_form extends base_form
{
    const HTML_CONDITION_ELEMENT_LIST_NAME = 'ned_list_elements';
    const HTML_COURSE_LEVEL_NAME = NN::COURSE_LEVEL_NAME;

    protected function definition()
    {
        global $USER;
        $mform = $this->_form;
        $big_text_size = 255;
        $normal_text_size = 60;

        $id = $this->_customdata['id'];
        //Id
        $mform->addElement('hidden', 'id', $id);
        $mform->setType('id', PARAM_INT);

        //Name
        $mform->addElement('text', 'name', get_string('name'));
        $mform->addRule('name', get_string('required'), 'required', null, 'client');
        $mform->addRule('name', get_string('maximumchars', '', $big_text_size), 'maxlength', $big_text_size, 'client');
        $mform->setType('name', PARAM_TEXT);

        //Weight
        $weightopc = array_combine($r = range(NN::MIN_WEIGHT, NN::MAX_WEIGHT), $r);
        $mform->addElement('select', 'weight', NED::str('notifications:weight'), $weightopc);
        $mform->addHelpButton('weight', 'notifications:weight', NED::$PLUGIN_NAME);
        $mform->setDefault('weight', 0);

        //Disabled
        $mform->addElement('advcheckbox', 'disabled', NED::str('notifications:disabled'));
        $mform->addHelpButton('disabled', 'notifications:disabled', NED::$PLUGIN_NAME);
        $mform->setDefault('disabled', 0);

        // Notification type
        $mform->addElement('select', 'notification_type', NED::str('notifications:type'), [
            NN::TYPE_MESSAGE => NED::str('notifications:type:message'),
            NN::TYPE_INFO => NED::str('notifications:type:info'),
            NN::TYPE_REMINDER => NED::str('notifications:type:reminder'),
            NN::TYPE_NOTICE => NED::str('notifications:type:notice')
        ]);

        // Notification rule
        $mform->addElement('select', 'notification_rule', NED::str('notifications:rule'), [
            NN::RULE_OR => NED::str('notifications:rule:or'),
            NN::RULE_AND => NED::str('notifications:rule:and')
        ]);
        $mform->addHelpButton('notification_rule', 'notifications:rule', NED::$PLUGIN_NAME);

        // Description
        $mform->addElement('text', 'description', NED::str('notifications:description'), ['size' => $big_text_size]);
        $mform->setType('description', PARAM_TEXT);
        $mform->addRule('description', get_string('maximumchars', '', $big_text_size), 'maxlength', $big_text_size);

        // Course level
        $mform->addElement('advcheckbox', static::HTML_COURSE_LEVEL_NAME, NED::str('notifications:courselevel'));
        $mform->addHelpButton(static::HTML_COURSE_LEVEL_NAME, 'notifications:courselevel', NED::$PLUGIN_NAME);
        $mform->setDefault(static::HTML_COURSE_LEVEL_NAME, 0);

        // Conditions
        $this->add_condition_elements();

        // Message
        $editoroptions = [];
        $mform->addElement('editor', 'message', NED::str('notifications:modaltext'), null, $editoroptions);
        $mform->setType('message', PARAM_RAW);

        // Redirect options
        // - redirect: general
        $mform->addElement('select', 'redirect_option_general', NED::str('notifications:redirect:title:selector'),
            NED::strings2menu(NN::REDIRECTS_GENERAL));

        $mform->addElement('url', 'redirect_url_general', NED::str('notifications:redirect:title:page'),
            ['size' => $normal_text_size], ['usefilepicker' => false]);
        $mform->setType('redirect_url_general', PARAM_URL);
        $mform->hideIf('redirect_url_general', 'redirect_option_general', 'neq', NN::REDIRECT_PAGE);

        foreach (['redirect_option_general', 'redirect_url_general'] as $elem){
            $mform->hideIf($elem, static::HTML_COURSE_LEVEL_NAME, 'eq', 1);
        }

        // - redirect: course
        $mform->addElement('select', 'redirect_option_course', NED::str('notifications:redirect:title:selector'),
            NED::strings2menu(NN::REDIRECTS_COURSE));

        $mform->addElement('text', 'redirect_name', NED::str('notifications:redirect:title:activity'), ['size' => $normal_text_size]);
        $mform->setType('redirect_name', PARAM_TEXT);
        $mform->addRule('redirect_name', get_string('maximumchars', '', $normal_text_size), 'maxlength', $normal_text_size);
        $mform->hideIf('redirect_name', 'redirect_option_course', 'neq', NN::REDIRECT_ACTIVITY_OTHER);

        $mform->addElement('url', 'redirect_url_course', NED::str('notifications:redirect:title:page'),
            ['size' => $normal_text_size], ['usefilepicker' => false]);
        $mform->setType('redirect_url_course', PARAM_URL);
        $mform->hideIf('redirect_url_course', 'redirect_option_course', 'neq', NN::REDIRECT_PAGE);

        foreach (['redirect_option_course', 'redirect_name', 'redirect_url_course'] as $elem){
            $mform->hideIf($elem, static::HTML_COURSE_LEVEL_NAME, 'eq', 0);
        }

        // For each course
        $mform->addElement('advcheckbox', 'foreachcourse', NED::str('notifications:foreachcourse'));
        $mform->addHelpButton('foreachcourse', 'notifications:foreachcourse', NED::$PLUGIN_NAME);
        $mform->setDefault('foreachcourse', 0);
        $mform->hideIf('foreachcourse', static::HTML_COURSE_LEVEL_NAME, 'eq', 0);

        // Repeat option
        $mform->addElement('select', 'iteration', NED::str('notifications:iteration'), [
            NN::REPEAT_ONCE => NED::str('notifications:iteration:once'),
            NN::REPEAT_THRICE => NED::str('notifications:iteration:thrice'),
            NN::REPEAT_UNLIMITED => NED::str('notifications:iteration:unlimited')
        ]);

        // Frequency
        $mform->addElement('select', 'timeout', NED::str('notifications:timeout'), [
            NN::TIMEOUT_NO => NED::str('notifications:timeout:no'),
            NN::TIMEOUT_15MIN => NED::str('notifications:timeout:15min'),
            NN::TIMEOUT_HOUR => NED::str('notifications:timeout:hour'),
            NN::TIMEOUT_BY_LOGIN => NED::str('notifications:timeout:bylogin'),
            NN::TIMEOUT_DAY => NED::str('notifications:timeout:day')
        ]);

        // Checkbox "Don't show again"
        $mform->addElement('advcheckbox', 'dontshow', NED::str('notifications:dontshow'));
        $mform->addHelpButton('dontshow', 'notifications:dontshow', NED::$PLUGIN_NAME);
        $mform->setDefault('dontshow', 1);
        $mform->hideIf('dontshow', 'iteration', 'eq', NN::REPEAT_ONCE);

        // Manual add & remove users
        $userids_options = [
            'ajax' => 'tool_lp/form-user-selector',
            'multiple' => true,
            'valuehtmlcallback' => function($value) {
                global $OUTPUT, $PAGE;
                if (empty($value)){
                    return '';
                }
                $keys = ['email', 'phone1', 'phone2', 'department', 'institution'];
                $identity = [];
                $user = \core_user::get_user($value, '*');
                if (empty($user)){
                    return '';
                }
                foreach ($keys as $key){
                    if (isset($user->$key) && !empty($user->$key)){
                        $identity[] =  $user->$key;
                    }
                }
                $user->fullname = fullname($user);
                $userpicture = new \user_picture($user);
                $user->profileimageurlsmall = $userpicture->get_url($PAGE)->out(false);
                $user->hasidentity = !empty($identity);
                $user->identity = join(', ', $identity);
                return $OUTPUT->render_from_template('tool_lp/form-user-selector-suggestion', $user);
            }
        ];
        $mform->addElement('autocomplete', 'userids_remove', NED::str('notifications:userids_remove'), [], $userids_options);
        $mform->addElement('autocomplete', 'userids_add', NED::str('notifications:userids_add'), [], $userids_options);
        $mform->addHelpButton('userids_remove', 'notifications:userids_remove', NED::$PLUGIN_NAME);
        $mform->addHelpButton('userids_add', 'notifications:userids_add', NED::$PLUGIN_NAME);
        $mform->setDefault('userids_remove', [$USER->id]);

        // Buttons
        $buttonarray = [];
        $buttonarray[] = $mform->createElement('submit', 'submitbutton', get_string('savechanges'));
        if ($id > 0) {
            // Checkbox "Reset views"
            $mform->addElement('advcheckbox', 'resetviews', NED::str('notifications:resetviews'));
            $mform->addHelpButton('resetviews', 'notifications:resetviews', NED::$PLUGIN_NAME);
            $mform->setDefault('resetviews', 0);
            // Buttons
            $buttonarray[] = $mform->createElement('submit', 'saveasbutton', NED::str('saveasnew'));
            $buttonarray[] = $mform->createElement('cancel');
            $msg = NED::str('notifications:confirm_removed');
            $buttonarray[] = $mform->createElement('submit', 'deletebutton',
                get_string('delete'), ['onClick' => "javascript:return confirm('$msg');"]);
        } else {
            $buttonarray[] = $mform->createElement('cancel', null, NED::str('close'));
        }
        $mform->addGroup($buttonarray, 'buttonar', '', ' ', false);
    }

    /**
     * Render & return form as html
     * @return string
     */
    public function draw() {
        //finalize the form definition if not yet done
        if (!$this->_definition_finalized) {
            $this->_definition_finalized = true;
            $this->definition_after_data();
        }

        return $this->_form->toHtml();
    }

    /**
     * Add to form data additional data from elements created by js
     * @return object
     */
    public function get_data(){
        global $_POST;
        $data = parent::get_data();
        if (!$data) return $data;

        $i=0;
        if ($this->is_submitted() && isset($_POST[NN::CONDITIONS_NAME])){
            while(isset($_POST[NN::CONDITIONS_NAME][$i])){
                $data->conditions[$i] = $_POST[NN::CONDITIONS_NAME][$i];

                foreach ($_POST[NN::CONDITIONS_NAME][$i] as $elem_name => $value){
                    if (is_array($value)){
                        if (isset($value['duration-input']) && isset($value['duration-select'])){
                            $data->conditions[$i][$elem_name] = $value['duration-input'] * $value['duration-select'];
                        } else {
                            $data->conditions[$i][$elem_name] = '';
                        }
                    }
                }
                $i++;
            }
        }
        $data->conditions['count'] = $i;

        // update redirection data
        $redirect_activity = 'redirect_name';
        [$redirect_option, $redirect_url] = static::get_redirect_field_names($data->{static::HTML_COURSE_LEVEL_NAME} ?? false);
        if (isset($data->$redirect_option)){
            $data->redirect_option = $data->$redirect_option;
            switch ($data->redirect_option){
                default:
                    $data->redirect_data = null;
                    break;
                case NN::REDIRECT_ACTIVITY_OTHER:
                    $data->redirect_data = $data->$redirect_activity ?? null;
                    break;
                case NN::REDIRECT_PAGE:
                    $data->redirect_data = $data->$redirect_url ?? null;
                    break;
            }
        }

        return $data;
    }

    // New functions
    /**
     * Create and add element, which will be filled by other elements by js
     * Call in form definition
     */
    public function add_condition_elements(){
        $FE = static::$FE;

        $conditions = [];
        $before_conditions = [];
        $condition_options_global = $condition_options_course = [NED::str('notifications:choose_condition')];
        $condition_types = NN::get_condition_types();
        foreach ($condition_types as $type){
            $name = $type . '_base';
            $cond_group = $this->get_condition_element($name, ['type' => $type], true);

            $condition_options_course[$name] = $cond_group->getLabel();
            if (!NN::is_course_level_type($type)){
                $condition_options_global[$name] = $condition_options_course[$name];
            }
            $conditions[] = $cond_group;
        }

        $group_elems = [];
        $fe_select_global = $FE::select('add_condition_global', '', $condition_options_global, null);
        $fe_select_global->setHtmlClass('add_condition');
        $fe_select_global->addHideIf(static::HTML_COURSE_LEVEL_NAME, 'eq', 1);
        $group_elems[] = $fe_select_global;

        $fe_select_course = $FE::select('add_condition_course', '', $condition_options_course, null);
        $fe_select_course->setHtmlClass('add_condition');
        $fe_select_course->addHideIf(static::HTML_COURSE_LEVEL_NAME, 'eq', 0);
        $group_elems[] = $fe_select_course;

        $group_selector = $FE::group('add_condition_group', NED::str('notifications:add_condition'), $group_elems, '', true);
        $group_selector->setHtmlClass('ned-selector-elements');

        $before_conditions[] = $group_selector;
        $ol = $FE::html(static::get_html_condition_element_list());
        $ol->setName(static::HTML_CONDITION_ELEMENT_LIST_NAME);
        $before_conditions[] = $ol;

        $conditions = array_merge($before_conditions, $conditions);
        $cond_group = $FE::group(NN::CONDITIONS_NAME, NED::str('notifications:conditions'), $conditions, ' ', true);
        $cond_group->setHtmlClass('ned-select-elements');
        $this->addNedElement($cond_group);

        NED::js_call_amd('/ned_select_elements', 'init');
    }

    /**
     * Update form elements by stored data
     * Call after definition and before show
     *
     * @param $data
     */
    public function update($data){
        if (empty($data)) return;

        $data = (array)$data;
        $data2upd = [];

        $mform = $this->_form;
        $c_name = NN::CONDITIONS_NAME;

        // update redirection data
        $redirect_activity = 'redirect_name';
        [$redirect_option, $redirect_url] = static::get_redirect_field_names($data[static::HTML_COURSE_LEVEL_NAME] ?? false);
        if (isset($data['redirect_option'])){
            $data[$redirect_option] = $data['redirect_option'];
            $data[$redirect_url] = $data[$redirect_activity] = null;
            switch ($data['redirect_option']){
                default:
                    break;
                case NN::REDIRECT_ACTIVITY_OTHER:
                    $data[$redirect_activity] = $data['redirect_data'] ?? null;
                    break;
                case NN::REDIRECT_PAGE:
                    $data[$redirect_url] = $data['redirect_data'] ?? null;
                    break;
            }
        }

        // Set up data for the base elements
        /** @var \HTML_QuickForm_group | \HTML_QuickForm_select | \HTML_QuickForm_element  $elem */
        foreach ($mform->_elements as $elem){
            $name = $elem->getName();
            if (isset($elem->_elements) || !isset($data[$name])) continue;

            if ($elem->getType() == 'editor'){
                $data2upd[$name] = (['text' => $data[$name], 'format' => 1]);
            } else {
                $data2upd[$name] = $data[$name];
            }
        }
        // update base elements through Moodle form method
        parent::set_data($data2upd);

        if (isset($data[$c_name]) && isset($data[$c_name]['count']) && $data[$c_name]['count'] > 0){
            $content = '';
            for ($i=0; $i < $data[$c_name]['count']; $i++){
                $name = $c_name."[$i]";
                $group = $this->get_condition_element($name, $data[$c_name][$i]);
                $content .= \html_writer::tag('li', NED::O()->mform_element($group, false, false, false, true));
            }

            $group = $mform->getElement($c_name);
            /** @var \HTML_QuickForm_html | \HTML_QuickForm_element $elem */
            foreach ($group->_elements as $elem){
                $name = $elem->getName();
                if ($name != static::HTML_CONDITION_ELEMENT_LIST_NAME){
                    continue;
                }
                $elem->setText(static::get_html_condition_element_list($content));
                break;
            }

        }

    }

    /**
     * Create list for new elements which will be added by js
     * @param string $content
     *
     * @return string
     */
    static public function get_html_condition_element_list($content=''){
        return \html_writer::tag('ol', $content, ['class' => 'ned-list-elements']);
    }

    /**
     * Create one condition_element, which get from stored data, or base element for js
     *
     * @param       $name
     * @param array $data
     * @param bool  $base_element
     *
     * @return \HTML_QuickForm_group
     */
    public function get_condition_element($name, $data=[], $base_element=false){
        $FE = static::$FE;
        $mform_error = function($text) {
            return $this->createNedElement(static::$FE::html(NED::div($text, 'error px-1')));
        };

        $type = $data['type'] ?? NN::COND_TYPE_PROFILE;
        $elements = [];
        $group_classes = ['ned-add-element'];
        $hide_label = false;
        $complex_elems = [];

        switch ($type){
            /** @noinspection PhpMissingBreakStatementInspection */
            default:
                trigger_error ("Unknown condition type \"$type\".", \E_USER_NOTICE);
                $type = NN::COND_TYPE_PROFILE;
            case NN::COND_TYPE_PROFILE:
                $profile_fields = NN::get_profile_fields();
                $elements[] = $FE::select('select_value', '', $profile_fields);
                $elements[] = $FE::select('equal', '', []);
                $elements[] = $FE::text('text_value', '', null, null, 25);
                break;
            case NN::COND_TYPE_COHORT:
                $cohort_list = NN::get_cohort_list();
                $elements[] = $FE::select('equal', '', []);
                $elements[] = $FE::select('select_value', '', $cohort_list);
                break;
            case NN::COND_TYPE_BADGE:
                $badge_list = NN::get_badge_list();
                $elements[] = $FE::select('equal', '', []);
                $elements[] = $FE::select('select_value', '', $badge_list);
                break;
            case NN::COND_TYPE_DM:
            case NN::COND_TYPE_KICA_GRADEBOOK:
                $elements[] = $FE::select('equal', '', []);
                break;
            case NN::COND_TYPE_ACTIVITY:
                $hide_label = true;
                $complex_elems['name_value'] = $FE::text('name_value', '', null, null, 25);
                $complex_elems['equal'] = $FE::select('equal');
                break;
            case NN::COND_TYPE_CLASS_TIME:
                $hide_label = true;
                $class_time_types = NN::get_class_time_types();
                $more_less = NN::get_more_less_list();
                $before_after = NN::get_before_after_list();

                $complex_elems['select_type'] = $FE::select('select_type', '', $class_time_types);
                $complex_elems['more_less'] = $FE::select('more_less', '', $more_less);
                $complex_elems['before_after'] = $FE::select('before_after', '', $before_after);
                $complex_elems['time_value'] = $FE::duration('time_value', '', null, null, null, false, DAYSECS);

                break;
        }

        if (!empty($complex_elems)){
            $complex_text = NED::str_check("notifications:$type:complex_text", null, false);
            if (!empty($complex_text)){
                $group_classes[] = 'ned-notification-group-complex-element';
                $complex_tokens = NED::str_split_tokens($complex_text);
                foreach ($complex_tokens as $i => $item){
                    if (empty($item)) continue;

                    if ($i % 2){
                        // $item is token
                        $elem = $complex_elems[$item] ?? null;
                        if (!empty($elem)){
                            if (is_string($elem)){
                                // add string
                                $elements[] = $FE::html($elem);
                            } else {
                                // add real element
                                $elements[] = $elem;
                            }
                        } else {
                            $elements[] = $mform_error(NED::str("notifications:error:element",
                                ['element' => $item, 'condition' => $type]));
                        }
                    } else {
                        // $item is simple text
                        $elements[] = $FE::html($item);
                    }
                }
            } else {
                $elements[] = $mform_error(NED::str("notifications:error:condition", ['condition' => $type]));
            }
        }

        if ($base_element){
            $group_classes[] = 'ned-add-base-element';
        }
        if ($hide_label){
            $group_classes[] = 'ned-hidden-label';
        }

        if (NN::is_course_level_type($type)){
            $elements[] = $FE::html(NED::fa('fa-copyright course-level-icon', '', NED::str('notifications:courselevel:icon_title')));
            $group_classes[] = 'ned-element-course-level';
        }
        $elements[] = $FE::hidden('type', $type, \PARAM_RAW);

        /** @var \HTML_QuickForm_group $group */
        $fe_group = $FE::group($name, NED::str("notifications:$type"), $elements, '', true);
        $fe_group->setHtmlClass($group_classes);
        $group = $this->createNedElement($fe_group);

        /** @var \HTML_QuickForm_select | \HTML_QuickForm_element | \MoodleQuickForm_duration $elem */
        foreach ($group->_elements as $elem){
            $elem_name = $elem->getName();
            if ($elem_name == 'equal'){
                $elem->addOption(NED::str("notifications:$type:match"), NN::MATCH);
                $elem->addOption(NED::str("notifications:$type:not_match"), NN::NOT_MATCH);
            }
            $elem->_attributes['data-nedtype'] = $elem_name;
            $value = $data[$elem_name] ?? null;

            if ($elem->getType() == 'duration'){
                $elem->_attributes['data-duration'] = 1;
                if (isset($value)){
                    [$number, $unit] = $elem->seconds_to_unit($value);
                    $value = ['number' => $number, 'timeunit' => $unit];
                }
            }

            if (isset($value)){
                $elem->setValue($value);
            }
        }

        return $group;
    }

    /**
     * Return field names for redirect options based on courselevel
     * @param bool $courselevel
     *
     * @return string[] - [$redirect_option, $redirect_url]
     */
    static public function get_redirect_field_names($courselevel=false){
        if ($courselevel){
            $redirect_option = 'redirect_option_course';
            $redirect_url = 'redirect_url_course';
        } else {
            $redirect_option = 'redirect_option_general';
            $redirect_url = 'redirect_url_general';
        }

        return [$redirect_option, $redirect_url];
    }
}
