<?php
/**
 * Class for storing all necessary grade info (from grade_grades, grade_items and scales)
 *
 * @package    local_ned_controller
 * @subpackage NED
 * @copyright  2021 NED {@link http://ned.ca}
 * @author     NED {@link http://ned.ca}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace local_ned_controller;
use local_ned_controller\shared_lib as NED;
use local_ned_controller\support\grade_grade;
use local_ned_controller\support\grade_item;
use local_ned_controller\support\scale;

defined('MOODLE_INTERNAL') || die();

/**
 * Class grade_info
 *
 * @package local_ned_controller
 *
 * @method static void prepare_s($courseid, $userid=null)
 * @method static bool is_prepare_s($courseid, $userid=null)
 * @method static grade_item get_grade_item_s($itemmodule, $iteminstance, $courseid)
 * @method static grade_grade get_grade_grade_s($itemid, $userid)
 * @method static scale get_scale_s($scaleid)
 * @method static grade_grade get_grade_grade_by_grade_item_s(grade_item $grade_item, $userid)
 * @method static scale get_scale_by_grade_item_s(grade_item $grade_item)
 * @method static grade_item get_grade_item_by_mod_s(\cm_info $mod)
 * @method static grade_grade get_grade_grade_by_mod_s(\cm_info $mod, $userid)
 * @method static scale get_scale_by_mod_s(\cm_info $mod)
 * @method static array get_all_s($itemmodule, $iteminstance, $courseid, $userid) - Return [grade_items, grade_grades, scale]
 * @method static array get_all_by_mod_s(\cm_info $mod, $userid) - Return [grade_items, grade_grades, scale]
 */
class grade_info
{
    protected static $_prepared = [];
    protected static $_grade_items = [];
    protected static $_grade_grades = [];
    protected static $_scale = [];

    public $courseid;
    public $userid=null;


    /**
     * grade_info constructor.
     *
     * @param      $courseid
     * @param null $userid
     * @param bool $prepare
     */
    public function __construct($courseid, $userid=null, $prepare=true){
        $this->courseid = $courseid;
        $this->userid = $userid;
        if ($prepare){
            $this->prepare();
        }
    }

    /**
     * @param $name
     * @param $arguments
     *
     * @return null
     */
    public static function __callStatic($name, $arguments){
        $name = substr($name, 0, -2);
        $gi = new self(null, null, false);
        if (method_exists($gi, $name)){
            return $gi->$name(...$arguments);
        }
        return null;
    }

    /**
     * Return default $courseid & userid
     * @param null $courseid
     * @param null $userid
     *
     * @return array($courseid, $userid)
     */
    protected function def($courseid=null, $userid=null){
        return [$courseid ?? $this->courseid, $userid ?? $this->userid];
    }

    /**
     * @param null $courseid
     * @param null $userid
     */
    public function prepare($courseid=null, $userid=null){
        list($courseid, $userid) = $this->def($courseid, $userid);
        if ($this->is_prepare($courseid, $userid)) return;

        list($gg_l, $gi_l, $sc_l) = array([], [], []);
        $t = ['grade_grades' => 'g_g', 'grade_items' => 'g_i', 'scale' => 'sc'];
        $dl = '___';
        $fields = [];
        foreach ($t as $t_name => $alias){
            $t_cols = static::get_db()->get_columns($t_name);
            foreach ($t_cols as $col => $col_info){
                $fields[] = "$alias.$col AS {$alias}{$dl}{$col}";
            }
            $t[$t_name] = $alias.$dl;
        }

        $sql = 'SELECT ' . join(',', $fields) . '
                FROM 
                  {grade_items} g_i
                LEFT JOIN {grade_grades} g_g
                  ON g_g.itemid = g_i.id
                LEFT JOIN {scale} sc 
                  ON sc.id = g_i.scaleid
                WHERE g_i.courseid = :courseid AND g_i.itemtype = "mod"
        ';
        $params = ['courseid' => $courseid];
        if (!is_null($userid)){
            $sql .= ' AND g_g.userid = :userid';
            $params['userid'] = $userid;
        }
        $grade_info_records = static::get_db()->get_records_sql($sql, $params);
        if (empty($grade_info_records)){
            return;
        }
        foreach ($grade_info_records as $r){
            $pr = $t['grade_items'];
            if (!isset($gi_l[$courseid][$r->{$pr.'itemmodule'}][$r->{$pr.'iteminstance'}])){
                $gi = new grade_item($r, $pr);
                $gi_l[$courseid][$gi->itemmodule][$gi->iteminstance] = $gi;
            }

            $pr = $t['grade_grades'];
            if (!isset($gg_l[$r->{$pr.'userid'}][$r->{$pr.'itemid'}])){
                $gg = new grade_grade($r, $pr);
                $gg_l[$gg->userid][$gg->itemid] = $gg;
            }

            $pr = $t['scale'];
            if (!isset($sc_l[$r->{$pr.'id'}])){
                $sc = new scale($r, $pr);
                $sc_l[$sc->id] = $sc;
            }
        }

        list(static::$_grade_items, static::$_grade_grades, static::$_scale) = [$gi_l, $gg_l, $sc_l];

        if (!is_null($userid)){
            static::$_prepared[$courseid][$userid] = true;
        } else {
            static::$_prepared[$courseid] = true;
        }
    }

    /**
     * @param null $courseid
     * @param null $userid
     *
     * @return bool
     */
    public function is_prepare($courseid=null, $userid=null){
        list($courseid, $userid) = $this->def($courseid, $userid);
        return (isset(static::$_prepared[$courseid]) && static::$_prepared[$courseid] === true) ||
            (isset(static::$_prepared[$courseid][$userid]) && static::$_prepared[$courseid][$userid]);
    }

    /**
     * @param $courseid
     * @param $itemmodule
     * @param $iteminstance
     *
     * @return grade_item|false
     */
    public function get_grade_item($itemmodule, $iteminstance, $courseid=null){
        list($courseid, $userid) = $this->def($courseid);
        if (!isset(static::$_grade_items[$courseid][$itemmodule][$iteminstance])){
            $cm = NED::get_cm_by_params($courseid, $itemmodule, $iteminstance);
            if ($cm){
                $gi_record = static::get_db()->get_record('grade_items',
                    ['itemtype' => 'mod', 'courseid' => $courseid,
                     'itemmodule' => $itemmodule, 'iteminstance' => $iteminstance, 'itemnumber' => NED::grade_get_gi_itemnumber($cm)]);
                static::$_grade_items[$courseid][$itemmodule][$iteminstance] = new grade_item($gi_record);
            } else {
                static::$_grade_items[$courseid][$itemmodule][$iteminstance] = false;
            }
        }
        return static::$_grade_items[$courseid][$itemmodule][$iteminstance];
    }

    /**
     * @param $userid
     * @param $itemid
     *
     * @return grade_grade|false
     */
    public function get_grade_grade($itemid, $userid=null){
        list($courseid, $userid) = $this->def(null, $userid);
        if (!isset(static::$_grade_grades[$userid][$itemid])){
            $gg_record = static::get_db()->get_record('grade_grades', ['userid' => $userid, 'itemid' => $itemid]);
            static::$_grade_grades[$userid][$itemid] = new grade_grade($gg_record);

        }
        return static::$_grade_grades[$userid][$itemid];
    }

    /**
     * @param $scaleid
     *
     * @return scale|false
     */
    public function get_scale($scaleid){
        if (!isset(static::$_scale[$scaleid])){
            $sc_record = static::get_db()->get_record('scale', ['id' => $scaleid]);
            static::$_scale[$scaleid] = new scale($sc_record);

        }
        return static::$_scale[$scaleid];
    }

    /**
     * @param grade_item $grade_item
     * @param $userid
     *
     * @return grade_grade|null
     */
    public function get_grade_grade_by_grade_item(grade_item $grade_item, $userid=null){
        if (!$grade_item) return null;
        return $this->get_grade_grade($grade_item->id, $userid);
    }

    /**
     * @param grade_item $grade_item
     *
     * @return scale|null
     */
    public function get_scale_by_grade_item(grade_item $grade_item){
        if (!$grade_item) return null;
        return $this->get_scale($grade_item->scaleid);
    }

    /**
     * @param \cm_info|\stdClass $mod
     *
     * @return grade_item|null
     */
    public function get_grade_item_by_mod($mod){
        if (!$mod) return null;
        return $this->get_grade_item($mod->modname, $mod->instance, $mod->course);
    }

    /**
     * @param \cm_info|\stdClass $mod
     * @param null     $userid
     *
     * @return grade_grade|null
     */
    public function get_grade_grade_by_mod($mod, $userid=null){
        if (!$mod) return null;
        $gi = $this->get_grade_item($mod->modname, $mod->instance, $mod->course);
        return $this->get_grade_grade_by_grade_item($gi, $userid);
    }

    /**
     * @param \cm_info|\stdClass $mod
     *
     * @return scale|null
     */
    public function get_scale_by_mod($mod){
        if (!$mod) return null;
        $gi = $this->get_grade_item($mod->modname, $mod->instance, $mod->course);
        return $this->get_scale_by_grade_item($gi);
    }

    /**
     * @param      $itemmodule
     * @param      $iteminstance
     * @param null $courseid
     * @param null $userid
     *
     * @return array [grade_items, grade_grades, scale]
     */
    public function get_all($itemmodule, $iteminstance, $courseid=null, $userid=null){
        $gi = $this->get_grade_item($itemmodule, $iteminstance, $courseid);
        $gg = $this->get_grade_grade_by_grade_item($gi, $userid);
        $sc = $this->get_scale_by_grade_item($gi);
        return [$gi, $gg, $sc];
    }

    /**
     * @param \cm_info|\stdClass $mod
     * @param null $userid
     *
     * @return array of grade_items|grade_grades|scale
     */
    public function get_all_by_mod($mod, $userid=null){
        return $this->get_all($mod->modname, $mod->instance, $mod->course, $userid);
    }


    /**
     * You can see, how many times this class use $DB, if set breakpoint here
     * @return \moodle_database
     */
    public static function get_db(){
        global $DB;
        return $DB;
    }

}
