<?php
/**
 * @package    local_ned_controller
 * @subpackage marking_manager
 * @category   NED
 * @copyright  2021 NED {@link http://ned.ca}
 * @author     NED {@link http://ned.ca}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 * @noinspection PhpUnused
 */

namespace local_ned_controller\marking_manager;
use block_ned_teacher_tools\output\menu_bar as MB;
use local_ned_controller\shared_lib as NED;

defined('MOODLE_INTERNAL') || die();

/**
 *
 * @property-read \stdClass $course;
 * @property-read int $courseid;
 * @property-read \context_course|\context $context;
 * @property-read \stdClass $group;
 * @property-read array $allgroups;
 * @property-read int $groupid;
 * @property-read \stdClass $user;
 * @property-read int $userid;
 * @property-read bool $show_inactive;
 * @property-read bool $cmstudents;
 * @property-read array $students;
 * @property-read array $studentids;
 * @property-read array $allstudents;
 * @property-read \stdClass $studuser = null;
 * @property-read int $studuserid = 0;
 * @property-read marking_manager_mod[] $mod_types = [];
 * @property-read bool $nousers = [];
 * @property-read bool has_kica = false;
 * @property-read array $base_filter = [];
 * @property-read array|\stdClass $status_data = null;
 * @property-read \stdClass|\cm_info $cm = null;
 * @property-read \stdClass|\cm_info $cm_data = null;
 * @property-read \stdClass $kica = null;
 * @property-read array $activity_filter = [];
 * @property-read array|\stdClass $activity_data = null;
 * @property-read \moodle_url $base_url = null;
 * @property-read string $show_status = null;
 */
class marking_manager extends MM_base{

    const P_SHOW_INACTIVE = 'show_inactive';
    const P_KICA_ACTIVITIES = 'kica_activities';
    const P_STATUS = 'show';
    const P_VIEW = 'view';
    const P_ACTIVITY_TYPE = 'activity_type';
    const P_PARTICIPANTS = 'participants';
    const P_COURSEID = 'courseid';
    const P_MID = 'mid';
    const P_GROUPID = 'group';
    const P_DEADLINE = 'deadline';
    const P_GETSHOW = 'getshow';
    const P_ASSIGN_GRADER = 'assign_grader';

    const OPTIONS = ['course', 'context', 'type', 'show_inactive', 'cmstudents', 'groupid', 'studuserid', 'load_users', 'set_user', 'set_students'];

    /** @var \stdClass $_course */
    protected $_course;
    protected $_courseid = 0;
    /** @var \context_course|\context $_context */
    protected $_context = null;
    /** @var \stdClass $_group */
    protected $_group = null;
    protected $_allgroups = [];
    protected $_groupid = 0;
    /** @var \stdClass $_user */
    protected $_user = null;
    protected $_userid = 0;
    protected $_show_inactive = false;
    protected $_cmstudents = false;
    protected $_allstudents = [];
    protected $_students = [];
    protected $_studentids = [];
    protected $_nousers = false;
    /** @var \stdClass $_studuser */
    protected $_studuser = null;
    protected $_studuserid = 0;
    /** @var marking_manager_mod[] $_mod_types */
    protected $_mod_types = [];
    protected $_has_kica = false;

    protected $_base_filter = null;
    protected $_status_data = null;
    /** @var \stdClass|\cm_info $_cm */
    protected $_cm = null;
    /** @var \stdClass|\cm_info $_cm */
    protected $_cm_data = null;
    /** @var \stdClass */
    protected $_kica = null;
    protected $_activity_filter = null;
    protected $_activity_data = null;
    /** @var \moodle_url $_base_url */
    protected $_base_url = null;
    protected $_show_status = null;


    /**
     * marking_manager constructor.
     *
     * If there is no $course, MM object will not be ready for use
     *
     * @param              $course
     * @param null         $show_inactive
     * @param null         $groupid
     * @param null         $studuserid
     * @param string|array $type
     * @param null         $context
     * @param bool         $load_users
     * @param null         $set_students
     * @param null         $set_user
     * @param bool         $cmstudents
     */
    public function __construct($course=null, $show_inactive=null, $groupid=null, $studuserid=null, $type=self::MOD_ALL, $context=null,
        $load_users=true, $set_students=null, $set_user=null, $cmstudents=false){
        global $USER;
        $this->_user = $USER;
        $this->_userid = $USER->id ?? 0;

        if (!$course){
            return;
        }

        $options = [
            'course' => $course,
            'context' => $context,
            'type' => $type,
            'show_inactive' => $show_inactive,
            'groupid' => $groupid,
            'studuserid' => $studuserid,
            'load_users' => $load_users,
            'set_user' => $set_user,
            'set_students' => $set_students,
            'cmstudents' => $cmstudents,
        ];
        $this->set_params($options);
    }

    /**
     * @param           $options        - see static::OPTIONS
     * @param bool      $do_preparation - run do_preparation after that (if true)
     * @param bool|null $load_types     - if true, also load MM types, by default (null) use the same value as $do_preparation
     */
    public function set_params($options, $do_preparation=true, $load_types=null){
        foreach (static::OPTIONS as $option){
            $this->{'_'.$option} = $options[$option] ?? null;
        }

        $this->_context = $this->_context ?: NED::ctx($this->_course->id);
        $this->_cmstudents = $this->_cmstudents ?? false;
        $load_types = $load_types ?? $do_preparation;
        if ($do_preparation){
            $load_users = $options['load_users'] ?? true;
            $set_students = $options['set_students'] ?? null;
            $set_user = $options['set_user'] ?? null;
            $this->do_preparation($load_users, $set_students, $set_user);
        }
        if ($load_types){
            $types = $options['types'] ?? $options['type'] ?? static::MOD_ALL;
            $this->load_types($types);
        }
    }

    /**
     * Don't call this function, if you haven't set parameters yet
     *
     * @param bool   $load_users
     * @param array  $set_students
     * @param int    $set_user
     */
    public function do_preparation($load_users=true, $set_students=null, $set_user=null){
        $this->_courseid = $this->_course->id;
        if ($set_students === '*'){
            $this->_students = [];
            $this->_nousers = false;
            $this->_studentids = '*';
        } else {
            if ($set_students){
                $this->_students = NED::get_user_list($set_students);
            } elseif ($load_users){
                [$this->_show_inactive, $this->_allgroups, $this->_group, $this->_groupid, $this->_allstudents, $this->_students, $this->_studuser,
                    $this->_studuserid] =
                    MB::get_showinactive_groups_group_groupid_allstudents_students_user($this->_course, $this->_show_inactive, $this->_groupid,
                        $this->_studuserid, false, $set_user, $this->_cmstudents);
            }

            if ($load_users || $set_students){
                $this->_nousers = empty($this->_students);
                if ($this->_nousers){
                    return;
                }
                if ($this->_studuser){
                    $this->_studentids = [$this->_studuserid];
                } elseif (!empty($this->_students)){
                    $this->_studentids = array_keys($this->_students);
                } else {
                    $this->_studentids = [];
                }
            } else {
                $this->_nousers = true;
                $this->_studentids = [];
            }
        }

        $this->_kica = NED::get_kica($this->_courseid);
        $this->_has_kica = $this->_kica->enabled ?? false;
    }

    /**
     * @param array|string $types
     */
    public function load_types($types){
        $types = NED::val2arr($types) ?: [static::MOD_ALL];
        if (in_array(static::MOD_ALL, $types)){
            $types = static::MOD_TYPES;
        }

        foreach ($types as $type){
            $cls_name = static::get_type_mod($type);
            if (!$cls_name){
                continue;
            }
            /** @var marking_manager_mod $mm_mod */
            $mm_mod = new $cls_name($this, $this->_courseid, $this->_studentids, $this->_has_kica);
            $this->_mod_types[$type] = $mm_mod;
        }
    }

    /**
     * @param           $options        - see static::OPTIONS
     * @param bool      $do_preparation - run do_preparation after that (if true)
     * @param bool|null $load_types     - if true, also load MM types, by default (null) use the same value as $do_preparation
     *
     * @return marking_manager
     */
    static public function get_MM_by_params($options=[], $do_preparation=true, $load_types=null){
        $MM = new static();
        $MM->set_params($options, $do_preparation, $load_types);
        return $MM;
    }

    /**
     * If $force or there are no capability to view grades before midnight - add HIDE_GRADES_BEFORE_MIDN to the filter
     *
     * @param array $filter - array, where need to add filter keys
     * @param bool  $force  - if true, skip capabilities checking
     *
     * @return array - updated $filter
     */
    public function filter_hide_grades_before_midn_if_need(&$filter, $force=false){
        if (!$force && NED::cap_can_view_grades_before_midn($this->_courseid ?: $this->_context)) return $filter;
        if (!is_array($filter)) return $filter;

        $filter[static::HIDE_GRADES_BEFORE_MIDN] = true;
        return $filter;
    }

    /**
     * Return classname of mm mod
     * @param string $type
     *
     * @return string|null
     */
    static protected function get_type_mod($type){
        global $CFG;
        $t_locallib = $CFG->dirroot . "/mod/$type/locallib.php";
        $t_lib = $CFG->dirroot . "/mod/$type/lib.php";
        $cls_name = "marking_manager_$type";
        $mm_lib = __DIR__ . "/$cls_name.php";

        $req_files = [$mm_lib];
        $mod_files = [$t_lib, $t_locallib];
        $files = [];
        // we need at least one file from $mod_files
        foreach ($mod_files as $file){
            if (file_exists($file)){
                $files[] = $file;
            }
        }
        if (empty($files)){
            return null;
        }
        // we need all files from $mod_files
        foreach ($req_files as $file){
            if (!file_exists($file)){
                return null;
            }
            $files[] = $file;
        }
        // we have all files, so we can continue
        foreach ($files as $file){
            include_once($file);
        }

        $cls_name = '\\' . NED::$PLUGIN_NAME  . '\\marking_manager\\' . $cls_name;
        return $cls_name;
    }

    /**
     * @return array [$show_inactive, $groupid]
     */
    public function get_showinactive_groupid(){
        return [$this->_show_inactive, $this->_groupid];
    }

    /**
     * @param array $set_filter=null
     * @param bool  $full_replace=false
     *
     * @return array
     */
    public function get_base_filter($set_filter=null, $full_replace=false){
        $base_filter = [static::SUM, static::BY_ACTIVITY, static::SORT];
        $new_filter = null;

        if (!is_null($set_filter)){
            if ($full_replace){
                $new_filter = $set_filter;
            } else {
                $new_filter = array_merge($base_filter, $set_filter);
            }
        } elseif (is_null($this->_base_filter)){
            $new_filter = $base_filter;
        }

        if ($new_filter){
            $this->_base_filter = $this->prepare_filter($new_filter);
        }

        return $this->_base_filter;
    }

    /**
     * Get data and save it to the $status_data
     *
     * @see get_data()
     *
     * @param array $set_filter = null
     * @param string $get_type - to get data for only one mod type
     *
     * @return array|object[]|object[][]|mm_data_by_user[]|mm_data_by_activity[]|mm_data_by_activity_user[]
     */
    public function get_status_data($set_filter=null, $get_type=null){
        if (!is_null($set_filter) && $set_filter != $this->_base_filter){
            $this->_status_data = $this->get_data($set_filter, true, $get_type);
        } elseif (is_null($this->_status_data)){
            $this->_status_data = $this->get_data(null, false, $get_type);
        }
        return $this->_status_data;
    }

    /**
     * @param int   $cm_id
     * @param array $set_filter   = null
     * @param bool  $full_replace = false
     * @param array $add_filter
     *
     * @return array
     */
    public function get_activity_filter($cm_id=0, $set_filter=null, $full_replace=false, $add_filter=[]){
        if (!$cm_id && !isset($this->_cm->id)){
            debugging('You can\'t use activity_filter without course_module id!');
            $cm_id = 0;
        } else {
            $cm_id = $cm_id ?: $this->_cm->id;
        }

        $add_base_filter = [static::USE_DEADLINE, static::BY_USER, static::USE_KICA, static::COURSEMODULE_IDS => $cm_id];
        $add_base_filter = $this->prepare_filter($add_base_filter, false);
        $remove_keys = [static::BY_ACTIVITY, static::SUM, static::SORT, static::NOT_ZERO];
        $remove_keys = array_merge($remove_keys, array_keys(static::COUNT_STATUS));
        $new_filter = null;

        if (!is_null($set_filter)){
            $set_filter = $this->prepare_filter($set_filter, false);
            if ($full_replace){
                $new_filter = $set_filter;
            } else {
                $new_filter = array_merge($add_base_filter, $set_filter);
            }
        } elseif (is_null($this->_activity_filter)){
            $base_filter = $this->get_base_filter();
            $new_filter =  array_merge($base_filter, $add_base_filter);
        }

        if (!empty($add_filter)){
            $new_filter =  array_merge($new_filter, $add_filter);
        }

        if ($new_filter){
            foreach ($remove_keys as $key){
                unset($new_filter[$key]);
            }
            $this->_activity_filter = $this->prepare_filter($new_filter);
        }

        return $this->_activity_filter;
    }

    /**
     * @param int   $cm_id = null
     * @param array $set_filter = null
     *
     * @return array|null
     */
    public function get_activity_data($cm_id=null, $set_filter=null){
        if ((!is_null($cm_id) && $cm_id != NED::isset2($this->_cm, 'id')) || (!is_null($set_filter) && $set_filter != $this->_base_filter)){
            $filter = $this->get_activity_filter($cm_id, $set_filter);
            $this->_activity_data = $this->get_data($filter);
        } elseif (is_null($this->_activity_data)){
            $filter = $this->get_activity_filter();
            $this->_activity_data = $this->get_data($filter);
        }
        return $this->_activity_data;
    }

    /**
     * @param \moodle_url $set_base_url = null
     *
     * @return \moodle_url
     */
    public function get_base_url($set_base_url=null){
        if ($set_base_url){
            $this->_base_url = new \moodle_url($set_base_url);
        } elseif (is_null($this->_base_url)){
            global $PAGE;
            $this->_base_url = new \moodle_url($PAGE->url);
        }
        // return new copy every time
        return new \moodle_url($this->_base_url);
    }

    /**
     * @param null $set_show_status - one of MM_SELECT_STATUS
     *
     * @return string
     */
    public function get_show_status($set_show_status=null){
        if ($set_show_status){
            $this->_show_status = $set_show_status;
        } elseif (is_null($this->_show_status)){
            $this->_show_status = static::ST_ALL;
        }

        return $this->_show_status;
    }

    /**
     * @param      $cm = null
     * @param bool $error_if_not_exist
     *
     * @return \cm_info|\stdClass
     */
    public function get_cm($cm=null, $error_if_not_exist=false){
        if ($cm){
            $cmid = $cm->id ?? $cm;
            if ($cmid != NED::isset2($this->_cm, 'id', 0)){
                $this->_cm = NED::get_cm_by_cmid($cmid, $this->_courseid);
            }
            if ($cm->id ?? false){
                if (!$this->_cm_data){
                    $this->_cm_data = NED::stdClass2($this->_cm);
                }

                foreach ($cm as $key => $item){
                    if (is_null($this->_cm_data->$key)){
                        $this->_cm_data->$key = $item;
                    }
                }
            }
        }
        if ($error_if_not_exist && !$this->_cm){
            NED::print_error('You should set $cm before ask it!');
        }
        return $this->_cm;
    }

    /**
     * @param array       $filter      = null
     * @param string      $show_status = null
     * @param int         $cmid        = 0
     * @param \moodle_url $base_url    = null
     *
     * @return \html_table|null
     */
    public function render_mm_status($filter=null, $show_status=null, $cmid=0, $base_url=null){
        $this->get_base_filter($filter);
        $mod_data = $this->get_status_data();
        $mod_data = $mod_data[static::MOD_ALL];
        if (empty($mod_data)){
            return null;
        }

        $cmid = $cmid ?: NED::isset2($this->_cm, 'id', 0);
        $base_url = $this->get_base_url($base_url);
        $show_status = $this->get_show_status($show_status);

        if (!isset($mod_data[$cmid])){
            reset($mod_data);
            $cmid = key($mod_data);
        }
        $selected_cm = $cmid ? $mod_data[$cmid] : null;

        $fnmarkingblock = new \html_table();
        $fnmarking_class = ['fnmarkingblock'];
        $sum = 0;

        foreach($mod_data as $mod){
            $mod = NED::stdClass2($mod);
            $val = $mod->{$show_status} ?? 0;
            $sum += $val;
            $row_class = ['normal'];
            $mname_class = ['marked-name'];
            $mname_attr = [];

            if ($cmid == $mod->id) {
                $row_class[0] = 'highlight';
            }

            $main_tag = null;
            foreach (NED::IMPORTANT_TAGS as $tag_key => $tag){
                $is_tag = 'is_'.$tag_key;
                if ($mod->$is_tag ?? false){
                    $main_tag = $tag;
                    break;
                }
            }
            if ($main_tag){
                $mname_class[] = 'add-tag-icon';
                $mname_attr['data-tags'] = $main_tag;
            }

            $modurl = new \moodle_url($base_url);
            $modurl->param('mid', $mod->id);
            $add_class = 'assignmentlist';
            if (!$mod->visible){
                $add_class .= ' dimmed';
            }

            $mod_icon = NED::img('icon', 'mod-icon', $mod->modname, ['height' => 20, 'width' => 20]);
            $mod_text = $mod_icon .' '. $mod->name;
            $add_params['title'] = $mod->name;
            $mod_link = NED::link($modurl, $mod_text, 'mod-link', $add_params);

            $activity_text = NED::cell(\html_writer::div($mod_link, $add_class), join(' ', $mname_class), $mname_attr);

            $add_val = '';
            $activity_val = NED::cell($val.$add_val, 'marked-value');
            $fnmarkingblock->data[] = NED::row([$activity_text, $activity_val], join(' ', $row_class));
        }

        $s_show = NED::str($show_status);
        $fnmarkingblock->head = [NED::str('activity')." ($s_show)", $sum];
        $fnmarkingblock->attributes['class'] = join(' ', $fnmarking_class);

        $this->get_cm($selected_cm);
        return $fnmarkingblock;
    }

    /**
     * Get MM_data by filter params
     * If none $get_type, it will be an array with mm_data by all mod_types keys,
     *  including static::MOD_ALL key with all mod types
     *
     * If you use both BY_ACTIVITY and BY_USER keys, mm_data will be packing as [cmid => [userid => mm_data_by_activity_user]]
     *
     * @param array        $filter
     * @param bool         $update_filter
     * @param string|array $get_types - to get data for only one mod type
     *
     * @return array|object[]|object[][]|mm_data_by_user[]|mm_data_by_activity[]|mm_data_by_activity_user[]
     */
    public function get_data($filter=null, $update_filter=false, $get_types=null){
        $get_types = NED::val2arr($get_types);
        $load_all_mods = empty($get_types) || in_array(static::MOD_ALL, $get_types);
        $filter = empty($filter) ? $this->get_base_filter() : $filter;
        if ($update_filter){
            $this->get_base_filter($filter, true);
        }
        $data_all = [];
        /** @noinspection PhpArrayIndexImmediatelyRewrittenInspection */
        $data = [static::MOD_ALL => $data_all];
        $filter = $this->prepare_filter($filter);
        $get_object = NED::isset2($filter, [static::SUM]) && !NED::isset2($filter, [static::BY_ACTIVITY]) && !NED::isset2($filter, [static::BY_USER]);
        $by_activity_and_types = NED::isset2($filter, [static::BY_ACTIVITY]) && NED::isset2($filter, [static::BY_USER]);
        $f_sort_by_pos = function($a, $b){
            return $a->pos <=> $b->pos;
        };
        $f_sort_by_pos_arr = function($a, $b) use ($f_sort_by_pos){
            return $f_sort_by_pos(reset($a), reset($b));
        };
        if ($get_object){
            $data_all = NED::stdClass2();
        }
        foreach ($this->_mod_types as $type => $mm_mod){
            if (!$load_all_mods && !in_array($type, $get_types)) continue;

            $m_data = $mm_mod->get_data($filter);
            $data[$type] = $m_data;
            if (empty($m_data)){
                continue;
            }
            if ($get_object){
                foreach ($m_data as $key => $m_value){
                    $data_all->$key = isset($data_all->$key) ? $data_all->$key + $m_value : $m_value;
                }
            } elseif ($by_activity_and_types){
                $m_data2 = [];
                foreach ($m_data as $m_value){
                    $m_data2[$m_value->id][$m_value->userid] = $m_value;
                    $data_all[$m_value->id][$m_value->userid] = $m_value;
                }
                $data[$type] = $m_data2;
            } else { // we think, that it's key-array
                $data_all += $m_data;
            }
            $this->_records_count += $mm_mod->get_records_count();
        }

        if (!$get_object && NED::isset2($filter, [static::SORT])){
            if ($by_activity_and_types){
                $data[static::MOD_ALL] = $data_all;
                foreach ($data as $type => &$data_type){
                    if (!$load_all_mods && !in_array($type, $get_types)) continue;

                    foreach ($data_type as &$activity_data){
                        asort($activity_data); // sort by u.id
                    }
                    uasort($data_type, $f_sort_by_pos_arr); // sort by pos
                }
            } else {
                uasort($data_all, $f_sort_by_pos);
                $data[static::MOD_ALL] = $data_all;
            }
        } else {
            $data[static::MOD_ALL] = $data_all;
        }

        if (!empty($get_types)){
            if (count($get_types) == 1){
                return reset($data);
            }
        }
        return $data;
    }

    /**
     * Get MM_data by filter params
     * No sort, no smart merge or packing
     * No delegation by mod types, no sum result into static::MOD_ALL data
     *
     * @param      $filter
     * @param bool $update_filter
     *
     * @return array|object[]|mm_data_by_user[]|mm_data_by_activity[]|mm_data_by_activity_user[]
     */
    public function get_raw_data($filter=null, $update_filter=false){
        $filter = empty($filter) ? $this->get_base_filter() : $filter;
        if ($update_filter){
            $this->get_base_filter($filter, true);
        }
        $filter = $this->prepare_filter($filter);
        $data = [];
        foreach ($this->_mod_types as $mm_mod){
            $m_data = $mm_mod->get_data($filter);
            if (!empty($m_data)){
                $data = array_merge($data, $m_data);
                $this->_records_count += $mm_mod->get_records_count();
            }
        }

        return $data;
    }


    /**
     * Get MM_data by filter params for all mod types
     * alias for get_data
     *
     * @see get_data();
     *
     * @param array $filter
     *
     * @return array|object[]|object[][]|mm_data_by_user[]|mm_data_by_activity[]|mm_data_by_activity_user[]
     */
    public function get_data_all($filter=null){
        return $this->get_data($filter, false, static::MOD_ALL);
    }

    /**
     * Prepare $filter for using in get_data function
     *
     * @param      $filter
     * @param bool $check_rules
     *
     * @return array
     */
    public function prepare_filter($filter, $check_rules=true){
        $filter = (array)$filter;
        $filter2 = [];
        foreach ($filter as $key => $f){
            if (is_numeric($key)){
                $filter2[$f] = true;
            } else {
                $filter2[$key] = $f;
            }
        }
        $filter = $filter2;

        if (!$check_rules){
            return $filter;
        }

        $f = function($key) use (&$filter){
            return NED::isset2($filter, [$key]);
        };

        if ($f(static::BY_ACTIVITY_USER)){
            $filter[static::BY_ACTIVITY] = true;
            $filter[static::BY_USER] = true;
        } elseif ($f(static::BY_ACTIVITY) && $f(static::BY_USER)){
            $filter[static::BY_ACTIVITY_USER] = true;
        }

        if (!$this->_has_kica){
            unset($filter[static::ONLY_KICA]);
            unset($filter[static::USE_KICA]);
            unset($filter[static::KICAGRADE]);
        } elseif ($f(static::ONLY_KICA)){
            $filter[static::USE_KICA] = true;
            $filter[static::KICAGRADE] = true;
        }

        if (!NED::is_tt_exists()){
            unset($filter[static::USE_GM]);
            unset($filter[static::USE_GM_ATTEMPT]);
            unset($filter[static::ONLY_GM]);
            unset($filter[static::ONLY_NOT_GM]);
        } elseif ($f(static::ONLY_GM) || $f(static::ONLY_NOT_GM)){
            $filter[static::USE_GM] = true;
            if ($f(static::ONLY_GM) && $f(static::ONLY_NOT_GM)){
                debugging("You shouldn't use ONLY_GM & ONLY_NOT_GM together");
                unset($filter[static::ONLY_NOT_GM]);
            }
        } elseif ($f(static::USE_GM_ATTEMPT)){
            $filter[static::USE_GM] = true;
        }

        if ($f(static::USE_GM)){
            $filter[static::USE_GROUPS] = true;
        }

        if ($f(static::GROUP_REGEXP) || $f(static::GROUP_NOT_REGEXP)){
            $filter[static::USE_GROUPS] = true;
        }

        if ($f(static::ONLY_NGC) && $f(static::ONLY_NOT_NGC)){
            debugging("You shouldn't use ONLY_NGC & ONLY_NOT_NGC together");
            unset($filter[static::ONLY_NOT_NGC]);
        }

        if ($f(static::QUIZ_DETAILED_VIEW)){
            $rem_filter = [static::USE_DEADLINE, static::USE_GROUPS];
            foreach ($rem_filter as $item){
                unset($filter[$item]);
            }
        }

        if ($f(static::ONLY_EXCLUDED) && $f(static::NOT_EXCLUDED)){
            debugging("You shouldn't use ONLY_EXCLUDED & ONLY_NOT_EXCLUDED together");
            unset($filter[static::ONLY_EXCLUDED]);
            unset($filter[static::NOT_EXCLUDED]);
        }

        $tagid = NED::isset2($filter, [static::CHECK_TAG]);
        if ($tagid == static::TAGS_ALL){
            unset($filter[static::CHECK_TAG]);
        }

        $tag_keys = [static::TAG_NAME_LIST_HAVE_ALL, static::TAG_NAME_LIST_HAVE_ANY, static::TAG_NAME_LIST_HAVE_NONE];
        foreach ($tag_keys as $tag_key){
            if (!$f($tag_key)) {
                continue;
            }
            if (is_numeric($filter[$tag_key]) || is_string($filter[$tag_key])){
                $filter[$tag_key] = [$filter[$tag_key]];
            } elseif (!is_array($filter[$tag_key])){
                unset($filter[$tag_key]);
                continue;
            }

            $filter[static::USE_TAGS] = true;
        }

        if ($f(static::BY_ACTIVITY) || $f(static::CHECK_TAG)){
            $filter[static::USE_TAGS] = true;
        }

        // we can't sum information by user in something one
        if ($f(static::BY_USER) && $f(static::SUM)){
            debugging("You shouldn't use BY_USER & SUM together");
            unset($filter[static::SUM]);
        }
        // if we have deadline options, check it for correctness
        if (isset($filter[static::DEADLINE])){
            $deadline_options = static::get_deadline_options();
            $filter[static::DEADLINE] = NED::isset_key($deadline_options, $filter[static::DEADLINE]);
        }
        // if deadline == 0 (all), we don't touch other options
        if ($deadline = $f(static::DEADLINE)){
            $filter[static::HAVE_DEADLINE] = true;
            $filter = array_merge($filter, static::get_deadline_use($deadline));
        }

        if ($deadline_upto = $f(static::DEADLINE_UPTO)){
            if (is_numeric($deadline_upto)){
                $filter[static::USE_DEADLINE] = true;
            } else {
                unset($filter[static::DEADLINE_UPTO]);
            }
        }
        if ($deadline_after = $f(static::DEADLINE_AFTER)){
            if (is_numeric($deadline_after)){
                $filter[static::USE_DEADLINE] = true;
            } else {
                unset($filter[static::DEADLINE_AFTER]);
            }
        }

        if ($f(static::GET_PROXY_ENABLED)){
            if (!NED::has_proxy_deadline_plugins()){
                unset($filter[static::GET_PROXY_ENABLED]);
            }
        }

        $get_st = $f(static::GET_ST) ?: [];
        if (!empty($get_st)){
            if (!is_array($get_st)){
                $get_st = [$get_st];
            }
            $get_st2 = [];
            foreach ($get_st as $key => $st){
                if (is_int($key)){
                    $get_st2[$st] = $st;
                } elseif($st) {
                    $get_st2[$key] = $key;
                }
            }
            $get_st = $get_st2;
            $filter[static::GET_ST] = $get_st;
        }

        $status_use_deadlines = $f(static::HAVE_DEADLINE) || $f(static::USE_DEADLINE);
        if (!$status_use_deadlines){
            $statuses_count_with_deadlines = array_diff(array_keys(static::COUNT_STATUS), [static::COUNT_DRAFT]);
            $statuses_with_deadlines = array_diff(static::MM_ALL_STATUSES, static::NON_PERSONAL_STATUSES);
            $statuses_with_deadlines = array_merge($statuses_with_deadlines, $statuses_count_with_deadlines, static::VALUE_DEADLINE);
            foreach ($statuses_with_deadlines as $state){
                if ($f($state) || isset($get_st[$state])){
                    $status_use_deadlines = true;
                    break;
                }
            }
        }

        if ($status_use_deadlines){
            // you can use, but not have, but if you have - you use it
            $filter[static::USE_DEADLINE] = true;
        }

        if ($f(static::USE_DEADLINE)){
            $filter[static::USE_GROUPS] = true;
        }

        return $filter;
    }

    /**
     * Get deadline list options by deadline option number
     * @param int $deadline_value
     *
     * @return array
     */
    static public function get_deadline_use($deadline_value=0){
        switch ($deadline_value){
            default:
            case 0: return [static::DEADLINE_PAST => 1, static::DEADLINE_NOTSET => 1, static::DEADLINE_UPCOMING => 1];
            case 1: return [static::DEADLINE_PAST => 1, static::DEADLINE_NOTSET => 0, static::DEADLINE_UPCOMING => 0];
            case 2: return [static::DEADLINE_PAST => 0, static::DEADLINE_NOTSET => 1, static::DEADLINE_UPCOMING => 0];
            case 3: return [static::DEADLINE_PAST => 0, static::DEADLINE_NOTSET => 0, static::DEADLINE_UPCOMING => 1];
            case 4: return [static::DEADLINE_PAST => 1, static::DEADLINE_NOTSET => 1, static::DEADLINE_UPCOMING => 0];
            case 5: return [static::DEADLINE_PAST => 0, static::DEADLINE_NOTSET => 1, static::DEADLINE_UPCOMING => 1];
            case 6: return [static::DEADLINE_PAST => 1, static::DEADLINE_NOTSET => 0, static::DEADLINE_UPCOMING => 1];
            case 7: return [];
        }
    }

    /**
     * Get deadline option number by list options
     * @param $deadline_options
     *
     * @return int
     */
    static public function get_deadline_value($deadline_options){
        foreach (static::VALUE_DEADLINE as $val){
            if (!isset($deadline_options[$val])){
                $deadline_options[$val] = 0;
            } else {
                $deadline_options[$val] = $deadline_options[$val] ? 1 : 0;
            }
        }
        $i=0;
        while ($uses = static::get_deadline_use($i++)){
            foreach ($uses as $key => $use){
                if ($deadline_options[$key] != $use){
                    continue 2;
                }
            }
            return $i-1;
        }
        return 0;
    }

    /**
     * Get deadline selector options
     * @return array
     */
    static public function get_deadline_options(){
        static $options = [];
        if (empty($options)){
            $i = 0;
            $options[$i++] = NED::str('all');
            while ($uses = static::get_deadline_use($i++)){
                $words = [];
                foreach ($uses as $key => $use){
                    if ($use){
                        $words[] = NED::str($key);
                    }
                }
                $options[] = join(' + ', $words);
            }
        }

        return $options;
    }

    /**
     * Check, does page with this show_status use deadlines
     *
     * @param null $show_status
     *
     * @return bool
     */
    static public function is_using_deadlines($show_status){
        if (!in_array($show_status, static::NON_PERSONAL_STATUSES)){
            return true;
        }
        return false;
    }

    /**
     * Check, has mod_type personal_method for render content by this $show_status
     *  can change $this->_cm if $cm passed
     * @param null $show_status
     * @param null $cm
     *
     * @return bool|null
     */
    public function has_personal_method($show_status=null, $cm=null){
        $cm = $this->get_cm($cm);
        if (!(($cm->id ?? false) && ($this->_mod_types[$cm->modname] ?? false))){
            return false;
        }

        if ($this->_mod_types[$cm->modname] ?? false){
            $mm_type = $this->_mod_types[$cm->modname];
            return $mm_type->has_personal_method($show_status);
        }

        /** @var marking_manager_mod $mm_type */
        $mm_type = static::get_type_mod($cm->modname);
        if(!$mm_type){
            return false;
        }

        $show_status = !is_null($show_status) ? $show_status : $this->get_show_status();
        return $mm_type::class_has_personal_method($show_status);
    }

    /**
     * Render table by $cm & show
     * @param null $show_status
     * @param null $cm
     * @param null $filter
     * @param null $base_url
     *
     * @return string|null
     */
    public function render_by_show($show_status=null, $cm=null, $filter=null, $base_url=null){
        if (!$this->has_personal_method($show_status, $cm)){
            return null;
        }

        $cm = $this->get_cm();
        $mm_type = $this->_mod_types[$cm->modname];
        return $mm_type->render_by_show($show_status, $cm, $filter, $base_url);
    }

    /**
     * @return string
     */
    public function render_mm_warnings(){
        $mm_warnings = [];

        foreach (static::COUNT_STATUS as $count_status => $status){
            $count = $this->_cm_data->$count_status ?? 0;
            if ($count > 0){
                $url = $this->get_base_url();
                $url->param(static::P_STATUS, $status);
                $mm_warnings[] = \html_writer::div(
                    NED::str($count_status, $count) .
                    NED::link([$url], 'clickheretoviewreport'),
                    $count_status);
            }
        }

        if (!empty($mm_warnings)){
            return \html_writer::div(join('', $mm_warnings), 'mm-warnings');
        }
        return '';
    }

    /**
     * Return activity header as action menu
     *
     * @return string
     */
    public function render_activity_header(){
        $cm = $this->get_cm();
        $o = '';
        if ($cm){
            $mm_type = $this->_mod_types[$cm->modname] ?? false;
            if ($mm_type){
                [$activity, $grade_item, $use_kica, $grade_fun] = $mm_type->get_usual_cm_data();
                $o = $mm_type->get_activity_header($grade_item, $use_kica);
            }
        } else {
            $o = $this->render_mm_warnings();
        }
        return $o;
    }

    /**
     * Return status-links instead of status-select
     *
     * @param $show
     * @param $view_more
     * @param array|int[]|null $cm_keys
     *
     * @return string
     */
    public function get_status_selector($show, $view_more, $cm_keys=null){
        $filter = static::MM_WFG_TYPES;
        if (NED::is_tt_exists()){
            if (\block_ned_teacher_tools\cta_tracker::is_tracker_on($this->_courseid)){
                $filter[static::TAG_NAME_LIST_HAVE_NONE] = NED::TAG_CTA;
            }
        }

        $this->filter_hide_grades_before_midn_if_need($filter);
        if ($view_more){
            $filter = array_merge($filter, static::MM_FILTER_TYPES);
        }
        $filter[] = static::SUM;

        $url = $this->get_base_url();
        $tag = $url->param(MB::PAR_TAG);
        if (!is_null($cm_keys)){
            $filter[static::CMIDS] = $cm_keys;
        }
        if ($tag){
            $filter[static::CHECK_TAG] = $tag;
        }

        $filter = $this->prepare_filter($filter);
        unset($filter[static::ST_ALL]);

        $sum_mm_data = $this->get_data_all($filter);
        $a_status = function($st) use(&$sum_mm_data, $filter, $url, $show){
            $url->param(static::P_STATUS, $st);
            $text = \html_writer::span(NED::str($st), 'status-name');
            if (isset($filter[$st])){
                $val = isset($sum_mm_data->$st) ? ($sum_mm_data->$st ?: 0) : 0;
                $text .= \html_writer::span($val, 'status-count');
                $show_link = $val > 0;
            } else {
                $show_link = true;
            }

            $a_class = ['status-link', $st];
            if ($show == $st){
                $a_class[] = 'active';
            }

            if ($show_link){
                return NED::link($url, $text, join(' ', $a_class));
            } else {
                $a_class[] = 'disabled';
                return \html_writer::span($text, join(' ', $a_class));
            }
        };

        $o = '';
        $statuses = [];
        $statuses[] = ['actionrequired', static::MM_WFG_TYPES];
        if ($view_more){
            $statuses[] = ['filters', static::MM_FILTER_TYPES];
        } elseif (in_array($show,static::MM_FILTER_TYPES)){
            $statuses[] = ['filters', [$show]];
        }

        foreach ($statuses as $status_data){
            [$name, $types] = $status_data;

            $content = [\html_writer::span(NED::str($name), 'mm-label')];
            foreach ($types as $type){
                $content[$type] = $a_status($type);
            }

            if (!isset($this->_mod_types[static::MOD_ASSIGN]) && $show != static::ST_DRAFT){
                unset($content[static::ST_DRAFT]);
            }

            $o .= \html_writer::div(join('', $content), "mm-status $name-status");
        }

        return $o;
    }
}
