<?php

/**
 * @package    local_ned_controller
 * @subpackage marking_manager
 * @category   NED
 * @copyright  2021 NED {@link http://ned.ca}
 * @author     NED {@link http://ned.ca}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 * @noinspection PhpUnused
 * @noinspection PhpUnnecessaryStopStatementInspection
 * @noinspection PhpUnusedParameterInspection
 * @noinspection PhpUnnecessaryCurlyVarSyntaxInspection
 */

namespace local_ned_controller\marking_manager;
use local_ned_controller\shared_lib as NED;

defined('MOODLE_INTERNAL') || die();

/**
 * Base class for MM activity classes
 *
 * @property-read int $courseid;
 * @property-read array $studentids;
 * @property-read string $has_kica;
 */
abstract class marking_manager_mod extends MM_base{
    const HAVE_DEADLINES = false;
    const HAVE_GRADES = true;

    const SQL_USER_TIMEMODIFIED = '0';
    const SQL_SUBMIT_TIME = '';
    /** @see \local_ned_controller\shared\grade_util::grade_get_gi_itemnumber() */
    const SQL_GI_ITEMNUMBER = NED::GRADE_ITEMNUMBER_USUAL;

    const SQL_D_DAY_START = 'd.day_start'; // start of the current day by NED time
    const SQL_D_DAY_END = 'd.day_end'; // end of the current day by NED time
    const SQL_D_NOW = 'd.now'; // Current UNIX time
    const SQL_D_IN_TODAY = ' BETWEEN '.self::SQL_D_DAY_START.' AND '.self::SQL_D_DAY_END; // X in today day by NED time

    const SQL_GG_OVERTIMEMODIFIED = "IF(COALESCE(gg.overridden,0) > 0, gg.overridden, gg.timemodified)";
    const SQL_GRADE_TIME = "IF(gg.finalgrade IS NULL AND gg.feedback IS NULL, NULL, ".self::SQL_GG_OVERTIMEMODIFIED.")";
    const SQL_FIRST_LAST_GRADE_TIME = "IF(
        gg.finalgrade IS NULL AND gg.feedback IS NULL, NULL,
        LEAST(COALESCE(ggh_first_graded_time.timemodified, ".self::SQL_GG_OVERTIMEMODIFIED."), ".self::SQL_GG_OVERTIMEMODIFIED.")
    )";
    const SQL_FIRST_LAST_GRADE_TIME_COAL = "COALESCE(".self::SQL_FIRST_LAST_GRADE_TIME.", 0)";

    const SQL_IF_GRADE_MIDN_HIDDEN = "(".self::SQL_FIRST_LAST_GRADE_TIME_COAL.self::SQL_D_IN_TODAY.")";
    const SQL_RAWGRADE = 'IF(gg.overridden > 0, gg.finalgrade, gg.rawgrade)';
    const SQL_RAWGRADE_MIDN = 'IF('.self::SQL_IF_GRADE_MIDN_HIDDEN.', NULL, '.self::SQL_RAWGRADE.')';
    const SQL_DEADLINE = 'COALESCE(effective.cutoffdate, 0)';
    const SQL_FINAL_DEADLINE = self::SQL_DEADLINE;
    const SQL_REQUIRE_ONLY_SUBMIT = '0';
    const SQL_ACTIVITY_HAS_COMPLETION = 'cm.completion <> '.COMPLETION_TRACKING_NONE;
    const SQL_USER_HAS_COMPLETION = 'cmc.completionstate <> '.COMPLETION_INCOMPLETE;
    const SQL_USER_HAS_GRADE = '('.self::SQL_RAWGRADE.' IS NOT NULL)';
    const SQL_USER_HAS_GRADE_MIDN = '('.self::SQL_RAWGRADE_MIDN.' IS NOT NULL)';
    const SQL_USER_ANY_COMPLETED = '('.self::SQL_USER_HAS_COMPLETION.' OR '.self::SQL_USER_HAS_GRADE.')';
    const SQL_USER_ANY_COMPLETED_MIDN = '('.self::SQL_USER_HAS_COMPLETION.' OR '.self::SQL_USER_HAS_GRADE_MIDN.')';
    const SQL_ACTIVITY_HAS_NONE_COMPLETION = 'cm.completion = '.COMPLETION_TRACKING_NONE;
    const SQL_COMPLETE = '(cmc.completionstate = '.COMPLETION_COMPLETE_PASS.' OR cmc.completionstate = '.COMPLETION_COMPLETE.')';
    const SQL_COMPLETE_COMPLETION = '('.self::SQL_ACTIVITY_HAS_NONE_COMPLETION.' OR '.self::SQL_COMPLETE.')';
    const SQL_COMPLETE_GRADE = '('.self::SQL_COMPLETE.' AND (gi.gradepass <= 0 OR '.self::SQL_USER_HAS_GRADE.'))';
    const SQL_COMPLETE_GRADE_MIDN = '('.self::SQL_COMPLETE.' AND (gi.gradepass <= 0 OR '.self::SQL_USER_HAS_GRADE_MIDN.'))';
    const SQL_COMPLETED_SUCCESSFULLY = '(gi.gradepass > 0 AND '.self::SQL_RAWGRADE.' >= gi.gradepass)';
    const SQL_COMPLETED_SUCCESSFULLY_MIDN = 'IF('.self::SQL_IF_GRADE_MIDN_HIDDEN.', FALSE, '.self::SQL_COMPLETED_SUCCESSFULLY.')';
    const SQL_COMPLETE_GRADE_PASS = '(gi.gradepass <= 0 OR '.self::SQL_RAWGRADE.' >= gi.gradepass)';
    const SQL_COMPLETE_GRADE_PASS_MIDN = 'IF('.self::SQL_IF_GRADE_MIDN_HIDDEN.', FALSE, '.self::SQL_COMPLETE_GRADE_PASS.')';
    const SQL_COMPLETED_GRADE_SUCCESSFULLY =
        '('.self::SQL_USER_ANY_COMPLETED.' AND '.self::SQL_COMPLETE_COMPLETION.' AND '.self::SQL_COMPLETE_GRADE_PASS.')';
    const SQL_COMPLETED_GRADE_SUCCESSFULLY_MIDN =
        '('.self::SQL_USER_ANY_COMPLETED_MIDN.' AND '.self::SQL_COMPLETE_COMPLETION.' AND '.self::SQL_COMPLETE_GRADE_PASS_MIDN.')';

    const SQL_SHOULDPASS = 'gi.gradepass > 0';
    const SQL_NOT_OVERRIDDEN = 'COALESCE(gg.overridden, 0) = 0';
    const SQL_GRADED_BY_OVERRIDDEN = '(gg.overridden > 0 AND gg.finalgrade IS NOT NULL)';
    const SQL_UNGRADED_BY_OVERRIDDEN = '(gg.overridden > 0 AND gg.finalgrade IS NULL)';
    const SQL_ATTEMPT = '0';
    const SQL_GT_SHOULDPASS = '1';
    const SQL_TABLE_ALIAS = null;

    const SQL_GRADE_FEEDBACK = 'gg.feedback';
    const SQL_GRADE_FEEDBACK_FORMAT = 'gg.feedbackformat';

    const URL_GRADE_SINGLEVIEW = '/grade/report/singleview/index.php';
    const URL_MODEDIT = '/course/modedit.php';
    const URL_USER = '/user/view.php';
    const URL_MANUAL_GRADING = '';

    protected $_courseid;
    protected $_studentids = [];
    protected $_student_condition = null;
    protected $_has_kica = false;
    /** @var marking_manager $_mm */
    protected $_mm = null;
    /**
     * @var array $_filter_from_get_data - filter dictionary, saved only during {@see get_data()}
     * to get|check filter value, use {@see _filter_get()}
     */
    protected $_filter_from_get_data = null;

    /**
     * marking_manager_mod constructor.
     *
     * @param marking_manager   $mm
     * @param string | int      $courseid
     * @param array             $studentids
     * @param bool              $has_kica
     */
    public function __construct($mm, $courseid, $studentids=[], $has_kica=false){
        $this->_courseid = $courseid;
        $this->_studentids = $studentids;
        $this->_has_kica = $has_kica;
        $this->_mm = $mm;
    }

    /**
     * Return type of mod (static::MOD_*), and it's name of main DB-table
     * @return string
     */
    abstract public function get_mod_type();

    /**
     * Get value from the filter data or null
     * Note, that filter property has data only during {@see get_data()} method
     *
     * @param string $key
     *
     * @return mixed|null
     */
    protected function _filter_get($key=''){
        return $this->_filter_from_get_data[$key] ?? null;
    }

    /**
     * Check value from the filter data on empties
     * Note, that filter property has data only during {@see get_data()} method
     *
     * @param string $key
     *
     * @return bool - return true, if value by key is empty (or doesn't exist)
     */
    protected function _filter_empty($key=''){
        return empty($this->_filter_from_get_data[$key]);
    }

    /**
     * Add SQL UNMARKED condition to filter hidden grades before midnight, if need it
     *
     * @return string
     */
    protected function _add_filter_unmarked_cond(){
        return $this->_filter_get(static::HIDE_GRADES_BEFORE_MIDN) ? (" OR ".static::SQL_IF_GRADE_MIDN_HIDDEN) : '';
    }

    /**
     * Add SQL MARKED condition to filter hidden grades before midnight, if need it
     *
     * @return string
     */
    protected function _add_filter_marked_cond(){
        return $this->_filter_get(static::HIDE_GRADES_BEFORE_MIDN) ? (" AND NOT ".static::SQL_IF_GRADE_MIDN_HIDDEN) : '';
    }

    /**
     * Return condition by there name
     *  - rewrite in child class
     * @param string|array $name
     *
     * @return array|string
     */
    public function sql_get_condition_raw($name){
        if (!is_null($ans = $this->_call_fun_list($name, 'sql_get_condition_raw'))){
            return $ans;
        }

        if (!$this::HAVE_DEADLINES){
            if (in_array($name, static::SHOULD_HAVE_DEADLINE))
                return NED::SQL_NONE_COND;
        }

        if (empty($this::SQL_SUBMIT_TIME)){
            switch ($name){
                case static::ST_ZERO_GRADE:
                case static::ST_GRADE_LESS_50:
                case static::ST_GRADE_NOT:
                case static::ST_WAITING_ZERO:
                case static::ST_MISSED_DEADLINE:
                case static::ST_SUBMITTED_MISSED_DEADLINE:
                    return NED::SQL_NONE_COND;
            }
        }

        if ($this->get_mod_type() != static::MOD_ASSIGN){
            switch ($name){
                case static::ST_DRAFT:
                case static::ST_REOPENED:
                    return NED::SQL_NONE_COND;
            }
        }

        $sql_deadline = $this::SQL_DEADLINE;
        $sql_deadline_grace = static::sql_get_deadline_grace();
        if (static::HAVE_GRADES) {
            $sql_rawgrade = $this::SQL_RAWGRADE;
            $sql_kica_grade = "IF($sql_rawgrade IS NOT NULL AND kgi.id IS NOT NULL AND kgg.id IS NOT NULL, " .
                "COALESCE(kgg.finalgrade, $sql_rawgrade), $sql_rawgrade)";

            if ($this->_filter_get(static::HIDE_GRADES_BEFORE_MIDN)){
                $sql_rawgrade = $this::SQL_RAWGRADE_MIDN;
                // use $sql_kica_grade inside IF with original $sql_rawgrade
                $sql_kica_grade = "IF(".static::SQL_IF_GRADE_MIDN_HIDDEN.", NULL, $sql_kica_grade)";
            }

            $sql_grade = $this->_has_kica ? $sql_kica_grade : $sql_rawgrade;
        } else {
            $sql_grade = "NULL";
        }
        $sql_submit_coal = 'COALESCE('.$this::SQL_SUBMIT_TIME.', 0)';
        $sql_now = NED::SQL_NOW;
        $sql_now_12h = NED::SQL_NOW_12h;
        $sql_now_24h = NED::SQL_NOW_24h;
        $GRADE_TYPE_VALUE = GRADE_TYPE_VALUE;
        $conds = [];
        $use_or = false;
        switch ($name){
            case static::ST_ALL: return "u.id IS NOT NULL";
            case static::ST_WAITING_ZERO:
                $conds = [
                    $this->sql_get_condition(static::DEADLINE_PAST),
                    $this->sql_get_condition(static::ST_GRADE_NOT),
                ];
                break;
            case static::ST_ZERO_GRADE:
                return "$sql_grade = 0";
            case static::ST_GRADE_NOT:
                return "$sql_grade IS NULL";
            case static::ST_GRADE_LESS_50:
                if (static::HAVE_GRADES) {
                    return "gi.gradetype = $GRADE_TYPE_VALUE AND gi.grademax > 0 AND 
                    $sql_grade IS NOT NULL AND $sql_grade < (gi.grademax/2)";
                } else {
                    return NED::SQL_NONE_COND;
                }
            case static::ST_SUBMITTED_MISSED_DEADLINE:
                $conds = [
                    $this->sql_get_condition(static::DEADLINE_PAST),
                    "($sql_deadline < $sql_submit_coal)",
                ];
                break;
            case static::ST_UNSUBMITTED_MISSED_DEADLINE:
                $conds = [
                    $this->sql_get_condition(static::DEADLINE_PAST),
                    $this->sql_get_condition(static::ST_UNSUBMITTED),
                ];
                break;
            case static::ST_UNSUBMITTED_MISSED_DEADLINE_GRACE:
                $conds = [
                    $this->sql_get_condition(static::DEADLINE_PAST_GRACE),
                    $this->sql_get_condition(static::ST_UNSUBMITTED),
                ];
                break;
            case static::ST_MISSED_DEADLINE:
                $conds = [
                    $this->sql_get_condition(static::ST_SUBMITTED_MISSED_DEADLINE),
                    $this->sql_get_condition(static::ST_UNSUBMITTED_MISSED_DEADLINE),
                ];
                $use_or = true;
                break;
            case static::ST_UNMARKED_MISSED_DEADLINE:
                $conds = [
                    $this->sql_get_condition(static::DEADLINE_PAST),
                    $this->sql_get_condition(static::ST_UNSUBMITTED),
                    $this->sql_get_condition(static::ST_GRADE_NOT),
                ];
                break;
            case static::ST_UNMARKED_MISSED_DEADLINE_GRACE:
                $conds = [
                    $this->sql_get_condition(static::DEADLINE_PAST_GRACE),
                    $this->sql_get_condition(static::ST_UNSUBMITTED),
                    $this->sql_get_condition(static::ST_GRADE_NOT),
                ];
                break;
            case static::ST_UNMARKED_DUE_12h:
                $conds = [
                    $this->sql_get_condition(static::ST_UNSUBMITTED),
                    $this->sql_get_condition(static::ST_GRADE_NOT),
                    "($sql_deadline BETWEEN ($sql_now+1) AND $sql_now_12h)",
                ];
                break;
            case static::ST_UNMARKED_DUE_24h:
                $conds = [
                    $this->sql_get_condition(static::ST_UNSUBMITTED),
                    $this->sql_get_condition(static::ST_GRADE_NOT),
                    "($sql_deadline BETWEEN ($sql_now_12h+1) AND $sql_now_24h)",
                ];
                break;
            case static::ST_UNMARKED_DUE_IMPORTANT:
                $conds = [
                    $this->sql_get_condition(static::ST_UNSUBMITTED),
                    $this->sql_get_condition(static::ST_GRADE_NOT),
                    "($sql_deadline BETWEEN 1 AND $sql_now_24h)",
                ];
                break;
            case static::ST_GM_POSSIBLE:
                $conds = [
                    $this->sql_get_condition(static::ST_SUBMITTED),
                    $this->sql_get_condition(static::ST_UNSUBMITTED_MISSED_DEADLINE),
                ];
                $use_or = true;
                break;
            case static::DEADLINE_PAST:
                return "($sql_deadline BETWEEN 1 AND $sql_now)";
            case static::DEADLINE_PAST_GRACE:
                return "($sql_deadline_grace BETWEEN 1 AND $sql_now)";
            case static::DEADLINE_NOTSET:
                return "$sql_deadline = 0";
            case static::DEADLINE_UPCOMING:
                return "($sql_deadline > $sql_now)";
        }

        if (!empty($conds)){
            $sep = $use_or ? ' OR ' : ' AND ';
            return join($sep, $conds);
        }
        return null;
    }

    /**
     * Return processed condition by there name or empty string
     * @param $name
     *
     * @return array|string
     */
    public function sql_get_condition($name){
        if (!is_null($ans = $this->_call_fun_list($name, 'sql_get_condition'))){
            return $ans;
        } else {
            $cond = $this->sql_get_condition_raw($name);
            return empty($cond) ? '' : "($cond)";
        }
    }

    /**
     * Return $select as sum by condition name
     *
     * @param      $name
     * @param null $nickname - name by default, set to FALSE if you don't need "AS $nickname" here
     *
     * @return array|string
     */
    public function sql_get_select_sum($name, $nickname=null){
        $nickname = $nickname ?? $name;
        if (!is_null($ans = $this->_call_fun_list($name, 'sql_get_select_sum', true, [$nickname]))){
            return $ans;
        } else {
            $cond = $this->sql_get_condition($name);
            if (empty($cond)){
                return '';
            }
            return "SUM($cond)" . ($nickname ? " AS $nickname" : '');
        }
    }

    /**
     * Return SQL to join course_module tables
     *
     * @return string
     */
    public function sql_get_join_cm(){
        $table = $this->get_mod_type();
        $t = static::SQL_TABLE_ALIAS ?? $table[0];
        return "
        JOIN {modules} m
            ON m.name = :table
        JOIN {course_modules} cm
            ON cm.course = $t.course
            AND cm.instance = $t.id 
            AND cm.module = m.id
        JOIN {course_sections} cs
            ON cs.course = cm.course
            AND cs.id = cm.section
            
        JOIN {user_enrolments} ue
            ON ue.userid = u.id
        JOIN {enrol} e
            ON e.id = ue.enrolid
            AND e.courseid = cm.course
            
        LEFT JOIN {role} r 
            ON r.shortname = :student_role
            
        LEFT JOIN {context} ctx_course
            ON ctx_course.contextlevel = :ctx_course
            AND ctx_course.instanceid = e.courseid
        LEFT JOIN {role_assignments} ra_course
            ON ra_course.roleid = r.id
            AND ra_course.contextid = ctx_course.id
            AND ra_course.userid = ue.userid
            
        LEFT JOIN {context} ctx_cm
            ON ctx_cm.contextlevel = :ctx_module
            AND ctx_cm.instanceid = cm.id
        LEFT JOIN {role_assignments} ra_cm
            ON ra_cm.roleid = r.id
            AND ra_cm.contextid = ctx_cm.id
            AND ra_cm.userid = ue.userid
        ";
    }

    /**
     * Return SQL to join grades tables
     *
     * @return string
     */
    public function sql_get_join_grades(){
        if (!static::HAVE_GRADES) return "";

        $table = $this->get_mod_type();
        $t = static::SQL_TABLE_ALIAS ?? $table[0];
        return "
            LEFT JOIN {grade_items} gi
                ON gi.iteminstance = $t.id 
                AND gi.courseid = $t.course
                AND gi.itemmodule = '$table'
                AND gi.itemtype = 'mod'
                AND gi.itemnumber = ".static::SQL_GI_ITEMNUMBER."
            LEFT JOIN {grade_grades} gg
                ON gg.itemid = gi.id
                AND gg.userid = u.id
            LEFT JOIN {scale} sc
                ON sc.id = gi.scaleid
        ";
    }

    /**
     * Return full FROM and JOIN statements
     * Generally this method should not be overwritten by child classes.
     *
     * @return array - $from
     */
    public function sql_get_base_from(){
        return array_merge(
            $this->sql_get_base_from_prefix(),
            $this->sql_get_base_from_basis(),
            $this->sql_get_base_from_postfix()
        );
    }

    /**
     * Return basic FROM and JOIN statements by mod type
     * To be overwritten by child classes.
     *
     * @return array - $from
     */
    public function sql_get_base_from_basis(){
        return [];
    }

    /**
     * Return prefix general FROM and JOIN statements
     * Generally this method should not be overwritten by child classes.
     *
     * @return array - $from
     */
    public function sql_get_base_from_prefix(){
        $from = [];

        $table = $this->get_mod_type();
        $t = static::SQL_TABLE_ALIAS ?? $table[0];
        $join_cm = $this->sql_get_join_cm();
        $join_grades = $this->sql_get_join_grades();
        $grade_controller_t = NED::$ned_grade_controller::TABLE;

        $from[] = "
            {user} u 
            JOIN {{$table}} $t 
                ON 1=1
            JOIN (SELECT :day_start AS day_start, :day_end AS day_end, :now AS now) d -- data table
                ON 1=1
            $join_cm
            LEFT JOIN {course_modules_completion} AS cmc
                ON cmc.coursemoduleid = cm.id
                AND cmc.userid = u.id
            LEFT JOIN {{$grade_controller_t}} AS ngc
                ON ngc.cmid = cm.id
                AND ngc.userid = u.id
            $join_grades
            ";

        if (static::HAVE_GRADES && $this->_has_kica){
            $ki_table = \local_kica\kica_item::TABLE;
            $kg_table = \local_kica\kica_grade::TABLE;
            $from[] = "
            LEFT JOIN {{$ki_table}} kgi
                ON kgi.itemid = gi.id
                AND kgi.courseid = cm.course
            LEFT JOIN {{$kg_table}} kgg
                ON kgg.itemid = kgi.id
                AND kgg.userid = u.id
                AND kgg.courseid = cm.course
            ";
        }

        if ($this->_filter_get(static::USE_GROUPS)){
            $from[] = "
            LEFT JOIN (
                SELECT g_m.userid, g_m.groupid, grps.courseid, grps.name
                FROM {groups_members} AS g_m
                JOIN {groups} AS grps
                    ON grps.id = g_m.groupid
                WHERE 1=1 ".
                $this->if_courseid(' AND grps.courseid = :courseid ').
                $this->get_student_condition('g_m.userid', true).
                "
                GROUP BY g_m.userid, g_m.groupid
            ) AS gr 
                ON gr.userid = u.id AND gr.courseid = cm.course
            ";
        }

        return $from;
    }

    /**
     * Return postfix general FROM and JOIN statements,
     *      which should be included after mod tables
     *
     * @return array - $from
     */
    public function sql_get_base_from_postfix(){
        $from = [];

        if (static::HAVE_GRADES) {
            // join table {grade_grades_history} as
            // ggh_last_ungraded_time, ggh_last_ungraded_time_fake, ggh_first_graded_time, ggh_first_graded_time_fake
            $from[] = NED::sql_get_first_last_grade_join(static::SQL_USER_TIMEMODIFIED);
        }

        return $from;
    }

    /**
     * Return base WHERE statements
     *
     * @return array
     */
    public function sql_get_base_where(){
        $table = $this->get_mod_type();
        $t = static::SQL_TABLE_ALIAS ?? $table[0];
        $params = [];
        $params[] = $this->get_student_condition();
        if ($this->_courseid){
            $params[] = "$t.course = :courseid";
        }

        $params2 = [
            "cm.visible = 1",
            "cm.deletioninprogress = 0",
        ];

        if ($this->_filter_empty(static::DONT_CHECK_STUDENT_ROLE)){
            if (!$this->_filter_empty(static::USE_COURSE_STUDENTS_ONLY)){
                $params2[] = "COALESCE(ra_course.id, 0) > 0";
            } elseif (!$this->_filter_empty(static::USE_CM_STUDENTS_ONLY)){
                $params2[] = "COALESCE(ra_cm.id, 0) > 0";
            } else {
                // default behavior, if there are no any changing filters
                $params2[] = "COALESCE(ra_course.id, ra_cm.id, 0) > 0";
            }
        }

        $params3 = [];
        if (static::HAVE_GRADES) {
            $params3 = [
                "ggh_last_ungraded_time_fake.id IS NULL",
                "ggh_first_graded_time_fake.id IS NULL",
            ];
        }

        return array_merge($params, $params2, $params3);
    }

    /**
     * Return base parameters for sql
     *
     * @return array
     */
    public function sql_get_base_params(){
        $table = $this->get_mod_type();
        $now = time();
        [$day_start, $day_end] = NED::get_day_start_end($now, NED::NED_TIMEZONE);

        $params = [
            // data table
            'day_start' => $day_start,
            'day_end' => $day_end,
            'now' => $now,
            // table & cm
            'table' => $table,
            'tiitemtype' => 'course_modules',
            'ticomponent' => 'core',
            // context
            'ctx_course' => CONTEXT_COURSE,
            'ctx_module' => CONTEXT_MODULE,
            // roles
            'student_role' => NED::ROLE_STUDENT,
            // tags
            'tag_summative' => NED::TAG_SUMMATIVE,
            'tag_formative' => NED::TAG_FORMATIVE,
            'tag_midterm' => NED::TAG_MIDTERM,
            'tag_term1' => NED::TAG_TERM1,
            'tag_term2' => NED::TAG_TERM2,
            'tag_final_exam' => NED::TAG_FINAL_EXAM,
        ];
        if ($this->_courseid){
            $params['courseid'] = $this->_courseid;
        }

        if (!empty($this->_studentids) && is_array($this->_studentids)){
            $i = 0;
            foreach ($this->_studentids as $studentid){
                $params['student_'.$i.'_id'] = $studentid;
                $i++;
            }
        }

        return $params;
    }

    /**
     * Return sql condition by user ids
     *
     * @param string $field
     * @param false  $add_and
     *
     * @return string
     *
     * @see sql_get_base_params
     */
    public function get_student_condition($field='u.id', $add_and=false){
        if (is_null($this->_student_condition)){
            if (empty($this->_studentids)){
                $this->_student_condition = ' = -1';
            } elseif ($this->_studentids === '*'){
                $this->_student_condition = false;
            } elseif (is_array($this->_studentids)) {
                $list = [];
                for ($i=0; $i<count($this->_studentids); $i++){
                    $list[] = ':student_'.$i.'_id';
                }
                if (count($list) == 1){
                    $this->_student_condition = " = ".reset($list);
                } else {
                    $this->_student_condition = " IN (".join(', ', $list).')';
                }
            } else {
                debugging('Wrong studentids value!');
                $this->_student_condition = false;
            }
        }

        if (!$this->_student_condition){
            $cond = '1=1';
        } else {
            $cond = $field.$this->_student_condition;
        }

        if ($add_and){
            $cond = "AND ".$cond;
        }

        return $cond;
    }

    /**
     * @param $text
     *
     * @return string
     */
    public function if_courseid($text){
        return empty($this->_courseid) ? '' : $text;
    }

    /**
     * Return string for getting deadline with grace period
     *
     * @return string
     */
    static public function sql_get_deadline_grace(){
        $sql_deadline = static::SQL_DEADLINE;
        $grace_period = NED::DEADLINE_PENALTY_GRACE_PERIOD ?: 0;
        return "IF($sql_deadline > 0, ($sql_deadline + $grace_period), 0)";
    }

    /**
     * Get SQL from sql parameters
     *
     * @param string|array $select
     * @param string|array $from
     * @param string|array $where
     * @param string|array $groupby
     * @param string|array $orderby
     * @param string|array $having
     * @param array        $params
     *
     * @return array - [$sql, $params]
     */
    static protected function _sql_get_sql_and_params($select, $from, $where=null, $groupby=null, $orderby=null, $having=null, $params=null){
            if (empty($select) || empty($from)) return ['', $params];

            $check = function($param, $keyword, $glue=',', $bkt=false){
                if (!$param){
                    $param = '';
                } else {
                    if(is_array($param)) {
                        $param = join($glue, $param);
                    }
                    if($bkt){
                        $param = "($param)";
                    }
                    $param = $keyword . $param;
                }
                return $param;
            };

            if (is_array($select)){
                $select2 = [];
                foreach ($select as $name => $item){
                    $val = $item;
                    if (!is_int($name)){
                        $val .= " AS $name";
                    }
                    $select2[] = $val;
                }
                $select = join(",\n", $select2);
            }
            $select = 'SELECT ' . $select;
            $from = 'FROM ' . (is_array($from) ? join("\n", $from) : $from);
            $where = $check($where, 'WHERE ', ") AND \n(", true);
            $groupby = $check($groupby, 'GROUP BY ');
            $orderby = $check($orderby, 'ORDER BY ');
            $having = $check($having, 'HAVING ', ") AND \n(", true);

            $sql = join("\n", [$select, $from, $where, $groupby, $having, $orderby]);
            $add_params = [];
            foreach ($params as $key => $value){
                $search = ':'.$key;
                $count = substr_count($sql, $search);
                if ($count > 1){
                    $offset = 0;
                    $search_len = strlen($search);
                    for ($i=0; $i < $count; $i++){
                        $pos = strpos($sql, $search, $offset);
                        if ($pos === false){
                            break;
                        }
                        // check, that next symbol after param number is not number (otherwise - it's part of param name)
                        if (!is_numeric($sql[$pos+$search_len-1])){
                            $sql = substr_replace($sql, $search . $i, $pos, $search_len);
                            $add_params[$key . $i] = $value;
                        }
                        $offset = $pos+$search_len;
                    }
                }
            }

            $params = array_merge($params, $add_params);

            return [$sql, $params];
    }

    /**
     * Count SQL records by sql parameters
     *
     * @param string|array|null $count_field
     * @param string|array      $from
     * @param string|array      $where
     * @param string|array      $having
     * @param array             $params
     *
     * @return int
     */
    static public function sql_count_records($count_field=null, $from=null, $where=null, $having=null, $params=null){
        $select = "COUNT(1)";
        if (!empty($count_field)){
            if (is_array($count_field)){
                $count_field = reset($count_field);
            }
            [$count_field, ] = explode(" AS ", $count_field, 2);
            $select = "COUNT(DISTINCT $count_field)";
        }

        [$sql, $params] = static::_sql_get_sql_and_params($select, $from, $where, null, null, $having, $params);
        if (empty($sql)) return 0;

        return static::_db()->count_records_sql($sql, $params);
    }

    /**
     * Execute SQL by sql parameters
     *
     * @param string|array $select
     * @param string|array $from
     * @param string|array $where
     * @param string|array $groupby
     * @param string|array $orderby
     * @param string|array $having
     * @param array        $params
     * @param int          $limitfrom
     * @param int          $limitnum
     *
     * @return array|null
     */
    static public function sql_get_execute($select, $from, $where=null, $groupby=null, $orderby=null, $having=null,
        $params=null, $limitfrom=0, $limitnum=0){
        [$sql, $params] = static::_sql_get_sql_and_params($select, $from, $where, $groupby, $orderby, $having, $params);
        if (empty($sql)) return null;

        return static::_db()->get_records_sql($sql, $params, $limitfrom, $limitnum);
    }

    /**
     * Add tag table if you need
     *
     * @param $select
     * @param $from
     * @param $where
     * @param $groupby
     * @param $orderby
     * @param $having
     * @param $params
     */
    protected function _check_and_join_tag_table(&$select, &$from, &$where, &$groupby, &$orderby, &$having, &$params){
        if (!$this->_filter_get(static::USE_TAGS)){
            return;
        }

        $db = static::_db();
        $check_tags_keys = [static::TAG_NAME_LIST_HAVE_ALL, static::TAG_NAME_LIST_HAVE_ANY, static::TAG_NAME_LIST_HAVE_NONE];
        $find_other_tags = '';
        foreach ($check_tags_keys as $check_tags_key){
            $check_tags = $this->_filter_get($check_tags_key);
            if (empty($check_tags)) continue;

            $check_tags = NED::val2arr($check_tags);
            [$sql_tags, $param_tags] = $db->get_in_or_equal($check_tags, SQL_PARAMS_NAMED, $check_tags_key.'_');
            $params = array_merge($params, $param_tags);

            switch ($check_tags_key){
                case static::TAG_NAME_LIST_HAVE_ALL:
                    $find_other_tags .= "\n\t\t\t\tCOUNT(DISTINCT IF(tg.rawname $sql_tags, tg.rawname, NULL)) AS $check_tags_key,";
                    $params['all_tags_must_have_count'] = count($check_tags);
                    $where[] = "tg.$check_tags_key = :all_tags_must_have_count";
                    break;
                case static::TAG_NAME_LIST_HAVE_ANY:
                    $find_other_tags .= "\n\t\t\t\tSUM(DISTINCT tg.rawname $sql_tags) AS $check_tags_key,";
                    $where[] = "tg.$check_tags_key > 0";
                    break;
                case static::TAG_NAME_LIST_HAVE_NONE:
                    $find_other_tags .= "\n\t\t\t\tSUM(DISTINCT tg.rawname $sql_tags) AS $check_tags_key,";
                    $where[] = "tg.$check_tags_key = 0";
                    break;
            }
        }

        $from[] = "
            LEFT JOIN (
                SELECT ti.itemid AS itemid,
                cm.course AS courseid,
                SUM(DISTINCT tg.rawname = :tag_summative) AS is_summative,
                SUM(DISTINCT tg.rawname = :tag_formative) AS is_formative,
                SUM(DISTINCT tg.rawname = :tag_midterm) AS is_midterm,
                SUM(DISTINCT tg.rawname = :tag_term1) AS is_term1,
                SUM(DISTINCT tg.rawname = :tag_term2) AS is_term2,
                SUM(DISTINCT tg.rawname = :tag_final_exam) AS is_final_exam, $find_other_tags
                SUM(DISTINCT tg.id IS NOT NULL) AS tag_count,
                GROUP_CONCAT(DISTINCT tg.rawname) AS tags,
                SUM(DISTINCT tg.id = :tagid) AS tag_filter
                
                FROM {tag} tg 
                LEFT JOIN {tag_instance} ti
                    ON ti.tagid = tg.id
                    AND ti.itemtype = :tiitemtype 
                    AND ti.component = :ticomponent
                LEFT JOIN {course_modules} cm
                    ON cm.id = ti.itemid
                ".$this->if_courseid("WHERE cm.course = :courseid")."
                GROUP BY ti.itemid
            ) AS tg
                ON tg.itemid = cm.id AND tg.courseid = cm.course
        ";
    }

    /**
     * Check data before main get_data method
     *  for child classes
     *
     * @param $select
     * @param $from
     * @param $where
     * @param $groupby
     * @param $orderby
     * @param $having
     * @param $params
     */
    protected function _precheck_data_params(&$select, &$from, &$where, &$groupby, &$orderby, &$having, &$params){
        return;
    }

    /**
     * Check data after main get_data method
     *  for child classes
     *
     * @param $select
     * @param $from
     * @param $where
     * @param $groupby
     * @param $orderby
     * @param $having
     * @param $params
     */
    protected function _postcheck_data_params(&$select, &$from, &$where, &$groupby, &$orderby, &$having, &$params){
        return;
    }

    /**
     * You can get data by filter, or use prepared sql parameters
     * @param      $filter
     * @param null $select
     * @param null $from
     * @param null $where
     * @param null $groupby
     * @param null $orderby
     * @param null $having
     * @param null $params
     * @param bool $return_before_execute
     *
     * @return array|mixed|null
     */
    public function get_data($filter, $select=null, $from=null, $where=null, $groupby=null, $orderby=null, $having=null, $params=null,
        $return_before_execute=false){

        $this->_filter_from_get_data = $filter;
        $f = function($key){ return $this->_filter_get($key); };
        $error = false;

        // some precheck for possible errors
        $all_or_sum = $f(static::ST_ALL) || $f(static::SUM);

        if ($f(static::HAVE_DEADLINE)){
            $error = (!$f(static::DEADLINE_NOTSET) && !$this::HAVE_DEADLINES) ||
                (!$f(static::DEADLINE_NOTSET) && !$f(static::DEADLINE_PAST) && !$f(static::DEADLINE_UPCOMING));
        }

        if (!$error && !$this::HAVE_DEADLINES){
            if ($f(static::DEADLINE_UPTO) || $f(static::DEADLINE_AFTER)){
                $error = true;
            }

            if (!$error && !$all_or_sum){
                foreach (static::SHOULD_HAVE_DEADLINE as $status_with_deadline){
                    if ($f($status_with_deadline)){
                        $error = true;
                        break;
                    }
                }
            }
        }

        // error -> pass
        if ($error){
            $this->_filter_from_get_data = null;
            return false;
        }

        [$select, $groupby, $orderby, $having] = NED::val2arr_multi(true, $select, $groupby, $orderby, $having);
        $from = $from ?? $this->sql_get_base_from();
        $where = $where ?? $this->sql_get_base_where();
        $params = $params ?? $this->sql_get_base_params();

        $this->_precheck_data_params($select, $from, $where, $groupby, $orderby, $having, $params);
        $this->_check_and_join_tag_table($select, $from, $where, $groupby, $orderby, $having, $params);

        $table = $this->get_mod_type();
        $t = static::SQL_TABLE_ALIAS ?? $table[0];
        $conds = [];
        $NGC = NED::$ned_grade_controller;

        $sql_submit = $this::SQL_SUBMIT_TIME;
        $sql_submit_coal = "COALESCE($sql_submit, 0)";
        $sql_deadline = $this::SQL_DEADLINE;
        $sql_deadline_grace = static::sql_get_deadline_grace();
        $sql_first_last_grade_time_coal = static::SQL_FIRST_LAST_GRADE_TIME_COAL;
        $sql_now = static::SQL_D_NOW;
        $midn_grade_cond = function($cond){ return "IF(".static::SQL_IF_GRADE_MIDN_HIDDEN.", NULL, $cond)"; };

        if ($f(static::ST_ALL)){
            foreach (static::MM_SELECT_STATUS as $cond){
                $conds[$cond] = $cond;
            }
        }

        foreach (static::MM_ALL_STATUSES as $cond){
            if ($cond == static::ST_ALL) continue;
            if ($f($cond))
                $conds[$cond] = $cond;
        }

        if ($f(static::BY_ACTIVITY)){
            if ($f(static::BY_USER)){
                /** @see \local_ned_controller\marking_manager\mm_data_by_activity_user::$uniqid */
                $select[] = "CONCAT(cm.id, '_', u.id) AS uniqid";
            }
            if (!$this->_courseid){
                /** @see \local_ned_controller\marking_manager\mm_data_by_activity::$courseid */
                $select[] = "cm.course AS courseid";
                $groupby[] = "cm.course";
                $orderby[] = "cm.course";
            }

            /**
             * @see \local_ned_controller\marking_manager\mm_data_by_activity::$id
             * @see \local_ned_controller\marking_manager\mm_data_by_activity::$cmid
             * @see \local_ned_controller\marking_manager\mm_data_by_activity::$pos
             * @see \local_ned_controller\marking_manager\mm_data_by_activity::$instance
             */
            $select[] = "cm.id";
            $select[] = "cm.id AS cmid, (cs.section*1000000 + FIND_IN_SET(cm.id, cs.sequence)) AS pos, cm.instance AS instance";
            /**
             * @see \local_ned_controller\marking_manager\mm_data_by_activity::$modid
             * @see \local_ned_controller\marking_manager\mm_data_by_activity::$name
             * @see \local_ned_controller\marking_manager\mm_data_by_activity::$modname
             * @see \local_ned_controller\marking_manager\mm_data_by_activity::$visible
             */
            $select[] = "$t.id AS modid, $t.name AS name, m.name AS modname, (cs.visible AND cm.visible) AS visible";
            /** @see \local_ned_controller\marking_manager\mm_data_by_activity::$activity_completion */
            $select[] = static::SQL_ACTIVITY_HAS_COMPLETION." AS activity_completion";
            if (static::HAVE_GRADES) {
                /** @see \local_ned_controller\marking_manager\mm_data_by_activity::$grademax */
                $select[] = "gi.grademax AS grademax";
            }
            /** @see \local_ned_controller\marking_manager\mm_data_by_activity::$require_only_submit */
            $select[] = $this::SQL_REQUIRE_ONLY_SUBMIT." as require_only_submit";
            /** @see \local_ned_controller\marking_manager\mm_data_by_activity::$is_summative */
            $select[] = "tg.is_summative AS is_summative";
            /** @see \local_ned_controller\marking_manager\mm_data_by_activity::$is_formative */
            $select[] = "tg.is_formative AS is_formative";
            /** @see \local_ned_controller\marking_manager\mm_data_by_activity::$is_midterm */
            $select[] = "tg.is_midterm AS is_midterm";
            /** @see \local_ned_controller\marking_manager\mm_data_by_activity::$is_term1 */
            $select[] = "tg.is_term1 AS is_term1";
            /** @see \local_ned_controller\marking_manager\mm_data_by_activity::$is_term2 */
            $select[] = "tg.is_term2 AS is_term2";
            /** @see \local_ned_controller\marking_manager\mm_data_by_activity::$is_final_exam */
            $select[] = "tg.is_final_exam AS is_final_exam";
            if ($f(static::GET_TAGS)){
                /** @see \local_ned_controller\marking_manager\mm_data_by_activity::$tags */
                $select[] = "tg.tags AS tags";
            }

            if ($f(static::GET_PROXY_ENABLED)){
                /** @see \local_ned_controller\marking_manager\mm_data_by_activity::$has_proxy */
                if ($this->get_mod_type() == static::MOD_ASSIGN){
                    $pp_select = [];
                    foreach (NED::PROXY_DEADLINE_PLUGINS as $pp_name){
                        $pr = 'apc_'.$pp_name;
                        $from[] = "
                            LEFT JOIN {assign_plugin_config} AS $pr
                                ON $pr.assignment = $t.id 
                                AND $pr.plugin = '$pp_name'
                                AND $pr.name = 'enabled'
                                AND $pr.subtype = 'assignsubmission'
                                AND $pr.value = 1 
                        ";
                        $pp_select[] = $pr.'.value';
                    }
                    $select[] = 'COALESCE('.join(', ', $pp_select). ", 0) AS has_proxy";
                } else {
                    $select[] = "0 AS has_proxy";
                }
            }

            $groupby[] = "cm.id";
            $orderby[] = "pos";
        }

        $activity_id = $f(static::ACTIVITY_ID);
        if (!is_null($activity_id)){
            $where[] = "$t.id = :".static::ACTIVITY_ID;
            $params[static::ACTIVITY_ID] = $activity_id;
        }

        $coursemodule_id = $f(static::CMIDS);
        if (!is_null($coursemodule_id)){
            if (empty($coursemodule_id)){
                $where[] = "cm.id = ".NED::SQL_NONE_COND;
            } else {
                NED::sql_add_get_in_or_equal_options('cm.id', $coursemodule_id, $where, $params, 'mm_'.static::CMIDS);
            }
        }

        if ($f(static::BY_USER)){
            /** @see \local_ned_controller\marking_manager\mm_data_by_user::$userid */
            $select[] = "u.id AS userid";
            /** @see \local_ned_controller\marking_manager\mm_data_by_user::$username */
            $select[] = "CONCAT(u.firstname, ' ', u.lastname) AS username";
            if ($f(static::USE_GROUPS)){
                /** @see \local_ned_controller\marking_manager\mm_data_by_user::$groupid */
                $select[] = 'COALESCE(gr.groupid, 0) AS groupid';
            }
            if (static::HAVE_GRADES) {
                /** @see \local_ned_controller\marking_manager\mm_data_by_user::$shouldpass */
                $select[] = $this::SQL_SHOULDPASS . " AS shouldpass";
                /** @see \local_ned_controller\marking_manager\mm_data_by_user::$gt_shouldpass */
                $select[] = $this::SQL_GT_SHOULDPASS . " AS gt_shouldpass";
                /** @see \local_ned_controller\marking_manager\mm_data_by_user::$excluded */
                $select[] = "gg.excluded";
                /** @see \local_ned_controller\marking_manager\mm_data_by_user::$graderid */
                $select[] = "IF(gg.usermodified < 0, 0, gg.usermodified) AS graderid";

                /**
                 * Based on the {@see \grade_grade::get_dategraded()}
                 * It's more like upgraded gg.timemodified, than really grade time - see original function or MDL-31379
                 *
                 * @see \local_ned_controller\marking_manager\mm_data_by_user::$grade_time
                 */
                $select[] = static::SQL_GRADE_TIME." AS grade_time";

                /** @see \local_ned_controller\marking_manager\mm_data_by_user::$gg_timemodified */
                $select[] = "gg.timemodified AS gg_timemodified";

                /** @see \local_ned_controller\marking_manager\mm_data_by_user::$first_last_grade_time */
                $select[] = static::SQL_FIRST_LAST_GRADE_TIME." AS first_last_grade_time";
                $hide_grades_before_midn = $f(static::HIDE_GRADES_BEFORE_MIDN);
                if ($hide_grades_before_midn){
                    /** @see \local_ned_controller\marking_manager\mm_data_by_user::$completed_any */
                    $select[] = $this::SQL_USER_ANY_COMPLETED_MIDN." AS completed_any";
                    /** @see \local_ned_controller\marking_manager\mm_data_by_user::$completed */
                    $select[] = $this::SQL_COMPLETE_GRADE_MIDN." AS completed";
                    /** @see \local_ned_controller\marking_manager\mm_data_by_user::$completed_successfully */
                    $select[] = $this::SQL_COMPLETED_SUCCESSFULLY_MIDN . " AS completed_successfully";
                    /** @see \local_ned_controller\marking_manager\mm_data_by_user::$completed_grade_successfully */
                    $select[] = $this::SQL_COMPLETED_GRADE_SUCCESSFULLY_MIDN." AS completed_grade_successfully";

                    /** @see \local_ned_controller\marking_manager\mm_data_by_user::$gradepass */
                    $select[] = $midn_grade_cond("gi.gradepass")." AS gradepass";
                    /** @see \local_ned_controller\marking_manager\mm_data_by_user::$finalgrade */
                    $select[] = $this::SQL_RAWGRADE_MIDN." AS finalgrade";
                    /** @see \local_ned_controller\marking_manager\mm_data_by_user::$overridden */
                    $select[] = $midn_grade_cond("gg.overridden")." AS overridden";

                    if ($f(static::GET_GRADE_COMMENT)){
                        /** @see \local_ned_controller\marking_manager\mm_data_by_user::$grade_feedback */
                        $select[] = $midn_grade_cond(static::SQL_GRADE_FEEDBACK)." AS grade_feedback";
                    }
                } else {
                    /** @see \local_ned_controller\marking_manager\mm_data_by_user::$completed_any */
                    $select[] = $this::SQL_USER_ANY_COMPLETED." AS completed_any";
                    /** @see \local_ned_controller\marking_manager\mm_data_by_user::$completed */
                    $select[] = $this::SQL_COMPLETE_GRADE." AS completed";
                    /** @see \local_ned_controller\marking_manager\mm_data_by_user::$completed_successfully */
                    $select[] = $this::SQL_COMPLETED_SUCCESSFULLY . " AS completed_successfully";
                    /** @see \local_ned_controller\marking_manager\mm_data_by_user::$completed_grade_successfully */
                    $select[] = $this::SQL_COMPLETED_GRADE_SUCCESSFULLY." AS completed_grade_successfully";

                    /** @see \local_ned_controller\marking_manager\mm_data_by_user::$gradepass */
                    $select[] = "gi.gradepass";
                    /** @see \local_ned_controller\marking_manager\mm_data_by_user::$finalgrade */
                    $select[] = $this::SQL_RAWGRADE." AS finalgrade";
                    /** @see \local_ned_controller\marking_manager\mm_data_by_user::$overridden */
                    $select[] = "gg.overridden";
                    if ($f(static::GET_GRADE_COMMENT)){
                        /** @see \local_ned_controller\marking_manager\mm_data_by_user::$grade_feedback */
                        $select[] = static::SQL_GRADE_FEEDBACK." AS grade_feedback";
                    }
                }

                if ($f(static::GET_GRADE_COMMENT)){
                    /** @see \local_ned_controller\marking_manager\mm_data_by_user::$grade_feedback_format */
                    $select[] = static::SQL_GRADE_FEEDBACK_FORMAT." AS grade_feedback_format";
                }

                // Filter by grade value
                $grade_filters = [
                    static::F_GRADE_EQ => NED::SQL_COND_EQ,
                    static::F_GRADE_GTE => NED::SQL_COND_GTE,
                    static::F_GRADE_LTE => NED::SQL_COND_LTE,
                ];
                foreach ($grade_filters as $gr_filter_key => $cond){
                    $grade2check = $f($gr_filter_key);
                    if (!is_numeric($grade2check)) continue;

                    $add_where = '';
                    if (is_int(+$grade2check)){
                        $add_where .= 'ROUND';
                    }

                    $add_where .= "(".($hide_grades_before_midn ? $this::SQL_RAWGRADE_MIDN : $this::SQL_RAWGRADE).") ";
                    $add_where .= $cond." :".$gr_filter_key;

                    $where[] = $add_where;
                    $params[$gr_filter_key] = $grade2check;
                }
            } else {
                /** @see \local_ned_controller\marking_manager\mm_data_by_user::$completed */
                $select[] = $this::SQL_COMPLETE." AS completed";
                $select[] = "null gradepass, null AS finalgrade, 0 overridden, 0 excluded, null grade_time";
            }

            /** @see \local_ned_controller\marking_manager\mm_data_by_user::$attempt */
            $select[] = $this::SQL_ATTEMPT. " AS attempt";
            if (!empty($this::SQL_SUBMIT_TIME)){
                /** @see \local_ned_controller\marking_manager\mm_data_by_user::$submit_time */
                $select[] = $sql_submit." AS submit_time";
            }

            // from ned_grade_controller table
            if (static::HAVE_GRADES && $f(static::HIDE_GRADES_BEFORE_MIDN)){
                /** @see \local_ned_controller\marking_manager\mm_data_by_user::$ngc_record */
                $select[] = $midn_grade_cond("ngc.id")." AS ngc_record";
            } else {
                /** @see \local_ned_controller\marking_manager\mm_data_by_user::$ngc_record */
                $select[] = "ngc.id AS ngc_record";
            }
            /** @see \local_ned_controller\marking_manager\mm_data_by_user::$ngc_grade_type */
            $select[] = "ngc.grade_type AS ngc_grade_type";
            /** @see \local_ned_controller\marking_manager\mm_data_by_user::$ngc_grade_change */
            $select[] = "ngc.grade_change AS ngc_grade_change";
            /** @see \local_ned_controller\marking_manager\mm_data_by_user::$ngc_reason */
            $select[] = "ngc.reason AS ngc_reason";
            /** @see \local_ned_controller\marking_manager\mm_data_by_user::$ngc_status */
            $select[] = "ngc.status AS ngc_status";
            /** @see \local_ned_controller\marking_manager\mm_data_by_user::$ngc_relatedid */
            $select[] = "ngc.relatedid AS ngc_relatedid";

            if ($f(static::USE_DEF_STUDENTS_ONLY)){
                [$default_role_field_id, $default_role_default_value] = NED::role_get_default_role_data();
                if (!empty($default_role_field_id)){
                    $from[] = "LEFT JOIN {".NED::TABLE_USER_INFO_DATA."} AS uid ON uid.userid = u.id AND uid.fieldid = :default_role_field_id";
                    $def_r_where = ["uid.data = :default_role_student"];
                    if ($default_role_default_value == NED::DEFAULT_ROLE_STUDENT){
                        $def_r_where[] = "uid.data IS NULL";
                    }
                    $where[] = NED::sql_condition($def_r_where, "OR");

                    $params['default_role_field_id'] = $default_role_field_id;
                    $params['default_role_student'] = NED::DEFAULT_ROLE_STUDENT;
                }
            }

            if ($f(static::ONLY_NGC)){
                $where[] = "ngc.id IS NOT NULL";
            } elseif ($f(static::ONLY_NOT_NGC)){
                $where[] = "ngc.id IS NULL";
            }

            if ($f(static::NGC_MISSED_DEADLINE)){
                if (empty($this::SQL_SUBMIT_TIME) || !$this::HAVE_DEADLINES){
                    $where[] = NED::SQL_NONE_COND;
                } else {
                    $where1 = [];
                    $where1[] = static::sql_get_condition(static::ST_UNMARKED_MISSED_DEADLINE_GRACE);
                    $where1[] = "ngc.id IS NULL";

                    $where2 = [];
                    $where2[] = static::sql_get_condition(static::ST_UNSUBMITTED_MISSED_DEADLINE_GRACE);
                    $where2[] = "NOT COALESCE(".static::sql_get_condition(static::ST_DRAFT).", 0)";
                    $where2[] = "ngc.id IS NOT NULL";
                    $where2[] = "ngc.reason = :ngc_reason_file";
                    $params['ngc_reason_file'] = NED::$ned_grade_controller::REASON_FILE;
                    $where[] = NED::sql_condition([NED::sql_condition($where1), NED::sql_condition($where2)], 'OR');

                    if ($this->get_mod_type() == static::MOD_ASSIGN){
                        $where[] = 'COALESCE('.static::SQL_ATTEMPT.', 0) = 0';
                    }
                }
            } elseif ($f(static::NGC_LATE_SUBMISSIONS)){
                if (empty($this::SQL_SUBMIT_TIME) || !$this::HAVE_DEADLINES){
                    $where[] = NED::SQL_NONE_COND;
                } else {
                    $where[] = "$sql_submit > 0 AND $sql_deadline_grace > 0 AND $sql_submit > $sql_deadline_grace";
                    $where[] = "(ngc.reason = :ngc_reason_file OR (ngc.grade_type = :ngc_grade_zero AND ngc.reason = :ngc_reason_submission))";
                    $params['ngc_grade_zero'] = $NGC::GT_AWARD_ZERO;
                    $params['ngc_reason_file'] = $NGC::REASON_FILE;
                    $params['ngc_reason_submission'] = $NGC::REASON_SUBMISSION;
                    $where[] = "(ngc.authorid = 0 OR $sql_submit > ngc.timecreated)";
                    $where[] = "ngc.status = :ngc_status";
                    $params['ngc_status'] = $NGC::ST_DONE;

                    if ($this->get_mod_type() == static::MOD_ASSIGN){
                        $where[] = 'COALESCE('.static::SQL_ATTEMPT.', 0) = 0';
                    }
                }
            } elseif ($f(static::NGC_FIXED_SUBMISSIONS)) {
                $where[] = "$sql_submit > 0";
                $where[] = "ngc.grade_type = :ngc_grade_type";
                $params['ngc_grade_type'] = $NGC::GT_AWARD_ZERO;
                $where[] = "ngc.reason = :ngc_reason";
                $params['ngc_reason'] = $NGC::REASON_FILE;
                $where[] = "$sql_submit > ngc.timecreated";
                $where[] = "ngc.status = :ngc_status";
                $params['ngc_status'] = $NGC::ST_DONE;
            }

            $groupby[] = "u.id";
            $orderby[] = "u.id";
        }

        if (static::HAVE_GRADES && $f(static::ONLY_KICA) && $this->_has_kica){
            $where[] = "kgi.id IS NOT NULL";
        }

        if (static::HAVE_GRADES && $f(static::USE_KICA) && $this->_has_kica){
            /** @see \local_ned_controller\marking_manager\mm_data_common::$kica_activity */
            $select[] = "SUM(kgi.id IS NOT NULL) AS kica_activity";
            if ($f(static::BY_USER)){
                if ($f(static::HIDE_GRADES_BEFORE_MIDN)){
                    /**
                     * @see \local_ned_controller\marking_manager\mm_data_by_user::$knowledge
                     * @see \local_ned_controller\marking_manager\mm_data_by_user::$inquiry
                     * @see \local_ned_controller\marking_manager\mm_data_by_user::$communication
                     * @see \local_ned_controller\marking_manager\mm_data_by_user::$application
                     */
                    $select[] = $midn_grade_cond("kgg.knowledge").       " AS knowledge";
                    $select[] = $midn_grade_cond("kgg.inquiry").         " AS inquiry";
                    $select[] = $midn_grade_cond("kgg.communication").   " AS communication";
                    $select[] = $midn_grade_cond("kgg.application").     " AS application";
                    /** @see \local_ned_controller\marking_manager\mm_data_by_user::$kica_zerograde */
                    $select[] = $midn_grade_cond("(
                        (kgg.knowledge = 0 OR COALESCE(kgi.knowledge, 0) = 0) AND 
                        (kgg.inquiry = 0 OR COALESCE(kgi.inquiry, 0) = 0) AND 
                        (kgg.communication = 0 OR COALESCE(kgi.communication, 0) = 0) AND 
                        (kgg.application = 0 OR COALESCE(kgi.application, 0) = 0)
                    )")." AS kica_zerograde";
                } else {
                    /**
                     * @see \local_ned_controller\marking_manager\mm_data_by_user::$knowledge
                     * @see \local_ned_controller\marking_manager\mm_data_by_user::$inquiry
                     * @see \local_ned_controller\marking_manager\mm_data_by_user::$communication
                     * @see \local_ned_controller\marking_manager\mm_data_by_user::$application
                     */
                    $select[] = "kgg.knowledge, kgg.inquiry, kgg.communication, kgg.application";
                    /** @see \local_ned_controller\marking_manager\mm_data_by_user::$kica_zerograde */
                    $select[] = "(
                        (kgg.knowledge = 0 OR COALESCE(kgi.knowledge, 0) = 0) AND 
                        (kgg.inquiry = 0 OR COALESCE(kgi.inquiry, 0) = 0) AND 
                        (kgg.communication = 0 OR COALESCE(kgi.communication, 0) = 0) AND 
                        (kgg.application = 0 OR COALESCE(kgi.application, 0) = 0)
                    ) AS kica_zerograde";
                }
            }
        }

        if (static::HAVE_GRADES){
            if ($f(static::ONLY_EXCLUDED)){
                $where[] = "gg.excluded > 0";
            } elseif ($f(static::NOT_EXCLUDED)){
                $where[] = "COALESCE(gg.excluded, 0) = 0";
            }
        }

        if ($f(static::GM_SHOULDPASS)){
            $where[] = static::SQL_GT_SHOULDPASS;
        }

        $gt_startdate = $f(static::GM_STARTDATE);
        if ($gt_startdate){
            if (!empty($this::SQL_SUBMIT_TIME) && $this::HAVE_DEADLINES){
                $where[] = "GREATEST($sql_submit_coal, $sql_first_last_grade_time_coal, ".
                    "IF($sql_submit_coal = 0 AND $sql_first_last_grade_time_coal = 0, $sql_deadline, 0)) >= :gt_startdate";
            } elseif (!empty($this::SQL_SUBMIT_TIME) && !$this::HAVE_DEADLINES){
                $where[] = "GREATEST($sql_submit_coal, $sql_first_last_grade_time_coal) >= :gt_startdate";
            } elseif (empty($this::SQL_SUBMIT_TIME) && $this::HAVE_DEADLINES){
                $where[] = "GREATEST($sql_first_last_grade_time_coal, ".
                    "IF($sql_first_last_grade_time_coal = 0, $sql_deadline, 0)) >= :gt_startdate";
            } else {
                // empty($this::SQL_SUBMIT_TIME) && !$this::HAVE_DEADLINES
                $where[] = $sql_first_last_grade_time_coal." >= :gt_startdate";
            }

            $params['gt_startdate'] = $gt_startdate;
        }

        if ($f(static::GM_NGC_FILTER)){
            $where[] = "(ngc.id IS NULL OR ".
                "NOT(ngc.grade_type = :gm_ngc_grade_zero AND ngc.reason = :gm_ngc_reason_submission AND ngc.status = :gm_ngc_status_done))";
            $params['gm_ngc_grade_zero'] = $NGC::GT_AWARD_ZERO;
            $params['gm_ngc_reason_submission'] = $NGC::REASON_SUBMISSION;
            $params['gm_ngc_status_done'] = $NGC::ST_DONE;
        }

        if ($f(static::USE_GM) && NED::get_GT()){
            $gt_table = NED::get_GT()::TABLE;
            $sql_gt = "LEFT JOIN {{$gt_table}} AS gt
                ON gt.courseid = cm.course AND gt.cmid = cm.id AND gt.userid = u.id";

            if ($f(static::USE_GM_ATTEMPT)){
                $sql_attempt = static::SQL_ATTEMPT;
                $sql_gt .= " AND ($sql_attempt IS NULL OR gt.attempt = $sql_attempt)";
            } else {
                $sql_gt .= " AND gt.latest = 1";
            }

            if ($f(static::ONLY_GM)){
                $where[] = "gt.id IS NOT NULL";
            } elseif ($f(static::ONLY_NOT_GM)){
                $where[] = "gt.id IS NULL";
            }

            $from[] = $sql_gt;

            if ($f(static::BY_USER)){
                /** @see \local_ned_controller\marking_manager\mm_data_by_user::$is_active */
                $select[] = "(
                    ue.status = :active AND e.status = :enabled AND 
                    ue.timestart <= $sql_now AND (ue.timeend = 0 OR ue.timeend > $sql_now) AND
                    u.suspended = 0 AND u.deleted = 0
                ) AS is_active";
                $params['enabled'] = ENROL_INSTANCE_ENABLED;
                $params['active'] = ENROL_USER_ACTIVE;
            }
        }

        if ($this::HAVE_DEADLINES){
            $deadline_cond = [];

            if ($f(static::USE_DEADLINE)){
                $sql_final_deadline = $this::SQL_FINAL_DEADLINE;
                /** @see \local_ned_controller\marking_manager\mm_data_common::$duedate */
                $select[] = "$sql_deadline AS duedate"; // after this submission will be late
                /** @see \local_ned_controller\marking_manager\mm_data_common::$cutoffdate */
                $select[] = "$sql_final_deadline AS cutoffdate"; // after this submission impossible
                if ($f(static::DEADLINE_UPTO) || $f(static::DEADLINE_AFTER)){
                    $where[] = "$sql_deadline > 0";
                    if ($deadline_upto = $f(static::DEADLINE_UPTO)){
                        $where[] = "$sql_deadline <= :deadline_upto";
                        $params['deadline_upto'] = $deadline_upto;
                    }
                    if ($deadline_after = $f(static::DEADLINE_AFTER)){
                        $where[] = "$sql_deadline >= :deadline_after";
                        $params['deadline_after'] = $deadline_after;
                    }
                }
            }

            if ($f(static::HAVE_DEADLINE)){
                foreach (static::VALUE_DEADLINE as $deadline){
                    if ($f($deadline)){
                        $deadline_cond[] = $this->sql_get_condition($deadline);
                    }
                }
                if (!empty($deadline_cond) && count($deadline_cond) < count(static::VALUE_DEADLINE)){
                    $where[] = join(' OR ', $deadline_cond);
                }

            }

            if (($f(static::USE_DEADLINE) || $f(static::HAVE_DEADLINE)) && count($deadline_cond) != 1){
                /**
                 * @see \local_ned_controller\marking_manager\mm_data_common::$deadline_notset
                 * @see \local_ned_controller\marking_manager\mm_data_common::$deadline_past
                 * @see \local_ned_controller\marking_manager\mm_data_common::$deadline_upcoming
                 */
                foreach (static::VALUE_DEADLINE as $deadline){
                    $select[] = $this->sql_get_select_sum($deadline);
                }
            }
        }

        foreach (static::COUNT_STATUS as $count_status => $status){
            /**
             * @see \local_ned_controller\marking_manager\mm_data_common::$count_waitingzero
             * @see \local_ned_controller\marking_manager\mm_data_common::$count_draft
             * @see \local_ned_controller\marking_manager\mm_data_common::$count_missed_deadline
             * @see \local_ned_controller\marking_manager\mm_data_common::$count_unmarked_missed_deadline
             */
            if ($f($count_status)){
                $select[] = $this->sql_get_select_sum($status, $count_status);
            }
        }

        $params['tagid'] = 0;
        $tagid = $f(static::CHECK_TAG);
        if ($tagid){
            if ($tagid == static::TAGS_NONE){
                $where[] = 'tg.tag_count IS NULL OR tg.tag_count = 0';
            } else {
                $where[] = 'tg.tag_filter > 0';
                $params['tagid'] = $tagid;
            }
        }

        // check group name
        $group_regexp = 'REGEXP ' . ($f(static::GROUP_REGEXP_BINARY) ? 'BINARY ' : '');
        if ($f(static::GROUP_REGEXP)){
            $where[] = "gr.groupid IS NOT NULL";
            $where[] = 'gr.name '.$group_regexp.'"'.$f(static::GROUP_REGEXP).'"';
        }
        if ($f(static::GROUP_NOT_REGEXP)){
            $where[] = '(gr.groupid IS NULL OR gr.name NOT '.$group_regexp.'"'.$f(static::GROUP_NOT_REGEXP).'")';
        }

        if ($f(static::CTA_ORDER)){
            $prior_order = [];
            if (static::HAVE_GRADES){
                $prior_order[] = "COALESCE(grade_time, 0) ASC";
            }
            if (!empty(static::SQL_SUBMIT_TIME)){
                $prior_order[] = "COALESCE(submit_time, 0) DESC";
            }
            if (!empty($prior_order)){
                $orderby = array_merge($prior_order, $orderby);
            }
        }

        /**
         * it should be last check, select ST_* as SUM or as separate values
         *
         * @see \local_ned_controller\marking_manager\mm_data_common::$marked
         * @see \local_ned_controller\marking_manager\mm_data_common::$unmarked
         * @see \local_ned_controller\marking_manager\mm_data_common::$submitted
         * @see \local_ned_controller\marking_manager\mm_data_common::$unsubmitted
         * @see \local_ned_controller\marking_manager\mm_data_common::$draft
         * @see \local_ned_controller\marking_manager\mm_data_common::$reopened
         * @see \local_ned_controller\marking_manager\mm_data_common::$missed_deadline
         * @see \local_ned_controller\marking_manager\mm_data_common::$unmarked_missed_deadline
         * @see \local_ned_controller\marking_manager\mm_data_common::$unmarked_due_12h
         * @see \local_ned_controller\marking_manager\mm_data_common::$unmarked_due_24h
         * @see \local_ned_controller\marking_manager\mm_data_common::$unmarked_due_important
         * @see \local_ned_controller\marking_manager\mm_data_common::$submitted_missed_deadline
         * @see \local_ned_controller\marking_manager\mm_data_common::$unsubmitted_missed_deadline
         * @see \local_ned_controller\marking_manager\mm_data_common::$waitingzero
         * @see \local_ned_controller\marking_manager\mm_data_common::$zerograde
         * @see \local_ned_controller\marking_manager\mm_data_common::$grade_less_50
         * @see \local_ned_controller\marking_manager\mm_data_common::$grade_not
         * @see \local_ned_controller\marking_manager\mm_data_common::$gt_possible
         */
        if ($all_or_sum){
            if (!empty($conds)){
                $select = array_merge($select, $this->sql_get_select_sum($conds, false));
                if ($f(static::NOT_ZERO)){
                    $having_cond = [];
                    foreach ($conds as $cond){
                        $having_cond[] =  $cond . ' > 0';
                    }
                    $having[] = join(' OR ', $having_cond);
                }
            }
        } else {
            $get_st = $f(static::GET_ST);
            if (!empty($get_st)){
                $add_select = [];
                $get_all = $get_st[static::ST_ALL] ?? false;
                foreach (static::MM_ALL_STATUSES as $cond){
                    if ($get_all || ($get_st[$cond] ?? false))
                        $add_select[$cond] = $cond;
                }
                $select = array_merge($select, $this->sql_get_condition($add_select));
            }

            /** @see \local_ned_controller\marking_manager\mm_data_by_user::$userid */
            $select[] = 'u.id AS userid';
            $where = array_merge($where, $this->sql_get_condition($conds));
        }

        $this->_postcheck_data_params($select, $from, $where, $groupby, $orderby, $having, $params);

        if ($return_before_execute){
            $data = [$select, $from, $where, $groupby, $orderby, $params];
        } else {
            $check_count = $f(static::GET_COUNT);
            if ($check_count){
                $this->_records_count += static::sql_count_records($select, $from, $where, $having, $params);
            }

            if (empty($select)){
                $data = null;
            } else {
                if ($check_count && empty($this->_records_count)){
                    $data = [];
                } else {
                    $limitfrom = $f(static::LIMIT_FROM) ?: 0;
                    $limitnum = $f(static::LIMIT_NUM) ?: 0;
                    if ($check_count){
                        $last = $this->_records_count - 1;
                        if ($last == 0 || !$limitnum){
                            $limitfrom = 0;
                        } else {
                            if ($limitfrom > $last){
                                $limitfrom = floor($last/$limitnum) * $limitnum;
                            }
                            $limitfrom = max(min($limitfrom, $last), 0);
                        }
                    }

                    $data = static::sql_get_execute($select, $from, $where, $groupby, $orderby, $having, $params, $limitfrom, $limitnum);
                }

                if ($f(static::SUM) && !$f(static::BY_ACTIVITY)){
                    $data = $data ? reset($data) : $data;
                }
            }
        }

        $this->_filter_from_get_data = null;
        return $data;
    }

    /**
     * @param int   $cm_id = null
     * @param array $set_filter = null
     *
     * @return array|null
     */
    public function get_activity_data($cm_id=null, $set_filter=null){
        if ((!is_null($cm_id) && $cm_id != NED::isset2($this->_mm->cm, 'id')) || (!is_null($set_filter) && $set_filter != $this->_mm->base_filter)){
            $filter = $this->_mm->get_activity_filter($cm_id, $set_filter);
            $this->_activity_data = $this->get_data($filter);
        } elseif (is_null($this->_activity_data)){
            $filter = $this->_mm->get_activity_filter();
            $this->_activity_data = $this->get_data($filter);
        }
        return $this->_activity_data;
    }

    /**
     * Check, has mod_type personal_method for render content by this $show_status
     *
     * @param null $show_status
     *
     * @return bool
     */
    static public function class_has_personal_method($show_status){
        if (!in_array($show_status, static::NON_PERSONAL_STATUSES)){
            return true;
        }
        return false;
    }

    /**
     * Check, has mod_type personal_method for render content by this $show_status
     *  can use previous $show_status
     *
     * @param null $show_status
     *
     * @return bool
     */
    public function has_personal_method($show_status=null){
        $show_status = !is_null($show_status) ? $show_status : $this->_mm->get_show_status();
        return $this::class_has_personal_method($show_status);
    }

    /**
     * @param null $show_status
     *
     * @return \Closure|string|null
     */
    protected function _get_personal_method($show_status=null){
        $show_status = $this->get_show_status($show_status);
        if(!$this::class_has_personal_method($show_status)){
            return null;
        }
        return '_render_activity_table';
    }

    /**
     * Render table by $cm & show
     *
     * @param null $show_status
     * @param null $cm
     * @param null $filter
     * @param null $base_url
     *
     * @return string|null
     */
    public function render_by_show($show_status=null, $cm=null, $filter=null, $base_url=null){
        $fun = $this->_get_personal_method($show_status);
        if ($fun){
            return $this->$fun($show_status, $cm, $filter, $base_url);
        }

        return null;
    }

    /**
     * @return string
     */
    static public function render_empty_data(){
        return \html_writer::div(NED::str('noselectedactivities'), 'no-selected');
    }

    /**
     * Return warning text, if something wrong, false otherwise
     *
     * @param null $cm
     * @param null $filter
     * @param bool $load_data
     * @param bool $use_js_sorting
     *
     * @return bool|string
     */
    protected function _prerender($cm=null, $filter=null, $load_data=true, $use_js_sorting=true){
        global $OUTPUT;
        $MM = $this->_mm;
        if ($MM->nousers){
            return $OUTPUT->box(NED::str('nostudents'), 'box generalbox generalboxcontent boxaligncenter', 'intro');
        }

        $cm = $MM->get_cm($cm);
        if (empty($cm->id ?? 0)){
            return $this::render_empty_data();
        }

        if ($load_data){
            $MM->get_activity_filter($cm->id, $filter);

            $mm_data = $this->get_activity_data();
            if (empty($mm_data)){
                return $this::render_empty_data();
            }
        }

        if ($use_js_sorting){
            NED::js_call_amd('add_sorter', 'add_sort', ['table.mm-table']);
            NED::js_call_amd('marking_manager', 'init');
        }

        return false;
    }

    /**
     * Return DB data which useful for render activity table: activity, grade_item, use_kica (bool), grade_fun (\Closure)
     *
     * @return array [$activity, $grade_item, $use_kica, $grade_fun]
     */
    public function get_usual_cm_data(){
        static $data=[];

        $MM = $this->_mm;
        $cm = $MM->get_cm(null, true);
        if ($data[$cm->id] ?? false){
            return $data[$cm->id];
        }

        $activity = NED::get_module_instance_by_cm($cm);
        $grade_item = NED::get_grade_item_by_cm_course($cm, $cm->course);
        if ($grade_item){
            $scale = NED::db()->get_record('scale', ['id' => $grade_item->scaleid]);
        } else {
            $scale = null;
        }
        [$use_kica, $grade_fun] = static::get_grade_function($grade_item, NED::isset2($scale, 'scale', ''), $MM->kica);


        $data[$cm->id] = [$activity, $grade_item, $use_kica, $grade_fun];

        return $data[$cm->id];
    }

    /**
     * Get parameters to add manual_grading_link to the action menu
     *
     * @return array|bool
     */
    public function get_manual_grading_link_params(){
        $url = $this::URL_MANUAL_GRADING;
        if (empty($url)){
            return false;
        }
        $url_params=[]; $class='moodledefaultview'; $text=null; $link_attr=[];
        return [$url, $url_params, $class, $text, $link_attr];
    }

    /**
     * Return use_kica and grade format function (by final_grade)
     * @param \grade_item|\stdClass $grade_item
     * @param                       $scale
     * @param \stdClass             $kica
     *
     * @return array [bool, \Closure]
     */
    static public function get_grade_function($grade_item, $scale, $kica){
        $gi = $grade_item;
        $def = '-';
        $def_fun = function($finalgrade) use($def) { return $def; };

        $kica_coef = [];
        $kica_max = 0;

        if ($gi && NED::is_kica_exists() && $kica){
            $kicaitem = NED::ki_get_by_grade_item($gi);
            if (!empty($kicaitem->id)){
                foreach (NED::KICA_KEYS as $item) {
                    $k_section = $kicaitem->get_section($item);
                    $kica_coef[$item] = 0;
                    if ($k_section) {
                        $kica_coef[$item] = $kica->{$item} / $k_section;
                        $kica_max += $kica->{$item};
                    }
                }

                if (!$kica_max){
                    $f = function($grade){ return 0; };
                } else {
                    $f = function($grade) use ($kica_coef, $kica_max, $def) {
                        $kicagrade = null;
                        foreach ($kica_coef as $item => $coef){
                            if (!is_null($grade->{$item})){
                                $kicagrade += $grade->{$item} * $coef;
                            }
                        }
                        if (is_null($kicagrade)){
                            return $def;
                        }
                        $kicagrade = $kicagrade * 100 / $kica_max;
                        return round(grade_floatval($kicagrade), 2) . '%';
                    };
                }
                return [true, $f];
            }
        }

        if (!$gi || is_null($gi->gradetype) || $gi->gradetype == GRADE_TYPE_NONE || $gi->gradetype == GRADE_TYPE_TEXT){
            $f = $def_fun;
        } elseif ($gi->gradetype == GRADE_TYPE_SCALE) {
            if (!empty($scale)){
                $scales = explode(',', $scale);
                $f = function($finalgrade) use ($scales, $def){
                    $finalgrade = is_object($finalgrade) ? $finalgrade->finalgrade : $finalgrade;
                    return !empty($finalgrade) ? NED::isset2($scales, (int)($finalgrade-1), $def) : $def;
                };
            } else {
                $f = $def_fun;
            }
        } elseif ($gi->gradetype == GRADE_TYPE_VALUE) {
            if (!is_null($gi->grademax) && $gi->grademax > 0){
                $grademax = $gi->grademax;
                $f = function($finalgrade) use ($grademax, $def){
                    $finalgrade = is_object($finalgrade) ? $finalgrade->finalgrade : $finalgrade;
                    return !empty($finalgrade) ? round($finalgrade).'/'.round($grademax) : $def;
                };
            } else {
                $f = $def_fun;
            }
        } else {
            $f = $def_fun;
        }

        return [false, $f];
    }

    /**
     * Return $show_status or MM->_show_status
     *
     * @param null $show_status
     *
     * @return string|null
     */
    function get_show_status($show_status=null){
        return $show_status ?? $this->_mm->get_show_status();
    }

    /**
     * Return activity header as action menu
     *
     * @param \grade_item|\stdClass $grade_item
     * @param bool                  $use_kica
     * @param array                 $add_menu_list - you can use classname, after what element add menu link, as keys of array
     *
     * @return string
     */
    public function get_activity_header($grade_item, $use_kica=false, $add_menu_list=[]){
        $MM = $this->_mm;
        $cm = $MM->get_cm(null, true);
        $modname = get_string('modulename', $cm->modname);

        $activity_menu = \html_writer::span($modname, 'mod-name') . NED::mod_link($cm, 16);
        return \html_writer::div($activity_menu,'mm_mod_header') . $MM->render_mm_warnings();
    }

    /**
     * Return function(user) to getting username as action menu
     *
     * @return \Closure
     */
    public function get_student_function(){
        $MM = $this->_mm;
        $courseid = $MM->courseid;

        $fun = function($student) use ($courseid){
            return NED::O()->user_picture($student, ['link' => false, 'includefullname' => true, 'courseid' => $courseid]);
        };

        return $fun;
    }

    /**
     * Render table and its container from html_table
     *
     * @param \html_table $table
     * @param array       $add_class
     *
     * @return string
     */
    public function render_mm_table(\html_table $table, $add_class=[]){
        $add_class = is_array($add_class) ? $add_class : (array)$add_class;
        $add_class = array_merge(['mm-table', $this->get_mod_type()], $add_class);
        if ($table->attributes['class'] ?? false){
            $add_class[] = $table->attributes['class'];
        }
        $table->attributes['class'] = join(' ', $add_class);
        return \html_writer::div(\html_writer::table($table), 'mm-table-container');
    }

    /**
     * Render usual table by show & cm
     * @noinspection DuplicatedCode
     *
     * @param null $show_status
     * @param null $cm
     * @param null $filter
     * @param null $base_url
     *
     * @return string
     */
    protected function _render_activity_table($show_status=null, $cm=null, $filter=null, $base_url=null){
        $MM = $this->_mm;
        if ($warning = $this->_prerender($cm, $filter)){
            return $warning;
        }

        $mm_data = $this->get_activity_data();
        [$activity, $grade_item, $use_kica, $grade_fun] = $this->get_usual_cm_data();

        $st_data = [];
        $d = function($key, $def='') use(&$st_data) {
            return NED::isset2($st_data, $key, $def);
        };

        $show_status = $this->get_show_status($show_status);
        $add_status = $show_status == static::ST_ALL;
        $base_url = $MM->get_base_url($base_url);

        $o = $this->get_activity_header($grade_item, $use_kica);

        $table = new \html_table();
        $table->head = [get_string('name'), get_string('group')];
        if ($add_status){
            $table->head[] = get_string('status');
        }
        $table->head[] = NED::str('duedate');
        $table->head[] = NED::str('submitted');
        $table->head[] = get_string('grade_', 'local_ned_controller') . ($use_kica ? '*' : '');

        $st_menu = $this->get_student_function();
        $students = $MM->studuser ? [$MM->studuser] : $MM->students;
        foreach ($students as $student){
            $st_data = NED::isset2($mm_data, $student->id); // used by $d function
            if (!$st_data && $show_status != static::ST_ALL){
                continue;
            }
            $st_data = NED::stdClass2($st_data);
            $add_class = '';
            $cells = [];
            // username
            $cells[] = NED::cell($st_menu($student), 'username', ['data-sort-value' => fullname($student)]);
            // group name
            $groupsort = 'zzzzzzz';
            $grouplink = '-';
            if (isset($student->group->id)){
                $groupsort = $student->group->name;
                $base_url->param('group', $student->group->id);
                $grouplink = NED::link($base_url, $groupsort);
            }
            $cells[] = NED::cell($grouplink, 'groupname', ['data-sort-value' => $groupsort]);
            // status
            if ($add_status){
                $statuses = [];
                foreach (static::MM_SELECT_STATUS as $status){
                    if ($status == static::ST_ALL){
                        continue;
                    }
                    if ($d($status, false)){
                        $statuses[] = NED::str($status);
                    }
                }

                $statuses = join(', ', $statuses);
                $cells[] = NED::cell($statuses, 'status', ['data-sort-value' => $statuses]);
            }
            // duedate
            $duedate = $d('duedate', 0);
            $cells[] = NED::cell(NED::ned_date($duedate), 'duedate', ['data-sort-value' => $duedate]);
            // submit time
            $submit_time = $d('submit_time', 0);
            $cells[] = NED::cell(NED::ned_date($submit_time), 'submitted', ['data-sort-value' => $submit_time]);
            // grade
            $grade = $grade_fun($st_data);
            $cells[] = NED::cell($grade, 'user-grade', ['data-sort-value' => $grade == '-' ? -1 : $grade]);

            $table->data[] = NED::row($cells, $add_class);
        }

        $page_title = NED::str_check(["$show_status:pagetitle", $show_status]);
        $o .= \html_writer::tag('h4', $page_title, ['class' => 'mm_header mm_msg']) .
           $this->render_mm_table($table, $show_status);

        return $o;
    }
}
