<?php
/** @noinspection PhpUnusedParameterInspection */
/**
 * @package    local_ned_controller
 * @subpackage marking_manager
 * @category   NED
 * @copyright  2021 NED {@link http://ned.ca}
 * @author     NED {@link http://ned.ca}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace local_ned_controller\marking_manager;

use local_ned_controller\shared_lib as NED;

defined('MOODLE_INTERNAL') || die();

/**
 * Class marking_manager_quiz
 *
 * @package local_ned_controller\marking_manager
 */
class marking_manager_quiz extends marking_manager_mod
{
    const HAVE_DEADLINES = true;
    const SQL_USER_TIMEMODIFIED = 'qa.timefinish';
    const SQL_SUBMIT_TIME = "IF(qa.state = '".\quiz_attempt::FINISHED."', ".self::SQL_USER_TIMEMODIFIED.", 0)";
    const SQL_DEADLINE = 'COALESCE(qo_user.timeclose, qo_group.timeclose, q.timeclose, 0)';
    const SQL_FINAL_DEADLINE = self::SQL_DEADLINE;
    const SQL_ATTEMPT = 'IF(qa.attempt > 0, qa.attempt-1, qa.attempt)'; // with compatibility to the GT

    const NON_PERSONAL_STATUSES = [];

    const URL_QUIZ_REVIEW = '/mod/quiz/review.php';
    const URL_QUIZ_REVIEWQUESTION = '/mod/quiz/reviewquestion.php';
    const URL_MANUAL_GRADING = '/mod/quiz/report.php';

    /**
     * Return type of mod (static::MOD_*), and it's name of main DB-table
     * @return string
     */
    public function get_mod_type(){
        return static::MOD_QUIZ;
    }

    /**
     * Return condition by there name
     * @param string|array $name
     *
     * @return array|string
     */
    public function sql_get_condition_raw($name){
        if (!is_null($ans = parent::sql_get_condition_raw($name))){
            return $ans;
        }

        $sql_rawgrade = $this::SQL_RAWGRADE;
        $finished = "'" . \quiz_attempt::FINISHED . "'";

        $sql_not_overridden = $this::SQL_NOT_OVERRIDDEN;
        $sql_graded_by_overridden = $this::SQL_GRADED_BY_OVERRIDDEN;
        $sql_ungraded_by_overridden = $this::SQL_UNGRADED_BY_OVERRIDDEN;
        switch($name){
            case static::ST_UNSUBMITTED:
                return "qa.state IS NULL OR qa.state <> $finished";
            case static::ST_SUBMITTED:
                return "qa.state = $finished";
            case static::ST_UNMARKED:
                return "((q.grade > 0 AND qa.state = $finished AND mg.needsgrading > 0) AND $sql_not_overridden) OR $sql_ungraded_by_overridden".
                    $this->_add_filter_unmarked_cond();
            case static::ST_MARKED:
                return "((q.grade > 0 AND COALESCE(mg.needsgrading, 0) = 0 AND $sql_rawgrade IS NOT NULL) OR $sql_graded_by_overridden)".
                    $this->_add_filter_marked_cond();
            default:
                debugging("MM: Unknown condition '$name'");
                return NED::SQL_NONE_COND;
        }
    }

    /**
     * Return SQL for getting detail information about user progress in a quiz
     *
     * @return string
     */
    static public function sql_get_detail_join(){
        return "
        JOIN {quiz_slots} AS q_s
            ON q_s.quizid = q.id
        JOIN {quiz_attempts} AS qz_a
            ON qz_a.quiz = q.id
            AND qz_a.userid = u.id
        JOIN {question_attempts} AS qn_a
            ON qn_a.questionusageid = qz_a.uniqueid
            AND qn_a.slot = q_s.slot
        JOIN {question_attempt_steps} AS qn_as 
            ON qn_as.questionattemptid = qn_a.id
        LEFT JOIN {question_attempt_steps} AS qn_as2 
            ON qn_as2.questionattemptid = qn_a.id
            AND qn_as2.sequencenumber > qn_as.sequencenumber
            
        -- since Moodle 4
        -- -- Not-random questions
        LEFT JOIN {question_references} qre 
            ON qre.itemid = q_s.id 
            AND qre.usingcontextid = ctx_cm.id AND qre.component = 'mod_quiz' AND qre.questionarea = 'slot'
        LEFT JOIN {question_bank_entries} AS qbe 
            ON qbe.id = qre.questionbankentryid
        -- -- Random questions
        LEFT JOIN {question_set_references} qsr 
            ON qsr.itemid = q_s.id 
            AND qsr.usingcontextid = ctx_cm.id AND qsr.component = 'mod_quiz' AND qsr.questionarea = 'slot'
        LEFT JOIN {question_bank_entries} AS qbe_r
            ON qbe_r.questioncategoryid = trim(BOTH '\"' FROM JSON_EXTRACT(qsr.filtercondition, '$.questioncategoryid'))
               
        JOIN {question_versions} AS qve 
            ON qve.questionbankentryid = COALESCE(qbe.id, qbe_r.id)
        JOIN {question} AS qn 
            ON qn.id = qve.questionid    
        ";
    }

    /**
     * Return base parameters for sql
     *
     * @return array
     */
    public function sql_get_base_params(){
        $params = parent::sql_get_base_params();
        $cmid = $this->_filter_get(static::CMIDS);
        if (!is_null($cmid) && (!is_array($cmid) || count($cmid) == 1)){
            $params[static::CMIDS] = is_array($cmid) ? reset($cmid) : $cmid;
        }

        return $params;
    }

    /**
     * Return base FROM and JOIN statements
     *
     * @return array
     */
    public function sql_get_base_from_basis(){
        $from = [];

        $add_where = [];
        $add_join = [];
        $add_join[] = $this->sql_get_join_cm();

        $cmid = $this->_filter_get(static::CMIDS);
        if (!is_null($cmid) && (!is_array($cmid) || count($cmid) == 1)){
            $cmid = is_array($cmid) ? reset($cmid) : $cmid;
            $add_where[] = "cm.id = :".static::CMIDS;
        }

        $add_join = !empty($add_join) ? join('', $add_join) : '';
        $add_where = !empty($add_where) ? ' AND ' . join(' AND ', $add_where) : '';
        $detail_join = static::sql_get_detail_join();

        $from[] = "
        LEFT JOIN {quiz_attempts} qa
            ON qa.quiz = q.id
            AND qa.userid = u.id
        LEFT JOIN {quiz_attempts} qa2
            ON qa2.quiz = q.id
            AND qa2.userid = u.id
            AND qa2.attempt > qa.attempt
            
        LEFT JOIN (
            SELECT 
                q.id AS quizid, u.id AS userid,
                SUM(qn_as.state IN ('needsgrading', 'manfinished', 'mangaveup', 'mangrwrong', 'mangrpartial', 'mangrright')) AS manualgrading,
                SUM(qn_as.state = 'needsgrading') AS needsgrading
            FROM {user} AS u
            JOIN {quiz} AS q
                ON 1=1
            $add_join
            $detail_join
            WHERE 1=1
	        ".$this->if_courseid('AND q.course = :courseid').
            $this->get_student_condition('u.id', true).
            "$add_where
	            AND qn_as2.id IS NULL
	            AND q_s.maxmark > 0
	            AND qn.length > 0
	        GROUP BY q.id, u.id
        ) AS mg
            ON mg.quizid = q.id 
            AND mg.userid = u.id
        ";

        if ($this->_filter_get(static::QUIZ_DETAILED_VIEW)){
            $from[] = $detail_join;
        }

        if ($this->_filter_get(static::USE_DEADLINE)){
            $from[] = "
            LEFT JOIN {quiz_overrides} AS qo_group
                ON qo_group.quiz = q.id
                AND qo_group.groupid = gr.groupid
            LEFT JOIN {quiz_overrides} AS qo_user
                ON qo_user.quiz = q.id
                AND qo_user.userid = u.id
            ";
        }
        return $from;
    }

    /**
     * @param $select
     * @param $from
     * @param $where
     * @param $groupby
     * @param $orderby
     * @param $having
     * @param $params
     */
    protected function _postcheck_data_params(&$select, &$from, &$where, &$groupby, &$orderby, &$having, &$params){
        if ($this->_filter_get(static::QUIZ_DETAILED_VIEW)){
            $select_ = [];
            $select_[] = "CONCAT(u.id, '#', COALESCE(qz_a.attempt, 0), '#', COALESCE(q_s.slot, 0)) AS uniqueid";
            $select_[] = "qz_a.id AS attempt_id, qz_a.attempt AS attempt, qz_a.sumgrades";
            $select_[] = "qn_a.maxmark AS attempt_maxmark, qn_as.fraction, qn_as.state, qn_a.slot";
            $select = array_merge($select_, $select);
            $where[] = 'qn_as2.id IS NULL';
            $where[] = 'qn.length > 0';
            $where[] = 'q_s.maxmark > 0';
            $orderby = $groupby = ['u.id', 'qz_a.attempt', 'qn_a.slot'];
        }
        $where[] = 'qa2.id IS NULL';
    }

    /**
     * @param null $show_status
     *
     * @return \Closure|\string|null
     */
    protected function _get_personal_method($show_status=null){
        $show_status = $this->get_show_status($show_status);
        if ($show_status == static::ST_MARKED || $show_status == static::ST_UNMARKED){
            return '_render_mark_table';
        }

        return parent::_get_personal_method($show_status);
    }

    /**
     * Get quiz slots data from DB
     *
     * @param int $quizid
     * @param \context_module $ctx - if null, load context by quizid
     *
     * @return array
     */
    static public function get_quiz_slots($quizid, $ctx=null){
        global $DB;
        $sql = "
        SELECT q_s.*, qn.name
        FROM {quiz_slots} AS q_s
            
        -- since Moodle 4
        LEFT JOIN {question_references} qre 
            ON qre.itemid = q_s.id 
            AND qre.usingcontextid = :ctxid AND qre.component = 'mod_quiz' AND qre.questionarea = 'slot'
        LEFT JOIN {question_set_references} qsr 
            ON qre.itemid IS NULL AND qsr.itemid = q_s.id 
            AND qsr.usingcontextid = :ctxid AND qsr.component = 'mod_quiz' AND qsr.questionarea = 'slot'
        LEFT JOIN {question_bank_entries} AS qbe 
            ON (qre.id IS NOT NULL AND qbe.id = qre.questionbankentryid) 
        LEFT JOIN {question_bank_entries} AS qbe_r
            ON (qsr.id IS NOT NULL AND 
               qbe_r.questioncategoryid = trim(BOTH '\"' FROM JSON_EXTRACT(qsr.filtercondition, '$.questioncategoryid')))
        JOIN {question_versions} AS qve 
            ON qve.questionbankentryid = COALESCE(qbe.id, qbe_r.id)
        JOIN {question} AS qn 
            ON qn.id = qve.questionid    
            
        WHERE q_s.quizid = :quizid AND qn.length > 0
        ORDER BY q_s.slot
        ";

        $params = ['quizid' => $quizid];
        if (!$ctx){
            $quiz_obj = \quiz::create($quizid);
            $ctx = $quiz_obj->get_context();
        }
        $params['ctxid'] = $ctx->id;

        $slots_r = $DB->get_records_sql($sql, $params);
        $slots = [];
        if ($slots_r){
            $i = 1;
            foreach ($slots_r as $slot){
                $slot->i = $i++;
                if ($slot->maxmark > 0){
                    $slots[] = $slot;
                }
            }
        }
        return $slots;
    }

    /**
     * Get parameters to add manual_grading_link to the action menu
     *
     * @return array
     */
    public function get_manual_grading_link_params(){
        list($url, $url_params, $class, $text, $link_attr) = parent::get_manual_grading_link_params();
        $url_params['mode'] = 'grading';
        $class = 'openmanualgrading';
        return [$url, $url_params, $class, $text, $link_attr];
    }

    /**
     * Render table for marked and unmarked status
     *
     * @param null $show_status
     * @param null $cm
     * @param null $filter
     * @param null $base_url
     *
     * @return string
     * @noinspection PhpUnused
     */
    protected function _render_mark_table($show_status=null, $cm=null, $filter=null, $base_url=null){
        $MM = $this->_mm;
        $cm = $this->_mm->get_cm($cm);

        $filter = $MM->get_activity_filter(0, null, false, [static::QUIZ_DETAILED_VIEW]);
        if ($warning = $this->_prerender($cm, $filter)){
            return $warning;
        }

        $show_status = $this->get_show_status($show_status);
        $mm_data = $this->get_activity_data();
        [$quiz, $grade_item, $use_kica, $grade_fun] = $this->get_usual_cm_data();

        $sortf = function($val){
            return ['data-sort-value' => $val];
        };
        $check_grade = function($val){
            return $val === '-' ? -1 : $val;
        };
        $grade_pick = function($name, $val='', $can_edit=false){
            switch($name){
                case 'correct':
                    return NED::fa('fa-check correct', '', NED::str('correct')) . $val;
                case 'incorrect':
                    return NED::fa('fa-times incorrect', '', NED::str('incorrect')) . $val;
                case 'partiallycorrect':
                    return NED::fa('fa-check partiallycorrect', '', NED::str('partiallycorrect')) . $val;
                case 'needsgrading':
                    if ($can_edit){
                        return NED::fa('fa-pencil needsgrading', '', NED::str('edit'));
                    }
                    return NED::icon_get_nge_by_status(NED::STATUS_UNMARKED);
                default:
                    return $val > 0 ? NED::fa($name).$val : NED::fa("fa-minus minus $name");
            }
        };

        $o = $this->get_activity_header($grade_item, $use_kica);

        $data = []; // userid => [attempt => [slot => q_data]]
        foreach ($mm_data as $mm_d){
            $data[$mm_d->userid ?? 0][$mm_d->attempt ?? 0][$mm_d->slot ?? 0] = $mm_d;
        }

        $table = new \html_table();
        $table->head = [
            NED::str('student'),
            NED::str('gradeslink') . '<br>/' . round($quiz->grade)
        ];

        $q_coeff = $quiz->sumgrades != 0 ? ($quiz->grade / $quiz->sumgrades) : 0;
        $slots = $this::get_quiz_slots($quiz->id, $cm->context);
        if (!$slots){
            return $this::render_empty_data();
        }

        foreach ($slots as $slot){
            $table->head[] = NED::cell('Q.' . $slot->i . '<br>/' . round($slot->maxmark * $q_coeff),
                'quiz-name', ['title' => $slot->name]);
        }

        $st_menu = $this->get_student_function();
        $can_grade_mod = NED::cm_can_grade_cm($cm);
        $students = $MM->studuser ? [$MM->studuser] : $MM->students;
        foreach ($students as $student){
            $st_data = $data[$student->id] ?? false;
            if (empty($st_data)){
                continue;
            }

            $st_rows = [];
            // we checked attempts from last to first, so we can use last attempt data as sorting data for all rows
            $first_attempt = key($st_data);
            $last = true;
            $st_data = array_reverse($st_data, true);
            $st_params = $sortf(fullname($student));
            $cell_name = NED::cell($st_menu($student), 'username', $st_params);
            $cell_name->rowspan = count($st_data);

            $sort_st_attr = $sortf(-1);
            $sort_cells_attr_list = [];

            foreach ($st_data as $attempt => $at_data){
                $first = $first_attempt == $attempt;
                $d = reset($at_data);

                $row = NED::row();
                if ($first){
                    $row->cells[] = $cell_name;
                } else {
                    $row->cells[] = NED::cell('', 'username-hide', $st_params);
                }

                // SUMGRADES.
                $add_class = [];
                if ($d->attempt_id){
                    $title = NED::str('reviewattempt');
                    if (!$first || !$last){
                        $title .= ' №' . $attempt;
                    }
                    $attempt_url = new \moodle_url(static::URL_QUIZ_REVIEW, ['attempt' => $d->attempt_id]);
                    $add_class[] = 'reviewattempt';
                    if ($d->sumgrades) {
                        $attempt_value = $text = round($d->sumgrades * $q_coeff, 2);
                        $add_class[] = 'graded';
                    } else {
                        $text = NED::str('notyetgraded');
                        $attempt_value = -1;
                        $add_class[] = 'notyetgraded';
                    }
                } else {
                    $title = NED::str('moodlegradebook');
                    $text = $grade = $grade_fun($d);
                    $attempt_value = $check_grade($grade);
                    $add_class[] = 'finalgrade';
                    $attempt_url = new \moodle_url(static::URL_GRADE_SINGLEVIEW,
                        ['id' => $MM->courseid, 'item' => 'user', 'group' => $student->group->id ?? $MM->groupid, 'itemid' => $student->id]);
                }

                if ($last){
                    $sort_st_attr = $sortf($attempt_value);
                }

                // KICA Quiz Grader.
                $kicagraderbutton = '';
                if ($can_grade_mod && $d->attempt_id && method_exists('\local_kicaquizgrading\helper', 'render_kica_grade_button')){
                    $kicagraderbutton = \local_kicaquizgrading\helper::render_kica_grade_button($d->attempt_id);
                }

                $cellsumgardes = $text;
                if ($can_grade_mod){
                    $cellsumgardes = NED::link($attempt_url, $cellsumgardes, $add_class, ['title' => $title, 'target' => '_blank']);
                }
                $row->cells[] = NED::cell($cellsumgardes . ' ' . $kicagraderbutton, 'attemptgrade', $sort_st_attr);

                foreach ($slots as $slot){
                    /** @var \stdClass $sl_data */
                    $sl_data = $at_data[$slot->slot] ?? false;
                    if ($sl_data){
                        $step_grade = round($sl_data->fraction * $sl_data->attempt_maxmark * $q_coeff, 2);
                        $st_status = $sl_data->state;
                        $url = new \moodle_url(static::URL_QUIZ_REVIEWQUESTION, ['attempt' => $d->attempt_id, 'slot' => $slot->slot]);
                    } else {
                        $step_grade = '-';
                        $st_status = '-';
                        $url = $attempt_url;
                    }

                    $add_class = ['slot-cell'];
                    switch($st_status){
                        case 'gradedright':
                        case 'mangrright':
                            $status = 'correct';
                            break;
                        case 'gradedwrong':
                        case 'mangrwrong':
                            $status = 'incorrect';
                            break;
                        case 'mangrpartial':
                        case 'gradedpartial':
                            $status = 'partiallycorrect';
                            break;
                        case 'needsgrading':
                            $status = 'needsgrading';
                            $add_class[] = 'fn-highlighted';
                            break;
                        default:
                            $status = $st_status;
                            if (!$step_grade){
                                $step_grade = -1;
                            }
                    }

                    if ($last){
                        $sort_cells_attr_list[$slot->slot] = $sortf($check_grade($step_grade));
                    } elseif (!isset($sort_cells_attr_list[$slot->slot])){
                        $sort_cells_attr_list[$slot->slot] = $sortf(-1);
                    }

                    $cell_attr = $sort_cells_attr_list[$slot->slot];
                    $cell_attr['title'] = NED::str('reviewresponse');
                    $content = $grade_pick($status, $step_grade, $can_grade_mod);
                    if ($can_grade_mod){
                        $content = NED::link($url, $content, '', ['target' => '_blank']);
                    }
                    $row->cells[] = NED::cell($content, $add_class, $cell_attr);
                }
                $last = false;
                $st_rows[] = $row;
            }
            $table->data = array_merge($table->data, array_reverse($st_rows));

        }

        $o .= $this->render_mm_table($table, $show_status);
        return $o;
    }
}
