<?php
/**
 * Class for storing assign info (about assign, submissions and grade data)
 *
 * @package    local_ned_controller
 * @subpackage mod_assign
 * @copyright  2021 NED {@link http://ned.ca}
 * @author     NED {@link http://ned.ca}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace local_ned_controller\mod_assign;

use local_ned_controller\shared_lib as NED;
use local_ned_controller\support\grade_grade;
use local_ned_controller\support\grade_item;
use local_ned_controller\support\scale;
use local_ned_controller\support\submission_info;
use local_ned_controller\tt_config_manager as CM;

defined('MOODLE_INTERNAL') || die();

NED::require_file("/mod/assign/locallib.php");

/**
 * Class assign_info
 *
 * @package local_ned_controller
 *
 * NOTE: if you want to use this class many times with same $userid, use
 * @see \local_ned_controller\mod_assign\assign_info::prepare_submission_info_records($course_id, $userid) before!
 *
 * @property-read bool                                    $exist = false;
 * @property-read submission_info[]                       $submissions;
 * @property-read int                                     $submissions_count = 0;
 * @property-read int                                     $user_signed_submissions_count = 0;
 * @property-read numeric                                 $last_submit_time = null;
 * @property-read numeric                                 $last_grade_time = null;
 * @property-read numeric                                 $last_grade = null;
 * @property-read numeric                                 $first_last_grade_time = null;
 *
 * @property-read numeric                                 $id;
 * @property-read \local_ned_controller\mod_assign\assign $assign;
 * @property-read numeric                                 $user_id;
 * @property-read numeric                                 $course_id;
 *
 * @property-read numeric                                 $finalgrade;
 * @property-read bool                                    $finalgraded = false;
 * @property-read numeric                                 $final_grade_time;
 * @property-read numeric                                 $override_time;
 * @property-read numeric                                 $extensionduedate;
 * @property-read bool                                    $overriden = false;
 * @property-read bool                                    $grantedextension = false;
 * @property-read numeric $groupid = 0;
 * @property-read string $groupname = '';
 *
 * @property-read grade_item $grade_item
 * @property-read grade_grade $grade_grade
 * @property-read scale $scale
 */
class assign_info{
    use \local_ned_controller\base_empty_class;

    protected static $submission_info_list = [];
    protected static $submission_info_list2 = [];
    protected static $submission_info_records = [];

    protected $_exist = false;
    protected $_submissions = [];
    protected $_submissions_count = 0;
    protected $_user_signed_submissions_count = 0;
    protected $_last_submit_time = null;
    protected $_last_grade_time = null;
    protected $_last_grade = null;
    protected $_first_last_grade_time = null;

    protected $_id;
    /* @var \local_ned_controller\mod_assign\assign */
    protected $_assign;
    protected $_user_id;
    protected $_course_id;

    protected $_finalgrade;
    protected $_finalgraded = false;
    protected $_final_grade_time;
    protected $_override_time;
    protected $_extensionduedate;
    protected $_overriden = false;
    protected $_grantedextension = false;
    protected $_groupid = 0;
    protected $_groupname = '';

    /**
     * @var grade_item
     */
    protected $_grade_item;
    /**
     * @var grade_grade
     */
    protected $_grade_grade;
    /**
     * @var scale
     */
    protected $_scale;


    /**
     * assign_info constructor.
     *
     * @param \assign | \cm_info | \local_ned_controller\mod_assign\assign | int $mod_or_assign - mod id, of assign/cm_info object
     * @param $userid
     */
    public function __construct($mod_or_assign, $userid){

        $assign = null;
        $mod_id = null;
        $mod = null;
        $classname = get_class($mod_or_assign);
        if ($classname && NED::str_ends_with($classname, 'assign')){
            $classname = 'assign';
        }
        switch($classname){
            case false:
                if (is_null($mod_or_assign)){
                    return;
                }
                $mod_id = $mod_or_assign;
                break;
            case 'cm_info':
                $mod = $mod_or_assign;
                $mod_id = $mod->id;
                break;
            case 'assign':
                $assign = $mod_or_assign;
        }

        if ($mod_id){
            $submission_info = static::get_submission_info_by_modid($mod_id, $userid);
            if ($submission_info && $submission_info->_exist){
                $this->import($submission_info->export('', false), true);
                return;
            } elseif ($submission_info === false){
                return;
            }

            if ($mod){
                $assign = static::get_assign_by_mod($mod);
            } else {
                $assign = static::get_assign_by_modid($mod_id);
            }
        }

        if (!$assign){
            return;
        }

        $submission_info = static::get_submission_info_by_assignid($assign->get_instance()->id, $userid);
        if ($submission_info && $submission_info->_exist){
            $this->import($submission_info->export('', false), true);
        } else {
            $this->create_submission_info_by_assign($assign, $userid);
            static::$submission_info_list[$userid][$assign->get_instance()->id] = $this->_exist ? $this : false;
            static::$submission_info_list2[$userid][$assign->get_course_module()->id] = $this->_exist ? $this : false;
        }
    }

    /**
     * Get group/user submission
     *
     * @param     $userid
     * @param int $attemptnumber = -1
     * @param bool $create If set to true a new submission object will be created in the database with the status set to "new".
     *
     * @return false|\stdClass|null
     */
    public function get_needed_user_submission($userid, $attemptnumber=-1, $create=false){
        $submission = null;
        $attemptnumber = $attemptnumber ?? -1;
        if ($this->assign->get_instance()->teamsubmission) {
            $submission = $this->assign->get_group_submission($userid, 0, $create, $attemptnumber);
        } else {
            $submission = $this->assign->get_user_submission($userid, $create, $attemptnumber);
        }

        return $submission;
    }

    /**
     * It's array construct from array of mods
     *
     * @param array $mods - array of mods or mod ids (can be mixed)
     * @param       $courseid
     * @param $user_id
     *
     * @return array
     */
    public static function get_assign_infos_by_mods($mods, $courseid, $user_id){
        $assigns = [];
        $assign_ids = [];
        $assign_infos = [];

        foreach ($mods as $mod){
            $mod_obj = null;
            if (is_object($mod)){
                $mod_obj = $mod;
                $mod_id = $mod->id;
            } else {
                $mod_id = $mod;
            }

            $submission_info = static::get_submission_info_by_modid($mod_id, $user_id);
            if ($submission_info && $submission_info->_exist){
                if ($courseid && $submission_info->course_id != $courseid){
                    continue;
                }
                $assign_infos[] = $submission_info;
                continue;
            } elseif ($submission_info === false){
                continue;
            }

            if ($mod_obj){
                $assign = static::get_assign_by_mod($mod, $courseid);
            } else {
                $assign = static::get_assign_by_modid($mod_id, $courseid);
            }

            if (!$assign){
                continue;
            }

            $assigns[] = $assign;
            $assign_ids[] = $assign->get_instance()->id;
        }

        static::prepare_submission_info_records($courseid, $user_id, $assign_ids);

        foreach ($assigns as $assign){
            $assign_infos[] = new self($assign, $user_id);
        }
        return $assign_infos;
    }

    /**
     * Get the submission status/grading status for all submissions in this assignment for the
     * given $userid.
     * Function is based on @see \assign::get_submission_info_for_participants()
     *
     * @param \assign|\local_ned_controller\mod_assign\assign $assign
     * @param         $userid
     *
     * These statuses match the available filters (requiregrading, submitted, notsubmitted, grantedextension).
     * If this is a group assignment, group info is also returned.
     */
    protected function create_submission_info_by_assign($assign, $userid){
        $assign_id = $assign->get_instance()->id;
        $course_id = $assign->get_course()->id;
        $submission_infos = static::get_submission_info_records($course_id, $userid, $assign_id);
        if (empty($submission_infos[$assign_id])){
            return;
        }
        $submission_info = $submission_infos[$assign_id];
        $this->copy($submission_info,'_');
        $this->_id = $assign_id;
        $this->_assign = \local_ned_controller\mod_assign\assign::get_ned_assign_from_assign($assign);
        $this->_user_id = $userid;
        $this->_course_id = $course_id;

        if (!empty($this->_final_grade_time) && !empty($this->_finalgrade)){
            $this->_finalgraded = true;
        }

        if ($this->_finalgraded && !empty($this->_override_time)){
            $this->_overriden = true;
        }

        if (!empty($this->_extensionduedate)) {
            $this->_grantedextension = true;
        }

        if ($assign->get_instance()->teamsubmission) {
            $group = $assign->get_submission_group($userid);
            if ($group) {
                $this->_groupid = $group->id;
                $this->_groupname = $group->name;
            }
        }

        $this->_submissions = [];
        $this->_user_signed_submissions_count = 0;
        foreach ($submission_info->submissions as $submission_number => $record){
            $this->_submissions[$submission_number] = new submission_info($record);
            if ($this->_submissions[$submission_number]->user_sign){
                $this->_user_signed_submissions_count++;
            }
        }
        $this->_submissions_count = count($this->_submissions);
        /** @var submission_info $last_submission */
        $last_submission = $this->_submissions[$this->_submissions_count-1];
        $this->_last_submit_time = $last_submission->submitted ? $last_submission->submit_time : null;
        if ($this->_overriden ||
            (!is_null($last_submission->grade_time) && $this->_final_grade_time > $last_submission->grade_time) ||
            (!is_null($last_submission->submit_time) && $this->_final_grade_time > $last_submission->submit_time)){
            $this->_last_grade_time = $this->_finalgraded ? $this->_final_grade_time : null;
            $this->_last_grade = $this->_finalgraded ? $this->_finalgrade : null;
        } else {
            $this->_last_grade_time = $last_submission->graded ? $last_submission->grade_time : null;
            $this->_last_grade = $last_submission->graded ? $last_submission->grade : null;
        }
        $this->_first_last_grade_time = $this->_last_grade_time ? \local_ned_controller\mod_assign\assign::get_first_last_grade_time_by_gi_id($this->_grade_item->id, $userid) : 0;

        $this->_exist = true;
    }


    /**
     * Get and store DB data for getting submission_info (see function get_submission_info)
     * Function is based on @see \assign::get_submission_info_for_participants()
     *
     * @param      $course_id
     * @param      $user_id
     * @param null|array|int|string $assign_id
     *
     * @return array|false
     */
    protected static function get_submission_info_records($course_id, $user_id, $assign_id=null){
        $bnttsi = static::$submission_info_records;
        $a = !is_null($assign_id);
        $assign_id = ($a && !is_array($assign_id)) ? [$assign_id] : $assign_id;
        $assign_id_need = [];
        $ans = [];
        if($a){
            foreach ($assign_id as $aid){
                if (isset($bnttsi[$user_id][$course_id][$aid])){
                    $ans[$aid] = $bnttsi[$user_id][$course_id][$aid];
                } else {
                    $assign_id_need[] = $aid;
                }
            }
            if (empty($assign_id_need)){
                return $ans;
            }
        } elseif (isset($bnttsi[$user_id][$course_id]) && $bnttsi[$user_id][$course_id] === false){
            return false;
        }

        $a = !empty($assign_id_need);
        $fields =
            'SELECT a_s.id AS submission, a.id AS assign, 
            a_s.status, a_s.timemodified AS submit_time, a_s.latest AS submission_latest, a_s.attemptnumber AS submission_number,
            a_g.timemodified AS grade_time, a_g.grade, g_g_h.loggeduser as authorization';
        $from =
            ' FROM {assign_submission} a_s
                 LEFT JOIN {assign} a
                        ON a_s.assignment = a.id
                 LEFT JOIN {assign_grades} a_g
                        ON a_s.assignment = a_g.assignment
                        AND a_g.userid = a_s.userid
                        AND a_g.attemptnumber = a_s.attemptnumber
                 LEFT JOIN {grade_items} g_i
                        ON a_s.assignment = g_i.iteminstance
                        AND g_i.itemtype = "mod"
                        AND g_i.itemmodule = "assign"
                        AND g_i.courseid = a.course
                 LEFT JOIN {grade_grades_history} g_g_h
                        ON g_i.id = g_g_h.itemid
                        AND g_g_h.timemodified = a_s.timecreated
                        AND g_g_h.userid = a_s.userid
                        AND g_g_h.usermodified = a_s.userid
                        AND g_g_h.source = "mod/assign"
            ';
        $where = "WHERE a.course = $course_id AND a_s.userid = $user_id";
        $where .= $a ? (" AND a.id IN (" . implode(',', $assign_id_need) . ")") : '';
        $order = "ORDER BY a_s.timemodified";
        $sql = "$fields $from $where $order";

        $submission_info_records = static::get_db()->get_records_sql($sql);
        $submission_infos = [];
        if (empty($submission_info_records)){
            if ($a){
                foreach ($assign_id_need as $aid){
                    $submission_infos[$aid] = false;
                }
            } else {
                $submission_infos = false;
            }
        } else {
            foreach ($submission_info_records as $submission_id => $record){
                $id = $record->assign;
                $record->userid = $user_id;
                $record->courseid = $course_id;
                if (isset($submission_infos[$id])){
                    $submission_infos[$id]->submissions[$record->submission_number] = $record;
                } else {
                    $submission_infos[$id] = $record;
                    $submission_infos[$id]->submissions = [$record->submission_number => $record];
                    /**
                     * @var grade_grade $grade_grade
                     * @var grade_item $grade_item
                     * @var scale $scale
                     */
                    [$grade_item, $grade_grade, $scale] =
                        \local_ned_controller\grade_info::get_all_s('assign', $id, $course_id, $user_id);
                    $submission_infos[$id]->grade_item = $grade_item;
                    $submission_infos[$id]->grade_grade = $grade_grade;
                    $submission_infos[$id]->scale = $scale;
                    $submission_infos[$id]->finalgrade = $grade_grade->finalgrade;
                    $submission_infos[$id]->override_time = $grade_grade->overridden;
                    $submission_infos[$id]->final_grade_time = $grade_grade->timemodified;
                }
            }
        }

        if ($a){
            foreach($assign_id_need as $aid){
                $val = $submission_infos[$aid] ?? false;
                $bnttsi[$user_id][$course_id][$aid] = $val;
                $ans[$aid] = $val;
            }
        } else {
            $bnttsi[$user_id][$course_id] = $submission_infos;
            $ans = $submission_infos;
        }

        static::$submission_info_records = $bnttsi;
        return $ans;
    }

    /**
     * Counts only submissions, which user opened by himself & only from activities, which have been chosen in TT
     *
     * @param      $courseid
     * @param      $user_id
     *
     * @return int
     */
    public static function get_course_submissions_count($courseid, $user_id){
        $sum = 0;

        if (!CM::is_resubmission_enabled($courseid)) return $sum;

        $resubmission_assignments = CM::get_enabled_resubmission_activities($courseid);
        if (empty($resubmission_assignments)) return $sum;

        $assign_infos = static::get_assign_infos_by_mods($resubmission_assignments, $courseid, $user_id);
        foreach ($assign_infos as $info){
            $sum += $info->_user_signed_submissions_count;
        }

        return $sum;
    }

    /**
     * Return assign if can, otherwise null
     *
     * @param \cm_info $mod
     * @param int      $courseid - check course id too, if it sent & $valid_only is true
     * @param bool     $valid_only
     *
     * @return \local_ned_controller\mod_assign\assign|null
     */
    public static function get_assign_by_mod(\cm_info $mod, $courseid=0, $valid_only=true){
        do {
            if ($mod->modname != 'assign') break;

            $ctx = \context_module::instance($mod->id, IGNORE_MISSING);
            if (!$ctx) break;

            $assign = new \local_ned_controller\mod_assign\assign($ctx, $mod, $mod->get_course());
            if ($valid_only && !$assign->is_valid($courseid)) break;

            return $assign;
        } while (false);

        return null;
    }

    /**
     * Return assign if can, otherwise null
     *
     * @param      $modid
     * @param int  $courseid - check course id too, if it sent & $valid_only is true
     * @param bool $valid_only
     *
     * @return \local_ned_controller\mod_assign\assign|null
     */
    public static function get_assign_by_modid($modid, $courseid=0, $valid_only=true){
        do {
            $ctx = \context_module::instance($modid, IGNORE_MISSING);
            if (!$ctx) break;

            $assign = new \local_ned_controller\mod_assign\assign($ctx, null, null);
            if ($valid_only && !$assign->is_valid($courseid)) break;

            return $assign;
        } while (false);

        return null;
    }

    /**
     * Full static::$submission_info_records array by one request
     *
     * @param      $course_id
     * @param      $user_id
     * @param null $assign_id
     */
    public static function prepare_submission_info_records($course_id, $user_id, $assign_id=null){
        static::get_submission_info_records($course_id, $user_id, $assign_id);
    }

    /**
     * Get assign_info from cache if it exists
     * @param $assignid
     * @param $userid
     *
     * @return null|assign_info
     */
    public static function get_submission_info_by_assignid($assignid, $userid){
        return NED::isset2(static::$submission_info_list, [$userid, $assignid]);
    }

    /**
     * Get assign_info by mod id from cache if it exists
     * @param $modid
     * @param $userid
     *
     * @return null|assign_info
     */
    public static function get_submission_info_by_modid($modid, $userid){
        return NED::isset2(static::$submission_info_list2, [$userid, $modid]);
    }

    /**
     * You can see, how many times this class use $DB, if set breakpoint here
     * @return \moodle_database
     */
    public static function get_db(){
        global $DB;
        return $DB;
    }
}
