<?php
/**
 * @package    local_ned_controller
 * @category   NED
 * @copyright  2021 NED {@link http://ned.ca}
 * @author     NED {@link http://ned.ca}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 * @noinspection PhpUnused
 */

namespace local_ned_controller;
use local_ned_controller\shared_lib as NED;
use local_ned_controller\support\ned_grade_controller_record as NGC_record;

/**
 * Class ned_grade_controller
 *
 * @package local_ned_controller
 */
class ned_grade_controller {
    use base_empty_class;

    const TABLE = 'local_ned_controller_grades';
    const RELATED_TABLE = 'local_academic_integrity_inf';
    const S_PREFIX = 'ngc_';
    const CM_TYPES_KEYS = [
        NED::MOD_ASSIGN_KEY => NED::MOD_TYPES[NED::MOD_ASSIGN_KEY],
        NED::MOD_QUIZ_KEY => NED::MOD_TYPES[NED::MOD_QUIZ_KEY],
    ];
    const CM_TYPES_VALS = [
        NED::MOD_ASSIGN => NED::MOD_ASSIGN_KEY,
        NED::MOD_QUIZ => NED::MOD_QUIZ_KEY
    ];

    const GT_AWARD_ZERO = 1;
    const GT_DEDUCTION = 2;
    const S_GT_PREFIX = self::S_PREFIX.'gradetype_';
    const GRADE_TYPES = [
        self::GT_AWARD_ZERO =>  self::S_GT_PREFIX.self::GT_AWARD_ZERO,
        self::GT_DEDUCTION =>   self::S_GT_PREFIX.self::GT_DEDUCTION,
    ];

    const REASON_AI = 1;
    const REASON_FILE = 2;          // Wrong submission
    const REASON_SUBMISSION = 3;    // Missed deadline + deadline grace period
    const REASON_OTHER = 4;
    const REASONS_GT = [
        self::GT_AWARD_ZERO => [
            self::REASON_AI,
            self::REASON_FILE,
            self::REASON_SUBMISSION,
            self::REASON_OTHER,
        ],
        self::GT_DEDUCTION => [
            self::REASON_AI,
            self::REASON_FILE,
            self::REASON_SUBMISSION,
            self::REASON_OTHER,
        ]
    ];
    const S_REASON_PREFIX = self::S_PREFIX.'reason_';
    const REASONS = [
        self::REASON_AI =>          self::S_REASON_PREFIX.self::REASON_AI,
        self::REASON_FILE =>        self::S_REASON_PREFIX.self::REASON_FILE,
        self::REASON_SUBMISSION =>  self::S_REASON_PREFIX.self::REASON_SUBMISSION,
        self::REASON_OTHER =>       self::S_REASON_PREFIX.self::REASON_OTHER,
    ];

    const ST_WAIT = 1;
    const ST_DONE = 2;
    const ST_PAUSED = 3;
    const ST_OBSOLETED = 4;
    const ST_ERROR = 10;
    const S_STATUS_PREFIX = self::S_PREFIX.'status_';
    const STATUSES = [
        self::ST_DONE => self::S_STATUS_PREFIX.self::ST_DONE,
        self::ST_PAUSED => self::S_STATUS_PREFIX.self::ST_PAUSED,
        self::ST_OBSOLETED => self::S_STATUS_PREFIX.self::ST_OBSOLETED,
        self::ST_WAIT => self::S_STATUS_PREFIX.self::ST_WAIT,
        self::ST_ERROR => self::S_STATUS_PREFIX.self::ST_ERROR,
    ];
    const HIDDEN_STATUSES = [
        self::ST_PAUSED => true,
        self::ST_ERROR => true,
    ];

    const SUBT_WS_DRAFT = 'wrongsubmission_draft';
    const SUBT_WS_CANCELLED = 'wrongsubmission_cancelled';
    const SUBT_WS_FIXED = 'wrongsubmission_fixed';
    const SUBT_MISSED_DEADLINE = 'misseddeadline';
    const SUBT_LATE_SUBMISSION = 'latesubmission';
    const SUBTYPES = [
        self::SUBT_WS_DRAFT,
        self::SUBT_WS_CANCELLED,
        self::SUBT_WS_FIXED,
        self::SUBT_MISSED_DEADLINE,
        self::SUBT_LATE_SUBMISSION,
    ];

    const CAP_CANT_VIEW = 0;
    const CAP_SEE_ME = 1;
    const CAP_SEE_OWN_ANY = 10;      // own something, only for the comparisons
    const CAP_SEE_OWN_SCHOOL = 12;  // own school, but not grader
    const CAP_SEE_OWN_GRADER = 15;  // own grader, but not school
    const CAP_SEE_OWN_ALL = 20;     // own school & grader
    const CAP_SEE_ALL = 99;

    const CAP_NAME_SEE_ME  = 'gradecontroller_seeme';
    const CAP_NAME_SEE_OWN = 'gradecontroller_seeown'; // own grader, but not school
    const CAP_NAME_SEE_SCH = 'gradecontroller_seeschool'; // own school, but not grader
    const CAP_NAME_SEE_ALL = 'gradecontroller_seeall';
    const CAP_NAME_EDIT    = 'gradecontroller_edit';

    /** @var \local_ned_controller\output\ned_grade_controller_render $NGC_RENDER */
    static public $NGC_RENDER = '\local_ned_controller\output\ned_grade_controller_render';

    protected $_ctx;
    protected $_user;
    protected $_userid;

    /**
     * ned_grade_controller constructor.
     */
    public function __construct(){
        global $USER;

        $this->_ctx = \context_system::instance();
        $this->_user = $USER;
        $this->_userid = $USER->id;
    }

    /**
     * @param \context  $ctx
     * @param bool      $is_admin
     *
     * @return int
     */
    static public function get_see_capability($ctx=null, $is_admin=null){
        static $_data = [];

        $ctx = $ctx ?? \context_system::instance();
        if (!isset($_data[$ctx->id])){
            $res = static::CAP_CANT_VIEW;
            $is_admin = $is_admin ?? is_siteadmin();
            if ($is_admin || NED::has_capability(static::CAP_NAME_SEE_ALL, $ctx)){
                $res = static::CAP_SEE_ALL;
            } else {
                $cap_own_grader = NED::has_capability(static::CAP_NAME_SEE_OWN, $ctx);
                $cap_own_school = NED::has_capability(static::CAP_NAME_SEE_SCH, $ctx);
                if ($cap_own_grader || $cap_own_school){
                    if ($cap_own_grader && $cap_own_school){
                        $res = static::CAP_SEE_OWN_ALL;
                    } elseif ($cap_own_school) {
                        $res = static::CAP_SEE_OWN_SCHOOL;
                    } else { // $cap_own_grader
                        $res = static::CAP_SEE_OWN_GRADER;
                    }
                } elseif (NED::has_capability(static::CAP_NAME_SEE_ME, $ctx)){
                    $res = static::CAP_SEE_ME;
                }
            }
            $_data[$ctx->id] = $res;
        }

        return $_data[$ctx->id];
    }

    /**
     * Return true, if user can see at least some NGC records
     *
     * @param \context|null  $ctx
     * @param bool|null      $is_admin
     *
     * @return bool
     */
    static public function has_any_see_capability($ctx=null, $is_admin=null){
        return static::get_see_capability($ctx, $is_admin) > static::CAP_CANT_VIEW;
    }

    /**
     * @param \context|null $ctx
     * @param bool|null     $is_admin
     *
     * @return bool
     */
    static public function has_edit_capability($ctx=null, $is_admin=null){
        $is_admin = $is_admin ?? is_siteadmin();
        return $is_admin || NED::has_capability(static::CAP_NAME_EDIT, $ctx ?? NED::ctx());
    }

    /**
     * Return subtypes, which are AI infractions + NGC subtypes
     *
     * @return array
     */
    static public function get_subtypes(){
        static $_data = null;
        if (is_null($_data)){
            $subtypes = [NED::ALL => NED::str('all')];
            if (NED::is_ai_exists()){
                $subtypes = array_merge($subtypes, \local_academic_integrity\infraction::get_penalty_list());
            }
            $_data = array_merge($subtypes, NED::strings2menu(static::SUBTYPES, true));
        }

        return $_data;
    }

    /**
     * Get related render
     *
     * @return string|\local_ned_controller\output\ned_grade_controller_render
     */
    static public function get_render(){
        return static::$NGC_RENDER;
    }

    /**
     * Get record and id from $record_or_id
     *
     * @param numeric|NGC_record|object $record_or_id
     *
     * @return array[$id, $record] - list($id, $record)
     */
    static public function get_id_and_record($record_or_id){
        $id = NED::get_id($record_or_id);
        if (is_object($record_or_id)){
            $record = $record_or_id;
        } else {
            $record = static::get_record_by_id($id);
        }

        return [$id, $record];
    }

    /**
     * @param int   $cmid
     * @param int   $userid
     * @param array $options
     *
     * @return array
     * @noinspection PhpUnusedParameterInspection
     */
    static public function check_db_params($cmid=0, $userid=0, &$options=[]){
        $keys = ['cmid', 'userid'];
        foreach ($keys as $key){
            if ($$key){
                $options[$key] = $$key;
            }
        }
        return $options;
    }

    /**
     * @return array - table keys
     */
    static public function get_table_columns(){
        return array_keys(NED::db()->get_columns(static::TABLE));
    }

    /**
     * @param array        $options array $fieldname => requested_value with AND in between
     * @param string|array $fields a comma separated list of fields to return
     * @param int          $limitfrom return a subset of records, starting at this point
     * @param int          $limitnum return a subset comprising this many records in total
     * @param string|array $sort an order to sort the results in
     *
     * @return NGC_record[]|array
     */
    static public function get_records($options=[], $fields='*', $limitfrom=0, $limitnum=0, $sort=''){
        global $DB;
        $fields = NED::arr2str($fields, '', ', ');
        $sort = NED::arr2str($sort, '', ', ');
        return $DB->get_records(static::TABLE, $options, $sort, $fields, $limitfrom, $limitnum);
    }

    /**
     * @param int   $id
     * @param array $options
     *
     * @return NGC_record|object|null
     */
    static public function get_record_by_id($id=0, $options=[]){
        if (!$id){
            return null;
        }

        $options['id'] = $id;
        $records = static::get_records($options) ?: [];

        return reset($records) ?: null;
    }

    /**
     * @param numeric $relatedid
     *
     * @return NGC_record|object|null
     */
    static public function get_record_by_related_id($relatedid){
        $options['relatedid'] = $relatedid;
        $records = static::get_records($options) ?: [];

        return reset($records) ?: null;
    }

    /**
     * Get records with construction 'IN()' or '=' sql fragment for all options
     * If you will use only simple options, may it will be better to use get_records()
     * @see get_records()
     * NOTE: this function doesn't work with empty conditions (options)
     *
     * @param array $options - conditions for "where", only table columns are correct keys
     *
     * @return NGC_record[]|object[]|array
     */
    static public function get_records_by_in_or_equal_options($options=[]){
        $options  = NED::val2arr($options);
        [$where, $params] = NED::sql_get_in_or_equal_options($options, 'ngc', static::get_table_columns());

        if (empty($where)){
            /**
             * Do not use empty conditions
             * @see get_records()
             */
            debugging('This function could not be called with empty "where", use get_records() instead');
            return [];
        }

        return NED::db()->get_records_select(static::TABLE, $where, $params);
    }

    /**
     * Get records by lists of course modules id(s) and user id(s)
     * You can leave some parameters empty, but you need set at least something
     *
     * @param numeric|array $cmids - id(s) of course module; if empty - load for all items
     * @param numeric|array $userids - user id(s); if empty - load for all users
     * @param bool          $by_userid_cmid - (optional) if true, result array will be by userid and cmid
     *
     * @return array|NGC_record[][]|object[][] - [cmid => [userid => NGC_record]], or [userid => [cmid => NGC_record]]
     */
    static public function get_records_by_cmids_userids($cmids=[], $userids=[], $by_userid_cmid=false){
        $options = [];
        if (!empty($cmids)){
            $options['cmid'] = NED::val2arr($cmids);
        }
        if (!empty($userids)){
            $options['userid'] = NED::val2arr($userids);
        }
        if (empty($options)){
            return [];
        }

        $data = static::get_records_by_in_or_equal_options($options);
        $res = [];
        foreach ($data as $datum){
            if ($by_userid_cmid){
                $res[$datum->userid][$datum->cmid] = $datum;
            } else {
                $res[$datum->cmid][$datum->userid] = $datum;
            }
        }

        return $res;
    }

    /**
     * @param array $ids
     * @param array $options
     *
     * @return NGC_record[]|object[]|array
     */
    static public function get_records_by_ids($ids, $options=[]){
        if (empty($ids)){
            return [];
        }

        $options = NED::val2arr($options);
        $options['id'] = $options['id'] ?? $ids;
        return static::get_records_by_in_or_equal_options($options);
    }

    /**
     * @param int   $cmid
     * @param int   $userid
     * @param array $options
     *
     * @return NGC_record[]|array
     */
    static public function get_records_by_params($cmid=0, $userid=0, $options=[]){
        static::check_db_params($cmid, $userid, $options);
        return static::get_records($options);
    }

    /**
     * Test whether a record exists in a table where all the given conditions met.
     *
     * @param array $conditions
     *
     * @return bool
     */
    static public function record_exists($conditions=[]){
        if (empty($conditions)) return false;

        return NED::db()->record_exists(static::TABLE, $conditions);
    }

    /**
     * Return records by user and cmid and other params, with checkin status = Done
     *
     * @param int   $cmid
     * @param int   $userid
     * @param array $options
     *
     * @return NGC_record[]|array
     */
    static public function get_records_by_params_done($cmid=0, $userid=0, $options=[]){
        $options['status'] = static::ST_DONE;
        return static::get_records_by_params($cmid, $userid, $options);
    }

    /**
     * Return ngc_record by mm_data and courseid
     *
     * WARNING: This object hasn't graderid, note, authorid, timecreated, timemodified fields, as mm_data hasn't it too.
     *
     * @param object $mm_data
     * @param int    $courseid - optional, course id for the record, as mm_data hasn't it
     *
     * @return NGC_record
     */
    static public function get_ngc_record_by_mm_data($mm_data, $courseid=0){
        $ngc = new NGC_record();
        $ngc->id = $mm_data->ngc_record ?? 0;
        $ngc->cmid = $mm_data->cmid ?? 0;
        $ngc->courseid = $courseid ?? NED::get_courseid_by_cmorid($ngc->cmid);
        $ngc->cm_type = NED::MOD_VALS[$mm_data->modname];
        $ngc->userid = $mm_data->userid ?? 0;
        $ngc->deadline = $mm_data->duedate ?? 0;

        $ngc->reason = $mm_data->ngc_reason ?? 0;
        $ngc->status = $mm_data->ngc_status ?? 0;
        $ngc->relatedid = $mm_data->ngc_relatedid ?? 0;
        $ngc->grade_type = $mm_data->ngc_grade_type ?? 0;
        $ngc->grade_change = $mm_data->ngc_grade_change ?? 0;

        return $ngc;
    }

    /**
     * Update records in the table
     *
     * @param $records
     *
     * @return bool
     */
    static public function update_records($records){
        global $DB;
        if (empty($records)){
            return false;
        }

        $records = NED::val2arr($records);
        $records = array_values($records);
        $last_i = count($records) - 1;
        foreach ($records as $i => $record){
            $DB->update_record(static::TABLE, $record, $i != $last_i);
        }

        return true;
    }

    /**
     * @param array $options
     *
     * @return bool
     */
    static public function delete_records_by_options($options=[]){
        global $DB;
        if (empty($options)){
            return false;
        }

        return $DB->delete_records(static::TABLE, $options);
    }

    /**
     * @param int[]|int $ids
     *
     * @return bool
     */
    static public function delete_records_by_ids($ids){
        global $DB;
        if (empty($ids)){
            return false;
        }

        $ids = NED::val2arr($ids);
        return $DB->delete_records_list(static::TABLE, 'id', $ids);
    }

    /**
     * @param object[]|NGC_record[]|object|NGC_record $records
     *
     * @return bool
     */
    static public function delete_records_by_records($records){
        $records = NED::val2arr($records);
        $ids = [];
        foreach ($records as $record){
            $ids[] = $record->id;
        }
        return static::delete_records_by_ids($ids);
    }

    /**
     * @param int   $cmid
     * @param int   $userid
     * @param array $options
     *
     * @return bool
     */
    static public function delete_records_by_params($cmid=0, $userid=0, $options=[]){
        static::check_db_params($cmid, $userid, $options);
        return static::delete_records_by_options($options);
    }

    /**
     * Return true, if all is fine, false otherwise
     * There is none checks here - use static::save_record if you are not sure, that's send data is correct
     *
     * @param object[]|array[]|NGC_record[] $records
     * @param bool|int|string               $update_timemodified - update timemodified to current date if it's true, or for its value, if it's number
     *
     * @return bool
     */
    static public function save_records($records, $update_timemodified=true){
        global $DB;
        $records = NED::val2arr($records);
        $new_records = [];
        foreach ($records as $record){
            $res = static::save_record($record, $update_timemodified, true);
            if (is_object($res)){
                $new_records[] = $res;
            }
        }

        if (!empty($new_records)){
            $DB->insert_records(static::TABLE, $records);
        }

        return true;
    }

    /**
     * @param array|\object|NGC_record $data
     * @param bool|int|string          $update_timemodified - update timemodified to current date if it's true, or for its value, if it's number
     * @param bool                     $return_instead_of_insert - if true and there is no id in $data - return updated $data
     *
     * @return bool|int|\object|NGC_record
     */
    static public function save_record($data, $update_timemodified=true, $return_instead_of_insert=false){
        global $DB;

        /** @var \object|NGC_record $data */
        $data = NED::val2obj($data);
        if ($update_timemodified || !($data->timemodified ?? false)){
            $data->timemodified = is_numeric($update_timemodified) ? (int)$update_timemodified : time();
        }
        if ($data->id ?? false){
            return $DB->update_record(static::TABLE, $data) ? $data->id : false;
        } else {
            if (!($data->timecreated ?? false)){
                $data->timecreated = $data->timemodified ?? time();
            }

            if ($return_instead_of_insert){
                return $data;
            }

            return $DB->insert_record(static::TABLE, $data);
        }
    }

    /**
     * Set new value for all records, which fit some params
     *
     * @param string            $field - name of changed column
     * @param \mixed            $newvalue - new value for this column
     * @param array             $where_params - where keys - columns of TABLE, and values - checked params
     * @param array|string      $where - you can add sql where conditions here
     * @param array             $params - array of sql parameters
     * @param bool|int|string   $update_timemodified - update timemodified to current date if it's true, or for its value, if it's number
     *
     * @return bool
     */
    static public function update_records_value($field, $newvalue, $where_params=[], $where=[], $params=[], $update_timemodified=true){
        global $DB;
        $where  = NED::val2arr($where);
        $params = NED::val2arr($params);

        $columns = static::get_table_columns();
        $has_field = in_array($field, $columns);
        if ($has_field){
            [$other_where, $other_params] = NED::sql_get_in_or_equal_options($where_params, 'ngc', $columns, true);
            $where = array_merge($where, $other_where);
            $params = array_merge($params, $other_params);
        }

        if (!$has_field || empty($where)){
            // we don't want change nothing or change all the table
            return false;
        }

        $where = NED::sql_condition($where);
        $res = $DB->set_field_select(static::TABLE, $field, $newvalue, $where, $params);
        if ($res && $update_timemodified && $field != 'timemodified'){
            $timemodified = is_numeric($update_timemodified) ? (int)$update_timemodified : time();
            $DB->set_field_select(static::TABLE, 'timemodified', $timemodified, $where, $params);
        }

        return $res;
    }

    /**
     * Set timemodified value for some amount of records (by id)
     *
     * @param array|string|int $ids  - id(s) which should change
     * @param int|null  $value       - timemodified value, if null, uses 'now'
     * @param int|null  $before_time - if set, update values only with timemodified earlier than $before_time
     *
     * @return bool
     */
    static public function update_timemodified_value($ids=[], $value=null, $before_time=null){
        if (empty($ids)){
            return false;
        }

        $value = $value ?? time();
        $where_params  = [];
        $where = [];
        $params = [];

        if ($ids === '*'){
            // change full table, use with caution!
            $where[] = NED::SQL_TRUE_COND;
        } else {
            $where_params['id'] = NED::val2arr($ids);
        }

        if ($before_time){
            $where[] = 'timemodified < :before_time';
            $params['before_time'] = $before_time;
        }

        return static::update_records_value('timemodified', $value, $where_params, $where, $params, false);
    }

    /**
     * Check data and save it as record (if there are no errors)
     *
     * @param array|object|NGC_record $data
     * @param numeric|null            $editorid - who create/edit record
     * @param bool                    $apply    - if true, try to apply penalty after creation
     *
     * @return bool|int
     */
    static public function check_and_save($data, $editorid=null, $apply=true){
        $errors = [];
        $check_errors = function($txt='') use(&$errors) {
            if (!empty($txt)){
                $errors[] = $txt;
            }
            if (!empty($errors)){
                NED::print_simple_error('NED Grade Controller Error!', $errors);
            }
        };

        /** @var object|NGC_record $data */
        $data = NED::val2obj($data);
        $check_params = ['courseid', 'cmid', 'userid', 'grade_type', 'reason'];
        $data->userid = $data->userid ?? ($data->studentid ?? 0);

        foreach ($check_params as $param){
            if (is_null($data->$param ?? null)){
                $errors[] = "There no necessary param $param!";
            }
        }
        $check_errors();

        $prev_record = null;
        if ($data->id ?? false){
            $prev_record = static::get_record_by_id($data->id);
        }

        if (!$prev_record){
            $records = static::get_records_by_params($data->cmid, $data->userid);
            if (!empty($records)){
                $check_errors("We already have record in course $data->courseid, cm $data->cmid, user $data->userid!");
            }
        }

        $cm = NED::get_cm_by_cmid($data->cmid, $data->courseid);
        if (!$cm){
            $check_errors("There are no cm $data->cmid in course $data->courseid!");
        }

        $data->cm_type = NED::MOD_VALS[$cm->modname] ?? 0;
        if (!isset(static::CM_TYPES_KEYS[$data->cm_type])){
            $check_errors("Wrong cm type $cm->modname!");
        }

        $data->graderid = NED::get_graderid_by_studentid($data->userid, $data->courseid, $data->cmid);
        if ($data->grade_type == static::GT_DEDUCTION){
            if (($data->grade_change ?? -1) < 0){
                $check_errors("Wrong grade_change!");
            }
        } else {
            $data->grade_change = 0;
        }

        if ($data->reason == static::REASON_AI){
            $related_record = null;
            if ($data->relatedid ?? 0){
                $related_record = static::get_related_data_by_id($data->relatedid);
            }
            if (!$related_record){
                $check_errors("Wrong relatedid!");
            }
        } else {
            $data->relatedid = 0;
        }

        $data->deadline = $data->deadline ?? NED::get_deadline_by_cm($data->cmid, $data->userid, $data->courseid);
        $data->note = $data->note ?? '';

        if (!isset($data->authorid)){
            $data->authorid = $editorid ?? NED::get_userid_or_global();
        }
        $data->editorid = $editorid ?? NED::get_userid_or_global();

        $data->status = ($data->status ?? 0) ?: static::ST_WAIT;
        if ($apply){
            if ($prev_record && $prev_record->status == static::ST_DONE){
                if ($prev_record->grade_type != $data->grade_type){
                    static::unapply_record($prev_record);
                    $data->status = static::ST_PAUSED;
                }
            }

            if ($data->status != static::ST_DONE){
                if (static::apply_record($data)){
                    $data->status = static::ST_DONE;
                }
            }
        }

        return static::save_record($data);
    }

    /**
     * Create and save records from mm_data (if there are no errors)
     * Return count of saved records
     *
     * @param array|object|object[]|marking_manager\mm_data_by_activity_user|marking_manager\mm_data_by_activity_user[] $mm_data
     * @param int $grade_type   - one of the grade_type key, @see self::GRADE_TYPES
     * @param int $reason       - one of the reason key, @see self::REASONS
     * @param int $grade_change - (optional) grade change, only for deduction
     * @param int $courseid     - (optional) Course id, if you already know it and there is no in $mm_data
     * @param string $note      - (optional) note
     * @param int $authorid     - (optional) ID of author, by default uses 0  (System)
     *
     * @return int
     */
    static public function check_and_save_from_mm($mm_data, $grade_type, $reason, $grade_change=0, $courseid=null, $note='', $authorid=0){
        $saved = 0;
        if (empty($mm_data)){
            return $saved;
        }

        $add_change_note = function($a, $b){
            return NED::str($a).' => '.NED::str($b);
        };

        /** @var array|object[]|marking_manager\mm_data_by_activity_user[] $mm_data */
        $mm_data = NED::val2arr($mm_data);
        foreach ($mm_data as $mm_datum){
            $r_courseid = $courseid ?: ($mm_datum->courseid ?? 0);
            $cmid = $mm_datum->cmid;
            if (!$r_courseid){
                $r_courseid = NED::get_courseid_by_cmorid($cmid);
            }
            $userid = $mm_datum->userid;

            /** @var object|NGC_record $data */
            if ($mm_datum->ngc_record){
                $data = static::get_record_by_id($mm_datum->ngc_record);
                if (!$data || ($data->status ?? 0) == static::ST_ERROR){
                    continue;
                }

                if (empty($note)){
                    $note = NED::str('automaticchangedrecord', NED::ned_date(time()));
                }

                if ($data->grade_type != $grade_type || $data->reason != $reason){
                    static::unapply_record($data);
                    $add2note = [];
                    if ($data->grade_type != $grade_type){
                        $add2note[] = $add_change_note(static::GRADE_TYPES[$data->grade_type] ?? '', static::GRADE_TYPES[$grade_type] ?? '');
                    }
                    if ($data->reason != $reason){
                        $add2note[] = $add_change_note(static::REASONS[$data->reason] ?? '', static::REASONS[$reason] ?? '');
                    }
                    $note .= "\n(".join(', ', $add2note).')';
                }
            } else {
                if (!NED::get_cm_visibility_by_user($cmid, $userid, false, false)){
                    continue;
                }

                $data = new \stdClass();
                if (empty($note)){
                    $note = NED::str('automaticcreatedrecord', NED::ned_date(time()));
                }
            }

            $data->courseid = $r_courseid;
            $data->cmid = $cmid;
            $data->cm_type = NED::MOD_VALS[$mm_datum->modname] ?? 0;
            $data->userid = $userid;
            $data->deadline = $mm_datum->duedate;

            $data->grade_type = $grade_type;
            $data->reason = $reason;
            $data->grade_change = $grade_change;
            if ($data->note ?? false){
                $data->note .= "\n".$note;
            } else {
                $data->note = $note;
            }

            $data->authorid = $authorid;
            $data->status = static::ST_WAIT;

            $saved += static::check_and_save($data, $authorid) ? 1 : 0;
        }

        return $saved;
    }

    /**
     * @param object|NGC_record $record
     * @param bool              $format
     *
     * @return bool|string - false, if all is OK, otherwise error string
     */
    static public function has_errors_for_deleting($record, $format=false){
        if ($record->status == static::ST_ERROR){
            return false;
        }

        if ($record->reason == static::REASON_SUBMISSION){
            $deadline = NED::get_deadline_by_cm($record->cmid, $record->userid, $record->courseid);
            if ($record->deadline != $deadline){
                $record->deadline = $deadline;
                if ($record->id ?? false){
                    static::update_records($record);
                }
            }

            $deadline_penalty = $deadline ? ($deadline + NED::DEADLINE_PENALTY_GRACE_PERIOD) : 0;
            if ($deadline_penalty && $deadline_penalty < time()){
                if (NED::get_cm_visibility_by_user($record->cmid, $record->userid, false, false)){
                    $submitted_time = NED::get_submitted_time_by_cm($record->cmid, $record->userid);
                    if (!($submitted_time && $deadline_penalty > $submitted_time)){
                        if ($format){
                            return static::get_format_record_string($record, 'ngc_delete_error_missed_deadlines_format');
                        } else {
                            return NED::str('ngc_delete_error_missed_deadlines');
                        }
                    }
                }
            }
        }

        return false;
    }

    /**
     * @param object|NGC_record $record
     * @param bool              $format
     *
     * @return bool|string - false, if all is OK, otherwise error string
     */
    static public function has_errors_for_change_status($record, $format=false){
        if ($record->status == static::ST_ERROR){
            if ($format){
                return static::get_format_record_string($record, 'ngc_change_status_error_format');
            } else {
                return NED::str('ngc_change_status_error');
            }
        }

        return false;
    }

    /**
     * Check data and delete record(s) (if there are no errors)
     *
     * @param numeric|object|NGC_record|array|NGC_record[] $records_or_ids
     * @param bool $check - (optional) if true, do some checks before allow records deleting
     * @param bool $skip_unapply - (optional) if true, do not actually change grade
     *
     * @return bool|int - count of deleted records or false
     */
    static public function check_and_delete($records_or_ids, $check=false, $skip_unapply=false){
        $records_or_ids = NED::val2arr($records_or_ids);
        $ids = [];
        foreach ($records_or_ids as $record_or_id){
            [$id, $record] = static::get_id_and_record($record_or_id);
            if (!$id || !$record){
                continue;
            }

            if ($check){
                $error = static::has_errors_for_deleting($record, true);
                if ($error){
                    NED::notification_add($error, NED::NOTIFY_WARNING);
                    continue;
                }
            }

            if (!$skip_unapply && $record->status == static::ST_DONE){
                static::unapply_record($record);
            }

            $ids[] = $id;
        }

        return static::delete_records_by_ids($ids) ? count($ids) : false;
    }

    /**
     * Check data and change state (as pause) of record(s) (if there are no errors)
     *
     * @param int                                          $state          - state to set or unset
     * @param numeric|object|NGC_record|array|NGC_record[] $records_or_ids
     * @param bool                                         $set            - if false, it's unset
     * @param bool                                         $check
     * @param bool                                         $skip_ai_update - if true, don't update AI table
     * @param numeric|object|null                          $editor_or_id   - if not null, update editor for all changed records
     *
     * @return bool|int - count of changed records or false
     */
    static public function check_and_change_about_pause_state($state=self::ST_PAUSED, $records_or_ids=[], $set=true, $check=false,
        $skip_ai_update=false, $editor_or_id=null){
        $records_or_ids = NED::val2arr($records_or_ids);
        $ids = [];
        $ai_ids = [];
        foreach ($records_or_ids as $record_or_id){
            /** @var NGC_record $record */
            [$id, $record] = static::get_id_and_record($record_or_id);
            if (!$id || !$record){
                continue;
            }

            if ($check){
                $error = static::has_errors_for_change_status($record, true);
                if ($error){
                    NED::notification_add($error, NED::NOTIFY_WARNING);
                    continue;
                }
            }

            $add_id = false;
            if ($set){
                if ($record->status == $state){
                    continue;
                } elseif ($record->status == static::ST_DONE){
                    if (static::unapply_record($record)){
                        $add_id = true;
                    }
                } else {
                    $add_id = true;
                }
            } else {
                // unset
                if ($record->status != static::ST_DONE){
                    if (static::apply_record($record)){
                        $add_id = true;
                    }
                }
            }

            if ($add_id){
                $ids[] = $id;
                if ($record->reason == static::REASON_AI && $record->relatedid){
                    $ai_ids[] = $record->relatedid;
                }
            }
        }

        if (!empty($ids)){
            $new_status = $set ? $state : static::ST_DONE;
            if (static::update_records_value('status', $new_status, ['id' => $ids])){
                if (!is_null($editor_or_id)){
                    static::update_records_value('editorid', NED::get_id($editor_or_id), ['id' => $ids], [], [], false);
                }
                if (!$skip_ai_update && NED::is_ai_exists() && !empty($ai_ids)){
                    \local_academic_integrity\infraction::update_states_by_ngc_status($new_status, $ai_ids);
                }
                return count($ids);
            }
        }

        return false;
    }

    /**
     * Check data and change "pause" state of record(s) (if there are no errors)
     *
     * @param numeric|object|NGC_record|array|NGC_record[] $records_or_ids
     * @param bool                                         $pause        - if false, it's unpause
     * @param bool                                         $check
     * @param numeric|object|null                          $editor_or_id - if not null, update editor for all changed records
     *
     * @return bool|int - count of changed records or false
     */
    static public function check_and_change_pause_state($records_or_ids, $pause=true, $check=false, $editor_or_id=null){
        return static::check_and_change_about_pause_state(static::ST_PAUSED, $records_or_ids, $pause, $check, false, $editor_or_id);
    }

    /**
     * Check data and change "obsolete" state of record(s) (if there are no errors)
     *
     * @param numeric|object|NGC_record|array|NGC_record[] $records_or_ids
     * @param bool                                         $obsolete     - if false, it's unobsolete (relevant)
     * @param bool                                         $check
     * @param numeric|object|null                          $editor_or_id - if not null, update editor for all changed records
     *
     * @return bool|int - count of changed records or false
     */
    static public function check_and_change_obsolete_state($records_or_ids, $obsolete=true, $check=false, $editor_or_id=null){
        return static::check_and_change_about_pause_state(static::ST_OBSOLETED, $records_or_ids, $obsolete, $check, false, $editor_or_id);
    }

    /**
     * Apply grade changing by record
     *
     * @param NGC_record|object $record
     *
     * @return bool
     */
    static public function apply_record($record){
        switch ($record->grade_type){
            case static::GT_AWARD_ZERO:
                return !empty(static::apply_award_zero($record));
            case static::GT_DEDUCTION:
                return true;
            default:
                break;
        }

        return false;
    }

    /**
     * Apply grade changing by record
     *
     * @param NGC_record|object $record
     *
     * @return bool
     */
    static public function unapply_record($record){
        switch ($record->grade_type){
            case static::GT_AWARD_ZERO:
                return !empty(static::unapply_award_zero($record));
            case static::GT_DEDUCTION:
                return true;
            default:
                break;
        }

        return false;
    }

    /**
     * Apply or Unapply Award Zero by records or its id
     *
     * @param int[]|NGC_record[]|object[]|int|NGC_record|object $records_or_ids
     * @param bool                                              $apply - apply or unapply AZ
     *
     * @return array - return list if ids, for which
     */
    static public function change_award_zero($records_or_ids, $apply=true){
        $ids = [];
        $error_ids = [];
        if (empty($records_or_ids)){
            return $ids;
        }

        $records_or_ids = NED::val2arr($records_or_ids);
        $check = reset($records_or_ids);
        if (is_object($check)){
            $records = $records_or_ids;
        } else {
            $records = static::get_record_by_id($records_or_ids, ['grade_type' => static::GT_AWARD_ZERO]);
            if (empty($records)){
                return $ids;
            }
        }

        $now = time();
        /** @var object|NGC_record $record */
        foreach ($records as $record){
            if ($record->grade_type != static::GT_AWARD_ZERO){
                continue;
            }

            try {
                $cm = NED::get_cm_by_cmid($record->cmid, $record->courseid);
                // change submission for Wrong Files
                if ($record->reason == static::REASON_FILE && $cm->modname == NED::MOD_ASSIGN){
                    $assign = NED::$ned_assign::get_assign_by_cm($cm, $cm->course);
                    if ($apply){
                        $assign->submitted2draft($record->userid, false);
                    } else {
                        $assign->draft2submit($record->userid, false);
                    }
                }
                // change grade
                $gg = NED::get_grade_grade($cm, $record->userid, true, true);
                if ($gg){
                    if ($apply){
                        $gg->usermodified = $record->authorid;
                        $gg->finalgrade = 0;
                        $gg->overridden = $now;
                    } else {
                        $gg->usermodified = $record->editorid;
                        $gg->finalgrade = null;
                        $gg->overridden = 0;
                    }
                    $gg->timemodified = $now;
                    $ids[] = $record->id ?? 0;
                    NED::grade_grade_update($cm, $gg);
                } else {
                    $error_ids[] = $record->id;
                }
            } catch (\Throwable){
                $error_ids[] = $record->id;
            }
        }

        if (!empty($error_ids)){
            static::update_records_value('status', static::ST_ERROR, ['id' => $error_ids]);
        }

        return $ids;
    }

    /**
     * Apply AZ by records or its id
     * @see change_award_zero()
     *
     * @param int[]|NGC_record[]|object[]|int|NGC_record|object $records_or_ids
     *
     * @return array - return list if ids, for which
     */
    static public function apply_award_zero($records_or_ids){
        return static::change_award_zero($records_or_ids, true);
    }

    /**
     * Unapply AZ by records or its id
     * @see change_award_zero()
     *
     * @param int[]|NGC_record[]|object[]|int|NGC_record|object $records_or_ids
     *
     * @return array - return list if ids, for which
     */
    static public function unapply_award_zero($records_or_ids){
        return static::change_award_zero($records_or_ids, false);
    }

    /**
     * Check, if current course id can be used for NGC by this user
     *
     * @param int           $courseid - ID of course
     * @param \stdClass|int $userorid - Optional userid (default = current)
     *
     * @return bool
     */
    static public function check_courseid($courseid, $userorid=null){
        return NED::check_grader_courseid($courseid, $userorid);
    }

    /**
     * Check, if current cmid can be used for NGC by this user
     *
     * @param int|\cm_info  $cm_or_id       - Id of course-module, or database object
     * @param \stdClass|int $courseorid - Optional course object (or its id) if already loaded
     * @param \stdClass|int $userorid   - Optional userid (default = current)
     *
     * @return bool
     */
    static public function check_cmid($cm_or_id, $courseorid=null, $userorid=null){
        $cm = NED::get_cm_by_cmorid($cm_or_id, $courseorid, $userorid);
        if (!$cm){
            return false;
        }

        if (!isset(static::CM_TYPES_VALS[$cm->modname])){
            return false;
        }

        return $cm->uservisible;
    }

    /**
     * Return cms, which can be used for NGC by this user
     *
     * @param \stdClass|int $courseorid - Course object (or its id)
     * @param \stdClass|int $userorid   - Optional userid (default = current)
     *
     * @return array|\cm_info[]
     */
    static public function get_NGC_cms($courseorid=null, $userorid=null){
        $checked_cms = [];
        $cms = NED::get_course_cms($courseorid,  $userorid);

        foreach ($cms as $cm){
            if (!isset(static::CM_TYPES_VALS[$cm->modname])){
                continue;
            }

            $checked_cms[$cm->id] = $cm;
        }

        return $checked_cms;
    }

    /**
     * Get dictionary of reasons by grade_type, or all of them,
     *  where keys - REASON_*, and values - keys for them translated strings
     *
     * @param null $grade_type
     *
     * @return array|string[]
     */
    static public function get_reasons($grade_type=null){
        if ($grade_type && isset(static::REASONS_GT[$grade_type])){
            $grade_reason_keys = static::REASONS_GT[$grade_type];
            $grade_reasons_keys_dir = array_combine($grade_reason_keys, $grade_reason_keys);
            $reasons = array_intersect_key(static::REASONS, $grade_reasons_keys_dir);
        } else {
            $reasons = static::REASONS;
        }

        if (!NED::is_ai_exists()){
            unset($reasons[static::REASON_AI]);
        }

        return $reasons;
    }

    /**
     * Return record(s) form the related table by student id and other parameters
     *
     * @param numeric $studentid
     * @param numeric $courseid
     * @param numeric $cmid
     * @param false   $only_first - if true, return object (or null), otherwise return array
     *
     * @return array|object|null
     */
    static public function get_related_data($studentid=0, $courseid=0, $cmid=0, $only_first=false){
        global $DB;
        $params = [];
        if ($studentid){
            $params['student'] = $studentid;
        }
        if ($courseid){
            $params['courseid'] = $courseid;
        }
        if ($cmid){
            $params['cmid'] = $cmid;
        }

        if (empty($params)){
            return $only_first ? null : [];
        }

        $records = $DB->get_records(static::RELATED_TABLE, $params) ?: [];
        if ($only_first){
            return reset($records) ?: null;
        }
        return $records;
    }

    /**
     * Return record(s) form the related table by record id
     *
     * @param int $id - id of the record in the related table
     *
     * @return object|null
     */
    static public function get_related_data_by_id($id=0){
        global $DB;
        if (empty($id)){
            return null;
        }

        $params = ['id' => $id];
        return $DB->get_record(static::RELATED_TABLE, $params) ?: null;
    }

    /**
     * @param NGC_record|object $record
     *
     * @return bool
     */
    static public function can_see_record($record){
        $viewerid = NED::get_userid_or_global();
        $cap = static::get_see_capability();
        switch ($cap){
            default:
            case static::CAP_CANT_VIEW:
                return false;

            case static::CAP_SEE_ME:
                return $record->userid == $viewerid;

            case static::CAP_SEE_OWN_GRADER:
                return $record->graderid == $viewerid;
            case static::CAP_SEE_OWN_SCHOOL:
                return NED::is_in_same_school($record->userid, $viewerid);
            case static::CAP_SEE_OWN_ANY:
            case static::CAP_SEE_OWN_ALL:
                return $record->graderid == $viewerid || NED::is_in_same_school($record->userid, $viewerid);

            case static::CAP_SEE_ALL:
                return true;
        }
    }

    /**
     * Check, can grader see some student (on specific course) by grader capabilities
     *
     * @param numeric|object $grader_or_id
     * @param numeric|object $student_or_id
     * @param numeric|object $course_or_id
     *
     * @return bool
     */
    static public function can_see_student($grader_or_id=null, $student_or_id=null, $course_or_id=null){
        $viewerid = NED::get_userid_or_global($grader_or_id);
        $userid = NED::get_id($student_or_id);
        if ($course_or_id === SITEID){
            $course_or_id = null;
        }

        $cap = static::get_see_capability();
        switch ($cap){
            default:
            case static::CAP_CANT_VIEW:
                return false;

            case static::CAP_SEE_ME:
                return $userid == $viewerid;

            case static::CAP_SEE_OWN_GRADER:
                return static::ngc_is_grader($grader_or_id, $student_or_id, $course_or_id);
            case static::CAP_SEE_OWN_SCHOOL:
                return NED::is_in_same_school($userid, $viewerid);
            case static::CAP_SEE_OWN_ANY: // shouldn't be such option really
            case static::CAP_SEE_OWN_ALL:
                return NED::is_in_same_school($userid, $viewerid) || static::ngc_is_grader($grader_or_id, $student_or_id, $course_or_id);

            case static::CAP_SEE_ALL:
                return true;
        }
    }

    /**
     * @param NGC_record|object $record
     *
     * @return string
     */
    static public function get_see_text($record){
        if (!static::can_see_record($record)){
            return '';
        }

        return NED::str('see_sth', NED::str(static::GRADE_TYPES[$record->grade_type]));
    }

    /**
     * @param NGC_record|object $record
     *
     * @return string
     */
    static public function get_human_record_name($record){
        if (!static::can_see_record($record)){
            return '';
        }

        return static::get_human_type_reason_name($record->grade_type, $record->reason);
    }

    /**
     * Return human name of the NGC type (grade type + reason)
     *
     * @param int $grade_type - type, one of the NGC::GRADE_TYPES
     * @param int $reason - reason, one of the NGC::REASONS
     *
     * @return string
     */
    static public function get_human_type_reason_name($grade_type, $reason){
        if ($reason == static::REASON_SUBMISSION){
            $name = NED::str($grade_type == static::GT_AWARD_ZERO ? static::SUBT_MISSED_DEADLINE : static::SUBT_LATE_SUBMISSION);
        } else {
            $name = isset(static::REASONS[$reason]) ? NED::str(static::REASONS[$reason]) : '?';
        }

        return $name;
    }

    /**
     * Return grade status to the NED grade icon
     * For getting extension status right, you should provide $cmid or $tags
     *
     * @param int $grade_type - type, one of the NGC::GRADE_TYPES
     * @param int $reason - reason, one of the NGC::REASONS
     * @param int $deadline - UNIX time
     * @param int $activity_type - activity key, @see NED::MOD_TYPES
     *
     * @return array
     */
    static public function get_grade_status($grade_type, $reason, $deadline=0, $activity_type=NED::MOD_ASSIGN_KEY){
        $icon = [];
        $main_type = null;
        $add_icon = function($add) use (&$icon){
            $icon[$add] = $add;
        };
        if ($grade_type == static::GT_AWARD_ZERO){
            $add_icon(NED::ICON_TYPE_ZERO);
            $now = time();
            $main_type = NED::STATUS_HARD_ZERO;
            if ($activity_type == NED::MOD_ASSIGN_KEY && $reason != static::REASON_AI){
                if ($deadline && $now <= ($deadline + NED::SOFT_ZERO_DELAY + NED::DEADLINE_PENALTY_GRACE_PERIOD)){
                    $main_type = NED::STATUS_SOFT_ZERO;
                }
            }
        } elseif ($grade_type == static::GT_DEDUCTION){
            $main_type = NED::ICON_TYPE_DEDUCTION;
        }

        switch ($reason){
            case static::REASON_AI:
                $add_icon(NED::ICON_REASON_AI); break;
            case static::REASON_FILE:
                $add_icon(NED::ICON_REASON_FILE); break;
            case static::REASON_SUBMISSION:
                $add_icon(NED::ICON_REASON_MISSED); break;
            case static::REASON_OTHER:
                $add_icon(NED::ICON_REASON_OTHER); break;
        }

        if ($main_type){
            $icon = [$main_type => $main_type] + $icon;
        }
        $add_icon(NED::ICON_LETTER);
        $icon['title'] = static::get_human_type_reason_name($grade_type, $reason);

        return $icon;
    }

    /**
     * Used by checking cron tasks
     * Return null, if nothing to check, otherwise return course id and mm_data_list
     *
     * @param object|numeric $course_or_id
     * @param array          $filter
     * @param numeric        $userid (optional)
     *
     * @return array|null - [$courseid, $mm_data_list]
     */
    static protected function _get_courseid_and_mm_data_by_filter_for_check($course_or_id, $filter, $userid=null){
        $course = NED::get_chosen_course($course_or_id, false);
        if (!$course){
            return null;
        }

        $courseid = $course->id;
        $MM = NED::$MM;
        $context = \context_course::instance($courseid, IGNORE_MISSING);
        if (!$context){
            return null;
        }

        $mm_params = [
            'course' => $course, 'context' => $context, 'type' => static::CM_TYPES_KEYS,
            'groupid' => 0, 'set_students' => '*',
        ];
        if ($userid){
            $user = NED::get_user($userid);
            if ($user){
                $mm_params['set_students'] = [$userid => $user];
            }
        }

        $manager = $MM::get_MM_by_params($mm_params);
        $mm_data_list = $manager->get_raw_data($filter);
        if (empty($mm_data_list)){
            return null;
        }

        return [$courseid, $mm_data_list];
    }

    /**
     * Check missed deadlines by course
     * Return number of successfully added (and saved) records
     *
     * @param object|numeric $course_or_id
     * @param numeric        $userid (optional)
     *
     * @return int
     */
    static public function check_missed_deadlines_by_course($course_or_id, $userid=null){
        $MM = NED::$MM;
        $filter = [$MM::BY_ACTIVITY_USER, $MM::USE_DEADLINE, $MM::NGC_MISSED_DEADLINE, $MM::NOT_EXCLUDED, $MM::USE_DEF_STUDENTS_ONLY];
        $data = static::_get_courseid_and_mm_data_by_filter_for_check($course_or_id, $filter, $userid);
        if (empty($data)){
            return 0;
        }

        [$courseid, $mm_data_list] = $data;
        NED::cli_debugging('[check_missed_deadlines] Found '.count($mm_data_list).' records');
        return static::check_and_save_from_mm($mm_data_list, static::GT_AWARD_ZERO, static::REASON_SUBMISSION,
            0, $courseid);
    }

    /**
     * Check late submissions by course
     * Return number of successfully added (and saved) records
     *
     * @param object|numeric $course_or_id
     * @param numeric        $userid (optional)
     *
     * @return int
     */
    static public function check_late_submissions_by_course($course_or_id, $userid=null){
        $MM = NED::$MM;
        $filter = [$MM::BY_ACTIVITY_USER, $MM::USE_DEADLINE, $MM::NGC_LATE_SUBMISSIONS, $MM::ONLY_NGC, $MM::NOT_EXCLUDED, $MM::USE_DEF_STUDENTS_ONLY];
        $data = static::_get_courseid_and_mm_data_by_filter_for_check($course_or_id, $filter, $userid);
        if (empty($data)){
            return 0;
        }

        [$courseid, $mm_data_list] = $data;
        NED::cli_debugging('[check_late_submissions] Found '.count($mm_data_list).' records');
        return static::check_and_save_from_mm($mm_data_list, static::GT_DEDUCTION, static::REASON_SUBMISSION,
            20, $courseid);
    }

    /**
     * Check fixed submissions by course
     * Return number of successfully added (and saved) records
     *
     * @param object|numeric $course_or_id
     * @param numeric        $userid (optional)
     *
     * @return int
     */
    static public function check_fixed_submissions_by_course($course_or_id, $userid=null){
        $MM = NED::$MM;
        $filter = [$MM::BY_ACTIVITY_USER, $MM::USE_DEADLINE, $MM::NGC_FIXED_SUBMISSIONS, $MM::ONLY_NGC, $MM::USE_DEF_STUDENTS_ONLY];
        $data = static::_get_courseid_and_mm_data_by_filter_for_check($course_or_id, $filter, $userid);
        if (empty($data)){
            return 0;
        }

        [$courseid, $mm_data_list] = $data;
        NED::cli_debugging('[check_fixed_submissions] Found '.count($mm_data_list).' records');
        return static::check_and_save_from_mm($mm_data_list, static::GT_DEDUCTION, static::REASON_FILE,
            0, $courseid);
    }

    /**
     * Check and fix graders and deadlines
     * Update and delete records, if it needs
     *
     * @param array|object[]|NGC_record|NGC_record[] $records - check records after this number
     *
     * @return array|int[] - array($c_upd, $c_del) - number of updated and deleted records
     */
    static public function check_and_fix_records($records){
        $records = NED::val2arr($records);
        $to_upd = [];
        $to_del = [];
        $now = time();
        /** @var NGC_record $record */
        foreach ($records as $record){
            if ($record->status == static::ST_ERROR){
                continue;
            }
            $error = [];

            $user = NED::get_user($record->userid);
            if (!$user || !empty($user->deleted)){
                // no such user or it was deleted
                $to_del[] = $record;
                continue;
            }

            if (!NED::role_is_user_default_student($user)){
                // user is not "default student" anymore
                $to_del[] = $record;
                continue;
            }

            $cmid = NED::get_cm_by_cmid($record->cmid, $record->courseid);
            if (!$cmid){
                $error[] = 'Can not find this course module!';
            }

            if (!empty($error)){
                $record->note = ($record->note ?? '') . "\nSYSTEM ERROR: ".join("\n", $error);
                $record->status = static::ST_ERROR;
                $to_upd[] = $record;
                continue;
            }

            $gg = NED::get_grade_grade($cmid, $record->userid, false);
            if (!empty($gg->excluded)){
                $to_del[] = $record;
                continue;
            }

            $upd = false;
            $deadline = NED::get_deadline_by_cm($record->cmid, $record->userid, $record->courseid);
            if ($record->deadline != $deadline){
                $record->deadline = $deadline;
                $upd = true;
            }

            if ($record->reason == static::REASON_SUBMISSION){
                $deadline_penalty = $deadline ? ($deadline + NED::DEADLINE_PENALTY_GRACE_PERIOD) : 0;
                if ($deadline_penalty > $now){
                    $to_del[] = $record;
                    continue;
                }

                $submitted_time = NED::get_submitted_time_by_cm($record->cmid, $record->userid);
                if ($submitted_time && $deadline_penalty > $submitted_time){
                    $to_del[] = $record;
                    continue;
                }

                if ($record->cm_type == NED::MOD_ASSIGN_KEY && $record->status == static::ST_DONE){
                    if (NED::get_attempt_by_cm($record->cmid, $record->userid) > 0){
                        $to_del[] = $record;
                        continue;
                    }
                }
            }

            $graderid = NED::get_graderid_by_studentid($record->userid, $record->courseid, $record->cmid);
            if ($record->graderid != $graderid){
                $record->graderid = $graderid;
                $upd = true;
            }

            if ($upd){
                $to_upd[] = $record;
            }
        }

        $c_del = count($to_del);
        if ($c_del){
            static::check_and_delete($to_del);
        }
        $c_upd = count($to_upd);
        if ($c_upd){
            static::update_records($to_upd);
        }

        return [$c_upd, $c_del];
    }

    /**
     * Check and fix graders and deadlines
     * Update and delete records, if it needs
     *
     * @param int   $cmid
     * @param int   $userid
     * @param array $options
     *
     * @return array|int[] - array($c_upd, $c_del) - number of updated and deleted records
     */
    static public function check_and_fix_records_by_params($cmid=0, $userid=0, $options=[]){
        $ngc_records = static::get_records_by_params($cmid, $userid, $options);
        if (!empty($ngc_records)){
            return static::check_and_fix_records($ngc_records);
        }
        return [0, 0];
    }

    /**
     * Check data and delete record(s) (if there are no errors)
     *
     * @param int   $cmid
     * @param int   $userid
     * @param array $options
     * @param bool  $check
     *
     * @return bool|int - count of deleted records or false
     */
    static public function check_and_delete_records_by_params($cmid=0, $userid=0, $options=[], $check=false){
        $ngc_records = static::get_records_by_params($cmid, $userid, $options);
        if (!empty($ngc_records)){
            return static::check_and_delete($ngc_records, $check);
        }
        return false;
    }

    /* EVENTS */

    /**
     * @param \core\event\base | \mod_assign\event\group_override_created | \mod_assign\event\group_override_updated | \mod_assign\event\group_override_deleted | \mod_quiz\event\group_override_created | \mod_quiz\event\group_override_updated | \mod_quiz\event\group_override_deleted  $event
     */
    static public function group_override_updated_event($event){
        $groupid = $event->other['groupid'] ?? 0;
        if (!$groupid){
            return;
        }

        $userids = NED::get_group_users($groupid, true);
        if (empty($userids)){
            return;
        }

        $options = ['userid' => $userids];
        $options['courseid'] = $event->courseid;
        $cmid = $event->contextinstanceid ?? 0;
        if ($cmid){
            $options['cmid'] = $cmid;
        }

        $records = static::get_records_by_in_or_equal_options($options);
        static::check_and_fix_records($records);
    }

    /**
     * @param \core\event\base | \mod_assign\event\user_override_created | \mod_assign\event\user_override_updated | \mod_assign\event\user_override_deleted | \mod_quiz\event\user_override_created | \mod_quiz\event\user_override_updated | \mod_quiz\event\user_override_deleted  $event
     */
    static public function user_override_updated_event($event){
        $options = [];
        $graderid = $event->userid; // who update
        // if relateduserid is null, then user updated himself
        $options['userid'] = $event->relateduserid ?? $graderid;
        $options['courseid'] = $event->courseid;
        $cmid = $event->contextinstanceid ?? 0;
        if ($cmid){
            $options['cmid'] = $cmid;
        }
        $records = static::get_records($options);
        static::check_and_fix_records($records);
    }


    /**
     * @param \mod_assign\event\assessable_submitted $event
     */
    static public function assign_submission_submitted_event($event){
        $cmid = $event->contextinstanceid ?? 0;
        if (!$cmid){
            return;
        }

        $graderid = $event->userid; // who update
        // if relateduserid is null, then user updated himself
        $userid = $event->relateduserid ?? $graderid;
        $courseid = $event->courseid;

        // check, should we update draft wrong submission record
        $records = static::get_records_by_params_done($cmid, $userid, ['reason' => static::REASON_FILE]);
        if (!empty($records)){
            $record = reset($records);
            $deadline_penalty = $record->deadline ? ($record->deadline + NED::DEADLINE_PENALTY_GRACE_PERIOD) : 0;
            if ($deadline_penalty < time()){
                static::check_late_submissions_by_course($courseid, $userid);
            } else {
                static::check_fixed_submissions_by_course($courseid, $userid);
            }
        }
    }

    /**
     * @param \local_ned_controller\event\submission_removed $event
     */
    static public function assign_submission_removed_event($event){
        $cmid = $event->contextinstanceid ?? 0;
        if (!$cmid){
            return;
        }

        $graderid = $event->userid; // who update
        // if relateduserid is null, then user updated himself
        $userid = $event->relateduserid ?? $graderid;
        $courseid = $event->courseid;

        // check, should we update draft wrong submission record
        $records = static::get_records_by_params_done($cmid, $userid, ['reason' => static::REASON_FILE]);
        if (!empty($records)){
            $record = reset($records);
            $deadline_penalty = $record->deadline ? ($record->deadline + NED::DEADLINE_PENALTY_GRACE_PERIOD) : 0;
            if ($deadline_penalty < time()){
                static::check_missed_deadlines_by_course($courseid, $userid);
            }
        }
    }

    /**
     * Load all "human" data by record and send it to the string manager
     *
     * @param object|NGC_record $record
     * @param string            $str_key
     *
     * @return string
     */
    static public function get_format_record_string($record, $str_key){
        $data = static::$NGC_RENDER::get_description_object($record);
        return NED::str($str_key, $data);
    }

    /**
     * Returns count of active Wrong submissions, Late submissions, Missed Deadlines ngc records by students ids
     *
     * @param array $student_ids
     * @param numeric|null $lastdays - count only for some last days (num of days)
     *
     * @return object[] - array with userid as key and
     *                    objects with properties "userid", "wrong_submissions", "late_submissions" and "missed_deadlines"
     */
    static public function get_students_ngc_records_count($student_ids, $lastdays=0){
        if (empty($student_ids) || !is_array($student_ids)) return [];

        $show_statuses = [];
        foreach (static::STATUSES as $status => $st_lang_key){
            if (self::HIDDEN_STATUSES[$status] ?? false) continue;
            $show_statuses[] = $status;
        }

        if (empty($show_statuses)) return [];

        $add_sum_select = function($reason, $grade_type = null) use ($show_statuses){
            $conditions = ['ngc.reason = '.$reason];
            if (!empty($grade_type)){
                $conditions[] = 'ngc.grade_type = '.$grade_type;
            }
            $conditions[] = 'ngc.status IN ('.join(',', $show_statuses).')';

            return 'SUM('.NED::sql_condition($conditions).')';
        };

        $select = ['ngc.userid'];
        $select[] = $add_sum_select(static::REASON_FILE).' AS wrong_submissions';
        $select[] = $add_sum_select(static::REASON_SUBMISSION, static::GT_DEDUCTION).' AS late_submissions';
        $select[] = $add_sum_select(static::REASON_SUBMISSION, static::GT_AWARD_ZERO).' AS missed_deadlines';

        [$col_sql, $col_params] = NED::db()->get_in_or_equal($student_ids, SQL_PARAMS_QM);
        $where = ["ngc.userid $col_sql"];
        $groupby = ['ngc.userid'];

        if ($lastdays) {
            $time = time() - $lastdays * DAYSECS;
            $where[] = "ngc.timecreated > $time";
        }

        $sql = NED::sql_generate($select, [], static::TABLE, 'ngc', $where, $groupby);
        return NED::db()->get_records_sql($sql, $col_params);
    }

    /**
     * Check, that some grader exists in the table
     * Additional, you can check grader for some specific student or course
     *
     * @param numeric|object $grader_or_id
     * @param numeric|object $student_or_id
     * @param numeric|object $course_or_id
     *
     * @return bool
     */
    static public function check_grader_exists($grader_or_id=null, $student_or_id=null, $course_or_id=null){
        $conditions = [
            'graderid' => NED::get_id($grader_or_id),
            'userid' => NED::get_id($student_or_id),
            'courseid' => NED::get_id($course_or_id),
        ];

        return static::record_exists(NED::sql_filter_params($conditions));
    }

    /**
     * Check, that some user can be grader for the NGC
     * Additional, you can check grader ability for some specific student or course
     *
     * @param numeric|object $grader_or_id
     * @param numeric|object $student_or_id
     * @param numeric|object $course_or_id
     *
     * @return bool
     */
    static public function ngc_is_grader($grader_or_id=null, $student_or_id=null, $course_or_id=null){
        // simple check, that much faster then others
        if (static::check_grader_exists($grader_or_id, $student_or_id, $course_or_id)) return true;

        $studentid = NED::get_id($student_or_id);
        $courseid = NED::get_id($course_or_id);
        if ($studentid){
            $grader_courses = NED::get_courses_for_grader_and_student($grader_or_id, $studentid);
            if ($courseid){
                return !empty($grader_courses[$courseid]);
            }

            return !empty($grader_courses);
        } else {
            return NED::user_is_grader($grader_or_id, $courseid ?: false);
        }
    }
}
