<?php

/**
 * Badges renderer
 *
 * @package    local_ned_controller
 * @category   core
 * @copyright  2021 NED {@link http://ned.ca}
 * @author     NED {@link http://ned.ca}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace local_ned_controller\output\core;
use local_ned_controller\shared_lib as NED;

defined('MOODLE_INTERNAL') || die;

/** @var \stdClass $CFG */
require_once($CFG->dirroot.'/badges/renderer.php');
require_once($CFG->dirroot.'/lib/enrollib.php');

/**
 * Class badges_renderer
 * @package local_ned_controller\output\core
 */
class badges_renderer extends \core_badges_renderer
{
    const TYPE_CERTIFICATES = 'certificates';
    const TYPE_RECOGNITION = 'recognition';
    const TYPE_COURSES = 'courses';
    const TYPE_OTHER = 'other';
    const USER_DEFAULT_ROLE = 'Student';

    const BADGES_TYPES = [
        self::TYPE_CERTIFICATES,
        self::TYPE_RECOGNITION,
        self::TYPE_COURSES,
        self::TYPE_OTHER,
    ];

    const FIELD_CODE = 'badge_code';

    static protected $_badge_filed_id = null;

    /**
     * badges_renderer constructor.
     *
     * @param \moodle_page $page
     * @param string       $target
     */
    public function __construct(\moodle_page $page, string $target){
        global $DB;
        parent::__construct($page, $target);
        if (is_null(static::$_badge_filed_id)){
            static::$_badge_filed_id = $DB->get_field('customfield_field', 'id', ['shortname' => static::FIELD_CODE]);
        }
    }

    /**
     * @param null $badges
     * @param int  $userid
     * @param bool $profile
     * @param bool $external
     *
     * @return string
     */
    public function print_badges_list($badges=null, $userid=0, $profile=false, $external=false){
        global $USER, $OUTPUT, $COURSE;

        $str = function($identifier, $a = null){
            return get_string($identifier, 'badges', $a);
        };
        $alink = function($url, $text='', $icon=null){
            return new \action_menu_link_secondary(new \moodle_url($url), $icon, $text);
        };

        $userid = $userid ?: $USER->id;
        $badges = $badges ?? badges_get_user_badges($userid);
        $my_courses = enrol_get_users_courses($userid, false,'*');
        $all_badges = [];
        $cat_badges = [];
        foreach ($badges as $badge){
            $all_badges[$badge->id] =   $badge;
        }
        foreach (static::BADGES_TYPES as $type){
            $cat_badges[$type] = [];
        }
        foreach ($my_courses as $my_course) {
            $code = $this->get_course_badge_code($my_course);
            if (empty($code)){
                continue;
            }

            $badge_by_code = static::get_badge_by_code($code);
            if (empty($badge_by_code) || isset($all_badges[$badge_by_code->id ?? 0])){
                continue;
            }

            $badge_postfix = mb_substr($badge_by_code->version, -2);
            $key = $this->get_type_code_badge($badge_postfix);
            if ($key == static::TYPE_COURSES){
                if ($USER->profile[NED::FIELD_DEFAULT_ROLE] == static::USER_DEFAULT_ROLE){
                    $all_badges[$badge_by_code->id] = $badge_by_code;
                }
            } else {
                $all_badges[$badge_by_code->id] = $badge_by_code;
            }

            foreach (static::get_user_badge_by_code($code, $userid) as $badge) {

                $badge_postfix = mb_substr($badge->version, -2);
                $key = $this->get_type_code_badge($badge_postfix);
                if ($key == static::TYPE_COURSES){
                    if ($USER->profile[NED::FIELD_DEFAULT_ROLE] != static::USER_DEFAULT_ROLE){
                        continue;
                    }
                }

                $all_badges[$badge_by_code->id] = $badge;
            }
        }

        $base_actions = [
            'badge_page' => $alink('/badges/badge.php', NED::str('openbadgepage')),
            'overview' => $alink('/badges/overview.php', $str('boverview')),
            'recipients' => $alink('/badges/recipients.php', $str('awards')),
        ];
        $add_badge_revoke = local_ned_controller_has_capability_revokebadge($COURSE->id);

        if (empty($all_badges)) {
            return \html_writer::div(get_string('nothingtodisplay', 'block_badges'), 'nothingtodisplay');
        }
        foreach ($all_badges as $badge){
            $badge->url = '';
            if (!empty($badge->uniquehash)) {
                $badge->url = new \moodle_url('/badges/badge.php', array('hash' => $badge->uniquehash));
            }
            $badge_image = $this->badge_image_html($badge);
            $action_bname = $bname = $badge_image.$badge->name;

            if (!empty($badge->uniquehash)){
            $badge->url = new \moodle_url('/badges/badge.php', array('hash' => $badge->uniquehash));
            $menu = new \action_menu();
            $menu->set_menu_trigger($bname);

            /** @var \action_menu_link_secondary | string $action */
            foreach ($base_actions as $key => $action) {
                if (!is_number($key)){
                    switch ($key){
                        case 'badge_page':
                                $action->url->param('hash', $badge->uniquehash);
                            break;
                        case 'overview':
                        case 'recipients':
                            $action->url->param('id', $badge->id);
                            break;
                        default:break;
                    }
                }
                $menu->add($action);
            }
            if ($add_badge_revoke){
                $menu->add(
                    \html_writer::link('#',
                        NED::str('revokebadge', 'local_ned_controller'),
                        ['class' => 'ned_revoke_badge', 'data-hash' => $badge->uniquehash])
                );
            }
            $action_bname = $OUTPUT->render($menu);
            }

            $badge_postfix = mb_substr($badge->version, -2);

            if (empty($badge_postfix)){
                $key =  static::TYPE_OTHER;
            } else {
                $key = $this->get_type_code_badge($badge_postfix);
            }
            $badge->html = $action_bname;
            $cat_badges[$key][] = $badge;
        }

        if ($add_badge_revoke){
            local_ned_controller_call_revokebadge_js($userid);
        }
        $output = '';
        foreach ($cat_badges as $type => $badge_cat){
            $output .= $this->badge_line_html(NED::str($type), $badge_cat);
        }
        $output = \html_writer::div($output, 'theme-ned-controller-badges');
        return $output;
    }

    /**
     * @param $badge_postfix
     * @return string
     */
    protected function get_type_code_badge($badge_postfix){
        $key = match ($badge_postfix) {
            '-C' => static::TYPE_CERTIFICATES,
            '-R' => static::TYPE_RECOGNITION,
            default => static::TYPE_COURSES,
        };
        return $key;
    }

    /**
     * @param $badge
     * @return string
     * @throws \dml_exception
     */
    function badge_image_html($badge){
        $context = ($badge->type == BADGE_TYPE_SITE) ? \context_system::instance() : \context_course::instance($badge->courseid);
        $badgeid = $badge->badgeid ?? $badge->id;

        $imgclass = 'nd-img-badge';
        $imgtitle = '';

        if (empty($badge->url)){
            $imgclass .= ' shadow-badge';
            $imgtitle = NED::str('inprogress');
        }

        $badge_image_url = \moodle_url::make_pluginfile_url($context->id, 'badges', 'badgeimage', $badgeid,
            '/', 'f1', false);
        return \html_writer::img($badge_image_url, $badge->version, ['class' => $imgclass, 'title' => $imgtitle]);
    }

    /**
     * @param $name
     * @param $badges
     *
     * @return string
     */
    protected function badge_line_html($name, $badges): string{
        if (empty($badges)){
            return '';
        }

        $rendered_badges = ['real' => '', 'shadow' => ''];
        foreach ($badges as $badge) {
            $key = 'real';
            if (empty($badge->url)){
                $key = 'shadow';
            }
            $rendered_badges[$key] .= \html_writer::div($badge->html,'nd-block-badge');
        }

        $output = \html_writer::span($name, 'badge-row-name') . $rendered_badges['real'] . $rendered_badges['shadow'];
        $output = \html_writer::div($output, 'badges-row');
        return $output;
    }

    /**
     * @param \stdClass $course_record
     *
     * @return string|null
     */
    protected function get_course_badge_code($course_record){
        if (!$course_record || !static::$_badge_filed_id){
            return null;
        }

        $course = new \core_course_list_element($course_record);
        $customfields = $course->get_custom_fields();
        $field = $customfields[static::$_badge_filed_id] ?? null;
        if (!$field){
            return null;
        }

        return $field->export_value();
    }

    /**
     * @param $code
     * @param $userid
     *
     * @return array
     */
    static public function get_user_badge_by_code($code, $userid){
        return static::get_badges_sql(
            'bi.*, b.version, b.courseid, b.type, b.name',
            'JOIN {badge_issued} bi ON bi.badgeid = b.id',
            'bi.userid = :userid',
            ['userid' => $userid], $code);
    }

    /**
     * @param $code
     *
     * @return object|null
     */
    static public function get_badge_by_code($code){
        return static::get_badges_sql('b.*', [], [], [], $code, true);
    }

    /**
     * @param array|string $select
     * @param array|string $join
     * @param array|string $where
     * @param array $params
     * @param null|int|string $version
     * @param bool  $only_one = false
     *
     * @return array|\stdClass|null
     */
    static public function get_badges_sql($select=['b.*'], $join=[], $where=[], $params=[], $version=null, $only_one=false){
        global $DB;
        $select = !is_array($select) ? [$select] : $select;
        $join = !is_array($join) ? [$join] : $join;
        $where = !is_array($where) ? [$where] : $where;

        $where[] = 'b.status != :deleted';
        $params['deleted'] = BADGE_STATUS_ARCHIVED;
        if (!is_null($version)){
            $where[] = 'b.version = :code';
            $params['code'] = $version;
        }

        $select = "SELECT ". join(', ', $select);
        $from = "\nFROM {badge} b ".join("\n", $join);
        $where = !empty($where) ? ("\nWHERE (" . join(') AND (', $where) . ')') : '';

        $records = $DB->get_records_sql("$select $from $where", $params);
        if ($only_one){
            return $records ? reset($records) : null;
        }
        return $records ?: [];
    }
}
