<?php

/**
 * The primary renderer for the calendar.
 *
 * @package    local_ned_controller
 * @subpackage output
 * @category   NED
 * @copyright  2021 NED {@link http://ned.ca}
 * @author     NED {@link http://ned.ca}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace local_ned_controller\output;

use local_ned_controller\shared_lib as NED;

defined('MOODLE_INTERNAL') || die;

/**
 * NED Class core_calendar_renderer
 *
 * @package local_ned_controller\output
 */
class core_calendar_renderer extends \core_calendar_renderer
{
    /**
     * Renders a template by name with the given context.
     *
     * The provided data needs to be array/stdClass made up of only simple types.
     * Simple types are array, stdClass, bool, int, float, string
     *
     * NED:
     * If one of the event - is assign resubmission, change its name
     *
     * @param string          $templatename
     * @param array|\stdClass $context Context containing data for the template.
     *
     * @return string|boolean
     */
    public function render_from_template($templatename, $context){
        $context = NED::val2obj($context);

        if (!empty($context->events)){
            foreach ($context->events as &$event){
                static::check_event($event);
            }
        }

        return parent::render_from_template($templatename, $context);
    }

    /**
     * Check event, if it assign for resubmission - change it name
     *
     * @param \stdClass $event
     */
    static public function check_event(&$event){
        if (empty($event) || !is_object($event)) return;
        if (empty($event->modulename) || empty($event->instance) || empty($event->userid) || empty($event->name)) return;
        if ($event->modulename != 'assign') return;

        try {
            $assign = NED::$ned_assign::get_assign_by_cm($event->instance, $event->course ?? null);
        } catch (\Throwable){
            return;
        }

        $submission = $assign->get_needed_user_submission($event->userid);
        if (($submission->attemptnumber ?? 0) < 1) return;

        // OK, this is resubmission, change the name
        $postfix = ' - '.NED::str('resubmission');
        $event->name .= $postfix;
        if (!empty($event->popupname)){
            $event->popupname .= $postfix;
        }
    }
}
