<?php
/**
 * @package    local_ned_controller
 * @subpackage custom_ned_menu
 * @category   NED
 * @copyright  2022 NED {@link http://ned.ca}
 * @author     NED {@link http://ned.ca}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace local_ned_controller\output\custom_ned_menu;

use \local_ned_controller\shared_lib as NED;

/**
 * Class settings_parser
 * Used by {@see custom_ned_menu} as parser and root storage
 *
 * @package local_ned_controller\output\custom_ned_menu
 */
class settings_parser
{
    const SYMBOL_LINE_DELIMITER = '\n';
    const SYMBOL_SECTION_DELIMITER = '|';
    const SYMBOL_GROUP_DELIMITER = ';';
    const SYMBOL_COMMENT = '#';

    /** @see custom_ned_menu::_is_title() */
    /** @see custom_ned_menu::_format_title_input() */
    const SYMBOL_TITLE_START = '[';
    const SYMBOL_TITLE_END = ']';
    /** @see custom_ned_menu::_get_option_level() */
    /** @see custom_ned_menu::_format_subitem_label() */
    const SYMBOL_SUBITEM = '-';

    const SYMBOL_ROLE_NEG = '-';
    /** @see settings_parser::parse_url */
    const SYMBOL_URL_SPECIAL = '#';

    protected const _REGEX_SECTIONS_BEFORE = '(?:'.self::SYMBOL_LINE_DELIMITER.'|^)'.
        '(?:[^'.self::SYMBOL_SECTION_DELIMITER.self::SYMBOL_LINE_DELIMITER.']*['.self::SYMBOL_SECTION_DELIMITER.'])';
    protected const _REGEX_SECTION_FIND = '([^'.self::SYMBOL_SECTION_DELIMITER.'\\'.self::SYMBOL_TITLE_END.self::SYMBOL_LINE_DELIMITER.']+)';

    const ROLE_SECTION_INDEX = 2;

    /** @var string - if you will change it, check {@see settings_parser::parse_url()} */
    const REGEX_URL_SECTION = '/^'.self::SYMBOL_URL_SPECIAL.'section\-(\d*)/';
    const REGEX_ROLES = '/'.self::_REGEX_SECTIONS_BEFORE.'{'.self::ROLE_SECTION_INDEX.'}'.self::_REGEX_SECTION_FIND.'/';

    protected $_courseid = 0;
    protected $_userid = 0;
    /** @var string - unparsed settings */
    protected $_raw_settings = '';
    protected $_init = false;
    protected $_use_units = false;
    /** @var \moodle_url[] - [name => url] */
    protected $_activity_urls = [];
    protected $_user_roles = [];
    protected $_is_admin = false;

    /** @var custom_ned_menu|null|false */
    protected $_current_item;
    /** @var custom_ned_menu|null|false */
    protected $_header_item;
    /** @var custom_ned_menu_header */
    protected $_menu_course_header;

    /**
     * @constructor settings_parser
     *
     * @param string         $raw_settings - initial raw text settings
     * @param bool           $use_units - set option of using units
     * @param numeric|object $course_or_id - course or its id (global by default)
     * @param numeric|object $user_or_id - user or its id (global by default)
     */
    public function __construct($raw_settings='', $use_units=false, $course_or_id=null, $user_or_id=null){
        $this->_courseid = NED::get_courseid_or_global($course_or_id);
        $this->_userid = NED::get_userid_or_global($user_or_id);
        // we need to save page url, as we will called it many times
        $this->_raw_settings = trim($raw_settings);
        $this->_use_units = $use_units;
        $this->_is_admin = is_siteadmin();
    }

    /**
     * Init general settings data
     *
     * @param string|null $set_input     - optional, another settings input
     * @param bool|null   $set_use_units - optional, change use_units option
     * @param bool        $force         - optional, if true, init object even if it have been already init
     */
    public function init($set_input='', $set_use_units=null, $force=false){
        if ($this->_init && !$force) return;

        if (isset($set_input[0])){
            $this->_raw_settings = trim($set_input);
        }
        if (isset($set_use_units)){
            $this->_use_units = $set_use_units;
        }
        $this->_init_activities();
        $this->_init_roles();

        $this->_init = true;
    }

    /**
     * Init course activities and urls
     */
    protected function _init_activities(){
        $this->_activity_urls = static::get_activity_urls($this->_courseid);
    }

    /**
     * Init settings and user roles, if need
     */
    protected function _init_roles(){
        if ($this->can_skip_role_checking()) return;

        $this->_user_roles = static::parse_user_roles($this->_raw_settings, $this->_courseid, $this->_userid);
    }

    /**
     * Return, does setting have such role (and it's real role)
     *
     * @param string $rolename - any role name, alias or id
     *
     * @return bool
     */
    protected function _is_role_exists($rolename){
        return isset($this->_user_roles[$rolename]);
    }

    /**
     * Parse string roles input into array of role condition, which roles user should or shouldn't have
     *
     * @param $input string - like "role1, -role2, role3"
     *
     * @return bool[] - [rolename => should/not have such role]
     */
    protected function _parse_input_roles($input){
        $roles = explode(static::SYMBOL_GROUP_DELIMITER, strtolower($input));
        $asked_roles = [];
        foreach ($roles as $role){
            $role = trim($role);
            if (!$role) continue;

            $has_role = $role[0] !== static::SYMBOL_ROLE_NEG;
            if (!$has_role){
                $role = substr($role, 1);
            }

            if (!$this->_is_role_exists($role)) continue;

            $asked_roles[$role] = $has_role;
        }

        return $asked_roles;
    }

    /**
     * Check parsed roles, does current user pass any of the role options
     *
     * @param bool[] $roles - array of role condition, which roles user should or shouldn't have: [rolename => should/not have such role]
     *                      {@see settings_parser::_parse_input_roles()}
     * @param bool $if_empty - default value, if there are no input roles
     *
     * @return bool
     */
    protected function _check_parsed_roles($roles, $if_empty=false){
        $roles = NED::val2arr($roles);
        foreach ($roles as $role => $has_role){
            if (!$this->_is_role_exists($role)) continue;

            if ($this->_user_roles[$role] === $has_role){
                return true;
            }
        }

        return (bool)$if_empty;
    }

    /**
     * @return bool
     */
    public function is_use_units(){
        return $this->_use_units;
    }

    /**
     * Get current active menu item from the menu
     *
     * @param custom_ned_menu $menu
     *
     * @return custom_ned_menu|null
     */
    public function get_current_menu_item($menu=null){
        if (is_null($this->_current_item) && $menu){
            $page_url = NED::page()->url;

            $p_url_params = $page_url->params();
            $p_params_count = count($p_url_params);

            $last_item = false;
            $last_match = 0;
            $last_count = 0;
            $last_delta = 0;

            foreach ($menu->iterate() as $item){
                $url = $item->get_url();
                if (!$url) continue;

                $check = $url->compare($page_url, URL_MATCH_PARAMS);
                if ($check){
                    if (!$p_params_count){
                        $last_item = $item;
                        break;
                    }

                    $item_params = $url->params();
                    $item_params_count = count($item_params);
                    $need_check = ($last_match < $p_params_count && $item_params_count > $last_match) ||
                        ($last_count > $p_params_count && $item_params_count < $last_count && $item_params_count >= $last_match);

                    if (!$need_check) continue;

                    $match = 0;
                    foreach ($item_params as $param => $val){
                        if (isset($p_url_params[$param]) && $p_url_params[$param] == $val) $match++;
                    }

                    $delta = $item_params_count - $match;
                    if ($match > $last_match || $delta < $last_delta){
                        $last_item = $item;
                        $last_count = $item_params_count;
                        $last_delta = $delta;
                        $last_match = $match;
                        if ($match == $p_params_count && $p_params_count == $item_params_count){
                            // 100% match
                            break;
                        }
                    }
                }
            }

            if ($last_item){
                $last_item->set_current();
            }

            $this->_current_item = $last_item;
        }

        return $this->_current_item ?: null;
    }

    /**
     * Reset current item, so you will need to search the new one
     */
    public function reset_current_menu_item(){
        $this->_current_item = null;
    }

    /**
     * Get current active menu item from the menu
     *
     * @param custom_ned_menu $menu
     *
     * @return custom_ned_menu|null
     */
    public function get_header_menu_item($menu=null){
        if (is_null($this->_header_item) && $menu){
            $top_parent = $menu->get_top_parent();
            if ($this->is_use_units()){
                $menu->init_current_active_item();
                $this->_header_item = $top_parent->find_active_child() ?: $top_parent;
            } else {
                $this->_header_item = $top_parent;
            }
        }

        return $this->_header_item ?: null;
    }

    /**
     * Reset header item, so you will need to search the new one
     */
    public function reset_header_menu_item(){
        $this->_header_item = null;
    }

    /**
     * Get custom_ned_menu__header object
     *
     * @param custom_ned_menu $menu
     *
     * @return custom_ned_menu_header
     */
    public function get_menu_course_header($menu=null){
         if (is_null($this->_menu_course_header) && $menu){
             $this->_menu_course_header = new custom_ned_menu_header($menu);
         }
         return $this->_menu_course_header;
    }

    /**
     * If true, we can skip parsing and checking of roles
     *
     * @return bool
     */
    public function can_skip_role_checking(){
        return $this->_is_admin;
    }

    /**
     * Check user capability to view this element
     *
     * @param string $input_roles_condition - string of the list of the roles, which user should or shouldn't have,
     *                                      like "role1, -role2, role3"
     *
     * @return bool - pass user or not this role conditions
     */
    public function check_unparsed_roles($input_roles_condition){
        // if there are nothing here, then we pass
        if ($this->can_skip_role_checking() || !isset($input_roles_condition[0])) return true;

        $roles = $this->_parse_input_roles($input_roles_condition);
        // if there are no roles here, but there were something before, user doesn't pass
        return $this->_check_parsed_roles($roles, false);
    }

    /**
     * If url - "#", then return activity link (by $activity_label),
     *  if url - "#section-$num" - return link to course with section $num (if $num - is number),
     *  otherwise return $unparsed_url as new moodle url
     *
     * @param string $unparsed_url - raw option value which should be url
     * @param string $item_label   - used if our url for activity, then we find it by label
     *
     * @return \moodle_url|false
     */
    public function parse_url($unparsed_url, $item_label){
        if (empty($unparsed_url)) return false;

        if (NED::str_starts_with_s($unparsed_url, static::SYMBOL_URL_SPECIAL)){
            // Check activity names
            if (!isset($unparsed_url[1])){
                if (isset($this->_activity_urls[$item_label])){
                    return new \moodle_url($this->_activity_urls[$item_label]);
                } else {
                    return false;
                }
            }

            // Get the full url for section anchors.
            $section_num = static::_get_parsed_section($unparsed_url);
            if (!is_null($section_num)){
                return new \moodle_url("/course/view.php", ['id' => $this->_courseid, 'section' => $section_num]);
            }
        }

        return new \moodle_url($unparsed_url);
    }

    // static
    /**
     * Check section in input string and return its id (or null)
     *
     * @param string $input url
     *
     * @return int|null
     */
    static protected function _get_parsed_section($input){
        if (!empty($input)) {
            $res = [];
            preg_match(static::REGEX_URL_SECTION, $input, $res);
            // Section number here
            if (isset($res[1]) && is_numeric($res[1])) return $res[1];
        }

        return null;
    }

    /**
     * Get activities names and its urls
     *
     * @param numeric $courseid
     *
     * @return \moodle_url[] - array [cm_name => cm_url]
     */
    static public function get_activity_urls($courseid){
        $cms = NED::get_course_cms($courseid);
        $activity_urls = [];
        foreach ($cms as $cm){
            if (!NED::check_activity_visible_by_cm($cm)) continue;

            $cm_name = s(trim(strip_tags($cm->name)));
            if (empty($cm_name)) continue;

            $activity_urls[$cm_name] = $cm->url;
        }
        return $activity_urls;
    }

    /**
     * Parse raw input and checks if the user has any role by any of its name, alias or id
     * Case insensitive
     *
     * @param string         $raw_input    - full string from config->text variable, like ...|...|role1,-role2,role3
     * @param numeric|object $course_or_id - optional, if provided, checks roles in the course context (and there names) too
     * @param numeric|object $user_or_id   - optional, user for whom check roles (global USER by default)
     *
     * @return bool[] - role array as [role_name => (user has or not any role with id)]
     *                where array keys will be from the $rolenames
     */
    static public function parse_user_roles($raw_input='', $course_or_id=null, $user_or_id=null){
        $res = [];
        preg_match_all(static::REGEX_ROLES, $raw_input, $res);
        if (empty($res[1])) return [];

        $ask_rolenames = [];
        foreach ($res[1] as $str_roles){
            $str_roles = trim($str_roles);
            if (!isset($str_roles[0])) continue;

            $roles = explode(static::SYMBOL_GROUP_DELIMITER, strtolower($str_roles));
            foreach ($roles as $role){
                $role = trim($role);
                if (!isset($role[0])) continue;

                if ($role[0] == static::SYMBOL_ROLE_NEG){
                    $role = substr($role, 1);
                }

                if (isset($ask_rolenames[$role])) continue;

                $ask_rolenames[$role] = true;
            }
        }

        return NED::role_check_user_roleids_by_any_names(array_keys($ask_rolenames), $course_or_id, $user_or_id, false);
    }
}
