<?php
/**
 * @package    local_ned_controller
 * @category   output
 * @copyright  2021 NED {@link http://ned.ca}
 * @author     NED {@link http://ned.ca}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace  local_ned_controller\output;

defined('MOODLE_INTERNAL') || die();

/**
 * Class html_list
 *
 * @package local_ned_controller
 */
class html_list implements \renderable
{
    const TYPE_MARK = 'ul';
    const TYPE_NUM = 'ol';

    public $type = self::TYPE_MARK;
    public $list = [];
    public $classes = '';
    public $attributes = [];

    /**
     * html_list constructor.
     *
     * @param array  $classes
     * @param string $type
     * @param array  $attributes
     * @param array  $list
     */
    public function __construct($classes=[], $type=self::TYPE_MARK, $attributes=[], $list=[])
    {
        $this->list = $list ?: [];
        if (is_string($classes)){
            $classes = explode(' ', $classes);
        }
        $this->classes = $classes;
        $this->type = $type;
        $this->attributes = $attributes ?: [];
    }

    /**
     * Static html_list constructor
     *
     * @param array  $classes
     * @param string $type
     * @param array  $attributes
     * @param array  $list
     *
     * @return static
     */
    static public function get_new_object($classes=[], $type=self::TYPE_MARK, $attributes=[], $list=[]){
        return new static($classes, $type, $attributes, $list);
    }

    /**
     * Add $element ot the list
     *
     * @param       $content
     * @param array $classes
     * @param array $attributes
     * @param false $rendered - added only $content if true
     */
    public function add($content, $classes=[], array $attributes=[], $rendered=false){
        if (is_string($classes)){
            $classes = explode(' ', $classes);
        }

        if ($rendered){
            $this->list[] = $content;
        } else {
            $this->list[] = [$content, $classes, $attributes];
        }
    }

    /**
     * Set/get attributes param
     *
     * @param      $name
     * @param \mixed $value
     *
     * @return \mixed
     */
    public function param($name, $value=null){
        $this->attributes[$name] = $value;
        return $this->attributes[$name];
    }

    /**
     * Unset attributes param
     *
     * @param $name
     */
    public function remove_param($name){
        unset($this->attributes[$name]);
    }

    /**
     * Renders the html list and returns the HTML to display it.
     *
     * @param bool $return - echo if false
     *
     * @return string html
     */
    public function render($return=false){
        $content = join('', static::render_list($this->list));
        $attributes = $this->attributes;
        $classes = $this->classes;
        $attributes['class'] = is_array($classes) ? join(' ', $classes) : $classes;
        $out = \html_writer::tag($this->type, $content, $attributes);
        if (!$return){
            echo $out;
        }

        return $out;
    }

    /**
     * Render each element in the list as <li>-s
     *
     * @param array $list
     * @param false $force_render_all - try to render all elements, even if they, can be rendered yet
     *
     * @return array
     */
    static public function render_list($list, $force_render_all=false){
        $r_list = [];
        foreach ($list as $item){
            if ($force_render_all || count($item) > 1){
                $item = $item + ['', [], []];
                list($content, $classes, $attributes) = $item;
                $attributes['class'] = is_array($classes) ? join(' ', $classes) : $classes;
                $r_list[] = \html_writer::tag('li', $content, $attributes);
            } else {
                $r_list[] = reset($item);
            }
        }

        return $r_list;
    }
}
