<?php
/**
 * @package    local_ned_controller
 * @subpackage NED
 * @copyright  2021 NED {@link http://ned.ca}
 * @author     NED {@link http://ned.ca}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace local_ned_controller\output;
use local_ned_controller\shared_lib as NED;
use local_ned_controller\ned_grade_controller as NGC;
use local_ned_controller\support\ned_grade_controller_record as NGC_record;
use local_academic_integrity\output\datatable_infractions as DI;
use local_academic_integrity\infraction as INF;

/**
 * Class ned_grade_controller_render
 *
 * Useful methods:
 *  Render class for page: {@see ned_grade_controller_render::render_full_page()}
 *
 *  Main render content method: {@see ned_grade_controller_render::_render_page_content()}
 *  Main get_data method: {@see ned_grade_controller_render::get_main_table_data()}
 *  Main get_data SQL method: {@see ned_grade_controller_render::get_db_table_data()}
 *
 *  Start init point: {@see ned_grade_controller_render::init()}
 *  Main check params: {@see ned_grade_controller_render::_check_params()}
 *  Export content for template: {@see ned_grade_controller_render::export_for_template()}
 *
 *
 * @package local_ned_controller\output
 *
 * @property-read int                         $view;
 * @property-read int                         $cm_type
 * @property-read int                         $grade_type
 * @property-read int                         $reason
 * @property-read int                         $status
 * @property-read string                      $note
 * @property-read int                         $relatedid
 * @property-read int                         $grade_change
 * @property-read string|int                  $subtype
 * @property-read string|\moodle_url          $returnurl
 * @property-read object|NGC_record           $record
 * @property-read array|object[]|NGC_record[] $records
 */
class ned_grade_controller_render extends \local_ned_controller\output\ned_base_table_page_render {
    protected const _NAVBAR_PLUGIN_URL = NED::PAGE_NED_CONTROLLER;
    protected const _USE_NED_SCHOOL_YEAR_FILTER = true;

    //region SQL data
    protected const _SQL_TABLE = NGC::TABLE;
    protected const _SQL_ALIAS = 'ngc';
    protected const _SQL_SUBMISSION_ALIAS = 'sbm';
    protected const _SQL_AI_ALIAS = 'lai';
    protected const _SQL_REPEATS = 'repeats';
    //endregion

    //region Repeats
    const REPEAT_HIDE = 0;
    const REPEAT_SHOW = 1;
    const REPEAT_1 = 2;
    const REPEAT_2 = 3;
    const REPEAT_3_PLUS = 4;
    const REPEAT_5_PLUS = 5;
    const REPEAT_10_PLUS = 6;
    const REPEAT_OPTIONS = [
        self::REPEAT_HIDE => 'ngc:repeats:hide',
        self::REPEAT_SHOW => 'ngc:repeats:show',
        self::REPEAT_1 => 'ngc:repeats:option_1',
        self::REPEAT_2 => 'ngc:repeats:option_2',
        self::REPEAT_3_PLUS => 'ngc:repeats:option_3_plus',
        self::REPEAT_5_PLUS => 'ngc:repeats:option_5_plus',
        self::REPEAT_10_PLUS => 'ngc:repeats:option_10_plus',
    ];
    //endregion

    //region Params
    const PAR_CM_TYPE = 'cm_type';
    const PAR_GRADE_TYPE = 'grade_type';
    const PAR_REASON = 'reason';
    const PAR_COURSE_VIEW = NED::PAR_COURSE_VIEW;
    const PAR_RELATEDID = 'relatedid';
    const PAR_GRADE_CHANGE = 'grade_change';
    const PAR_SUBTYPE = 'subtype';
    const PAR_REPEATS = 'repeats';

    // not for PARAMS
    const PAR_NOTE = NED::PAR_NOTE;
    const PAR_CONFIRM = NED::PAR_CONFIRM;

    const PARAMS = [
        NED::PAR_ID, NED::PAR_IDS, NED::PAR_FILTER_ID, NED::PAR_FILTER_IDS,
        NED::PAR_COURSE, NED::PAR_CM, NED::PAR_USER, NED::PAR_GROUP, NED::PAR_SCHOOL, self::PAR_CM_TYPE, self::PAR_GRADE_TYPE,
        self::PAR_REASON, NED::PAR_STATUS, NED::PAR_GRADER, NED::PAR_ACTION, self::PAR_COURSE_VIEW, NED::PAR_PAGE, NED::PAR_PERPAGE,
        NED::PAR_VIEW, NED::PAR_STEP, self::PAR_RELATEDID, self::PAR_GRADE_CHANGE, self::PAR_SUBTYPE, self::PAR_REPEATS,
        NED::PAR_SCHOOL_YEAR,
        NED::PAR_RETURN_URL,
    ];

    /**
     * By default, params uses _PARAM_TYPE_DEFAULT and _PARAM_VALUE_DEFAULT, but you can set defaults to others here
     * Possible keys for rewrite array:
     *  • type - change default type of param
     *  • default - change default value of param
     *  • property - load raw param value for $this as $this->{$property}
     * Also {@see \local_ned_controller\shared\C::PARAM_DATA}
     *
     * @var array[] - keys from PARAMS, value is array
     */
    const PARAM_DATA = [
        self::PAR_REASON => ['type' => PARAM_INT, 'default' => self::REASON_HIDE_ALL],
        self::PAR_SUBTYPE => ['type' => PARAM_TEXT, 'default' => NED::ALL],
        self::PAR_REPEATS => ['type' => PARAM_INT, 'default' => self::REPEAT_HIDE],
        NED::PAR_RETURN_URL => ['type' => PARAM_URL, 'default' => null],

        NED::PAR_COURSE => ['property' => '_courseid'],
        NED::PAR_CM => ['property' => '_cmid'],
        NED::PAR_USER => ['property' => '_userid'],
        NED::PAR_GROUP => ['property' => '_groupid'],
        NED::PAR_SCHOOL => ['property' => '_schoolid'],
        NED::PAR_GRADER => ['type' => PARAM_INT, 'default' => self::GRADER_ALL, 'property' => '_graderid'],
    ];
    //endregion

    //region Other Consts
    const URL = NED::PAGE_GRADE_CONTROLLER;

    const REASON_HIDE_ALL = -1;
    const REASON_SHOW_ALL = 0;

    const ACTION_VIEW = 0;
    const ACTION_ADD = 1;
    const ACTION_EDIT = 2;
    const ACTION_DELETE = 3;
    const ACTION_PAUSE = 4;
    const ACTION_UNPAUSE = 5;
    const ACTION_ADD_WRONG_SUBMISSION = 6;
    const ACTION_OBSOLETE = 7;
    const ACTIONS = [
        self::ACTION_VIEW => 'view',
        self::ACTION_ADD => 'add',
        self::ACTION_EDIT => 'edit',
        self::ACTION_DELETE => 'ngc_delete',
        self::ACTION_PAUSE => 'ngc_pause',
        self::ACTION_UNPAUSE => 'ngc_unpause',
        self::ACTION_OBSOLETE => 'ngc_obsolete',
        self::ACTION_ADD_WRONG_SUBMISSION => 'continue',
    ];
    const ACTION_NONE = -1;

    const AUTOMATIC_CONFIRM_ACTIONS_ANY = [
        self::ACTION_PAUSE => true,
        self::ACTION_UNPAUSE => true,
        self::ACTION_OBSOLETE => true,
    ];
    const AUTOMATIC_CONFIRM_ACTIONS_GROUP = [
        self::ACTION_DELETE => true,
    ];

    const VIEW_EXPANDED = 1;
    const VIEW_COMPACT = 2;
    const VIEWS = [
        self::VIEW_EXPANDED => 'expanded',
        self::VIEW_COMPACT => 'compact',
    ];

    const NORMAL_FINAL_STEP = 4;
    const FULLY_FINAL_STEP = 6;
    const SESSION_NOTE_KEY = 'NED_NGC_NOTE';
    const SESSION_EDITABLE_RECORD = 'NED_NGC_EDITABLE_RECORD';
    //endregion

    //region Properties
    protected $_view;
    protected $_cm_type = 0;
    protected $_grade_type = 0;
    protected $_reason = 0;
    protected $_status = 0;
    protected $_step = 0;
    protected $_relatedid = 0;
    protected $_grade_change = 0;
    protected $_subtype = NED::ALL;
    protected $_repeats = self::REPEAT_HIDE;
    /** @var \moodle_url */
    protected $_returnurl = null;

    protected $_note;
    protected $_confirm;
    /** @var object|NGC_record $_record */
    protected $_record;
    /** @var array|object[]|NGC_record[] $_records */
    protected $_records;
    //endregion

    //region Init methods
    /**
     * @constructor
     */
    public function __construct(){
        parent::__construct();

        $this->_cap_view = NGC::get_see_capability($this->_ctx, $this->_is_admin);
        $this->_cap_edit = false;
        $this->_view_all_schools = true;

        if ($this->_cap_view > NGC::CAP_CANT_VIEW){
            $this->_cap_edit = NGC::has_edit_capability($this->_ctx, $this->_is_admin);

            if ($this->_cap_view === NGC::CAP_SEE_OWN_SCHOOL){
                $this->_view_all_schools = false;
                $this->_limit_schoolids = NED::get_user_schools($this->_viewer, false, true);
                if (empty($this->_limit_schoolids)){
                    $this->_cap_view = NGC::CAP_CANT_VIEW;
                }
            }
        }
    }

    /**
     * Check loading record by ID
     * Normally called by {@see _check_params()}
     */
    protected function _check_params_record_id(){
        $this->_params[NED::PAR_ID] = max($this->_params[NED::PAR_ID] ?? 0, 0);
        $this->_id = $this->_params[NED::PAR_ID];
        $this->_params[NED::PAR_ID] = null;
        if (!$this->_id) return;

        $record = NGC::get_record_by_id($this->_id);
        if (!$record || NED::grade_is_hidden_now_before_midn($record->cmid, $record->userid)){
            NED::redirect(static::URL, NED::str('warn:wrongrecord'), null, NED::NOTIFY_ERROR);
            die;
        }

        if ($this->_action == static::ACTION_EDIT){
            $this->_filter_id = $this->_id;
            $this->_filter_ids = [$this->_id];
            $this->_params[NED::PAR_ID] = $this->_id;
            // do usual check for editing later
            return;
        }

        // If not edit, then we don't need usual checks at all
        $this->_set_record($record);
        switch($this->_action){
            case static::ACTION_ADD:
            case static::ACTION_ADD_WRONG_SUBMISSION:
                $this->_action = static::ACTION_VIEW;
                break;
        }
        // for single record view, make course_view is on by default
        $this->_course_view = (bool)($this->_params[NED::PAR_COURSE_VIEW] ?? true);
        $this->_check_course_view();
        $this->_returnurl = $this->_params[NED::PAR_RETURN_URL];
        // clear other params, as we don't need them
        $this->_params = [
            NED::PAR_ID => $this->_id,
            NED::PAR_COURSE => $this->_courseid,
            NED::PAR_COURSE_VIEW => $this->_course_view,
            NED::PAR_RETURN_URL => $this->_returnurl,
        ];
    }

    /**
     * Check *_ID params: PAR_IDS, PAR_FILTER_IDS
     * Load the record by id, if required
     * Normally called by {@see _check_params()}
     */
    protected function _check_params_ids(){
        if ($this->_id) return;

        /**
         * If admin - we can set id and that's all
         * If not - only can set id as filter for other 'where' conditions, and not allowed as later url params
         */
        if ($this->_is_admin){
            $this->_params[NED::PAR_IDS] = $this->_params[NED::PAR_IDS] ?? null;
            $this->_ids = $this->_params[NED::PAR_IDS] ? explode(',', $this->_params[NED::PAR_IDS]) : [];
        } else {
            $this->_ids = null;
            $this->_params[NED::PAR_IDS] = null;
        }

        $this->_params[NED::PAR_FILTER_IDS] = $this->_params[NED::PAR_FILTER_IDS] ?? $this->_params[NED::PAR_FILTER_ID];
        $this->_filter_ids = $this->_params[NED::PAR_FILTER_IDS] ? explode(',', $this->_params[NED::PAR_FILTER_IDS]) : [];
        if (count($this->_filter_ids) == 1){
            $this->_filter_id = reset($this->_filter_ids);
        }
        $this->_params[NED::PAR_FILTER_ID] = null;
        $this->_params[NED::PAR_FILTER_IDS] = null;
    }

    /**
     * Check loaded params with the view user capability
     * You need to load required params in the object before call this method
     * Normally called by {@see _check_params()} or {@see _set_record()}
     */
    protected function _check_params_by_view_cap(){
        switch ($this->_cap_view){
            case NGC::CAP_SEE_ALL:
            case NGC::CAP_SEE_OWN_ALL:
            case NGC::CAP_SEE_OWN_ANY: // shouldn't be really
            case NGC::CAP_SEE_OWN_SCHOOL:
                if (!is_null($this->_graderid) && !$this->_record){
                    $options = $this->get_graders_options();
                    $this->_graderid = NED::isset_key($options, $this->_graderid ?? static::GRADER_ALL, static::GRADER_ALL);
                }
                break;

            case NGC::CAP_SEE_OWN_GRADER:
                if (!empty($this->_graderid) && $this->_graderid != static::GRADER_ALL && $this->_graderid != $this->_viewerid){
                    NED::print_error('warn:seeyourrecords', null, $this->get_my_url([NED::PAR_GRADER => $this->_viewerid]));
                }

                $this->_graderid = $this->_viewerid;
                break;

            case NGC::CAP_SEE_ME:
                if (!empty($this->_userid) && $this->_userid != $this->_viewerid){
                    NED::print_error('warn:seeyourrecords', null, $this->get_my_url([NED::PAR_USER => $this->_viewerid]));
                }

                $this->_userid = $this->_viewerid;
                $this->_graderid = null;
                $this->_schoolid = null;
                $this->_groupid = null;
                break;

            default:
                $this->_action = static::ACTION_VIEW;
                $this->_graderid = static::GRADER_ALL;
                $this->_userid = $this->_viewerid;
        }
    }

    /**
     *  Check not-view params
     */
    protected function _check_params_not_view(){
        if ($this->_action != static::ACTION_VIEW){
            $this->_step = $this->_params[NED::PAR_STEP] ?? 0;
            $this->_grade_change = $this->_params[static::PAR_GRADE_CHANGE] ?? $this->_grade_change ?? 0;
            $this->_relatedid = $this->_params[static::PAR_RELATEDID] ?? $this->_relatedid ?? 0;
        } else {
            unset($this->_params[static::PAR_GRADE_CHANGE]);
            unset($this->_params[static::PAR_RELATEDID]);
        }
        unset($this->_params[NED::PAR_STEP]);
    }

    /**
     * Check loaded params
     * Normally called by {@see set_and_check_params()} after param loading
     */
    protected function _check_params(){
        if (!$this->can_see()) return;

        // some prechecked defaults
        $this->_view = static::VIEW_COMPACT;
        // check action first of all
        $this->_params[NED::PAR_ACTION] = NED::isset_key(static::ACTIONS, $this->_params[NED::PAR_ACTION], static::ACTION_VIEW);
        $this->_action = $this->_params[NED::PAR_ACTION];
        $this->_returnurl = $this->_params[NED::PAR_RETURN_URL];

        if (!$this->can_edit() && $this->_action != static::ACTION_VIEW){
            $this->add_notification(get_string('nopermissions', 'error', NED::str('ned_controller:gradecontroller_edit')), NED::NOTIFY_ERROR);
            $this->_action = static::ACTION_VIEW;
        }

        $this->_check_params_record_id();
        // if we got the record by _id, then we have already loaded all necessary params
        if ($this->_record) return;

        $this->_check_params_ids();
        $this->_check_params_by_view_cap();

        $this->_grade_type = NED::isset_key(NGC::GRADE_TYPES, $this->_params[static::PAR_GRADE_TYPE], NED::ALL);
        $this->_params[static::PAR_GRADE_TYPE] = $this->_grade_type;

        $this->_cm_type = NED::isset_key(NGC::CM_TYPES_KEYS, $this->_params[static::PAR_CM_TYPE], NED::MOD_ALL);
        $this->_params[static::PAR_CM_TYPE] = $this->_cm_type;

        switch ($this->_action){
            /** @noinspection PhpMissingBreakStatementInspection */
            default:
                $this->_action = static::ACTION_VIEW;
            case static::ACTION_EDIT:
            case static::ACTION_DELETE:
            case static::ACTION_PAUSE:
            case static::ACTION_OBSOLETE:
            /** @noinspection PhpMissingBreakStatementInspection */
            case static::ACTION_UNPAUSE:
                if (!$this->_filter_ids){
                    $this->_action = static::ACTION_VIEW;
                }
            case static::ACTION_VIEW:
                $this->_usual_check_params();
                break;
            case static::ACTION_ADD:
            case static::ACTION_ADD_WRONG_SUBMISSION:
                // need other param check, if we want to add new record
                $this->_add_check_params();
                break;
        }

        $this->_status = NED::isset_key(NGC::STATUSES, $this->_params[NED::PAR_STATUS], NED::ALL);
        $this->_params[NED::PAR_STATUS] = $this->_status;
        if (!$this->can_edit() && !empty(NGC::HIDDEN_STATUSES[$this->_status])){
            $this->_status = NED::ALL;
        }

        $this->_course_view = (bool)$this->_params[NED::PAR_COURSE_VIEW];
        $this->_view = NED::isset_key(static::VIEWS, $this->_params[NED::PAR_VIEW], static::VIEW_COMPACT);
        $this->_subtype = NED::isset_key(NGC::get_subtypes(), $this->_params[static::PAR_SUBTYPE], NED::ALL);
        $this->_repeats = NED::isset_key(static::REPEAT_OPTIONS, $this->_params[static::PAR_REPEATS], static::REPEAT_HIDE);

        $options = $this->get_perpage_options();
        $this->_params[NED::PAR_PERPAGE] = NED::isset_key($options, $this->_params[NED::PAR_PERPAGE] ?? 0);
        $this->_perpage = $options[$this->_params[NED::PAR_PERPAGE]];
        if ($this->_perpage === NED::PERPAGE_ALL){
            $this->_perpage = 0;
            $this->_page = 0;
        } else {
            $this->_page = (int)abs($this->_params[NED::PAR_PAGE]);
        }

        $this->_check_params_not_view();
        $this->_params_init = true;
    }

    /**
     * Check loaded params by TABLE
     */
    protected function _usual_check_params(){
        if ($this->_courseid != static::COURSE_NONE && $this->_courseid != NED::ALL){
            $course = NED::get_course($this->_courseid);
            $this->_courseid = $course->id ?? NED::ALL;
        }

        $options = $this->get_courses_options();
        $courseid = NED::isset_key($options, $this->_courseid, static::COURSE_NONE);
        if ($courseid != static::COURSE_NONE && $courseid != NED::ALL){
            $options = $this->get_group_options();
            $this->_groupid = NED::isset_key($options, $this->_groupid, NED::ALL);
            $options = $this->get_cm_options();
            $this->_cmid = NED::isset_key($options, $this->_cmid, NED::ALL);
        } else {
            $this->_groupid = 0;
            $this->_cmid = 0;
        }

        if ($this->_cap_view >= NGC::CAP_SEE_OWN_ANY){
            $options = $this->get_users_options();
            $this->_userid = NED::isset_key($options, $this->_userid, NED::ALL);
        }

        $options = $this->get_schools_options();
        $this->_schoolid = NED::isset_key($options, $this->_schoolid, NED::ALL);

        $options = $this->get_reasons_options();
        $this->_params[static::PAR_REASON] = NED::isset_key($options, $this->_params[static::PAR_REASON]);
        $this->_reason = $this->_params[static::PAR_REASON];

        if (static::_USE_NED_SCHOOL_YEAR_FILTER){
            $this->_school_year = static::SCHOOL_YEAR_CURRENT;
            $options = $this->get_school_year_options();
            $this->_school_year = NED::isset_key($options, $this->_params[NED::PAR_SCHOOL_YEAR], $this->_school_year);
        } else {
            $this->_school_year = null;
        }
    }

    /**
     * Check loaded params from ADD possibility
     */
    protected function _add_check_params(){
        $this->_courseid = NGC::check_courseid($this->_courseid, $this->_viewerid) ? $this->_courseid : 0;
        if ($this->_courseid > 0){
            if ($this->_is_admin){
                $groupids = NED::get_course_groupids($this->_courseid);
            } else {
                $groupids = NED::get_user_groupids($this->_courseid, $this->_viewerid);
            }
            $this->_groupid = NED::isset_in_list($groupids, $this->_groupid, 0);

            $this->_cmid = NGC::check_cmid($this->_cmid, $this->_courseid, $this->_viewerid) ? $this->_cmid : 0;
        } else {
            $this->_groupid = 0;
            $this->_cmid = 0;
        }

        if ($this->_action == static::ACTION_ADD_WRONG_SUBMISSION){
            $this->_grade_type = NGC::GT_AWARD_ZERO;
            $this->_reason = NGC::REASON_FILE;
            $this->_note = '';

            unset($this->_params[static::PAR_REASON]);
            unset($this->_params[static::PAR_GRADE_TYPE]);
        } else {
            $this->_params[static::PAR_REASON] = NED::isset_key(NGC::REASONS, $this->_params[static::PAR_REASON], NED::ALL);
            $this->_reason = $this->_params[static::PAR_REASON];
        }
    }

    /**
     * Set NGC record as the base for current view
     *
     * @param object $record
     */
    protected function _set_record($record=null){
        $this->_record = $record;
        if (!empty($record)){
            $this->copy($record, '_');
            $this->_check_params_by_view_cap();
        }
    }
    //endregion

    //region Capabilities methods
    /**
     * Return true, if current user is able to see this page
     *
     * @return bool
     */
    public function can_see(){
        return $this->_cap_view > NGC::CAP_CANT_VIEW;
    }

    /**
     * Return true, if current user has extra edit permission
     *
     * @return bool
     */
    public function can_extra_edit(){
        return $this->_is_admin;
    }
    //endregion

    //region Setup Page methods
    /**
     * @return string
     */
    public function get_page_title(){
        return NED::str('gradecontroller');
    }
    //endregion

    //region SQL menu filter methods
    /**
     * Add limitation by grader, changing $where and $params for sql query
     *
     * @param array $joins
     * @param array $where
     * @param array $params
     *
     * @return void
     */
    protected function _sql_add_view_limitation(&$joins=[], &$where=[], &$params=[]){
        if ($this->_cap_view == NGC::CAP_SEE_ME && $this->_userid > 0){
            static::_sql_add_equal(static::_SQL_USERID, $this->_userid);
        } elseif ($this->_cap_view >= NGC::CAP_SEE_OWN_ANY && $this->_graderid > static::GRADER_ALL){
            static::_sql_add_equal(static::_SQL_GRADERID, $this->_graderid);
        }

        if (!$this->_view_all_schools){
            if (empty($this->_limit_schoolids)){
                $where[] = NED::SQL_NONE_COND;
            } else {
                $this->_sql_add_limit_school($joins, $where, $params, $this->_limit_schoolids);
            }
        }

        if (!$this->can_edit()){
            $where[] = static::_SQL_ALIAS.".status NOT IN (".join(',', array_keys(NGC::HIDDEN_STATUSES)).")";
        }
    }

    /**
     * Load sql params for getting NGC graders with capability SEE_OWN
     *
     * @param array  $joins
     * @param array  $where
     * @param array  $params
     * @param string $sql_graderid = ngc.graderid
     * @param bool   $load_user_table - if true, load user table here
     * @param string $userid_sql - sql to check user id (u.id by default)
     * @param string $u_table_prefix - prefix for loading user table (if $load_user_table - true)
     *
     * @return void - result saves in the $joins, $where and $params
     */
    protected function _sql_get_grader_query(&$joins=[], &$where=[], &$params=[], $sql_graderid='',
        $load_user_table=true, $userid_sql='', $u_table_prefix=''){
        $u = $u_table_prefix.'u';
        $userid = $userid_sql ?: ($u.'.id');
        $sql_graderid = $sql_graderid ?: static::_sql_a(static::_SQL_GRADERID);
        if ($load_user_table){
            $joins[] = "LEFT JOIN {user} $u ON $userid = $sql_graderid";
        }

        $where_join = [];
        NED::sql_add_equal('rc.capability', NED::get_full_capability(NGC::CAP_NAME_EDIT), $where_join, $params);
        NED::sql_add_equal('rc.permission', CAP_ALLOW, $where_join, $params, 'allow');
        $params['system_context'] = NED::ctx()->id;

        if ($this->_ctx->contextlevel != CONTEXT_SYSTEM){
            NED::sql_add_get_in_or_equal_options('ra.contextid', $this->_ctx->get_parent_context_ids(true),
                $where_join, $params);
        }

        $joins[] = "
            LEFT JOIN (
                SELECT rc.id, rc.roleid, ra.userid
                FROM {role_assignments} ra 
                JOIN {role_capabilities} rc 
                    ON rc.roleid = ra.roleid AND rc.contextid = :system_context
                ".NED::sql_where($where_join)."
                GROUP BY ra.userid
            ) grader_permissions
            ON grader_permissions.userid = $userid
        ";

        $where[] = "(grader_permissions.id IS NOT NULL OR $userid IS NULL)";
    }
    //endregion

    //region Get filter options [get_*_options()]
    /**
     * @return array
     */
    public function get_graders_options(){
        if ($this->_cap_view < NGC::CAP_SEE_ALL){
            return [];
        }

        $data = $this->_static_data[__FUNCTION__] ?? null;
        if (is_null($data)){
            $key = static::_sql_a(static::_SQL_GRADERID);
            $val = "IF(u.id IS NOT NULL, CONCAT(u.firstname, ' ', u.lastname), 'None')";
            $this->_sql_get_grader_query($joins, $where, $params);
            $data = $this->_sql_menu_filter($key, $val, $joins, $where, $params, static::GRADER_ALL, true);
            $this->_static_data[__FUNCTION__] = $data;
        }
        return $data;
    }

    /**
     * @return array
     */
    public function get_reasons_options(){
        $def_options = [
            static::REASON_HIDE_ALL => 'hideall',
            static::REASON_SHOW_ALL => 'showall',
        ];
        return $def_options + NGC::REASONS;
    }

    /**
     * @return array
     */
    public function get_users_options(){
        if ($this->_cap_view < NGC::CAP_SEE_OWN_ANY) return [];

        return parent::get_users_options();
    }

    /**
     * @return array
     */
    public function get_schools_options(){
        if ($this->_cap_view < NGC::CAP_SEE_OWN_ANY) return [];

        return parent::get_schools_options();
    }

    /**
     * @return array
     */
    public function get_courses_options(){
        if (!$this->can_see()) return [];

        return parent::get_courses_options();
    }
    //endregion

    //region Get main table data methods
    /**
     * Get SQL-where data for penalty type based filter
     * Used data from "Grade Type", "Reason", "Status", "Infraction" (subtype)
     *
     * @param array  $where
     * @param array  $params
     * @param string $param_postfix - (optional) - add it to the all $param_name-s
     *
     * @return void - result saves in the $where and $params
     */
    protected function _sql_get_penalty_type_filter_params(&$where=[], &$params=[], $param_postfix=''){
        if ($this->_grade_type){
            static::_sql_add_equal('grade_type', $this->_grade_type, $where, $params, '', $param_postfix);
        }

        if ($this->_reason > 0){
            static::_sql_add_equal('reason', $this->_reason, $where, $params, '', $param_postfix);
        }

        if ($this->_status){
            static::_sql_add_equal('status', $this->_status, $where, $params, '', $param_postfix);
        }

        $subtypes = NGC::get_subtypes();
        $sbm_alias = static::_SQL_SUBMISSION_ALIAS;
        if ($this->_subtype && isset($subtypes[$this->_subtype])){
            if (in_array($this->_subtype,NGC::SUBTYPES)){
                $add_wrong_file = false;
                $add_missed_submission = false;
                switch ($this->_subtype){
                    case NGC::SUBT_WS_DRAFT:
                        $add_wrong_file = true;
                        NED::sql_add_equal("$sbm_alias.status", ASSIGN_SUBMISSION_STATUS_DRAFT,
                            $where, $params, '', $param_postfix);
                        break;
                    case NGC::SUBT_WS_FIXED:
                        $add_wrong_file = true;
                        NED::sql_add_equal("$sbm_alias.status", ASSIGN_SUBMISSION_STATUS_SUBMITTED,
                            $where, $params, '', $param_postfix);
                        break;
                    case NGC::SUBT_WS_CANCELLED:
                        $add_wrong_file = true;
                        $param_smb_new = 'sbm_new'.$param_postfix;
                        $param_smb_reopened = 'sbm_reopened'.$param_postfix;
                        $where[] = "($sbm_alias.status IS NULL OR $sbm_alias.status = :$param_smb_new OR $sbm_alias.status = :$param_smb_reopened)";
                        $params[$param_smb_new] = ASSIGN_SUBMISSION_STATUS_NEW;
                        $params[$param_smb_reopened] = ASSIGN_SUBMISSION_STATUS_REOPENED;
                        break;
                    case NGC::SUBT_MISSED_DEADLINE:
                        $add_missed_submission = true;
                        static::_sql_add_equal('grade_type', NGC::GT_AWARD_ZERO,
                            $where, $params, 'sbt_zero'.$param_postfix);
                        break;
                    case NGC::SUBT_LATE_SUBMISSION:
                        $add_missed_submission = true;
                        static::_sql_add_equal('grade_type', NGC::GT_DEDUCTION,
                            $where, $params, 'sbt_deduction'.$param_postfix);
                        break;
                }

                if ($add_wrong_file){
                    static::_sql_add_equal('reason', NGC::REASON_FILE,
                        $where, $params, 'sbt_reason'.$param_postfix);
                    static::_sql_add_equal('cm_type', NED::MOD_ASSIGN_KEY,
                        $where, $params, 'sbt_assign_type'.$param_postfix);
                } elseif ($add_missed_submission){
                    static::_sql_add_equal('reason', NGC::REASON_SUBMISSION,
                        $where, $params, 'sbt_reason'.$param_postfix);
                }
            } else {
                // AI infraction
                if (NED::is_ai_exists()){
                    NED::sql_add_equal(static::_SQL_AI_ALIAS.'.penalty', $this->_subtype,
                        $where, $params, '', $param_postfix);
                } else {
                    $where[] = NED::SQL_NONE_COND;
                }
            }
        }
    }

    /**
     * Get SQL join string for necessary module tables
     *
     * @param array  $params                    - array to save additional params
     * @param string $param_postfix             - (optional) - add it to the saved param names
     * @param string $t_cm_alias                - (optional) - alias for the course_modules table, 'cm' by default
     * @param string $t_assign_submission_alias - (optional) - alias for the assign_submission table, 'sbm' by default
     *
     * @return string of tables to join, also saved necessary data in the $params
     */
    static protected function _sql_get_modules_join(&$params=[], $param_postfix='',
        $t_cm_alias='cm', $t_assign_submission_alias=self::_SQL_SUBMISSION_ALIAS){
        $a_user = static::_sql_a(static::_SQL_USERID);
        $a_cm = static::_sql_a(static::_SQL_CMID);
        $a_cm_type = static::_sql_a('cm_type');

        $t_cm_alias = $t_cm_alias ?: 'cm';
        $t_sbm_alias = $t_assign_submission_alias ?: static::_SQL_SUBMISSION_ALIAS;
        $param_assign_type = 'assign_type'.$param_postfix;
        $params[$param_assign_type] = NED::MOD_ASSIGN_KEY;

        return "
            JOIN {course_modules} $t_cm_alias 
                ON $t_cm_alias.id = $a_cm
            LEFT JOIN {assign_submission} $t_sbm_alias
                ON $a_cm_type = :$param_assign_type
                AND $t_sbm_alias.assignment = $t_cm_alias.instance
                AND $t_sbm_alias.userid = $a_user
                AND $t_sbm_alias.latest = 1
        ";
    }

    /**
     * Get SQL join string for the local AI table
     * Can return empty string, if AI plugin doesn't exist
     *
     * @param array  $params        - array to save additional params
     * @param string $param_postfix - (optional) - add it to the saved param names
     * @param string $t_ai_alias    - (optional) - alias for the local AI table, 'lai' by default
     *
     * @return string of tables to join, also saved necessary data in the $params
     */
    static protected function _sql_get_ai_join(&$params=[], $param_postfix='', $t_ai_alias=self::_SQL_AI_ALIAS){
        if (!NED::is_ai_exists()) return '';

        $t_ai_alias = $t_ai_alias ?: static::_SQL_AI_ALIAS;
        $a_reason = static::_sql_a('reason');
        $a_relatedid = static::_sql_a('relatedid');
        $param_ai_reason = 'ai_reason'.$param_postfix;
        $params[$param_ai_reason] = NGC::REASON_AI;

        return "
            LEFT JOIN {".NGC::RELATED_TABLE."} $t_ai_alias
                ON $a_reason = :$param_ai_reason
                AND $t_ai_alias.id = $a_relatedid  
        ";
    }

    /**
     * Get SQL condition to get NGC subtype
     * Warning: it gets only NGC-native subtypes, without AI subtypes
     *
     * This logic should be the same as PHP logic for definition of subtype in {@see get_main_table_data()}
     *
     * @param string $subtype - one of the {@see \local_ned_controller\ned_grade_controller::SUBTYPES}
     *
     * @return string|false - SQL condition or false
     */
    static protected function _sql_get_subtype_condition($subtype=NGC::SUBT_WS_DRAFT){
        $sql_reason = static::_sql_a('reason');
        $sql_gt = static::_sql_a('grade_type');
        $sql_sbm_status = static::_SQL_SUBMISSION_ALIAS.'.status';

        switch ($subtype){
            case NGC::SUBT_WS_CANCELLED:
                $reason = NGC::REASON_FILE;
                $assign_new = ASSIGN_SUBMISSION_STATUS_NEW;
                $assign_reopened = ASSIGN_SUBMISSION_STATUS_REOPENED;
                return "$sql_reason = '$reason' AND ($sql_sbm_status IS NULL OR $sql_sbm_status IN ('$assign_new', '$assign_reopened'))";
            case NGC::SUBT_WS_DRAFT:
                $reason = NGC::REASON_FILE;
                $assign_draft = ASSIGN_SUBMISSION_STATUS_DRAFT;
                return "$sql_reason = '$reason' AND $sql_sbm_status = '$assign_draft'";
            case NGC::SUBT_WS_FIXED:
                $reason = NGC::REASON_FILE;
                $assign_submitted = ASSIGN_SUBMISSION_STATUS_SUBMITTED;
                return "$sql_reason = '$reason' AND $sql_sbm_status = '$assign_submitted'";

            case NGC::SUBT_MISSED_DEADLINE:
                $reason = NGC::REASON_SUBMISSION;
                $gt_zero = NGC::GT_AWARD_ZERO;
                return "$sql_reason = '$reason' AND $sql_gt = '$gt_zero'";
            case NGC::SUBT_LATE_SUBMISSION:
                $reason = NGC::REASON_SUBMISSION;
                $gt_deduction = NGC::GT_DEDUCTION;
                return "$sql_reason = '$reason' AND $sql_gt = '$gt_deduction'";

            default: return false;
        }
    }

    /**
     * Get SQL code to join custom "repeats" table
     *
     * @param array $params - array to save additional params
     *
     * @return string of joins, also saved necessary data in the $params
     */
    protected function _sql_get_repeats_table_join(&$params=[]){
        $r_select = [
            static::_sql_a(static::_SQL_USERID),
            "COUNT(1) AS ".static::_SQL_REPEATS,
        ];
        $r_joins = [];
        $r_groupby = [
            static::_sql_a(static::_SQL_USERID),
        ];
        $r_where = [];
        $r_param_postfix = '_repeats';

        $r_joins[] = static::_sql_get_modules_join($params, $r_param_postfix);
        $join_ai = static::_sql_get_ai_join($params, $r_param_postfix);
        if (!empty($join_ai)){
            $r_joins[] = $join_ai;
        }

        // we need the same type-filter values for repeat table as for the main table
        $this->_sql_get_penalty_type_filter_params($r_where, $params, $r_param_postfix);

        // also we need to check course
        if ($this->_courseid > NED::ALL){
            static::_sql_add_equal(static::_SQL_COURSEID, $this->_courseid,
                $r_where, $params, '', $r_param_postfix);

            $sql_courseid_a = static::_sql_a(static::_SQL_COURSEID);
            $r_select[] = $sql_courseid_a;
            $r_groupby[] = $sql_courseid_a;
        }

        return static::_sql_generate_sql($r_select, $r_joins, $r_where, $r_groupby);
    }

    /**
     * Get base joins for {@see _db_process_sql_query()}
     *
     * @param array $add_joins  - add additional joins to it
     * @param array $params     - array to save additional params
     * @param array $base_joins - joins for adding to the beginning of the join list
     *
     * @return array
     */
    protected function _db_get_base_join($add_joins=[], &$params=[], $base_joins=[]){
        $base_joins = $base_joins ?: [];
        $base_joins[] = static::_sql_get_modules_join($params);

        return parent::_db_get_base_join($add_joins, $params, $base_joins);
    }

    /**
     * Process SQL query for the getting data from the DB for main table
     *
     * @param array|string $select
     * @param array|string $joins
     * @param array|string $where
     * @param array        $params
     * @param array|string $groupby
     * @param array|string $orderby
     *
     * @return bool - saves result in the params, return true if you can continue query
     */
    protected function _db_process_sql_query(&$select=[], &$joins=[], &$where=[], &$params=[], &$groupby=[], &$orderby=[]){
        [$select, $joins, $where, $groupby, $orderby] = NED::val2arr_multi(true, $select, $joins, $where, $groupby, $orderby);

        if (empty($select)){
            $select[] = static::_sql_a('*');
        }

        $joins = $this->_db_get_base_join($joins, $params);
        $join_ai = static::_sql_get_ai_join($params);
        if (!empty($join_ai)){
            $joins[] = $join_ai;
            $select[] = static::_SQL_AI_ALIAS.".penalty AS penalty";
        }

        if ($this->_is_admin && !empty($this->_ids)){
            $ids = $this->_ids;
            // if we have filter, we should filter even admin call
            if (!empty($this->_filter_ids)){
                $ids = array_intersect($ids, $this->_filter_ids);
                if (empty($ids)){
                    return false;
                }
            }
            NED::sql_add_get_in_or_equal_options(static::_sql_a('id'), $ids, $where, $params);
        } else {
            $groupid = $graderid = $simple_school_id_filter = null;
            if ($this->_cap_view >= NGC::CAP_SEE_OWN_ANY){
                $groupid = $this->_groupid;
                $graderid = $this->_graderid;
                $simple_school_id_filter = $this->_sql_provide_school_limitation($joins,$where,$params,$groupby);
            }

            static::_sql_set_simple_filters($where, $params,
                $this->_courseid, $this->_cmid, $this->_userid, $groupid, $graderid, $simple_school_id_filter,
                $this->_filter_ids, $this->_filter_id, $this->_school_year
            );
            if (!$this->can_edit()){
                $a_status = static::_sql_a('status');
                $where[] = "$a_status NOT IN (".join(',', array_keys(NGC::HIDDEN_STATUSES)).")";
            }
        }

        return true;
    }

    /**
     * Get object data with infraction statistic by single user (and optionally course)
     *
     * @return object
     */
    protected function _get_user_infraction_info_data(){
        $res = [];
        $site_data_postfix = '_site_data';
        $course_data_postfix = '_course_data';
        $subtypes = NGC::SUBTYPES;

        $user_exists = NGC::record_exists([static::_SQL_USERID => $this->_userid]);
        $has_course = !empty($this->_courseid);
        $select = $joins = $where = $params = [];

        foreach ($subtypes as $subtype){
            $cond = static::_sql_get_subtype_condition($subtype);
            if (empty($cond)) continue;

            $res[$subtype.$site_data_postfix] = 0;
            if ($has_course){
                $res[$subtype.$course_data_postfix] = 0;
            }
            if ($user_exists){
                $select[] = "SUM($cond) AS $subtype";
            }
        }

        if (!$user_exists) return (object)$res;

        // get site data first
        $joins[] = static::_sql_get_modules_join($params);
        NED::sql_add_get_in_or_equal_options(static::_sql_a('status'), [NGC::ST_DONE, NGC::ST_OBSOLETED]);
        NED::sql_add_condition(static::_sql_a('reason'), NGC::REASON_AI, $where, $params, NED::SQL_COND_NEQ);
        static::_sql_set_simple_filters($where, $params,
            null, null, $this->_userid, null, null, null,
            null, null, $this->_school_year
        );
        $groupby = static::_sql_a(static::_SQL_USERID);

        $sql = NED::sql_generate($select, $joins, static::_SQL_TABLE, static::_SQL_ALIAS, $where, $groupby);
        $record = NED::db()->get_record_sql($sql, $params);

        $has_data = false;
        if (!empty($record)){
            foreach ($subtypes as $subtype){
                if (empty($record->$subtype)) continue;

                $has_data = true;
                $res[$subtype.$site_data_postfix] = $record->$subtype;
            }
        }

        if ($has_data && $has_course){
            // get course data now
            static::_sql_set_simple_filters($where, $params,
                $this->_courseid, null, null, null, null, null,
                null, null, null
            );
            $sql = NED::sql_generate($select, $joins, static::_SQL_TABLE, static::_SQL_ALIAS, $where, $groupby);
            $record = NED::db()->get_record_sql($sql, $params);
            if (!empty($record)){
                foreach ($subtypes as $subtype){
                    if (empty($record->$subtype)) continue;

                    $res[$subtype.$course_data_postfix] = $record->$subtype;
                }
            }
        }

        return (object)$res;
    }

    /**
     * Get and render data for main page table
     * @see get_db_table_data() for DB information
     *
     * @param bool $use_filter - if false, ignores some page params
     * @param bool $only_keys  - if true, return only list of ids
     *
     * @return array
     *
     * @noinspection PhpMissingBreakStatementInspection
     */
    public function get_main_table_data($use_filter=true, $only_keys=false){
        $use_filter = (int)$use_filter;
        if (isset($this->_static_data[__FUNCTION__][$use_filter])){
            $records = $this->_static_data[__FUNCTION__][$use_filter];
            if ($only_keys){
                return array_keys($records);
            }

            return $records;
        }

        if (!$this->can_see()){
            return [];
        }

        $group_t = static::_SQL_GROUP_ALIAS;
        $sbm_alias = static::_SQL_SUBMISSION_ALIAS;
        $select = [
            static::_sql_a('*'),
            "$group_t.group_ids AS group_ids",
            "$group_t.group_names AS group_names",
            "$sbm_alias.status AS submission_status",
            "$sbm_alias.timemodified AS submission_timemodified",
        ];

        $joins = [];
        $where = [];
        $params = [];
        $set_limit = $this->_perpage;

        if ($use_filter){
            $this->_sql_get_penalty_type_filter_params($where, $params);

            if ($this->_repeats != static::REPEAT_HIDE){
                $t_repeat = 't_repeats';
                $sql_userid = static::_SQL_USERID;
                $sql_repeats = static::_SQL_REPEATS;
                $sql_t_repeats = "$t_repeat.$sql_repeats";
                $r_join_cond = [
                    "$t_repeat.$sql_userid = ".static::_sql_a($sql_userid),
                ];

                if ($this->_courseid > NED::ALL){
                    $sql_courseid = static::_SQL_COURSEID;
                    $sql_courseid_a = static::_sql_a($sql_courseid);
                    $r_join_cond[] = "$t_repeat.$sql_courseid = $sql_courseid_a";
                }

                $select[] = $sql_t_repeats." AS ".$sql_repeats;
                $repeat_sql = $this->_sql_get_repeats_table_join($params);
                $joins[] = "LEFT JOIN \n($repeat_sql) AS $t_repeat \n ON ".NED::sql_condition($r_join_cond);

                // Add condition to the main WHERE
                $repeat_value = null;
                switch ($this->_repeats){
                    default:
                    case static::REPEAT_SHOW:
                        $repeat_condition = false;
                        break;
                    case static::REPEAT_1:
                        $repeat_value = $repeat_value ?? 1;
                    case static::REPEAT_2:
                        $repeat_value = $repeat_value ?? 2;
                        $repeat_condition = NED::SQL_COND_EQ;
                        break;
                    case static::REPEAT_3_PLUS:
                        $repeat_value = $repeat_value ?? 3;
                    case static::REPEAT_5_PLUS:
                        $repeat_value = $repeat_value ?? 5;
                    case static::REPEAT_10_PLUS:
                        $repeat_value = $repeat_value ?? 10;
                        $repeat_condition = NED::SQL_COND_GTE;
                        break;
                }
                if ($repeat_condition){
                    NED::sql_add_condition($sql_t_repeats, $repeat_value, $where, $params, $repeat_condition);
                }
            }
        }

        $raw_records = $this->get_db_table_data($select, $joins, $where, $params,
            static::_sql_a('id'), static::_sql_a('timecreated'), $set_limit);
        if ($only_keys){
            return array_keys($raw_records);
        }

        $records = [];
        /** @var object|NGC_record $r_record */
        foreach ($raw_records as $r_record){
            if (NED::grade_is_hidden_now_before_midn($r_record->cmid, $r_record->userid)) continue;

            if (isset($r_record->group_ids)){
                $groupids = NED::str2arr($r_record->group_ids, '', ',');
                $r_record->groupid = reset($groupids);
            }

            /** @var object|NGC_record $record */
            $record = static::get_description_object($r_record, $r_record, null, '', $this->_cap_view);
            $record->classes = [];
            $record->controls = $this->can_edit();

            $record->str_status = isset(NGC::STATUSES[$record->status]) ? NED::str(NGC::STATUSES[$record->status]) : '?';
            if ($record->status == NGC::ST_ERROR){
                $record->classes[] = 'error';
            }

            /**
             * Note, that this PHP definition of subtype should be the same as SQL condition in {@see _sql_get_subtype_condition()}
             */
            $record->subtype = null;
            $record->subtype_str = null;
            switch ($record->reason){
                case NGC::REASON_AI:
                    if (isset($record->penalty) && NED::is_ai_exists()){
                        $record->subtype = $record->penalty;
                        $record->subtype_str = INF::get_penalty_list(true)[$record->penalty] ?? '?';
                    }
                    break;
                case NGC::REASON_FILE:
                    $submission_status = $record->submission_status ?? ASSIGN_SUBMISSION_STATUS_NEW;
                    switch ($submission_status){
                        default:
                        case ASSIGN_SUBMISSION_STATUS_NEW:
                        case ASSIGN_SUBMISSION_STATUS_REOPENED:
                            $record->subtype = NGC::SUBT_WS_CANCELLED;
                            break;
                        case ASSIGN_SUBMISSION_STATUS_DRAFT:
                            $record->subtype = NGC::SUBT_WS_DRAFT;
                            break;
                        case ASSIGN_SUBMISSION_STATUS_SUBMITTED:
                            $record->subtype = NGC::SUBT_WS_FIXED;
                            break;
                    }
                    break;
                case NGC::REASON_SUBMISSION:
                    if ($record->grade_type == NGC::GT_AWARD_ZERO){
                        $record->subtype = NGC::SUBT_MISSED_DEADLINE;
                    } elseif ($record->grade_type == NGC::GT_DEDUCTION){
                        $record->subtype = NGC::SUBT_LATE_SUBMISSION;
                    }
                    break;
                default: break;
            }
            if (!$record->subtype_str && $record->subtype){
                $record->subtype_str = NED::str($record->subtype);
            }
            $record->action_menu = $this->get_ngc_action_menu($record);

            $records[$record->id] = $record;
        }

        $this->_static_data[__FUNCTION__][$use_filter] = $records;
        return $records;
    }
    //endregion

    //region NGC action_menu
    /**
     * Init ngc menu with links
     *
     * @param \action_menu           $menu   - menu object, to fill links
     * @param object|NGC_record|null $record - record, for which create menu (null for bulk menu)
     */
    protected function _init_ngc_action_menu_links_view(&$menu, $record=null){
        if (!$this->can_see() || !isset($record->id)) return;

        $m_add = function($text_key, $add_url=null, $class=['link']) use (&$menu){
            if ($add_url){
                if (is_array($add_url)){
                    $url = static::get_url($add_url);
                } else {
                    $url = $add_url;
                }
            } else {
                $url = ['#'];
            }
            $class = NED::val2arr($class);
            $class[] = $text_key;
            if (!$add_url){
                $class[] = 'calljs';
            }

            $menu->add(NED::link($url, NED::str($text_key), NED::arr2str($class)));
        };

        if (!$this->_id){
            $m_add('showdetails', null);
            $m_add('ngc_view_record', NED::ngc_record_view_url($record->id, $this->_course_view));
        }

        if ($this->_cap_view >= NGC::CAP_SEE_OWN_ANY){
            $base_params = [];
            if ($this->_course_view){
                $base_params[NED::PAR_COURSE] = $record->courseid;
                $base_params[NED::PAR_COURSE_VIEW] = $this->_course_view;
            }

            $m_add('see_student_records', $base_params + [NED::PAR_USER => $record->userid]);
            $m_add('see_cm_records', $base_params + [NED::PAR_COURSE => $record->courseid, NED::PAR_CM => $record->cmid]);

            if ($this->_cap_view >= NGC::CAP_SEE_ALL){
                if ($record->graderid != $this->_viewerid){
                    $m_add('see_grader_records', $base_params + [NED::PAR_GRADER => $record->graderid]);
                }
            }
        }
    }

    /**
     * Init ngc menu with links
     *
     * @param \action_menu           $menu   - menu object, to fill links
     * @param object|NGC_record|null $record - record, for which create menu (null for bulk menu)
     */
    protected function _init_ngc_action_menu_links_edit(&$menu, $record=null){
        if (!$this->can_see() || !$this->can_edit()) return;

        $m_add = function($call_form_or_name, $add_url=null, $class=[]) use (&$menu){
            $url = $add_url ?: ['#'];
            $class = NED::val2arr($class);
            $class[] = $call_form_or_name;
            if (!$add_url){
                $class[] = 'calljs';
            }

            $menu->add(NED::link($url, NED::str($call_form_or_name), NED::arr2str($class)));
        };
        $curr_url = $this->get_my_url();

        if (isset($record->id)){
            $par_id = $this->_get_param_id($record->id);
            $m_add_rec_menu = function($name, $action, $class=[]) use (&$m_add, &$record, &$par_id, &$curr_url){
                $class = NED::val2arr($class);
                $class[] = 'link';
                $m_add($name, $this->get_my_url([NED::PAR_ACTION => $action, $par_id => $record->id, NED::PAR_RETURN_URL => $curr_url]), $class);
            };

            $m_add_rec_menu('edit', static::ACTION_EDIT);
            if ($record->status == NGC::ST_PAUSED){
                $m_add_rec_menu('ngc_unpause', static::ACTION_UNPAUSE);
            } else {
                $m_add_rec_menu('ngc_pause', static::ACTION_PAUSE);
            }
            if ($record->status == NGC::ST_OBSOLETED){
                $m_add_rec_menu('ngc_unpause', static::ACTION_UNPAUSE);
            } else {
                $m_add_rec_menu('ngc_obsolete', static::ACTION_OBSOLETE);
            }
            $m_add_rec_menu('ngc_delete', static::ACTION_DELETE);
        } else {
            $m_add_bulk_menu = function($name, $action, $class=[]) use (&$m_add, &$curr_url){
                $class = NED::val2arr($class);
                $class[] = 'link';
                $m_add($name,
                    $this->get_my_url([NED::PAR_ACTION => $action, NED::PAR_FILTER_IDS => '{ids}', NED::PAR_RETURN_URL => $curr_url]), $class);
            };

            $m_add_bulk_menu('ngc_unpause', static::ACTION_UNPAUSE);
            $m_add_bulk_menu('ngc_pause', static::ACTION_PAUSE);
            $m_add_bulk_menu('ngc_obsolete', static::ACTION_OBSOLETE);
            $m_add_bulk_menu('ngc_deleterecords', static::ACTION_DELETE);
        }
    }

    /**
     * Get NGC menu for ngc-record or page
     *
     * @param null|NGC_record|\stdClass $record - record, for which menu is
     * @param string                    $title  - add title to menu trigger
     * @param int                       $type   - init only specific type of links, if it's ACTION_NONE - load all links
     * @param bool                      $render - if true, return string, otherwise return action_menu
     *
     * @return \action_menu|string
     * @noinspection PhpMissingBreakStatementInspection
     */
    public function get_ngc_action_menu($record=null, $title='', $type=self::ACTION_NONE, $render=true){
        $menu = new \action_menu();
        if (!empty($title)){
            $title = NED::span($title, 'ned-actionmenu-title mx-1');
        } else {
            $title = '';
        }
        $menu->set_menu_trigger($title.NED::fa('fa-caret-down'));

        $menu->attributes['class'] .= ' ned-actionmenu teacher-menu';

        $init_one = $type != static::ACTION_NONE;
        switch($type){
            case static::ACTION_NONE:
            case static::ACTION_VIEW:
                $this->_init_ngc_action_menu_links_view($menu, $record);
                if ($init_one) break;
            case static::ACTION_EDIT:
                $this->_init_ngc_action_menu_links_edit($menu, $record);
                if ($init_one) break;
        }

        if ($render){
            if ($menu->is_empty()) return '';

            return $this->_o->render($menu);
        }

        return $menu;
    }
    //endregion

    //region Main render page content methods
    /**
     * Render infraction info for some user and (optional) some course
     * It ignores most of the normal (page) render methods and checks
     * Normally should be call only for the user profile page.
     *
     * @param numeric|object|null $user_or_id - if null, try uses {@see _userid}
     * @param numeric|object|null $course_or_id - if null, try uses {@see _courseid}
     *
     * @return string|false
     */
    public function render_user_infraction_info($user_or_id, $course_or_id=null){
        $userid = NED::get_id($user_or_id);
        if (!$this->can_see() || (empty($userid) && empty($this->_userid))){
            // do not call an error, as this content for moodle pages
            return false;
        }

        // rewrite user and course
        $this->_userid = $userid ?: $this->_userid;
        $this->_courseid = NED::get_courseid_or_global($course_or_id, null) ?? $this->_courseid;

        $this->_school_year = static::SCHOOL_YEAR_CURRENT;
        $school_year_options = static::get_school_year_options();
        if (!empty($school_year_options)){
            $school_year = optional_param(NED::PAR_PROFILE_SCHOOL_YEAR, static::SCHOOL_YEAR_CURRENT, PARAM_INT);
            $this->_school_year = NED::isset_key($school_year_options, $school_year, static::SCHOOL_YEAR_CURRENT);
        }

        $site_grader = NGC::can_see_student($this->_viewerid, $this->_userid);
        if (!$site_grader) return false;

        if (empty($this->_courseid) || !NGC::can_see_student($this->_viewerid, $this->_userid, $this->_courseid)){
            $this->_courseid = NED::ALL;
        }

        $course_name = '';
        if (!empty($this->_courseid)){
            $course = NED::get_course($this->_courseid);
            if (!empty($course->shortname)){
                $course_name = $course->shortname;
            } else {
                $this->_courseid = NED::ALL;
            }
        }

        $data = $this->_get_user_infraction_info_data();
        if (empty($data)) return false;

        $this->content = $data;
        $this->content->form = [];
        $link_site_params = [NED::PAR_USER => $this->_userid];
        if ($this->_school_year != static::SCHOOL_YEAR_CURRENT){
            $link_site_params[NED::PAR_SCHOOL_YEAR] = $this->_school_year;
        }
        $link_course_params = $link_site_params + [NED::PAR_COURSE => $this->_courseid, NED::PAR_COURSE_VIEW => 1];
        $this->content->site_link = static::get_url($link_site_params);

        $has_course = !empty($course_name);
        if ($has_course){
            $this->content->show_course = true;
            $this->content->course_name = $course_name;
            $this->content->course_link = static::get_url($link_course_params);
        }

        foreach (NGC::SUBTYPES as $subtype){
            $this->content->{$subtype.'_site_link'} = static::get_url($link_site_params+[static::PAR_SUBTYPE => $subtype]);
            if ($has_course){
                $this->content->{$subtype.'_course_link'} = static::get_url($link_course_params+[static::PAR_SUBTYPE => $subtype]);
            }
        }

        if (!empty($school_year_options)){
            $school_year_selector = NED::single_select(NED::page()->url, NED::PAR_PROFILE_SCHOOL_YEAR,
                static::get_list_as_options($school_year_options), $this->_school_year, NED::str('ned_school_year_title'));
            $this->content->form[] = $school_year_selector;
        }

        $this->content->has_form = !empty($this->content->form);
        return NED::render_from_template('ned_grade_controller_user_infraction_info', $this->content);
    }

    /**
     * Init some content values before {@see export_for_template()}
     * Normally calling from the {@see export_for_template()}
     */
    protected function _before_start_export(){
        parent::_before_start_export();

        $this->content->admin_links = [];
        $this->content->single_record = null;

        if ($this->_is_admin){
            $this->content->show_ids = !empty($this->_ids);
        }
        $this->content->{static::VIEWS[$this->_view]} = true;
        $this->content->hide_fields_for_student = $this->_cap_view == NGC::CAP_SEE_ME;
        $this->content->show_repeats_column = $this->_repeats != static::REPEAT_HIDE;
    }

    /**
     * Some last content changes at final {@see export_for_template()}
     * Normally calling from the {@see export_for_template()}
     */
    protected function _before_finish_export(){
        parent::_before_finish_export();
        if ($this->content->has_data){
            NED::js_call_amd('add_sorter','add_sort', ['table.nedtable']);
            NED::js_call_amd('ned_grade_controller', 'init', [(int)$this->_course_view]);
        }
    }

    /**
     * Call this method before header output, then you can redirect here
     */
    public function before_header_output(){
        $this->_precheck_submit_form();
        $this->_check_submit_form();
    }

    /**
     * Render page content
     * Normally calling from the {@see export_for_template()}
     * You can rewrite this method and not changing original {@see export_for_template()}
     */
    protected function _render_page_content(){
        if ($this->_action != static::ACTION_VIEW) return;

        if (empty($this->_id)){
            $this->_render_view();
        } else {
            $this->_render_single_view();
        }
    }

    /**
     * Set data for single record
     */
    protected function _render_single_view(){
        $record = $this->_record;

        $buttons = [];
        if (!empty($this->_returnurl)){
            $buttons[] = NED::button_link($this->_returnurl, 'back', '', 1);
        }

        if ($record->relatedid){
            $buttons[] = NED::button_link(NED::ai_record_view_url($record->relatedid, $this->_course_view), 'See violation record');
        }
        $buttons[] = NED::button_link($this->get_main_url(), NED::str('see_all_records'));

        $menu = new \action_menu();
        $menu->set_menu_trigger(NED::span('See', 'mx-1').NED::fa('fa-caret-down'));
        $menu->attributes['class'] .= ' ned-actionmenu teacher-menu';

        $record->_course_view = $this->_course_view;
        $desc = static::get_description_object($record, $record, null, '', $this->_cap_view);

        $desc->buttons = $buttons;
        $desc->controls = true;
        $desc->bulk_menus = [];
        $menus = [static::ACTION_VIEW => 'see_ngc_others', static::ACTION_EDIT => 'change'];
        foreach ($menus as $type => $title){
            $menu = $this->get_ngc_action_menu($record, NED::str($title), $type);
            if (!empty($menu)){
                $desc->bulk_menus[] = $menu;
            }
        }
        $desc->has_bulk_menu = !empty($desc->bulk_menus);

        $this->content->single_record = $desc;
    }

    /**
     * Render for action view
     */
    protected function _render_view(){
        $show_own_selectors = $this->_cap_view >= NGC::CAP_SEE_OWN_ANY;
        $this->content->data = array_values($this->get_main_table_data());

        $this->content->pager = $this->r_pager();
        $url = $this->get_my_url();

        $grader_selector = $this->r_grader_selector();
        if (!empty($grader_selector)){
            $this->content->control_panel1[] = $grader_selector;
        }

        $options = static::get_list_as_options(NGC::GRADE_TYPES, true);
        $this->content->control_panel1[] =
            NED::single_select($url, static::PAR_GRADE_TYPE, $options, $this->_grade_type, NED::str('gradetype'));

        $options = static::get_list_as_options($this->get_reasons_options());
        $this->content->control_panel1[] =
            NED::single_select($url, static::PAR_REASON, $options, $this->_reason, NED::str('reason'));
        $this->content->show_reasons = $this->_reason == 0;

        $options = static::get_list_as_options(NGC::STATUSES, true);
        if (!$this->can_edit()){
            foreach (NGC::HIDDEN_STATUSES as $status => $hidden){
                if ($hidden) unset($options[$status]);
            }
        }
        $this->content->control_panel1[] =
            NED::single_select($url, NED::PAR_STATUS, $options, $this->_status, NED::str('status'));
        $this->content->control_panel1[] =
            NED::single_select($url, static::PAR_SUBTYPE, NGC::get_subtypes(), $this->_subtype, NED::str('infraction'));
        $options = static::get_list_as_options(static::REPEAT_OPTIONS);
        $this->content->control_panel1[] =
            NED::single_select($url, static::PAR_REPEATS, $options, $this->_repeats, NED::str('ngc:repeats:title'));
        $this->content->control_panel1[] = $this->r_school_year_selector();

        if ($show_own_selectors){
            $school_selector = $this->r_school_selector();
            if (!empty($school_selector)){
                $this->content->control_panel2[] = $school_selector;
            }
        }

        if (!$this->_course_view){
            $this->content->control_panel2[] = $this->r_course_selector();
        }

        if ($this->_courseid > 0){
            $this->content->control_panel2[] = $this->r_cm_selector();

            if ($show_own_selectors){
                $this->content->control_panel2[] = $this->r_group_selector();
            }
        }

        if ($show_own_selectors){
            $this->content->control_panel2[] = $this->r_users_selector();
        }

        if ($this->can_edit()){
            $this->content->buttons[] = NED::button_link($this->get_my_url([NED::PAR_ACTION => static::ACTION_ADD]), 'addrecord', '', true);
        }

        if ($this->_courseid > 0){
            $this->content->buttons[] = $this->r_course_view_button();
        }

        if (!empty($url->params()) || !empty($this->_filter_ids) || !empty($this->_ids)){
            $this->content->buttons[] = $this->r_reset_button();
        }

        $options = static::get_list_as_options(static::VIEWS);
        $this->content->control_panel3[] =
            NED::single_select($url, NED::PAR_VIEW, $options, $this->_view, NED::HTML_INVISIBLE);

        if ($this->_total_count > static::PERPAGES[0]){
            $this->content->control_panel3[] = $this->r_perpage_selector();
        }

        if (!empty($this->content->data) && $this->can_edit()){
            $this->content->bulk_menu = $this->get_ngc_action_menu();
        }
    }
    //endregion

    //region Form methods
    /**
     * Set SESSION_RETURN_URL and/or _record
     * Can redirect, if it finds an error
     */
    protected function _precheck_submit_form(){
        if (empty($this->_filter_ids) || $this->_action <= static::ACTION_ADD){
            return;
        }

        $cache = NED::get_user_cache();
        $return_url = $this->_returnurl ?: $cache->get(NED::SESSION_RETURN_URL);
        if (!$return_url){
            $cache->delete(static::SESSION_NOTE_KEY);

            $return_params = [NED::PAR_ACTION => static::ACTION_VIEW, NED::PAR_FILTER_ID => 0];
            if ($this->_action != static::ACTION_DELETE && $this->_id){
                $return_params[NED::PAR_ID] = $this->_id;
            } elseif ($userid = $this->_userid ?: ($this->_record->userid ?? false)){
                $return_params[NED::PAR_USER] = $userid;
            }
            $cache->set(NED::SESSION_RETURN_URL, $this->get_my_url($return_params));

            // redirect
            $params = [
                NED::PAR_ACTION => $this->_action,
                NED::PAR_COURSE => $this->_courseid,
                NED::PAR_COURSE_VIEW => $this->_course_view,
            ];
            if (!empty($this->_filter_ids)){
                $params[NED::PAR_FILTER_IDS] = join(',', $this->_filter_ids);
            } else {
                $params[NED::PAR_FILTER_ID] = $this->_filter_id;
            }
            NED::redirect(static::get_url($params));
        } else {
            $records = $this->get_main_table_data(false);
            if (!empty($records) && count($records) > 1){
                $this->_records = $records;
            }
            $record = reset($records);

            if (!$record){
                $cache->delete(NED::SESSION_RETURN_URL);
                $text = $this->_filter_id ? NED::str('norecordwithid', $this->_filter_id) : NED::str('nosuchrecords');
                NED::redirect($return_url, $text, null,NED::NOTIFY_ERROR);
            } else {
                $saved_record = $cache->get(static::SESSION_EDITABLE_RECORD);
                if ($record->id == ($saved_record->id ?? 0)){
                    $record = $saved_record;
                } else {
                    $cache->delete(static::SESSION_EDITABLE_RECORD);
                }
                $this->_set_record($record);
            }
        }
    }

    /**
     * Redirected to saved url or home NGC page
     *
     * @param string $message
     * @param string $messagetype
     */
    protected function _form_home_redirect($message='', $messagetype=NED::NOTIFY_INFO){
        $cache = NED::get_user_cache();
        $is_delete = $this->_action == static::ACTION_DELETE;
        $return_url = $this->_returnurl;

        if (!$is_delete && $this->_id){
            $return_url = $return_url ?: $this->get_main_url([NED::PAR_ID => $this->_id]);
        } else {
            $return_url = $return_url ?: $cache->get(NED::SESSION_RETURN_URL);
            if ($return_url && $is_delete){
                $return_url = new \moodle_url($return_url);
                if ($return_url->param(NED::PAR_ID)){
                    $return_url = null;
                }
            }

            $return_url = $return_url ?: $this->get_main_url();
        }

        $cache->delete(static::SESSION_NOTE_KEY);
        $cache->delete(static::SESSION_EDITABLE_RECORD);
        $cache->delete(NED::SESSION_RETURN_URL);

        NED::redirect($return_url, $message, null, $messagetype);
    }

    /**
     * Redirected to next options page
     *
     * @param int       $step
     * @param string    $message
     */
    protected function _form_next_redirect($step=0, $message=''){
        $params = [NED::PAR_STEP => $step, NED::PAR_ACTION => $this->_action];
        if ($this->_record->id ?? false){
            $params[$this->_get_param_id($this->_record->id)] = $this->_record->id;
        }

        NED::redirect($this->get_my_url($params), $message);
    }

    /**
     * Check and show form
     * @noinspection PhpUnnecessaryStopStatementInspection
     */
    protected function _check_submit_form(){
        if ($this->_action == static::ACTION_VIEW) return;

        $cache = NED::get_user_cache();
        $step = $this->get_step();
        $f_road = $step >= static::NORMAL_FINAL_STEP;

        // $simple_form has only buttons
        $simple_form = $this->get_form_by_step(-1);
        if ($simple_form->is_submitted()){
            if ($simple_form->is_cancelled()){
                if ($simple_form->is_cancelled('back')){
                    if ($step == static::FULLY_FINAL_STEP){
                        $step -= 2;
                    } else {
                        $step--;
                    }
                    $this->_form_next_redirect($step);
                } else {
                    $this->_form_home_redirect();
                }
            }

            if ($f_road){
                $data_form = $this->get_form_by_step($step);
                $data = $data_form->exportValues();
                $this->_note = $data[static::PAR_NOTE] ?? null;
                if (!is_null($this->_note)){
                    $this->_note = trim($this->_note);
                    $cache->set(static::SESSION_NOTE_KEY, $this->_note);
                }

                if ($this->_record){
                    foreach ($this->_record as $par => $val){
                        if (!is_null($data[$par] ?? null)){
                            $this->_record->$par = $data[$par];
                        }
                    }
                    $cache->set(static::SESSION_EDITABLE_RECORD, $this->_record);
                }
                $this->_confirm = $data[static::PAR_CONFIRM] ?? null;
            }

            if (!$this->_confirm){
                $this->_form_next_redirect($step+1);
            }
        }

        if ($f_road){
            if (!$this->_confirm){
                // don't ask confirm for multiply actions or if it's not delete
                $this->_confirm = (static::AUTOMATIC_CONFIRM_ACTIONS_ANY[$this->_action] ?? false) ||
                    (!empty($this->_records) && !empty(static::AUTOMATIC_CONFIRM_ACTIONS_GROUP[$this->_action]));
            }

            if (is_null($this->_note)){
                $this->_note = $cache->get(static::SESSION_NOTE_KEY) ?: null;
            }
        }

        // OK, there is no redirection, and we should show really form
        if ($f_road && $this->_confirm){
            $redirect_text = $this->_confirm_form();
            $this->_form_home_redirect(NED::str_check($redirect_text), NED::NOTIFY_SUCCESS);
            return;
        } else {
            $form = $this->get_form_by_step($step);
            if (!$form){
                return;
            }

            $this->content->form = $form->draw();
        }
    }

    /**
     * Sent form confirmation
     *
     * @return string
     *
     * @noinspection PhpMissingBreakStatementInspection
     */
    protected function _confirm_form(){
        $object = null;
        switch ($this->_action){
            case static::ACTION_EDIT:
                $object = $this->_record;
            case static::ACTION_ADD:
            case static::ACTION_ADD_WRONG_SUBMISSION:
                $object = $object ?? $this->export('', false, true);
                $this->_id = NGC::check_and_save($object);
                return 'savedsuccessfully';
            case static::ACTION_DELETE:
                if (!empty($this->_records)){
                    $count = NGC::check_and_delete($this->_records, true);
                    if ($count > 0){
                        return NED::str('ngc_removedsuccessfully_records', $count);
                    }
                } else {
                    if (NGC::check_and_delete($this->_record, true)){
                        return 'ngc_removedsuccessfully';
                    }
                }

                break;
            case static::ACTION_PAUSE:
                $set_state = $set_state ?? true;
            case static::ACTION_UNPAUSE:
                $state = $state ?? NGC::ST_PAUSED;
                $set_state = $set_state ?? false;
                $ans_records = $ans_records ?? ['ngc_unpausedsuccessfully_records', 'ngc_pausedsuccessfully_records'];
                $ans_state = $ans_state ?? ['ngc_unpausedsuccessfully', 'ngc_pausedsuccessfully'];

            case static::ACTION_OBSOLETE:
                $set_state = $set_state ?? true;
                $state = $state ?? NGC::ST_OBSOLETED;
                $set_state = $set_state ?? false;
                $ans_records = $ans_records ?? ['ngc_unpausedsuccessfully_records', 'ngc_obsoletedsuccessfully_records'];
                $ans_state = $ans_state ?? ['ngc_unpausedsuccessfully', 'ngc_obsoletesuccessfully'];

                // change state
                $editorid = NED::get_userid_or_global();
                if (!empty($this->_records)){
                    $count = NGC::check_and_change_about_pause_state($state, $this->_records, $set_state, true, false, $editorid);
                    if ($count > 0){
                        return NED::str($ans_records[(int)$set_state], $count);
                    }
                } else {
                    if (NGC::check_and_change_about_pause_state($state, $this->_record, $set_state, true, false, $editorid)){
                        return $ans_state[(int)$set_state];
                    }
                }

                break;
            default:
            case static::ACTION_VIEW:
                return '';
        }

        return '';
    }

    /**
     * @return int
     */
    public function get_min_step(){
        return match ($this->_action) {
            static::ACTION_DELETE, static::ACTION_PAUSE, static::ACTION_UNPAUSE, static::ACTION_OBSOLETE,
            static::ACTION_ADD_WRONG_SUBMISSION => static::FULLY_FINAL_STEP,

            static::ACTION_EDIT => static::NORMAL_FINAL_STEP,
            default => 0,
        };
    }

    /**
     * Get current add step by existing params
     *
     * @return int
     */
    public function get_step(){
        $this->_step = $this->_step ?: $this->get_min_step();

        switch ($this->_action){
            case static::ACTION_VIEW:
            case static::ACTION_DELETE:
            case static::ACTION_PAUSE:
            case static::ACTION_UNPAUSE:
            case static::ACTION_OBSOLETE:
                return $this->_step;
            default:
                break;
        }

        $step = 0;
        /** @var object|NGC_record|static $object */
        $object =  $this->export('', false, true);

        if (empty($object->courseid) || empty($object->grade_type)){
            $step = 1;
        } elseif (empty($object->cmid)){
            $step = 2;
        } elseif (empty($object->userid)){
            $step = 3;
        } elseif (empty($object->reason)){
            $step = static::NORMAL_FINAL_STEP;
        } elseif ($this->_step <= static::NORMAL_FINAL_STEP){
            $step = static::NORMAL_FINAL_STEP;
        } else {
            if (!$this->_record){
                $records = $this->get_main_table_data(false);
                if (!empty($records)){
                    $step = static::NORMAL_FINAL_STEP;
                }
            }

            if ($object->grade_type == NGC::GT_DEDUCTION){
                $object->grade_change = (int)$object->grade_change;
                if ($object->grade_change < 0 || $object->grade_change > 99){
                    $this->add_notification(NED::str('gradereduction_rule'), NED::NOTIFY_WARNING);
                    $step = static::NORMAL_FINAL_STEP;
                }
            }

            if (!$step){
                $related_record = null;
                $not_need_step5 = false;
                if ($object->reason == NGC::REASON_AI){
                    if (NED::is_ai_exists()){
                        if (!empty($object->relatedid)){
                            $related_record = NGC::get_related_data_by_id($object->relatedid);
                            if (!$related_record){
                                $this->_relatedid = 0;
                            }
                        } else {
                            $records = NGC::get_related_data($object->userid, $object->courseid, $object->cmid);
                            $c_records = count($records);
                            if ($c_records == 1){
                                $this->_relatedid = $object->relatedid = reset($records)->id;
                                if ($this->_record){
                                    $this->_record->relatedid = $this->_relatedid;
                                }
                                // there is only one record, not need to choose
                                $not_need_step5 = true;
                            }
                        }
                    } else {
                        // we shouldn't be here if all is fine
                        $object->reason = NGC::REASON_OTHER;
                        $not_need_step5 = true;
                    }
                } else {
                    $not_need_step5 = true;
                }

                if ($not_need_step5){
                    if ($this->_step == 5){
                        $this->_step = static::FULLY_FINAL_STEP;
                    }
                    $step = static::FULLY_FINAL_STEP;
                } elseif (!$related_record){
                    $step = 5;
                } else {
                    $step = static::FULLY_FINAL_STEP;
                }
            }
        }

        if ($this->_step ?? false){
            $step = min($this->_step, $step);
        }

        return max(0, $step);
    }

    /**
     * Return adding form by adding step
     *
     * @param int $step
     *
     * @return \local_ned_controller\form\base_form|string
     */
    public function get_form_by_step($step=0){
        $form_data = [];
        $FE = NED::$form_element;
        $F = NED::$base_form;
        $hiddens = [];
        $elements = [];
        $errors = [];
        $buttons = [];
        $last_step = false;
        $object = $this->_record ?? $this;

        if ($object->id ?? false){
            $hiddens[$this->_get_param_id($object->id)] = $object->id;
        }

        switch($step){
            case 1:
                if (!$object->courseid || !$this->_course_view){
                    $courses = NED::get_grader_courses($this->_viewerid);
                    if (empty($courses)){
                        $errors[] = NED::str('sorry_no_smth', get_string('course'));
                        break;
                    }

                    $course_menu = NED::records2menu($courses, 'shortname');
                    $elem = $FE::autocomplete(NED::PAR_COURSE, get_string('course'), $course_menu, $object->courseid);
                    $elem->setSimpleRequired();
                    $elements[] = $elem;
                }

                $options = NED::strings2menu(NGC::GRADE_TYPES);
                $elem = $FE::select(static::PAR_GRADE_TYPE, NED::str('gradetype'), $options, $object->grade_type);
                $elem->setSimpleRequired();
                $elements[] = $elem;
                break;
            case 2:
                $cms = NGC::get_NGC_cms($object->courseid, $this->_viewerid);
                if (empty($cms)){
                    $errors[] = NED::str('sorry_no_smth', NED::str('activity'));
                    break;
                }

                $options = NED::cms2menu($cms);
                $elem = $FE::autocomplete(NED::PAR_CM, NED::str('activity'), $options, $object->cmid);
                $elem->setSimpleRequired();
                $elements[] = $elem;

                $groups = NED::get_all_user_course_groups($object->courseid, $this->_viewerid, 'g.id, g.name');
                if (!empty($groups)){
                    $options = NED::records2menu($groups);
                    $options = [0 => get_string('all')] + $options;
                    $elements[] = $FE::autocomplete(NED::PAR_GROUP, NED::str('class'), $options, $object->groupid);
                }
                break;
            case 3:
                $students = NED::get_course_students_by_role($object->courseid, $object->cmid, $object->groupid);
                if (empty($students)){
                    $errors[] = NED::str('sorry_no_smth', NED::str('student'));
                    break;
                }

                $options = NED::users2menu($students);
                $elem = $FE::autocomplete(NED::PAR_USER, NED::str('student'), $options, $object->userid);
                $elem->setSimpleRequired();
                $elements[] = $elem;
                break;
            case static::NORMAL_FINAL_STEP:
                $extra_edit = $this->can_extra_edit() && $this->_action == static::ACTION_EDIT;

                if (!($object->id ?? false)){
                    $records = $this->get_main_table_data(false);
                    if (!empty($records)){
                        $record = reset($records);
                        $errors[] = NED::str('havethisstudent').' '.
                            NED::link(static::get_main_url([NED::PAR_ID => $record->id]),
                                NED::str('opensomepage', NED::str('gradecontroller')));
                        break;
                    }
                }

                // turn off reason editing, set only per creation
                if ($this->_action == static::ACTION_ADD || $extra_edit){
                    if ($extra_edit){
                        $options = NED::strings2menu(NGC::GRADE_TYPES);
                        $elem = $FE::select(static::PAR_GRADE_TYPE, NED::str('gradetype'), $options, $object->grade_type);
                        $elem->setSimpleRequired();
                        $elements[] = $elem;
                    }

                    $options = NGC::get_reasons($object->grade_type);
                    if (!empty($options)){
                        $options = NED::strings2menu($options);
                        $elem = $FE::select(static::PAR_REASON, NED::str('reason'), $options, $object->reason);
                        $elem->setSimpleRequired();
                        $elements[] = $elem;
                    }
                }

                if ($object->grade_type == NGC::GT_DEDUCTION || $extra_edit){
                    $elem = $FE::text(static::PAR_GRADE_CHANGE, NED::str('applygradereduction'));
                    $elem->addAttribute('size', 5);
                    $elem->setDefault($object->grade_change ?: 0);
                    $elem->addRule('numeric');
                    $elem->setRequired(null, 'server');
                    if ($extra_edit){
                        $elem->addHideIf(static::PAR_GRADE_TYPE, 'neq', NGC::GT_DEDUCTION);
                    }
                    $elements[] = $elem;
                }

                $elem = $FE::textarea(static::PAR_NOTE, NED::str('note'));
                $elem->setDefault($object->note ?? '');
                $elements[] = $elem;
                break;

            /** @noinspection PhpMissingBreakStatementInspection */
            case 5:
                if (NED::is_ai_exists()){
                    $q = NED::str('choose_ai_record') .' '.
                        NED::ext_link([NED::PAGE_AI_INFRACTIONS, [NED::PAR_USER => $object->userid, NED::PAR_COURSE => $object->courseid]],
                            NED::str('opensomepage', NED::str('academicintegrityinfractions')));
                    $elements[] = $FE::div($q, 'question');
                    $records = NGC::get_related_data($object->userid, $object->courseid, $object->cmid);
                    if (empty($records)){
                        $errors[] = NED::str('noaireport');
                        break;
                    }

                    $column_data = DI::get_table_columns(false, true);
                    $column_data = ['select' => get_string('select')] + $column_data;
                    $table = DI::fill_and_get_table($column_data, $records);
                    $table->align['select'] = 'center';
                    $fake_options = [];
                    // Add manual radio buttons to the table rows
                    /** @var \html_table_row $row */
                    foreach ($table->data as $row){
                        $params = ['type' => 'radio', 'name' => static::PAR_RELATEDID, 'value' => $row->id];
                        if ($row->id == $object->relatedid){
                            $params['checked'] = 'checked';
                        }
                        $row->cells['select'] = \html_writer::empty_tag('input', $params);
                        $fake_options[$row->id] = $row->id;
                    }
                    $elements[] = $FE::html(\html_writer::table($table));
                    // save hidden fake options, to get info through the form
                    $elem = $FE::radio(static::PAR_RELATEDID, $fake_options, null, ['class' => 'display-none']);
                    $elements[] = $elem;
                    break;
                }

                // if we here, then continue with the next step
                $step = static::FULLY_FINAL_STEP;
            case static::FULLY_FINAL_STEP:
                $elements[] = $FE::html('<br>');
                $action_error = false;
                switch ($this->_action){
                    case static::ACTION_DELETE:
                        $action_error = NGC::has_errors_for_deleting($object);
                        break;
                    case static::ACTION_PAUSE:
                    case static::ACTION_UNPAUSE:
                    case static::ACTION_OBSOLETE:
                        $action_error = NGC::has_errors_for_change_status($object);
                        break;
                    case static::ACTION_ADD_WRONG_SUBMISSION:
                        $elements[] = $FE::html(NED::render_from_template('ned_grade_controller_wrong_submission_description'));
                        break;
                }
                if ($action_error){
                    $errors[] = $action_error;
                }

                if (empty($errors)){
                    $q = NED::str_check('NGC_confirm_'.static::ACTIONS[$this->_action],[], NED::str('wishcontinue'));
                    $elements[] = $FE::div($q, 'question');
                    $hiddens[static::PAR_CONFIRM] = 1;
                }

                if (!empty($object->note)){
                    $elements[] = $FE::hidden(static::PAR_NOTE, $object->note);
                }

                $relatedid = $object->relatedid ?: $this->_relatedid;
                if (!empty($relatedid)){
                    $elements[] = $FE::hidden(static::PAR_RELATEDID, $relatedid);
                }

                $last_step = true;
                break;
        }

        if ($step > 0){
            $hiddens[NED::PAR_STEP] = $step;
        }
        // for step == 0 we want to get back button too
        if ($step == -1 || ($step != 1 && $step > $this->get_min_step())){
            $buttons[] = $FE::cancel('back', get_string('back'));
        }
        $buttons[] = $FE::cancel();
        if (empty($errors)){
            if ($this->_action >= static::ACTION_DELETE){
                $b_name = NED::str(static::ACTIONS[$this->_action] ?? 'ok');
                $buttons[] = $FE::submit('submit', $b_name, true,'btn-danger');
            } else {
                $buttons[] = $FE::submit('submit', get_string($last_step ? 'save' : 'next'));
            }
        }
        $elements[] = $FE::group('buttons', '', $buttons);
        $form_data[$F::DESCRIPTION] = $this->get_add_description($step, $object);
        $form_data[$F::ELEMENTS] = $elements;
        $form_data[$F::HIDDENS] = $hiddens;
        $form_data[$F::ERRORS] = $errors;

        $form = $F::create($this->get_my_url([NED::PAR_ACTION => $this->_action]), $form_data);
        return $form;
    }
    //endregion

    //region URL methods
    /**
     * Get param name filter_id or id, depends of object properties
     *
     * @param int|null $check_id - if provided, check this id with object _id
     *
     * @return string
     */
    protected function _get_param_id($check_id=null){
        if ($this->_id){
            if ($check_id){
                if ($check_id == $this->_id){
                    return NED::PAR_ID;
                } else {
                    return NED::PAR_FILTER_ID;
                }
            } else {
                return NED::PAR_ID;
            }
        }
        return NED::PAR_FILTER_ID;
    }
    //endregion

    //region Static Utils methods
    /**
     * Create bulk_menu link with 1 parameter
     *
     * @param $base_url
     * @param $text
     * @param $key
     * @param $value
     * @return string
     */
    static public function link($base_url, $text, $key=null, $value=null){
        $url = $base_url;
        if (!empty($key) && !empty($value)){
            $url = clone $base_url;
            $url->param($key, $value);
        }
        return NED::link($url, $text);
    }

    /**
     * @param string        $status - status value from NGC_record
     * @param string|null   $text_status - (optional) custom text title, by default it's translated status name
     *
     * @return string - html fa element
     * @noinspection PhpUnusedMatchConditionInspection
     */
    static public function get_status_icon($status, $text_status=null){

        $fa_class = match ($status) {
            NGC::ST_WAIT => 'fa-clock-o text-warning',
            NGC::ST_DONE => 'fa-check-circle-o text-success',
            NGC::ST_PAUSED => 'fa-pause-circle-o text-warning',
            NGC::ST_OBSOLETED => 'fa-history muted',
            NGC::ST_ERROR => 'fa-etsy text-danger',
            default => 'fa-question-circle-o text-black-50',
        };

        return NED::fa($fa_class, '', $text_status ?? NED::str_check($status, null, '?'));
    }
    //endregion

    //region Description methods
    /**
     * Render description screen by object
     *
     * @param object|NGC_record $obj
     * @param object $desc  (optional) use this object instead of new
     * @param int    $step  (optional) check info size by current step
     * @param string $title (optional) add title field
     * @param bool   $cap_view (optional) need to output|hide fields
     *
     * @return object  - rendered html
     */
    static public function get_description_object($obj, $desc=null, $step=null, $title='', $cap_view=null){
        $max_step = 100;
        $step = $step ?: $max_step;
        $desc = $desc ?? new \stdClass();
        $desc->title = $title;
        $cap_view = $cap_view ?? NGC::get_see_capability();

        if (empty($obj->courseid) || empty($obj->grade_type)){
            $max_step = 1;
        } elseif (empty($obj->cmid)){
            $max_step = 2;
        } elseif (empty($obj->userid)){
            $max_step = 3;
        } elseif (empty($obj->reason)){
            $max_step = static::NORMAL_FINAL_STEP;
        }

        $step = min($max_step, $step);

        if ($step > 1){
            $desc->course_name = NED::q_course_link($obj->courseid, false, true);
            $desc->course = NED::q_course_link($obj->courseid);
        }
        if ($step > 2){
            $desc->cm_name = NED::q_cm_name($obj->cmid, $obj->courseid, $obj->userid, true);
            if ($cap_view >= NGC::CAP_SEE_OWN_GRADER){
                $desc->cm = NED::q_cm_grade_link($obj->cmid, $obj->userid, $obj->courseid, true, true);
            } else {
                $desc->cm = NED::q_cm_link($obj->cmid, $obj->courseid, $obj->userid);
            }

            $group = null;
            if ($cap_view >= NGC::CAP_SEE_OWN_ANY){
                $desc->classname = " ";
                if (!empty($obj->group_ids)){
                    $groupids = NED::str2arr($obj->group_ids, '', ',');
                    $group = NED::get_group(reset($groupids));
                } elseif (!empty($obj->groupid)){
                    $group = NED::get_group($obj->groupid);
                } elseif (!empty($obj->userid)){
                    $groups = NED::get_all_course_groups($obj->courseid, $obj->userid);
                    $group = reset($groups);
                }

                if ($group){
                    $desc->classname = $group->name;
                    $desc->classenddate = $group->enddate;
                    $desc->classenddate_str = NED::ned_date($desc->classenddate);
                }
            }
        }
        if ($step > 3){
            if ($cap_view >= NGC::CAP_SEE_OWN_ANY){
                $desc->student = NED::q_user_link($obj->userid, $obj->courseid);
            }
            if (isset($obj->viewerid)){
                $desc->graderid = NED::get_graderid_by_studentid($obj->userid, $obj->courseid, $obj->cmid);
            } else {
                $desc->graderid = ($obj->graderid ?? 0) ?: NED::get_graderid_by_studentid($obj->userid, $obj->courseid, $obj->cmid);
            }
            $desc->gradername = NED::q_user_link($desc->graderid, $obj->courseid);
            $desc->deadline = $obj->deadline ?? NED::get_deadline_by_cm($obj->cmid, $obj->userid, $obj->courseid);
            $desc->deadline_str = NED::ned_date($desc->deadline);
            if ($cap_view >= NGC::CAP_SEE_OWN_GRADER && NED::is_tt_exists()){
                $desc->deadline_str = NED::ext_link(
                    [NED::PAGE_DM, [NED::$MM::P_COURSEID => $obj->courseid, NED::PAR_SETUSER => $obj->userid, 'p' => 'user']],
                    $desc->deadline_str
                );
            }
        } else {
            $desc->deadline = 0;
        }

        if ($step > static::NORMAL_FINAL_STEP || !empty($obj->reason)){
            $course_view = $obj->course_view ?? NED::get_courseid_or_global() != SITEID;
            $desc->reason = $obj->reason;
            $desc->reason_str = NED::str(NGC::REASONS[$obj->reason]);
            if ($obj->reason == NGC::REASON_AI && $obj->relatedid){
                $desc->reason_str = NED::link(NED::ai_record_view_url($obj->relatedid, $course_view), $desc->reason_str);
            }
        }

        if ($step > static::NORMAL_FINAL_STEP){
            $course_view = $course_view ?? $obj->course_view ?? NED::get_courseid_or_global() != SITEID;
            if ($obj->grade_type == NGC::GT_DEDUCTION && ($obj->grade_change ?? -1) >= 0){
                $desc->grade_change = $obj->grade_change;
            } else {
                $desc->grade_change = false;
            }
            $desc->grade_type = $obj->grade_type;

            if ($cap_view >= NGC::CAP_SEE_OWN_ANY){
                $desc->note = $obj->note ?? '';
            } else {
                $desc->note = null;
            }

            if ($obj->status ?? false){
                $desc->status_str = isset(NGC::STATUSES[$obj->status]) ? NED::str(NGC::STATUSES[$obj->status]) : '?';
                $desc->status_icon = static::get_status_icon($obj->status, $desc->status_str);
            }

            $desc->classenddate = $desc->classenddate ?? 0;
            $desc->deadline = $desc->deadline ?? 0;
            $desc->timecreated = $obj->timecreated ?: time();
            $desc->applying = $desc->timecreated;
            $desc->applying_str = NED::ned_date($desc->applying);

            $icon = NGC::get_grade_status($desc->grade_type, $desc->reason, $desc->deadline, $obj->cm_type);
            $icon = NED::ned_grade_icon_check_dm_extension($icon, $obj->cmid, $obj->userid, $desc->deadline);
            $icon[NED::ICON_URL_DATA] = [
                NED::PAR_ID          => $desc->id ?? $obj->id,
                NED::ICON_URL_AI_ID  => $obj->relatedid,
                NED::PAR_COURSE_VIEW => $course_view,
            ];
            $desc->icon = NED::get_ned_grade_element($icon, '');
        }

        NED::save_author_and_editor_to_obj($obj, $desc);
        $desc->show_timecreated = $cap_view >= NGC::CAP_SEE_OWN_ANY && $desc->timecreated;

        return $desc;
    }

    /**
     * Render description screen by object
     *
     * @param object $desc
     *
     * @return string - rendered html
     */
    static public function render_description_object($desc){
        return NED::render_from_template('ned_grade_controller_description', $desc);
    }

    /**
     * Render description for "Add" screen
     *
     * @param int $step
     * @param object|static|NGC_record $object - object, by which data set form
     *
     * @return string - rendered html
     */
    public function get_add_description($step=0, $object=null){
        $title = '';
        $object = $object ?? $this;
        switch ($this->_action){
            case static::ACTION_ADD:
                if ($step <= 1){
                    $title = NED::str('addrecord2smth', NED::str('gradecontroller'));
                } else {
                    $title = NED::str('award_smth', NED::str(NGC::GRADE_TYPES[$object->grade_type]));
                }
                break;
            case static::ACTION_ADD_WRONG_SUBMISSION:
                $title = NED::str(NGC::REASONS[NGC::REASON_FILE]);
                break;
            default:
                break;
        }

        $decr = static::get_description_object($object, null, $step, $title);
        return static::render_description_object($decr);
    }
    //endregion

    //region Main rendering class/page methods
    /** For render_full_page {@see static::render_full_page()} */

    /**
     * Render user profile infraction section
     *
     * @param numeric|object $user_or_id - student id to view
     * @param numeric|object $course_or_id - course id to additional course column
     *
     * @return string|false
     */
    static public function render_user_profile_data($user_or_id=null, $course_or_id=null){
        $userid = NED::get_id($user_or_id);
        if (empty($userid)){
            // try to get data from Moodle page
            $userid = optional_param('id', 0, PARAM_INT);
        }
        if (empty($userid)) return false;

        $courseid = NED::get_courseid_or_global($course_or_id, NED::ALL);

        $NGC= new static();
        // skip init, we don't need it here
        return $NGC->render_user_infraction_info($userid, $courseid);
    }
    //endregion
}
