<?php
/**
 * Plotly class for interaction with plotly js library
 *
 * @package    local_ned_controller
 * @subpackage shared
 * @category   NED
 * @copyright  2021 NED {@link http://ned.ca}
 * @author     NED {@link http://ned.ca}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace local_ned_controller\output\plotly;
use local_ned_controller\shared_lib as NED;

/**
 * Class plotly
 *
 * @package local_ned_controller\output\plotly
 *
 * @property-read string $id;
 * @property-read bool $rendered = false;
 */
class plotly
{
    const AMD_SCRIPT = 'local_ned_controller/plotly_add';
    const SIMPLE_DATA_KEYS = ['name', 'hovertext', 'hoverinfo', 'hovertemplate', 'textinfo', 'hole', 'type', 'domain', 'rotation', 'direction', 'sort', 'orientation'];
    const SIMPLE_LAYOUT_KEYS = ['title', 'showlegend', 'height', 'width', 'grid', 'paper_bgcolor', 'plot_bgcolor', 'autosize'];
    const SIMPLE_CONFIG_KEYS = ['displayModeBar', 'editable', 'staticPlot', 'responsive',
        'square', 'get_height', 'get_width', 'get_min', 'get_max', 'width_if_null', 'height_if_null',
    ];
    /** @var \local_ned_controller\output\plotly\plotly[] $_plots */
    static protected $_plots = [];
    static protected $_lastid = 0;

    protected $_id;
    protected $_rendered = false;

    public $selector = '';

    // data
    public $type = '';
    public $textinfo = '';
    public $hole = null;
    public $hovertext = '';
    public $hoverinfo = '';
    public $hovertemplate = '';
    public $hovertemplate2 = '';
    public $rotation = 0;
    public $direction = 'clockwise';
    public $sort = false;
    public $orientation = null;
    public $data_width = null;

    public $hover_size = null;
    // layout
    public $title = '';
    public $annotation = null;
    public $annotation_size = 20;
    public $annotation_color = '';
    public $annotation_x = 0.5;
    public $annotation_y = 0.5;
    public $showarrow = false;
    public $showlegend = false;
    public $height = null;
    public $width = null;
    public $paper_bgcolor = 'rgba(0,0,0,0)';
    public $plot_bgcolor = 'rgba(0,0,0,0)';
    public $autosize = false;
    public $font_size = null;
    public $font_color = null;
    // config
    public $displayModeBar = false;
    public $editable = false;
    public $staticPlot = true;
    public $responsive = true;
    public $square = false;
    public $get_height = false;
    public $get_width = false;
    public $get_min = true;
    public $get_max = false;
    public $width_if_null = 123;
    public $height_if_null = 123;
    // additional
    public $data_options;
    public $layout_options;
    public $config_options;

    /** @var \local_ned_controller\output\plotly\plotly_value[] $values */
    public $values = [];

    /**
     * plotly constructor.
     *
     * @param        $selector
     * @param string $type
     * @param null   $options - options to rewrite any class public properties
     * @param null   $data_options
     * @param null   $layout_options
     * @param null   $config_options
     */
    public function __construct($selector, $type='pie', $options=null, $data_options=null, $layout_options=null, $config_options=null){
        $this->_id = static::$_lastid++;
        static::$_plots[$this->_id] = $this;

        $this->selector = $selector;
        $this->type = $type;

        $options = (object)($options ?: []);
        $data_options = (object)($data_options ?: []);
        $layout_options = (object)($layout_options ?: []);
        $config_options = (object)($config_options ?: []);

        foreach ($options as $option => $value){
            if ($option[0] !== '_' && property_exists($this, $option)){
                $this->$option = $value;
            }
        }

        $this->data_options = $data_options;
        $this->layout_options = $layout_options;
        $this->config_options = $config_options;
    }

    /**
     * Method to get new plotly
     * @constructor
     *
     * @param        $selector
     * @param string $type
     * @param null   $options - options to rewrite any class public properties
     * @param null   $data_options
     * @param null   $layout_options
     * @param null   $config_options
     *
     * @return static
     */
    static public function new_plotly($selector, $type='pie', $options=null, $data_options=null, $layout_options=null, $config_options=null){
        return new static($selector, $type, $options, $data_options, $layout_options, $config_options);
    }

    public function __get($name){
        $name = '_' . $name;
        return property_exists($this, $name) ? $this->$name : null;
    }

    public function __clone(){
        $this->_id = static::$_lastid++;
        static::$_plots[$this->_id] = $this;
        $this->_rendered = false;

        $this->data_options = clone($this->data_options);
        $this->layout_options = clone($this->layout_options);
        $this->config_options = clone($this->config_options);

        foreach ($this->values as $key => $value){
            $this->values[$key] = clone($value);
        }
    }

    /**
     * Add value to plot
     *
     * @param        $value
     * @param string $label
     * @param string $color
     * @param bool   $as_first - add as first value, not last
     *
     * @return \local_ned_controller\output\plotly\plotly_value
     */
    public function add_value($value, $label='', $color='', $as_first=false){
        if (!($value instanceof plotly_value)){
            $value = new plotly_value($value, $label, $color);
        }
        if ($as_first){
            $this->values = array_merge([$value], $this->values);
        } else {
            $this->values[] = $value;
        }
        return $value;
    }

    /**
     * Fill plot to show something of full value (for example one value from 100%)
     *
     * @param int  $sum_value
     * @param bool $color
     */
    public function fill_up_to_100_percent($sum_value=100, $color=false){
        $sum = 0;
        foreach ($this->values as $key => $value){
            $sum += $value->value;
        }

        if ($sum >= $sum_value){
            return;
        }

        $v = $this->add_value($sum_value - $sum);
        $v->set_invisible(true, $color);
    }

    /**
     * @param string $annotation
     * @param float  $hole
     * @param null   $size
     */
    public function set_donut($annotation='', $hole=0.4, $size=null){
        $this->type = 'pie';
        $this->annotation = $annotation;
        $this->hole = $hole;
        $this->rotation = 180;
        if (!is_null($size)){
            $this->height = $this->width = $size;
        }
        $this->square = true;
    }

    /**
     * @param string $annotation
     * @param int    $sum_value
     * @param float  $hole
     * @param null   $size
     * @param string $color
     */
    public function set_donut_piece($annotation='', $sum_value=100, $hole=0.4, $size=null, $color='#c3c3c3'){
        $this->set_donut($annotation, $hole, $size);
        $this->hoverinfo = 'skip';
        $this->textinfo = 'text';
        $this->fill_up_to_100_percent($sum_value, $color);
    }

    /**
     * Set type "bar" and some default values
     *
     * @param bool   $horizontally
     * @param int    $max
     * @param int    $data_width
     * @param string $bgcolor
     */
    public function set_bar($horizontally=true, $max=100, $data_width=1, $bgcolor='#c3c3c3'){
        $this->type = 'bar';
        $this->hoverinfo = 'skip';
        $val = $horizontally ? 'x' : 'y';
        $this->hovertemplate = "%{{$val}}";
        $this->staticPlot = false;
        $this->orientation = $horizontally ? 'h' : 'v';
        $axis1 = ['visible' => false, 'range' => [0, $max]];
        $axis2 = ['visible' => false, 'linewidth' => 0];
        list($this->layout_options->xaxis, $this->layout_options->yaxis) = $horizontally ? [$axis1, $axis2] : [$axis2, $axis1];
        $this->data_width = $data_width;
        $this->plot_bgcolor = $bgcolor;
        $this->get_min = false;
    }

    /**
     * Save class object properties from $keys to $object
     *
     * @param       $object
     * @param array $keys
     *
     * @return mixed
     */
    public function copy_properties(&$object, $keys=[]){
        foreach ($keys as $key){
            if (!is_null($this->$key ?? null)){
                $object->$key = $this->$key;
            }
        }

        return $object;
    }

    /**
     * @return object
     */
    public function get_data(){
        if ($this->hovertemplate){
            $this->hovertemplate .= "<extra>{$this->hovertemplate2}</extra>";
        }

        $data = clone($this->data_options);
        $data->values = $data->values ?? [];
        $data->marker = $data->marker ?? [];
        $data->labels = $data->labels ?? [];
        $data->marker['colors'] = $data->marker['colors'] ?? [];

        $this->copy_properties($data, static::SIMPLE_DATA_KEYS);

        foreach ($this->values as $value){
            $data->values[] = $value->value;
            $data->labels[] = $value->label;
            $data->marker['colors'][] = $value->color;
        }

        if ($this->type == 'bar'){
            list($data->x, $data->y) = $this->orientation == 'h' ? [$data->values, $data->labels] : [$data->labels, $data->values];
            $data->x = array_reverse($data->x);
            $data->y = array_reverse($data->y);
            $data->marker['color'] = $data->marker['colors'];
        }

        $data->width = $this->data_width;
        if (!is_null($this->hover_size)){
            $data->hoverlabel = $data->hoverlabel ?? [];
            $data->hoverlabel['font'] = $data->hoverlabel['font'] ?? [];
            $data->hoverlabel['font']['size'] = $this->hover_size;
        }

        return $data;
    }

    /**
     * @return object
     */
    public function get_layout(){
        $layout = clone($this->layout_options);

        $this->copy_properties($layout, static::SIMPLE_LAYOUT_KEYS);

        if (!is_null($this->annotation)){
            $layout->annotations = $layout->annotations ?? [];
            $annotation = $layout->annotation ?? [];
            $annotation['text'] = $this->annotation;
            $annotation['showarrow'] = $this->showarrow;
            $annotation['x'] = $this->annotation_x;
            $annotation['y'] = $this->annotation_y;
            $annotation['font'] = $layout->annotation['font'] ?? [];
            $annotation['font']['size'] = $this->annotation_size;
            $annotation['font']['color'] = $this->annotation_color;

            $layout->annotations[] = $annotation;
        }

        $layout->margin = [
            'l' => 0,
            'r' => 0,
            't' => 0,
            'b' => 0,
        ];

        $layout->font = [];
        if (!is_null($this->textfont_color)){
            $layout->font['color'] = $this->textfont_color;
        }
        if (!is_null($this->textfont_size)){
            $layout->font['size'] = $this->textfont_size;
        }

        return $layout;
    }

    /**
     * @return object
     */
    public function get_config(){
        $config = clone($this->config_options);
        $this->copy_properties($config, static::SIMPLE_CONFIG_KEYS);

        return $config;
    }

    /**
     * Generate data for js lib from class object.
     *
     * @param bool $call_js_lib
     *
     * @return array
     */
    public function render($call_js_lib=true){
        $data = $this->get_data();
        $layout = $this->get_layout();
        $config = $this->get_config();

        $elem_data = [$this->selector, [$data], $layout, $config];
        $this->_rendered = true;

        if ($call_js_lib){
            NED::js_call_amd(static::AMD_SCRIPT, 'add', [$elem_data]);
        }

        return $elem_data;
    }

    /**
     * Generate data for js lib from all PLOTLY class object.
     *      Not recommended to call js lib here, if you have many objects or a lot of data in it
     *
     * @param bool $skip_rendered
     * @param bool $call_js_lib
     * @param bool $encode
     *
     * @return array|false|string
     */
    static public function render_all($skip_rendered=true, $call_js_lib=true, $encode=false){
        $elems_data = [];

        foreach (static::$_plots as $plot){
            if ($skip_rendered && $plot->rendered){
                continue;
            }
            $elems_data[] = $plot->render(false);
        }

        if ($call_js_lib){
            NED::js_call_amd(static::AMD_SCRIPT, 'add', $elems_data);
        }

        return $encode ? json_encode($elems_data) : $elems_data;
    }
}
