<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Сlass is used to create and check accessibility rules
 */

namespace local_ned_controller;

defined('MOODLE_INTERNAL') || die();


class rule  {


    protected $rule;

    /**
     * Constructs with item details.
     *
     * @param \section_info $section Section object
     */
    public function __construct($rule = null) {
        if ($rule){
            $this->rule = $rule;
        }
    }

    /**
     * Set rule
     *
     * @param $rule
     */
    public function set_rule($rule){
        $this->rule = $rule;
    }

    /**
     * Check availability rule
     *
     * @return bool
     */
    public function is_available() {

        $availabilitycon = $this->parse_availability();

        if (count($availabilitycon->c) == 0){
            return $this->rule->id;
        }

        $any = str_contains($availabilitycon->op, '|');
        foreach ($availabilitycon->c as $availability){
            $currentvalue = $this->get_field_value_by_type_and_name($availability);
            $need_eq = $availability->op == 'isequalto';
            $eq = stripos($currentvalue, $availability->value) !== false;
            if ($need_eq == $eq){
                if ($any){
                    return $this->rule->id;
                }
            } else {
                if (!$any){
                    return false;
                }
            }
        }

        return $any ? false : $this->rule->id;
    }

    /**
     * Get current field value
     *
     * @param $availability
     * @return null
     */
    private function get_field_value_by_type_and_name($availability){
        switch ($availability->type){
            case 'userprofile':
                global $USER;
                if (property_exists($USER,$availability->sf)){
                    return $USER->{$availability->sf};
                }
                $profile = $USER->profile ?? [];
                if (array_key_exists($availability->sf, $profile)){
                    return $profile[$availability->sf];
                }
                break;
            case 'category':
                global $COURSE;
                if (property_exists($COURSE,'category')){
                    return $COURSE->category;
                }
                break;
            default:
                return null;
        }
        return null;
    }

    /**
     * convert to associative array for js
     *
     * @param array $inarray
     * @param $keyname
     * @param $valuename
     * @return array
     */
    public function convert_associative_array_for_js(array $inarray,
            $keyname, $valuename) {
        $result = array();
        foreach ($inarray as $key => $value) {
            $result[] = (object)array($keyname => $key, $valuename => $value);
        }
        return $result;
    }

    /**
     * Get a list of user fields
     *
     * @return array
     */
    public function get_user_field(){

        $standardfields = array(
                'firstname' => \core_user\fields::get_display_name('firstname'),
                'lastname' => \core_user\fields::get_display_name('lastname'),
                'email' => \core_user\fields::get_display_name('email'),
                'city' => \core_user\fields::get_display_name('city'),
                'country' => \core_user\fields::get_display_name('country'),
                'url' => \core_user\fields::get_display_name('url'),
                'skype' => \core_user\fields::get_display_name('skypeid'),
                'idnumber' => \core_user\fields::get_display_name('idnumber'),
                'institution' => \core_user\fields::get_display_name('institution'),
                'department' => \core_user\fields::get_display_name('department'),
                'phone1' => \core_user\fields::get_display_name('phone1'),
                'phone2' => \core_user\fields::get_display_name('phone2'),
                'address' => \core_user\fields::get_display_name('address')
        );
        \core_collator::asort($standardfields);

        // Custom fields.
        $customfields = array();
        foreach ($this->get_custom_profile_fields() as $field) {
            $customfields[$field->shortname] = format_string($field->name, true);
        }
        \core_collator::asort($customfields);

        return $this->convert_associative_array_for_js($standardfields + $customfields,'field', 'display');
    }

    /**
     * Get a list of custom fields
     *
     * @return array
     */
    public function get_custom_profile_fields() {
        global $CFG;
        // Get fields and store them indexed by shortname.
        require_once($CFG->dirroot . '/user/profile/lib.php');
        $fields = profile_get_custom_fields(true);
        $customprofilefields = array();
        foreach ($fields as $field) {
            $customprofilefields[$field->shortname] = $field;
        }

        return $customprofilefields;
    }

    /**
     * Json parsing
     *
     * @return mixed
     */
    public function parse_availability(){
		if (empty($this->rule->availabilityconditionsjson)){
			return json_decode('{"op":"&amp;","c":[]}');
		}
        return json_decode($this->rule->availabilityconditionsjson);
    }
}
