<?php
/**
 * @package    local_ned_controller
 * @subpackage shared
 * @category   NED
 * @copyright  2021 NED {@link http://ned.ca}
 * @author     NED {@link http://ned.ca}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 * @noinspection DuplicatedCode
 */

namespace local_ned_controller\shared;

defined('MOODLE_INTERNAL') || die();
/** @var \stdClass $CFG */
require_once($CFG->dirroot . '/local/ned_controller/lib.php');
require_once($CFG->dirroot . '/user/profile/lib.php');
require_once($CFG->dirroot . '/cohort/lib.php');

/**
 * Trait data_util
 *
 * @package local_ned_controller\shared
 * @mixin cm_util
 */
trait data_util {
    use global_util, util, db_util, pd_util;

    /**
     * @return object
     */
    static public function session(){
        global $SESSION;

        // Initialise $SESSION if necessary.
        if (!is_object($SESSION)) {
            $SESSION = new \stdClass();
        }

        return $SESSION;
    }

    //region Cache
    /**
     * Return "shared_cache" application cache
     *
     * @return \cache_application
     */
    static public function cache_get_shared_cache(){
        $res = static::g_get(__FUNCTION__);
        if (is_null($res)){
            $res = \cache::make(C::CTRL, C::CACHE_SHARED);
            static::g_set(__FUNCTION__, [], $res);
        }

        return $res;
    }

    /**
     * Return "users" local (session) cache
     *
     * @return \cache_session
     */
    static public function get_user_cache(){
        $res = static::g_get(__FUNCTION__);
        if (is_null($res)){
            $res = \cache::make(C::CTRL, C::CACHE_USERS);
            static::g_set(__FUNCTION__, [], $res);
        }

        return $res;
    }

    /**
     * Return groups application cache from the TT plugin
     *
     * @return \cache_session|null
     */
    static public function get_tt_group_cache(){
        $res = static::g_get(__FUNCTION__);
        if (is_null($res)){
            if (!static::is_tt_exists()){
                return null;
            }

            $res = \cache::make(C::TT, 'block_group');
            static::g_set(__FUNCTION__, [], $res);
        }

        return $res;
    }
    //endregion

    // OBJECTS

    /**
     * Return saved groupid from block settings, or from the global SESSION
     *
     * @param numeric $courseid
     *
     * @return int|null - return null, if it finds nothing
     */
    static public function get_cached_groupid($courseid){
        $cache = static::get_tt_group_cache();
        if ($cache){
            $res = $cache->get($courseid);
            if ($res === false) return null;
            else return $res;
        } else {
            return static::session()->currentgroup[$courseid] ?? null;
        }
    }

    /**
     * Return user object by object, id, or return global $USER
     * NOTICE: it tries to return global $USER if id/object is incorrect, not null
     *
     * @param object|numeric|null $user_or_id
     * @param bool                      $check_global_user
     *
     * @return \stdClass|null $user
     */
    static public function get_chosen_user($user_or_id=null, $check_global_user=true){
        global $USER;

        $userid = $user_or_id->id ?? ($user_or_id ?? 0);
        $res = static::g_get(__FUNCTION__, $userid);
        if (!is_null($res)){
            return $res ?: null;
        }

        $user = null;
        if ($user_or_id){
            if (!is_object($user_or_id)){
                $user = static::get_user($user_or_id);
            } else {
                $user = $user_or_id;
            }
        }

        if ($check_global_user){
            $user = static::choose_obj_with_id($user, $USER);
        }

        if ($user->id ?? false){
            static::g_set(__FUNCTION__, $user->id, $user);
        }
        static::g_set(__FUNCTION__, $userid, $user);

        return $user ?: null;
    }

    /**
     * Return received userid or global USER id
     *
     * @param object|numeric $user_or_id
     *
     * @return int
     */
    static public function get_userid_or_global($user_or_id=0){
        global $USER;

        $userid = static::get_id($user_or_id);

        return $userid ?: $USER->id;
    }

    /**
     * Return received courseid or global COURSE id
     *
     * @param object|numeric $course_or_id
     * @params numeric|mixed $return_if_siteid - value to return, if the course id - is site id; by default - SITEID
     *
     * @return int
     */
    static public function get_courseid_or_global($course_or_id=null, $return_if_siteid=\SITEID){
        global $COURSE;

        $courseid = static::get_id($course_or_id);
        $courseid = $courseid ?: $COURSE->id;

        if ($courseid == \SITEID){
            return $return_if_siteid;
        }
        return $courseid;
    }

    /**
     * Return course object by object, id, or return global $COURSE
     * NOTICE: it tries to return global $COURSE if id/object is incorrect, not null
     *
     * @param object|numeric|null $course_or_id
     * @param bool                      $check_global_course
     *
     * @return \stdClass|null
     */
    static public function get_chosen_course($course_or_id=null, $check_global_course=true){
        global $COURSE;

        $courseid = $course_or_id->id ?? ($course_or_id ?? 0);
        $res = static::g_get(__FUNCTION__, $courseid);
        if (!is_null($res)){
            return $res ?: null;
        }

        $course = null;
        if ($course_or_id){
            if (!is_object($course_or_id)){
                $course = static::get_course($course_or_id);
            } else {
                $course = $course_or_id;
            }
        }

        if ($check_global_course){
            $course = static::choose_obj_with_id($course, $COURSE);
        }

        if ($course->id ?? false){
            static::g_set(__FUNCTION__, $course->id, $course);
        }
        static::g_set(__FUNCTION__, $courseid, $course);

        return $course ?: null;
    }

    /**
     * If provided $groupid and user has such group - return group with such group id,
     * otherwise, it tries to get group id from the TT or global cache, and return such group, if found it,
     * otherwise, return just the first group from the list
     *
     * Note: this function for the global USER only
     *
     * @param numeric|object|null $course_or_id - course or its id, otherwise load global $COURSE
     * @param numeric|null $groupid - group id
     *
     * @return object|null - return null, if there are none groups, otherwise some group object
     */
    static public function get_chosen_group($course_or_id=null, $groupid=null){
        $courseid = static::get_courseid_or_global($course_or_id);
        $groups  = static::get_all_user_course_groups($courseid);
        if (empty($groups)){
            return null;
        }

        if (count($groups) == 1){
            return reset($groups);
        }

        if (!empty($groups[$groupid])){
            return $groups[$groupid];
        }

        $cached_groupid = static::get_cached_groupid($courseid);
        if (!empty($groups[$cached_groupid])){
            return $groups[$cached_groupid];
        }

        return reset($groups);
    }

    /**
     * Check, can user view course info in any way
     *
     * @param numeric|object|null $course_or_id - uses global COURSE by default
     * @param numeric|object|null $user_or_id - uses global USER by default
     *
     * @return bool
     */
    static public function course_can_view_course_info($course_or_id=null, $user_or_id=null){
        if (empty($course_or_id)) return false;

        $courseid = static::get_courseid_or_global($course_or_id);
        if ($courseid == SITEID) return true;

        $userid = static::get_userid_or_global($user_or_id);
        $res = static::g_get(__FUNCTION__, [$courseid, $userid]);
        if (is_null($res)){
            $user = static::get_user($userid);
            $res = is_siteadmin($user) ||
                is_enrolled(static::ctx($courseid), $user, '', true) ||
                \core_course_category::can_view_course_info(static::get_course($courseid), $user);
            static::g_set(__FUNCTION__, [$courseid, $userid], $res);
        }

        return $res;
    }

    /**
     * Return current device type
     *
     * @return string
     */
    static public function get_devicetype(){
        global $CFG;
        $res = static::g_get(__FUNCTION__);
        if (is_null($res)){
            $enabledevicedetection_option = $CFG->enabledevicedetection ?? 0;
            $CFG->enabledevicedetection = 1;
            $cu = \core_useragent::instance(!$enabledevicedetection_option);
            $res = $cu::get_user_device_type();
            $CFG->enabledevicedetection = $enabledevicedetection_option;
            static::g_set(__FUNCTION__, [], $res);
        }

        return $res;
    }

    /**
     * Return ned_assign by course-module
     * Call moodle_exception if it can't create the object
     *
     * @param \cm_info|int|string $cm_or_id
     * @param object|numeric      $courseorid - Optional course object (or its id) if loaded, improves optimization if $cm_or_id is represented as ID
     *
     * @return \local_ned_controller\mod_assign\assign
     * @throws \moodle_exception
     */
    static public function ned_assign_by_cm($cm_or_id, $courseorid=null){
        return static::$ned_assign::get_assign_by_cm($cm_or_id, $courseorid);
    }

    /**
     * Returns current string_manager instance.
     *
     * @return \core_string_manager
     */
    static public function get_string_manager(){
        $res = static::g_get(__FUNCTION__);
        if (is_null($res)){
            $res = get_string_manager();
            static::g_set(__FUNCTION__, [], $res);
        }

        return $res;
    }

    /**
     * Gets a course object from database. If the course id corresponds to an
     * already-loaded $COURSE or $SITE object, then the loaded object will be used,
     * saving a database query.
     *
     * @param int|string $courseid Course id
     *
     * @return null|\stdClass A course object
     */
    static public function get_course($courseid) {
        if (!$courseid){
            return null;
        }
        $courseid = (int)$courseid;
        $res = static::g_get(__FUNCTION__, $courseid);
        if (is_null($res)){
            if (static::g_get(__FUNCTION__, [0, 0])){
                // we have load all courses already
                return null;
            }

            try{
                $res = get_course($courseid);
            } catch (\Throwable){
                $res = null;
            }
            static::g_set(__FUNCTION__, $courseid, $res);
        }

        return $res ?: null;
    }

    /**
     * Return all site courses
     *  (and save it in the cache)
     *
     * @param $skip_site - if true, there will be not result by SITEID key
     *
     * @return object[]
     */
    static public function get_all_courses($skip_site=true){
        global $DB;
        if (!static::g_get('get_course', [0, 0])){
            $courses = $DB->get_records('course');
            $courses[0] = true;
            static::g_set('get_course', [], $courses);
        } else {
            $courses = static::g_get('get_course');
        }

        unset($courses[0]);
        if ($skip_site){
            unset($courses[SITEID]);
        }

        return $courses;
    }

    /**
     * This function gets the list of courses that this user has a particular capability in.
     *
     * It is now reasonably efficient, but bear in mind that if there are users who have the capability
     * everywhere, it may return an array of all courses.
     *
     * @param string    $capability Capability in question
     * @param int|null  $userid User ID or null for current user
     * @param string    $fieldsexceptid Leave blank if you only need 'id' in the course records;
     *   otherwise use a comma-separated list of the fields you require, not including id.
     *   Add ctxid, ctxpath, ctxdepth etc. to return course context information for preloading.
     * @param string    $orderby If set, use a comma-separated list of fields from course
     *   table with sql modifiers (DESC) if needed
     * @param bool      $all_for_admin - if True - don't check permissions for admin (default)
     *
     * @return object[] list of courses
     *          Note: it's simple array-list AND it hasn't course ids as array keys
     *          If $fieldsexceptid is empty, it will be only course id in course objects
     */
    static public function get_course_by_capability($capability, $fieldsexceptid='', $userid=null, $orderby='', $all_for_admin=true){
        return get_user_capability_course($capability, $userid, $all_for_admin, $fieldsexceptid, $orderby, 0) ?: [];
    }

    /**
     * Return all courses, for which user has access
     *  You can add $capabilities to check, but then we can't save result in the local storage
     *
     * @param int|string|null  $userid User ID or null for current user
     * @param array|string     $capabilities - additional capability(es) to check
     *
     * @return array|mixed|object[]
     */
    static public function get_course_with_access($userid=null, $capabilities=[]){
        $userid = static::get_userid_or_global($userid);
        if (is_siteadmin($userid)){
            return static::get_all_courses();
        }

        $courses = static::g_get(__FUNCTION__, [$userid]);
        $check_access = false;
        $capabilities = static::val2arr($capabilities);
        $can_save = empty($capabilities); // we can't save all possible capabilities

        if (is_null($courses)){
            $check_access = true;
            if (empty($capabilities)){
                $courses = static::get_all_courses();
            } else {
                $main_cap = array_shift($capabilities);
                $courses = static::get_course_by_capability($main_cap, '*', $userid);
            }
        } elseif ($can_save){
            return $courses;
        }

        $access_courses = [];
        foreach ($courses as $course){
            if ($check_access){
                if (!can_access_course($course, $userid, '', true)){
                    continue;
                }
            }

            if (!empty($capabilities)){
                $ctx = \context_course::instance($course->id);
                if (!has_all_capabilities($capabilities, $ctx, $userid)){
                    continue;
                }
            }

            $access_courses[$course->id] = $course;
        }

        if ($can_save){
            static::g_set(__FUNCTION__, [$userid], $access_courses);
        }

        return $access_courses;
    }

    /**
     * Return grader courses
     *
     * @param numeric|null $graderid  User ID or null for current user
     * @param numeric|null $studentid (optional) Student id, if this student should be in courses
     *
     * @return array|object[]
     */
    static public function get_grader_courses($graderid=null, $studentid=null){
        $graderid = static::get_userid_or_global($graderid);
        $studentid = $studentid ?: 0;
        if (!$studentid && is_siteadmin($graderid)){
            return static::get_all_courses();
        }

        $courses = static::g_get(__FUNCTION__, [$graderid, $studentid]);
        if (is_null($courses)){
            if ($studentid){
                $grader_courses = static::get_grader_courses($graderid, 0);
                $st_courses = static::get_course_with_access($studentid, C::CAPABILITY_STUDENT);
                $courses = array_intersect_key($grader_courses, $st_courses);
            } else {
                $courses = static::get_course_with_access($graderid, C::CAPABILITY_COURSE_GRADER);
            }
            static::g_set(__FUNCTION__, [$graderid, $studentid], $courses);
        }

        return $courses;
    }

    /**
     * Check userid for be grader on the course (by courseid)
     *
     * @param numeric      $courseid
     * @param numeric|null $graderid User ID or null for current user
     * @param numeric|null $studentid Student id, if he should be in course(s)
     *
     * @return bool
     */
    static public function check_grader_courseid($courseid, $graderid=null, $studentid=null){
        if (!$courseid) {
            return false;
        }

        $graderid = static::get_userid_or_global($graderid);
        $studentid = $studentid ?: 0;
        $checked = static::g_get(__FUNCTION__, [$courseid, $graderid, $studentid]);
        if (is_null($checked)){
            $checked = false;
            do {
                $course = static::get_course($courseid);
                if (!$course){
                    break;
                }

                if ($studentid){
                    if (!static::check_grader_courseid($courseid, $graderid, 0)){
                        break;
                    }

                    if (!can_access_course($course, $studentid, C::CAPABILITY_STUDENT, false)){
                        break;
                    }
                } else {
                    if (is_siteadmin($graderid)){
                        $checked = true;
                        break;
                    }

                    if (!can_access_course($course, $graderid, C::CAPABILITY_COURSE_GRADER, true)){
                        break;
                    }
                }

                $checked = true;
            } while(false);
            static::g_set(__FUNCTION__, [$courseid, $graderid, $studentid], $checked);
        }

        return $checked;
    }

    /**
     * Return config for block on course page by course id
     * Don't use it in a course loop!
     *
     * @param        $courseid
     *
     * @param string $plugin_name - short plugin name, without "block_"
     *
     * @return bool|\stdClass
     */
    static public function get_block_config($courseid, $plugin_name){
        global $DB;

        $block_config = static::g_get_clone(__FUNCTION__, [$courseid, $plugin_name]);
        if (is_null($block_config)){
            $context = \context_course::instance($courseid);
            $blockdata = $DB->get_record('block_instances', ['blockname' => $plugin_name, 'parentcontextid' => $context->id]);
            if (!$blockdata){
                $block_config = false;
            } else {
                $config = unserialize(base64_decode($blockdata->configdata));
                if (!$config){
                    $config = new \stdClass();
                }
                $config->block_id = $blockdata->id;
                $config->course_id = $courseid;
                $block_config = $config;
            }
            static::g_set(__FUNCTION__, [$courseid, $plugin_name], $block_config);
        }

        return $block_config;
    }

    /**
     * Save config of block for course page by course id
     * Don't use it in a course loop!
     *
     * @param           $courseid
     * @param string    $plugin_name - short plugin name, without "block_"
     * @param \stdClass $block_config
     *
     * @return bool
     */
    static public function save_block_config($courseid, $plugin_name, $block_config){
        global $DB;

        $configdata = base64_encode(serialize($block_config));
        $context = \context_course::instance($courseid);
        if ($DB->set_field('block_instances', 'configdata', $configdata,
            ['blockname' => $plugin_name, 'parentcontextid' => $context->id])){
            static::g_set('get_block_config', [$courseid, $plugin_name], $block_config);
            return true;
        }

        return false;
    }


    /**
     * Get block config for course, based on site block config and course block settings
     *
     * @param numeric|object $course_or_id
     * @param string         $plugin_name - short plugin name, without "block_"
     * @param string|null    $option      - if option is set, return its value or null
     *
     * @return object|mixed|null - if not $option, return full config object
     */
    static public function get_site_and_course_block_config($course_or_id, $plugin_name, $option=null){
        $courseid = static::get_courseid_or_global($course_or_id);
        $config_data = static::g_get_clone(__FUNCTION__, [$courseid, $plugin_name]);
        if (is_null($config_data)){
            $site_config = get_config('block_' . $plugin_name);
            $block_config = static::get_block_config($courseid, $plugin_name);
            $block_config = $block_config ?: [];
            $config_data = (object)array_merge((array)$site_config, (array)$block_config);
            static::g_set(__FUNCTION__, [$courseid, $plugin_name], $config_data);
        }

        return $option ? ($config_data->$option ?? null) : $config_data;
    }

    // USERS

    /**
     * Return user object from db or create noreply or support user,
     * if userid matches core_user::NOREPLY_USER or core_user::SUPPORT_USER
     * respectively. If userid is not found, then return null.
     *
     * @param int|string $userid
     *
     * @return \stdClass|null
     */
    static public function get_user($userid){
        if (!$userid){
            return null;
        }

        $userid = (int)$userid;
        $res = static::g_get(__FUNCTION__, $userid);
        if (is_null($res)){
            global $USER;
            if ($userid == $USER->id){
                $res = clone($USER);
            } else {
                $res = \core_user::get_user($userid);
            }
            static::g_set(__FUNCTION__, $userid, $res);
        }

        return $res ?: null;
    }

    /**
     * Gets a group object from database.
     * If the group id corresponds to an already-loaded group object,
     *  then the loaded object will be used, saving a database query.
     *
     * @see groups_get_group
     *
     * @param int|string $groupid Group id
     * @param string     $field - get only field value
     *
     * @return null|\stdClass|mixed - group object or null if not found,
     *                                if field specified - return its value (or null)
     */
    static public function get_group($groupid, $field=null){
        if (!$groupid){
            return null;
        }

        $groupid = (int)$groupid;
        $res = static::g_get(__FUNCTION__, $groupid);
        if (is_null($res)){
            $res = groups_get_group($groupid);
            static::g_set(__FUNCTION__, $groupid, $res);
        }

        if ($field){
            return $res->$field ?? null;
        }
        return $res ?: null;
    }

    /**
     * Gets the name of a group with a specified id
     * Alias for @see \local_ned_controller\shared\data_util::get_group()
     *
     * @param int|string $groupid Group id
     *
     * @return string|null - The name of the group
     */
    static public function get_groupname($groupid){
        return static::get_group($groupid, 'name');
    }

    /**
     * Gets a group object from database.
     * If the group id corresponds to an already-loaded group object,
     *  then the loaded object will be used, saving a database query.
     *
     * @see groups_get_members
     *
     * @param int|string $groupid Group id
     * @param bool       $only_ids return only user id or full record
     *
     * @return array|object[]|int[] users by id or list of userids
     */
    static public function get_group_users($groupid, $only_ids=false){
        if (!$groupid){
            return null;
        }
        $groupid = (int)$groupid;
        $only_ids = (int)$only_ids;
        $res = static::g_get(__FUNCTION__, [$groupid, $only_ids]);
        if (is_null($res)){
            if ($only_ids){
                $users = static::g_get(__FUNCTION__, [$groupid, 0]);
                if (is_null($users)){
                    $users = groups_get_members($groupid, 'u.id');
                }
                $res = array_keys($users);
            } else {
                $res = groups_get_members($groupid, 'u.*');
            }

            static::g_set(__FUNCTION__, [$groupid, $only_ids], $res);
        }

        return $res ?: [];
    }

    /**
     * Returns info about user's groups in course (or all).
     * Can use cache created by load_all_user_groups
     *
     * @see static::load_all_user_groups
     * @see groups_get_user_groups
     *
     * @param int $courseid
     * @param int $userid $USER if not specified
     *
     * @return array Array[groupingid][groupid_1, groupid_2, ...] including grouping id 0 which means all groups if $courseid,
     *               else Array[courseid][groupid_1, groupid_2, ...] including course id 0 which means all groups
     */
    static public function get_user_groupings($courseid, $userid=0){
        $get_it = static::g_get('load_all_user_groups', $courseid) ||
            ($courseid && static::g_get('load_all_user_groups', 0));

        if (!$get_it){
            $usergroups = groups_get_user_groups($courseid, $userid);
            if ($courseid){
                return $usergroups;
            }
        }

        // else: we have checked all users, don't need DB query from the base function
        $cache = \cache::make('core', 'user_group_groupings');
        $usergroups = $cache->get($userid);
        if ($courseid || empty($usergroups)){
            return $usergroups[$courseid] ?? [0 => []];
        }

        $all_user_groups = [];
        foreach ($usergroups as $cid => $usergroup){
            $all_user_groups[$cid] = $usergroup[0] ?? [];
        }
        $all_user_groups[0] = array_merge(...$all_user_groups);

        return $all_user_groups;
    }

    /**
     * Returns user's group ids in course (or all).
     * Can use cache created by load_all_user_groups
     * alias for the get_user_groupings to get group ids without grouping ids
     *
     * @see get_user_groupings
     * @see static::load_all_user_groups
     * @see groups_get_user_groups
     *
     * @param int $courseid
     * @param int $userid $USER if not specified
     *
     * @return array [groupid_1, groupid_2, ...]
     */
    static public function get_user_groupids($courseid, $userid=0){
        $groupings = static::get_user_groupings($courseid, $userid);
        return $groupings[0] ?? [];
    }

    /**
     * Returns course's group ids.
     * alias for the get_all_course_groups to get only group ids
     * @see get_all_course_groups
     *
     * @param int $courseid
     *
     * @return array [groupid_1, groupid_2, ...]
     */
    static public function get_course_groupids($courseid){
        return array_keys(static::get_all_course_groups($courseid, 0, 'g.id') ?: []);
    }

    /**
     * Load all groups by courseid in the cache
     * BE CAREFUL with $courseid = 0, it will load ALL groups
     *
     * If you change this function, change also get_user_groups
     * @see static::get_user_groupids
     *
     * @param int $courseid
     */
    static public function load_all_user_groups($courseid=0){
        if (static::g_get(__FUNCTION__, $courseid) ||
            ($courseid && static::g_get(__FUNCTION__, 0))){
            return;
        }

        global $DB;
        $cache = \cache::make('core', 'user_group_groupings');

        $sql = "SELECT CONCAT(g.id, '_', gm.userid) AS id, 
                    g.id AS gropid, gm.userid, g.courseid, gg.groupingid
                  FROM {groups} g
                  JOIN {groups_members} gm ON gm.groupid = g.id
             LEFT JOIN {groupings_groups} gg ON gg.groupid = g.id
        ";
        $params = [];
        if ($courseid){
            $sql .= "\nWHERE g.courseid = :courseid";
            $params['courseid'] = $courseid;
        }

        $all_groups = [];
        $user_groups = [];

        $rs = $DB->get_recordset_sql($sql, $params);
        foreach ($rs as $group) {
            $all_groups[$group->userid][$group->courseid][$group->gropid] = $group->gropid;
            if (is_null($group->groupingid)) {
                continue;
            }
            $user_groups[$group->userid][$group->courseid][$group->groupingid][$group->gropid] = $group->gropid;
        }
        $rs->close();

        foreach ($all_groups as $userid => $allgroups){
            foreach (array_keys($allgroups) as $cid) {
                $user_groups[$userid][$cid]['0'] = array_keys($allgroups[$cid]); // All user groups in the course.
            }
            // Cache the data.
            $cache->set($userid, $user_groups[$userid]);
        }

        static::g_set(__FUNCTION__, $courseid, true);
    }

    /**
     * Gets array of all groups in a specified course (subject to the conditions imposed by the other arguments).
     * alias for the groups_get_all_groups
     * @see groups_get_all_groups
     *
     * @param int $courseid The id of the course.
     * @param int|int[] $userid optional user id or array of ids, returns only groups containing one or more of those users.
     * @param string $fields defaults to g.*. This allows you to vary which fields are returned.
     *      If $groupingid is specified, the groupings_groups table will be available with alias gg.
     *      If $userid is specified, the groups_members table will be available as gm.
     * @param bool $withmembers if true return an extra field members (int[]) which is the list of userids that
     *      are members of each group. For this to work, g.id (or g.*) must be included in $fields.
     *      In this case, the final results will always be an array indexed by group id.
     * @param int $groupingid optional returns only groups in the specified grouping.
     *
     * @return array returns an array of the group objects (unless you have done something very weird
     *      with the $fields option).
     */
    static public function get_all_course_groups($courseid, $userid=0, $fields='g.*', $withmembers=false, $groupingid=0){
        return groups_get_all_groups($courseid, $userid, $groupingid, $fields, $withmembers);
    }

    /**
     * Gets array of all groups in a specified course (subject to the conditions imposed by the other arguments).
     * If user has access to all groups on course - return all groups on course, otherwise only groups with user
     * @see get_all_course_groups
     *
     * @param int|object     $courseorid  The course object or its id
     * @param object|numeric $user_or_id  optional user id, who should have access to the groups (global $USER by default)
     * @param string         $fields      defaults to g.*. This allows you to vary which fields are returned.
     *      If $groupingid is specified, the groupings_groups table will be available with alias gg.
     *      If $userid is specified, the groups_members table will be available as gm.
     * @param bool           $withmembers if true return an extra field members (int[]) which is the list of userids that
     *      are members of each group. For this to work, g.id (or g.*) must be included in $fields.
     *      In this case, the final results will always be an array indexed by group id.
     * @param int            $groupingid  optional returns only groups in the specified grouping.
     *
     * @return array returns an array of the group objects (unless you have done something very weird
     *      with the $fields option).
     */
    static public function get_all_user_course_groups($courseorid, $user_or_id=0, $fields='g.*', $withmembers=false, $groupingid=0){
        $courseid = static::get_id($courseorid);
        $user_or_id = static::get_userid_or_global($user_or_id);
        if (!$courseid){
            return [];
        }

        $context = \context_course::instance($courseid);
        $aag = has_capability('moodle/site:accessallgroups', $context, $user_or_id);

        return static::get_all_course_groups($courseid, $aag ? 0 : $user_or_id, $fields, $withmembers, $groupingid);
    }

    /**
     * Get list with users by there id
     *
     * @param $ids_or_users - list of users, or there id
     *
     * @return array
     */
    static public function get_user_list($ids_or_users){
        if (empty($ids_or_users)){
            return [];
        }

        $ids_or_users = static::val2arr($ids_or_users);
        $k = key($ids_or_users);
        $v = reset($ids_or_users);
        $res = [];
        if ($k === 0){
            if (is_object($v)){
                foreach ($ids_or_users as $user){
                    $res[$user->id] = $user;
                }
            } else {
                foreach ($ids_or_users as $userid){
                    $res[$userid] = static::get_user($userid);
                }
            }
        } else {
            if (is_object($v)){
                return $ids_or_users;
            } else {
                foreach ($ids_or_users as $userid => $smth){
                    $res[$userid] = static::get_user($userid);
                }
            }
        }

        return $res;
    }

    /**
     * Check, does course categories (array of ids) has such course id
     * If none $courseid, return array of all course id from the current course categories
     *
     * @param array|string $course_cats - if string, id values should be delimiter by ','
     * @param numeric|null $courseid    - if provided, return true if $courseid exists in the course category, false otherwise
     * @param bool         $check_all   - if true, and 0 ("All") key presents in the $course_cats, returns true
     *
     * @return array|bool - return true, $check_all true and "All" is presents, or provided $courseid exists in the course category;
     *                    otherwise return array [course_id => course_id] from $course_cats
     */
    static public function course_cats_has_courseid($course_cats, $courseid=null, $check_all=true){
        global $DB;
        if (empty($course_cats) && !is_numeric($course_cats)) return false;

        if (!is_array($course_cats)){
            if (is_string($course_cats) && str_contains($course_cats, ',')){
                $course_cats = explode(',', $course_cats);
            } else {
                $course_cats = [$course_cats];
            }
        }

        if ($check_all){
            if (in_array(C::ALL, $course_cats)){
                return true;
            }
        }

        [$cc_sql, $params] = $DB->get_in_or_equal($course_cats, SQL_PARAMS_NAMED);
        $params['cat_contextlevel'] = CONTEXT_COURSECAT;
        $params['course_contextlevel'] = CONTEXT_COURSE;

        $sql = "SELECT DISTINCT c.id, c.id as courseid 
            FROM {course} c 
            JOIN {context} cat_ctx
                ON cat_ctx.instanceid $cc_sql
                AND cat_ctx.contextlevel = :cat_contextlevel
            JOIN {context} course_ctx
                ON course_ctx.instanceid = c.id 
                AND course_ctx.contextlevel = :course_contextlevel
                AND course_ctx.path LIKE CONCAT(cat_ctx.path, '/%')
        ";

        if ($courseid){
            $params['courseid'] = $courseid;
            $sql .= "\nWHERE c.id = :courseid";
            return $DB->record_exists_sql($sql, $params);
        } else {
            return $DB->get_records_sql_menu($sql, $params);
        }
    }

    /**
     * @param string    $rolename
     * @param int       $courseid
     * @param int       $cmid
     * @param int       $groupid
     * @param bool      $return_only_first
     * @param string    $fields
     * @param null|int  $check_userid
     *
     * @return array|\stdClass|null
     */
    static public function get_users_by_role($rolename, $courseid=0, $cmid=0, $groupid=0,
        $return_only_first=false, $fields='u.id', $check_userid=null){
        global $DB;
        $return_only_first = (int)$return_only_first;
        $check_userid = $check_userid ?: 0;
        $keys = [$rolename, $courseid, $cmid, $groupid, $return_only_first, $fields, $check_userid];
        $res = static::g_get(__FUNCTION__, $keys);
        if (is_null($res)){
            $where = [];
            $params = [];
            $join = '';
            $other = '';
            if ($check_userid){
                $where[] = 'u.id = :userid';
                $params['userid'] = $check_userid;
            }
            if ($return_only_first){
                $other = "\nLIMIT 1";
            }

            $sql = static::sql_user_enrolments("DISTINCT u.id, $fields", $where, $params, $rolename, $courseid, $groupid, $cmid,
                0,false, $other, $join);

            $users = $DB->get_records_sql($sql, $params) ?: [];
            if ($return_only_first){
                $res = !empty($users) ? reset($users) : null;
            } else {
                $res = $users;
            }

            static::g_set(__FUNCTION__, $keys, $res);
        }

        return $res;
    }

    /**
     * Return list of students ['user_id' => user_object] by $courseid & $groupid
     *
     * @param numeric|object|null $courseorid
     * @param numeric|null        $cmid             - if "true" numeric, user should have access to this activity as student
     * @param numeric|null        $groupid          - if "true" numeric, load students only from this group, otherwise load all from all groups
     * @param bool                $show_only_active - if true, loads active students only, otherwise loads all (active and inactive)
     * @param bool|null           $load_cm_students - load students from course or cm:
     *                                              • false - load only course students, ignore $cmid option;
     *                                                  if $courseorid are not provided, loads all course students from the site
     *                                              • true - load only cm students, but if both $cmid & $courseorid provided,
     *                                                  also load course students, who have access to this $cmid
     *                                              • null - load both course and cm students
     *
     * @return array|object[]
     */
    static public function get_course_students_by_role($courseorid=null, $cmid=0, $groupid=0, $show_only_active=null, $load_cm_students=null){
        global $DB;
        $load_cm_students_key = is_null($load_cm_students) ? 0 : ($load_cm_students ? 1 : 2);
        $courseid = static::get_id($courseorid);
        $keys = [$courseid, $cmid, $groupid, (int)$show_only_active, $load_cm_students_key];
        $res = static::g_get(__FUNCTION__, $keys);

        if (is_null($res)){
            $params = [];
            $where = [];
            if (!is_null($load_cm_students)){
                if ($load_cm_students){
                    $cmid = $cmid ?: true;
                } else {
                    $cmid = null;
                }
            } else {
                $cmid = $cmid ?: false;
            }

            $sql = static::sql_user_enrolments(
                "DISTINCT u.id, u.*, gr.id AS groupid",
                $where, $params, C::ROLE_STUDENT,
                $courseid,
                $groupid, $cmid, 0, $show_only_active, 'GROUP BY u.id ORDER BY u.firstname, u.lastname', 'group');
            $res = $DB->get_records_sql($sql, $params);
            static::g_set(__FUNCTION__, $keys, $res);
        }

        return $res ?: [];
    }

    /**
     * Checked, that user enrolled on course/cm with different options
     *
     * @param numeric            $userid
     * @param string|array       $rolenames - shortname(s) from the role table
     * @param numeric|false|null $courseid  - course id, null to check nothing, false to check all, or id
     * @param numeric|array      $groupids  - filter by the group
     * @param numeric|bool|null  $cmid      - course module id, null to check nothing, false to check all, true - check all by course, or id
     * @param numeric|array      $schoolids - filter by the school
     * @param bool|int           $is_active - if false - get all, if true, return only active users, if === -1 - return only inactive
     *
     * @return bool
     */
    static public function is_enrolled_user($userid, $rolenames, $courseid=null, $groupids=[], $cmid=null, $schoolids=[], $is_active=true){
        static::sql_add_equal('ue.userid', $userid, $where, $params);
        $sql = static::sql_user_enrolments('ue.id', $where, $params, $rolenames, $courseid, $groupids, $cmid, $schoolids, $is_active);
        return static::db()->record_exists_sql($sql, $params);
    }

    /**
     * Checked, that user enrolled on course/cm with some role with different filters
     *
     * @param string|array                      $rolenames    - shortname(s) from the role table
     * @param numeric|object                    $user_or_id   - user or its id to check, uses global $USER by default
     * @param numeric|false|object|null         $course_or_id - course or id; null to check nothing, false (or SITEID) to check all, or id
     * @param numeric|array                     $groupids     - filter by the group
     * @param numeric|bool|object|\cm_info|null $cm_or_id     - cm id, null to check nothing, false to check all, true - check all by course, or id
     * @param numeric|array                     $schoolids    - filter by the school
     * @param bool|int                          $is_active    - if false - get all, true - return only active users, if === -1 -> return only inactive
     *
     * @return bool
     */
    static public function user_has_role($rolenames=[], $user_or_id=null, $course_or_id=null, $groupids=[],
        $cm_or_id=null, $schoolids=[], $is_active=true){
        if (empty($rolenames)) return false;

        $userid = static::get_userid_or_global($user_or_id);
        $courseid = $cmid = null;
        if (!is_null($course_or_id)){
            $courseid = static::get_id($course_or_id);
            if ($courseid == SITEID){
                $courseid = false;
            }
        }
        if (!is_null($cm_or_id)){
            $cmid = static::get_id($cm_or_id);
        }

        return static::is_enrolled_user($userid, $rolenames, $courseid, $groupids, $cmid, $schoolids, $is_active);
    }

    /**
     * Checked, that user enrolled on course/cm as student with different filters
     * Alias {@see static::user_has_role()}
     *
     * @param numeric|object                    $user_or_id   - user or its id to check
     * @param numeric|false|object|null         $course_or_id - course or id; null to check nothing, false (or SITEID) to check all, or id
     * @param numeric|array                     $groupids     - filter by the group
     * @param numeric|bool|object|\cm_info|null $cm_or_id     - cm id, null to check nothing, false to check all, true - check all by course, or id
     * @param numeric|array                     $schoolids    - filter by the school
     * @param bool|int                          $is_active    - if false - get all, true - return only active users, if === -1 -> return only inactive
     *
     * @return bool
     */
    static public function user_is_student($user_or_id, $course_or_id=null, $groupids=[], $cm_or_id=null, $schoolids=[],  $is_active=true){
        return static::user_has_role(static::ROLE_STUDENT, $user_or_id, $course_or_id, $groupids, $cm_or_id, $schoolids,  $is_active);
    }

    /**
     * Checked, that user enrolled on course/cm as grader with different filters
     * Alias {@see static::user_has_role()}
     *
     * @param numeric|object                    $user_or_id   - user or its id to check
     * @param numeric|false|object|null         $course_or_id - course or id; null to check nothing, false (or SITEID) to check all, or id
     * @param numeric|array                     $groupids     - filter by the group
     * @param numeric|bool|object|\cm_info|null $cm_or_id     - cm id, null to check nothing, false to check all, true - check all by course, or id
     * @param numeric|array                     $schoolids    - filter by the school
     * @param bool|int                          $is_active    - if false - get all, true - return only active users, if === -1 -> return only inactive
     *
     * @return bool
     */
    static public function user_is_grader($user_or_id, $course_or_id=null, $groupids=[], $cm_or_id=null, $schoolids=[],  $is_active=true){
        return static::user_has_role(static::ROLE_OT, $user_or_id, $course_or_id, $groupids, $cm_or_id, $schoolids,  $is_active);
    }

    /**
     * Return grader id by group id, course id and course-module id.
     * Return only one grader!
     * Return zero if it finds nothing.
     *
     * @param int $groupid
     * @param int $courseid
     * @param int $cmid
     *
     * @return int
     */
    static public function get_graderid_by_groupid($groupid=0, $courseid=0, $cmid=0){
        global $DB;
        $keys = [$groupid, $courseid, $cmid];
        $res = static::g_get(__FUNCTION__, $keys);
        if (is_null($res)){
            $w = ["ot.groupid = :groupid"];
            $params = ['groupid' => $groupid];

            if ($courseid){
                $w[] = 'ot.courseid = :courseid';
                $params['courseid'] = $courseid;
            }
            if ($cmid){
                $w[] = '(ot.cmid = :cmid OR ot.cmid IS NULL)';
                $params['cmid'] = $cmid;
            }

            $where = static::sql_where($w);
            $ot_sql = static::sql_user_enrolments("ue.userid, e.courseid, COALESCE(gr.id, 0) AS groupid, ra_cm.id AS cmid", [], $params,
                static::ROLE_OT, $courseid, $groupid, $cmid, 0,true,
                'GROUP BY e.courseid, groupid, ra_cm.id', 'group');

            $sql = "SELECT COALESCE(ot.userid, 0) AS graderid
                FROM ($ot_sql) ot
                $where    
                LIMIT 1
            ";
            $record = $DB->get_record_sql($sql, $params);

            $res = $record->graderid ?? 0;
            static::g_set(__FUNCTION__, $keys, $res);
        }

        return $res;
    }

    /**
     * Return grader id by student id, course id and course-module id.
     * Return only one grader!
     * Return zero if it finds nothing.
     *
     * @param int    $studentid
     * @param int    $courseid
     * @param int    $cmid
     * @param string $grader_role
     *
     * @return int
     */
    static public function get_graderid_by_studentid($studentid, $courseid, $cmid=0, $grader_role=C::ROLE_OT){
        global $DB;
        $keys = [$studentid, $courseid, $cmid, $grader_role];
        $res = static::g_get(__FUNCTION__, $keys);
        if (is_null($res)){
            $w = [];
            $params = [];
            $w[] = 'u.id = :userid';
            $params['userid'] = $studentid;
            $params['courseid'] = $courseid;

            if ($cmid){
                $w[] = '(ot.cmid = :cmid OR ot.cmid IS NULL)';
                $params['cmid'] = $cmid;
            }
            $where = static::sql_where($w);
            $ot_sql = static::sql_user_enrolments("ue.userid, e.courseid, COALESCE(gr.id, 0) AS groupid, ra_cm.id AS cmid", [], $params,
                $grader_role, $courseid, 0, $cmid, 0,true,
                'GROUP BY e.courseid, groupid, ue.userid', 'group', 'ot_');

            $sql = "SELECT COALESCE(ot.userid, 0) AS graderid
                    FROM {user} u 
                    LEFT JOIN (
                        SELECT grp.id, g_m.userid, grp.courseid, grp.name
                        FROM {groups} AS grp
                        JOIN {groups_members} AS g_m
                           ON g_m.groupid = grp.id
                        GROUP BY grp.courseid, g_m.userid
                    ) gr 
                        ON gr.courseid = :courseid
                        AND gr.userid = u.id
                    JOIN ($ot_sql) ot
                        ON ot.courseid = gr.courseid
                        AND ot.groupid = gr.id
                    $where
                    LIMIT 1
            ";

            $record = $DB->get_record_sql($sql, $params);

            $res = $record->graderid ?? 0;
            static::g_set(__FUNCTION__, $keys, $res);
        }

        return $res;
    }

    /**
     * Return array of course ids where specific user is grader (by role) for the specific student
     * Note: it checks only active enrollment for the grader, but any enrollment for the student
     *
     * @param numeric|object $grader_or_id - if null, uses global $USER
     * @param numeric|object $student_or_id
     * @param string         $grader_role - role to specify, who is grader
     *
     * @return array [courseid => courseid]
     */
    static public function get_courses_for_grader_and_student($grader_or_id=null, $student_or_id=null, $grader_role=C::ROLE_OT){
        $graderid = static::get_userid_or_global($grader_or_id);
        $studentid = static::get_id($student_or_id);
        if (empty($graderid) || empty($studentid) || empty($grader_role)) return [];

        $keys = [$graderid, $studentid, $grader_role];
        $res = static::g_get(__FUNCTION__, $keys);
        if (is_null($res)){
            $w = [];
            $params = [];
            $w[] = 'u.id = :studentid';
            $w[] = 'ot.userid = :graderid';
            $params['studentid'] = $studentid;
            $params['graderid'] = $graderid;

            $where = static::sql_where($w);
            $ot_sql = static::sql_user_enrolments("ue.userid, e.courseid, COALESCE(gr.id, 0) AS groupid", [], $params,
                $grader_role, false, 0, null, 0,true,
                'GROUP BY e.courseid, groupid', 'group', 'ot_');

            $sql = "SELECT gr.courseid AS courseid, gr.courseid AS courseid2
                    FROM {user} u 
                    JOIN (
                        SELECT grp.id, g_m.userid, grp.courseid, grp.name
                        FROM {groups} AS grp
                        JOIN {groups_members} AS g_m
                           ON g_m.groupid = grp.id
                        GROUP BY grp.courseid, g_m.userid
                    ) gr 
                        ON gr.userid = u.id
                    JOIN ($ot_sql) ot
                        ON ot.courseid = gr.courseid
                        AND ot.groupid = gr.id
                    $where
                    GROUP BY gr.courseid
            ";

            $res = static::db()->get_records_sql_menu($sql, $params) ?: [];

            static::g_set(__FUNCTION__, $keys, $res);
        }

        return $res;
    }

    /**
     * Get & cache users by rolename
     *  return array, if there are no of any parameters, object if it gets all parameters or false, if found nothing by them
     *
     * If there is no $courseid, result will be dictionary [courseid][groupid] => [userids => userids]
     * If there is $courseid and no $groupid, result will be dictionary [groupid] => [userids => userids]
     * If there is $courseid and $groupid, result will be int array - [userids => userids]
     * But, if $return_only_userids is true, then result will be simple list of userids.
     *
     * @param string            $rolename            - short rolename
     * @param numeric|bool|null $courseid            - return array[courseid][groupid] when false
     * @param numeric|bool|null $groupid             - return array[groupid] when false
     * @param numeric|bool|null $cmid                - course module id, null to check nothing, false to check all, true - check all by course, or id
     * @param bool              $active              - if false - get all, if true, return only active users, if === -1 - return only inactive
     * @param array|int         $userids             - check only by this user ids, if set; function doesn't use cache with this parameter
     * @param bool              $return_only_userids - if true, then result will be always array of userids (even for single int result)
     *
     * @return array - return [userids => userids], return empty array if it finds nothing
     */
    static public function get_role_users_ids($rolename='', $courseid=null, $groupid=null, $cmid=null, $active=true, $userids=null,
            $return_only_userids=false){
        global $DB;
        if (empty($rolename)){
            return [];
        }

        $key_rolename = $rolename.'_'.($active ? '1' : '0');
        $res = null;
        $can_use_cache = empty($userids);

        if ($can_use_cache){
            if (!$courseid){
                if (static::g_get(__FUNCTION__, [$key_rolename, -1])){
                    $res = static::g_get(__FUNCTION__, [$key_rolename], []);
                    unset($res[-1]);
                }
            } elseif (!$groupid){
                if (static::g_get(__FUNCTION__, [$key_rolename, -1]) || static::g_get(__FUNCTION__, [$rolename, $courseid, -1])){
                    $res = static::g_get(__FUNCTION__, [$key_rolename, $courseid], []);
                    unset($res[-1]);
                }
            } else {
                $res = static::g_get(__FUNCTION__, [$key_rolename, $courseid, $groupid]);
            }
        }

        if (is_null($res)){
            $where = [];
            $params = [];

            if (!empty($userids)){
                [$sql_where, $params_where] = $DB->get_in_or_equal($userids, SQL_PARAMS_NAMED, 'userids_');
                $where[] = 'ue.userid '.$sql_where;
                $params = array_merge($params, $params_where);
            }

            $sql = static::sql_user_enrolments("CONCAT(e.courseid, '_', COALESCE(gr.id, 0), '_', ue.userid) as uniqueid, 
                e.courseid, COALESCE(gr.id, 0) AS groupid, ue.userid", $where, $params,
                $rolename, $courseid ?: false, $groupid, $cmid, 0, $active,  'GROUP BY e.courseid, groupid, ue.userid', 'group');
            $records = $DB->get_records_sql($sql, $params);

            do {
                $res = [];
                if (!$courseid){
                    foreach ($records as $record){
                        $res[$record->courseid][$record->groupid][$record->userid] = $record->userid;
                    }
                    if (!$can_use_cache) break;

                    static::g_set(__FUNCTION__, [$key_rolename], $res);
                    static::g_set(__FUNCTION__, [$key_rolename, -1], true);

                } elseif (!$groupid){
                    foreach ($records as $record){
                        $res[$record->groupid][$record->userid] = $record->userid;
                    }
                    if (!$can_use_cache) break;

                    static::g_set(__FUNCTION__, [$key_rolename, $courseid], $res);
                    static::g_set(__FUNCTION__, [$key_rolename, $courseid, -1], true);

                } else {
                    foreach ($records as $record){
                        $res[$record->userid] = $record->userid;
                    }
                    if (!$can_use_cache) break;

                    static::g_set(__FUNCTION__, [$key_rolename, $courseid, $groupid], $res);
                }
            } while(false);
        }

        if ($return_only_userids){
            if (empty($res)){
                $res = [];
            } else {
                $user_list = static::array_values_recursive($res);
                $res = array_values(array_unique($user_list));
                if (count($res) == 1 && empty($res[0])){
                    $res = [];
                }
            }
        }

        return $res;
    }

    /**
     * Get & cache online teachers (OT)
     *  return array, if there are no of any parameters, object if it gets all parameters or false, if found nothing by them
     *
     * If there is no $courseid, result will be dictionary [courseid][groupid] => [userids => userids]
     * If there is $courseid and no $groupid, result will be dictionary [groupid] => [userids => userids]
     * If there is $courseid and $groupid, result will be int array - [userids => userids]
     * But, if $return_only_userids is true, then result will be simple list of userids.
     *
     * @param numeric|bool|null $courseid            - return array[courseid][groupid] when false
     * @param numeric|bool|null $groupid             - return array[groupid] when false
     * @param numeric|bool|null $cmid                - course module id, null to check nothing, false to check all, true - check all by course, or id
     * @param bool              $active              - if false - get all, if true, return only active users, if === -1 - return only inactive
     * @param array|int         $userids             - check only by this user ids, if set; function doesn't use cache with this parameter
     * @param bool              $return_only_userids - if true, then result will be always array of userids (even for single int result)
     *
     * @return array - return [userids => userids], return empty array if it finds nothing
     */
    static public function get_online_teachers_ids($courseid=null, $groupid=null, $cmid=null, $active=true, $userids=null,
            $return_only_userids=false){
        return static::get_role_users_ids(C::ROLE_OT, $courseid, $groupid, $cmid, $active, $userids, $return_only_userids);
    }

    /**
     * Get & cache classroom teachers (CT)
     *  return array, if there are no of any parameters, object if it gets all parameters or false, if found nothing by them
     *
     * If there is no $courseid, result will be dictionary [courseid][groupid] => [userids => userids]
     * If there is $courseid and no $groupid, result will be dictionary [groupid] => [userids => userids]
     * If there is $courseid and $groupid, result will be int array [userids => userids]
     * But, if $return_only_userids is true, then result will be simple list of userids.
     *
     * @param numeric|bool|null $courseid            - return array[courseid][groupid] when false
     * @param numeric|bool|null $groupid             - return array[groupid] when false
     * @param numeric|bool|null $cmid                - course module id, null to check nothing, false to check all, true - check all by course, or id
     * @param bool              $active              - if false - get all, if true, return only active users, if === -1 - return only inactive
     * @param array|int         $userids             - check only by this user ids, if set; function doesn't use cache with this parameter
     * @param bool              $return_only_userids - if true, then result will be always array of userids (even for single int result)
     *
     * @return array - return [userids => userids], return empty array if it finds nothing
     */
    static public function get_classroom_teachers_ids($courseid=null, $groupid=null, $cmid=null, $active=true, $userids=null,
            $return_only_userids=false){
        return static::get_role_users_ids(C::ROLE_CT, $courseid, $groupid, $cmid, $active, $userids, $return_only_userids);
    }

    /**
     * Return users with rolenames2 from course & groups where users exist with rolenames1
     *
     * @param array|string  $rolenames1  - optional, but $rolenames1 or $rolenames2 is necessary
     * @param array|numeric $userids1    - check only by this user ids, if set, optional, but $userids1 or $userids2 is necessary
     * @param array|string  $rolenames2  - optional, but $rolenames1 or $rolenames2 is necessary
     * @param array|numeric $userids2    - check only by this user ids, if set, optional, but $userids1 or $userids2 is necessary
     * @param null          $courseids   - (optional) check only by this course ids, if set
     * @param null          $groupids    - (optional) check only by this group ids, if set
     * @param array|numeric $school_ids2 - (optional) if set, users with rolenames2 should be from this school(s)
     * @param array|numeric $school_ids1 - (optional) if set, users with rolenames1 should be from this school(s)
     * @param bool          $active      - (optional) return only active enrollments
     * @param numeric|bool|null $cmid    - (optional) course module id, null to check nothing, false to check all, true - check all by course, or id
     * @param string        $select_sql  - (optional) what you wish to select from the DB, by default return only user id
     *
     * @return array
     */
    static public function get_course_groups_connected_users_by_rolenames($rolenames1=[], $userids1=[], $rolenames2=[], $userids2=[],
        $courseids=null, $groupids=null, $school_ids2=0, $school_ids1=0, $active=true, $cmid=false, $select_sql=null){
        global $DB;

        if (empty($rolenames1) && empty($rolenames2)){
            debugging('You cann\'t use empty rolenames1 and rolenames2!');
            return [];
        }

        if (empty($userids1) && empty($userids2)){
            debugging('You cann\'t use empty userids1 and userids2!');
            return [];
        }

        [$userids1, $userids2, $courseids, $groupids] =
            static::val2arr_multi(true, $userids1, $userids2, $courseids, $groupids);
        $select = 'ue.id, ue.userid, e.courseid, gr.id AS groupid';
        $select_sql = $select_sql ?: 'DISTINCT users2.userid, users2.userid AS userid2';
        $other_sql = '';
        $add_join='group';
        $courseid = 0;
        if (count($courseids) == 1){
            $courseid = reset($courseids);
        }

        $check_params = [
            'users1.userid' => $userids1,
            'users2.userid' => $userids2,
            'users1.courseid' => $courseids,
            'users1.groupid' => $groupids,
        ];
        if (!static::is_school_manager_exists()){
            unset($check_params['school1.id']);
            unset($check_params['school2.id']);
        }
        [$where, $params] = static::sql_get_in_or_equal_options($check_params);

        $sql_users1 = static::sql_user_enrolments($select, [], $params, $rolenames1,
            $courseid, $groupids, $cmid, $school_ids1, $active, $other_sql, $add_join, 'ue1_');
        $sql_users2 = static::sql_user_enrolments($select, [], $params, $rolenames2,
            $courseid, $groupids, $cmid, $school_ids2, $active, $other_sql, $add_join, 'ue2_');

        $sql = "
            SELECT $select_sql
            FROM ($sql_users1) AS users1
            JOIN ($sql_users2) AS users2
                ON users2.courseid = users1.courseid
                AND users2.groupid = users1.groupid
        ";

        $sql .= static::sql_where($where);
        if (substr_count($select_sql, ',') < 2){
            $records = $DB->get_records_sql_menu($sql, $params);
        } else {
            $records = $DB->get_records_sql($sql, $params);
        }

        return $records;
    }

    /**
     * Return students from courses & groups of graders (OT & CT)
     * Get students both by course and cm enrollments
     * Alias for the @see data_util::get_course_groups_connected_users_by_rolenames
     *
     * @param array|numeric $graders            - if empty, uses global user id
     * @param array|numeric $courseids          - (optional) check only by this course ids, if set
     * @param array|numeric $groupids           - (optional) check only by this group ids, if set
     * @param array|numeric $student_ids_filter - check only by this user ids, if set
     * @param bool          $active             - (optional) return only active enrollments
     * @param string        $select             - (optional) what you wish to select from the DB, by default return only user id
     *
     * @return array
     */
    static public function get_grader_students($graders=[], $courseids=null, $groupids=null, $student_ids_filter=[], $active=true, $select=null){
        global $USER;
        if (empty($graders)){
            $graders = $USER->id;
        }

        return static::get_course_groups_connected_users_by_rolenames([C::ROLE_OT, C::ROLE_CT], $graders, C::ROLE_STUDENT, $student_ids_filter,
            $courseids, $groupids, null, null, $active, false, $select);
    }

    /**
     * Return students from courses & schools of graders (OT & CT)
     * Get students only by course enrollments, not cm
     * Alias for the @see data_util::get_course_groups_connected_users_by_rolenames
     *
     * @param array|numeric     $graders            - if empty, uses global user id
     * @param array|numeric     $courseids          - (optional) check only by this course ids, if set
     * @param array|numeric     $school_ids         - (optional) check only by this school ids, if set
     * @param array|numeric     $student_ids_filter - check only by this user ids, if set
     * @param bool              $active             - (optional) return only active enrollments
     * @param string            $select             - (optional) what you wish to select from the DB, by default return only user id
     *
     * @return array
     */
    static public function get_grader_students_by_school($graders=[], $courseids=null, $school_ids=null, $student_ids_filter=[], $active=true,
        $select=null){
        global $USER;
        if (empty($graders)){
            $graders = $USER->id;
        }

        return static::get_course_groups_connected_users_by_rolenames([C::ROLE_OT, C::ROLE_CT], $graders, C::ROLE_STUDENT, $student_ids_filter,
            $courseids, null, $school_ids, null, $active, null, $select);
    }

    /**
     * Get role ids by any of its name, alias or id
     * Course name rewrites site names, but not the system names (role.shortname)
     * Several roles can possibly have the same name/alias,
     *  so in array by names as key we return lists of role ids
     *
     * Note: if $case_sensitive is false, then all rolename keys in the result array will be lowercase.
     *
     * @param string|string[] $rolenames      - list of any role name (role.shortname or role.name), course alias or id
     * @param numeric|object  $course_or_id   - optional, course id, check course alias, if provided
     * @param bool            $case_sensitive - optional, if true, checking role names will be case-sensitive
     *
     * @return int[][] - rolenames => [list of roleids]
     */
    static public function role_get_roleids_by_any_names($rolenames, $course_or_id=null, $case_sensitive=false){
        if (empty($rolenames)) return [];

        $rolenames = static::val2arr($rolenames);
        $courseid = static::get_id($course_or_id);
        $case_sensitive = $case_sensitive ? 1 : 0;
        $res = [];
        $ask = [];

        foreach ($rolenames as $rolename){
            $rn = $case_sensitive ? $rolename : strtolower($rolename);
            $roleids = static::g_get(__FUNCTION__, [$case_sensitive, $courseid, $rn]);
            if (is_null($roleids)){
                $ask[] = $rn;
            } elseif (!empty($roleids)) {
                $res[$rn] = $roleids;
            }
        }

        if (!empty($ask)){
            $params = [];
            $joins = [];
            if ($courseid){
                $ctx = static::ctx($courseid);
                $join_clause = [
                    "rn.roleid = r.id",
                    "rn.contextid = :contextid"
                ];
                $params['contextid'] = $ctx->id;
                $joins[] = "LEFT JOIN {role_names} rn ON ".static::sql_condition($join_clause);
                unset($join_clause);
            }

            $binary = $case_sensitive ? "BINARY" : '';
            $ask_res = [];
            if (count($ask) === 1){
                // if there is only one role to request, we can use more easy query
                $select = 'r.id';
                $r = reset($ask);
                $params['rolename'] = $r;
                if ($courseid){
                    $in_cond = "r.id, r.shortname, COALESCE(rn.name, r.name)";
                } else {
                    $in_cond = "r.id, r.shortname, r.name";
                }
                $where = "$binary :rolename IN ($in_cond)";

                $sql = static::sql_generate($select, $joins, 'role', 'r', $where);
                $records = static::db()->get_records_sql($sql, $params);

                $roleids = empty($records) ? 0 : array_keys($records);
                $ask_res = [$r => $roleids];
            } else {
                $main_select = [
                    "CONCAT(CAST(all_role_names.name AS CHAR), '_', all_role_names.id) uniq_id",
                    "CAST(all_role_names.name AS CHAR) name",
                    "all_role_names.id id",
                ];
                $tables = [];
                $tables[] = static::sql_generate("r.id name, r.id", [], 'role');
                $tables[] = static::sql_generate("$binary r.shortname name, r.id", [], 'role');
                if ($courseid){
                    $tables[] = static::sql_generate("$binary COALESCE(rn.name, r.name), r.id", $joins, 'role');
                }

                $table = static::sql_union_tables($tables, true);
                $where = [];
                static::sql_add_get_in_or_equal_options("$binary all_role_names.name", $ask, $where, $params);

                $sql = static::sql_generate($main_select, [], $table, 'all_role_names', $where);
                $records = static::db()->get_records_sql($sql, $params);

                foreach ($records as $record){
                    $rn = $case_sensitive ? $record->name : strtolower($record->name);
                    if (!isset($ask_res[$rn])){
                        $ask_res[$rn] = [];
                    }
                    $ask_res[$rn][] = $record->id;
                }
                unset($records);
            }

            foreach ($ask_res as $rolename => $roleids){
                static::g_set(__FUNCTION__, [$case_sensitive, $courseid, $rolename], $roleids);
                if (!empty($roleids)){
                    $res[$rolename] = $roleids;
                }
            }
        }

        return $res;
    }

    /**
     * Checks if the user has any role by id from the array with lists of this ids
     *
     * @param int[][]        $roleid_lists - role array as [role_any_identification => [role_ids]]
     * @param numeric|object $course_or_id - course or its id, if provided - checked roles in course and parent contexts
     * @param numeric|object $user_or_id   - user or its id, uses global USER by default
     *
     * @return bool[] - role array as [role_any_identification => (user has or not any role with id)]
     *                where array keys will be from the $input_roles
     */
    static public function role_check_user_roleids($roleid_lists=[], $course_or_id=null, $user_or_id=null){
        if (empty($roleid_lists)) return [];

        $roleid_lists = static::arr_vals2arr($roleid_lists);
        $courseid = static::get_id($course_or_id);
        $userid = static::get_userid_or_global($user_or_id);
        $ctx = static::ctx($courseid);
        $params = ['userid' => $userid];
        $where = ['userid = :userid'];

        $all_role_ids = array_unique(array_merge(...array_values($roleid_lists)));
        static::sql_add_get_in_or_equal_options('ra.roleid', $all_role_ids, $where, $params);
        static::sql_add_get_in_or_equal_options('ra.contextid', $ctx->get_parent_context_ids(true), $where, $params);
        unset($all_role_ids);

        $sql = static::sql_generate('DISTINCT roleid', [], 'role_assignments', 'ra', $where);
        $user_has_roleids = static::db()->get_records_sql($sql, $params);

        $user_roles = [];
        foreach ($roleid_lists as $key => $roleids){
            $user_roles[$key] = false;

            foreach ($roleids as $roleid){
                if (isset($user_has_roleids[$roleid])){
                    $user_roles[$key] = true;
                    break;
                }
            }
        }

        return $user_roles;
    }

    /**
     * Checks if the user has any role by any of its name, alias or id
     * Course name rewrites site names, but not the system names (role.shortname)
     * Several roles can possibly have the same name/alias - result or this name will be true if user has at least one of these role
     *
     * Notes:
     * - if there are no role on server by some key, this key will be absent from the result array
     * - if $case_sensitive is false, then all rolename keys in the result array will be lowercase.
     *
     * Alias for calling
     * @see \local_ned_controller\shared\data_util::role_get_roleids_by_any_names() &
     * @see \local_ned_controller\shared\data_util::role_check_user_roleids()
     *
     * @param string|string[] $rolenames      - list of any role name (role.shortname or role.name), course alias or id
     * @param numeric|object  $course_or_id   - optional, course id, check course alias, if provided
     * @param numeric|object  $user_or_id     - user or its id, uses global USER by default
     * @param bool            $case_sensitive - optional, if true, checking role names will be case-sensitive
     *
     * @return bool[] - role array as [role_name => (user has or not any role with id)]
     *                where array keys will be from the $rolenames
     */
    static public function role_check_user_roleids_by_any_names($rolenames, $course_or_id=null, $user_or_id=null, $case_sensitive=false){
        $roleid_lists = static::role_get_roleids_by_any_names($rolenames, $course_or_id, $case_sensitive);
        return static::role_check_user_roleids($roleid_lists, $course_or_id, $user_or_id);
    }

    /**
     * Load and get data for default_role functions: default_role field ID and its default value
     *
     * @return array - [$default_role_field_id, $default_role_default_value]
     */
    static public function role_get_default_role_data(){
        $key_default_role_field_id = 'default_role_field_id';
        $key_default_role_default_value = 'default_role_default_value';

        $default_role_field_id = static::g_get(__FUNCTION__, $key_default_role_field_id);
        $default_role_default_value = null;
        if (is_null($default_role_field_id)){
            $record = static::db()->get_record(C::TABLE_USER_INFO_FIELD, ['shortname' => C::FIELD_DEFAULT_ROLE], 'id, defaultdata');
            $default_role_field_id = $record->id ?? false;
            $default_role_default_value = $record->defaultdata ?? '';

            static::g_set(__FUNCTION__, $key_default_role_field_id, $default_role_field_id);
            static::g_set(__FUNCTION__, $key_default_role_default_value, $default_role_default_value);
        }

        $default_role_default_value = $default_role_default_value ?? static::g_get(__FUNCTION__, $key_default_role_default_value);

        return [$default_role_field_id, $default_role_default_value];
    }

    /**
     * Return user default role
     * Note: it doesn't connect to the real user roles, it returns value of user profile data for field "Default role"
     *
     * @param object|numeric $user_or_id
     *
     * @return string - return empty string, if there is no such data
     */
    static public function role_get_user_def_role($user_or_id=null){
        [$default_role_field_id, $default_role_default_value] = static::role_get_default_role_data();
        if (empty($default_role_field_id)) return '';

        $userid = static::get_userid_or_global($user_or_id);
        return static::db()->get_field(C::TABLE_USER_INFO_DATA, 'data',
            ['fieldid' => $default_role_field_id, 'userid' => $userid]) ?: '';
    }

    /**
     * Return true, if user has provided default role
     * Note: it doesn't connect to the real user roles, it checks value of user profile data: field "Default role"
     *
     * @param object|numeric $user_or_id
     * @param string         $def_role_name - default role to check
     *
     * @return bool
     */
    static public function role_is_user_def_role($user_or_id=null, $def_role_name=C::DEFAULT_ROLE_STUDENT){
        [$default_role_field_id, $default_role_default_value] = static::role_get_default_role_data();
        if (empty($default_role_field_id)) return false;

        $field_data = static::role_get_user_def_role($user_or_id);
        return $field_data == $def_role_name || (empty($field_data) && $default_role_default_value == $def_role_name);
    }

    /**
     * Return true, if user has default role "Student"
     * Note: it doesn't connect to the real user roles, it checks value of user profile data: field "Default role"
     * Alias for @see role_is_user_def_role();
     *
     * @param object|numeric $user_or_id
     *
     * @return bool
     */
    static public function role_is_user_default_student($user_or_id=null){
        return static::role_is_user_def_role($user_or_id, static::DEFAULT_ROLE_STUDENT);
    }

    /**
     * Get count of users for grader, by school and course
     *
     * @param numeric           $courseid
     * @param numeric           $school_id
     * @param numeric|object    $grader_or_id
     *
     * @return int
     */
    static public function get_count_students_at_grader_school($courseid, $school_id, $grader_or_id=null){
        if (empty($courseid) || empty($school_id)){
            return 0;
        }

        $userid = static::get_userid_or_global($grader_or_id);
        $res = static::g_get(__FUNCTION__, [$userid, $courseid, $school_id]);
        if (is_null($res)){
            $students = static::get_grader_students_by_school($userid, $courseid, $school_id);
            $res = empty($students) ? 0 : count($students);
            static::g_set(__FUNCTION__, [$userid, $courseid, $school_id], $res);
        }

        return $res;
    }

    /**
     * Return school object by its id
     *
     * @param numeric $id
     *
     * @return object|\local_schoolmanager\school|null
     */
    static public function school_get_school($id){
        if (empty($id) || !static::is_schm_exists()) return null;

        $res = static::g_get(__FUNCTION__, [$id]);
        if (is_null($res)){
            $res = \local_schoolmanager\school::get_school_by_id($id);
            static::g_set(__FUNCTION__, [$id], $res);
        }

        return $res ?: null;
    }

    /**
     * Get school(s) by user (id)
     *
     * @param numeric|object    $user_or_id - for who get schools
     * @param bool              $only_one - if true, return only one record (or id)
     * @param bool              $only_id - if true, return id(s) o object(s)
     *
     * @return array|object[]|int[]|object|int|null
     */
    static public function get_user_schools($user_or_id=null, $only_one=false, $only_id=false){
        if (!static::is_school_manager_exists()){
            return $only_one ? null : [];
        }

        $userid = static::get_userid_or_global($user_or_id);
        $schools = static::g_get(__FUNCTION__, [$userid]);
        if (is_null($schools)){
            $from = [static::sql_get_school_join()];
            $sql = static::sql_generate('school.*', $from, 'user', 'u', 'u.id = :userid', [], 'school.id');
            $params = ['userid' => $userid];
            $schools = static::db()->get_records_sql($sql, $params);
            static::g_set(__FUNCTION__, [$userid], $schools);
        }

        $res = $schools;
        if ($only_id){
            $res = empty($res) ? [] : array_keys($res);
        }
        if ($only_one){
            $res = empty($res) ? null : reset($res);
        }

        return $res;
    }

    /**
     * Function get school name from user profile data and check if user is member of this school
     * If not, tries to get his first school from {@see data_util::get_user_schools()}
     *
     * @param numeric|object    $user_or_id - for whom get school
     * @param bool              $only_id - if true, return id of the school, otherwise return school object
     *
     * @return object|int|\local_schoolmanager\school|null
     */
    static public function get_user_school($user_or_id=null, $only_id=false){
        $userid = static::get_userid_or_global($user_or_id);
        $res = static::g_get(__FUNCTION__, $userid);
        if (is_null($res)){
            $res = false;
            $user = static::get_user($userid);
            if (!empty($user)){
                if (empty($user->profile_field_partner_school)){
                    profile_load_data($user);
                }
                if (!empty($user->profile_field_partner_school)){
                    $school = static::get_school_by_name($user->profile_field_partner_school);
                    if ($school && cohort_is_member($school->id, $userid)){
                        $res = $school;
                    }
                }
                if (!$res){
                    $res = static::get_user_schools($user_or_id, true) ?: false;
                }
            }

            static::g_set(__FUNCTION__, $userid, $res);
        }

        return $res ? ($only_id ? $res->id : $res) : null;
    }

    /**
     * @param numeric|object $user_or_id - for whom get school name
     * @param bool           $short      - if true, return school codes (shortnames), otherwise full names
     *
     * @return string
     */
    public static function get_user_school_name($user_or_id=null, $short=false){
        return static::get_user_school_names($user_or_id, $short, '', true);
    }


    /**
     * @param numeric|object $user_or_id - for whom get school name
     * @param bool           $short      - if true, return school codes (shortnames), otherwise full names
     * @param string         $separator  - separator to join several school names
     * @param bool           $only_one   - if true, return only one name
     *
     * @return string
     */
    public static function get_user_school_names($user_or_id=null, $short=false, $separator=', ', $only_one=false){
        if (!static::is_schm_exists()) return '';

        $schools = static::get_user_schools($user_or_id);
        $key = $short ? 'code' : 'name';
        $names = [];
        foreach ($schools as $school){
            if (isset($school->$key)){
                $names[] = trim($school->$key);
                if ($only_one) break;
            }
        }

        return join($separator, $names);
    }

    /**
     * Get school by its name
     *
     * @param string $name
     *
     * @return object|\local_schoolmanager\school|false
     */
    public static function get_school_by_name($name){
        $name = trim($name);
        if (empty($name) || isset(C::SCHOOL_EMPTY_LIST[$name])) return false;

        $len = strlen($name);
        if ($len != C::SCHOOL_CODE_LENGTH){
            if ($len < C::SCHOOL_CODE_LENGTH){
                return false;
            } else {
                // $len > C::SCHOOL_CODE_LENGTH
                $name = substr($name, 0, C::SCHOOL_CODE_LENGTH);
            }
        }

        if (static::is_schm_exists()){
            $data = \local_schoolmanager\school::get_records_select('code LIKE ?', [$name]);
            return reset($data);
        } else {
            return static::db()->get_record_select('cohort', 'idnumber LIKE ?', [$name]);
        }
    }

    /**
     * Determine if two users share a cohort in common.
     *
     * @param numeric|object      $user_or_id1 First user.
     * @param numeric|object|null $user_or_id2 Second user, if null - uses global user
     *
     * @return bool True if the do share a common school, false otherwise.
     */
    static public function is_in_same_school($user_or_id1, $user_or_id2=null) {
        $school_ids = static::get_user_schools($user_or_id2, false, true);
        $userid1 = static::get_id($user_or_id1);

        foreach ($school_ids as $schoolid) {
            if (cohort_is_member($schoolid, $userid1)){
                return true;
            }
        }

        return false;
    }

    /**
     * Get enabled manual enrol by courseid
     *
     * @param object|numeric    $course_or_id
     * @param numeric           $id - (optional) enrol id for additional check
     *
     * @return object|false
     */
    static public function enrol_get_manual_enrol_instances($course_or_id, $id=null){
        $courseid = static::get_id($course_or_id);
        $res = static::g_get(__FUNCTION__, [$courseid]);
        if (is_null($res)){
            $params = ['enrol' => 'manual', 'courseid' => $courseid, 'status' => ENROL_INSTANCE_ENABLED];
            if ($id){
                $params['id'] = $id;
            }

            $instances = static::db()->get_records('enrol', $params, 'sortorder, id');
            if (empty($instances)){
                $res = false;
            } else {
                $res = reset($instances);
            }
        } elseif ($res && $id){
            if ($res->id != $id){
                $res = false;
            }
        }

        return $res;
    }

    /**
     * Return last assign submission by course-module (id) and user id
     *
     * @param int|\cm_info|object   $cm_or_id  - Id of course-module, or database object
     * @param numeric|object|null   $user_or_id - optional user, by whom submitted time need, uses global USER by default
     *
     * @return false|\stdClass|null
     */
    static public function assign_get_last_submission_by_cm($cm_or_id, $user_or_id=null){
        $cm = static::get_cm_by_cmorid($cm_or_id);
        if (!$cm || $cm->modname != C::ASSIGN){
            return null;
        }

        $userid = static::get_userid_or_global($user_or_id);
        $ned_assign = static::$ned_assign::get_assign_by_cm($cm, $cm->course);
        return $ned_assign->get_needed_user_submission($userid);
    }

    /**
     * Get quiz_attempts by course-module (id) and user id
     *
     * @param int|\cm_info|object   $cm_or_id  - Id of course-module, or database object
     * @param numeric|object|null   $user_or_id - optional user, by whom submitted time need, uses global USER by default
     *
     * @return array
     */
    static public function quiz_get_quiz_attempts_by_cm($cm_or_id, $user_or_id=null){
        $cm = static::get_cm_by_cmorid($cm_or_id);
        if (!$cm || $cm->modname != C::QUIZ){
            return null;
        }

        static::require_file('/mod/quiz/lib.php');
        $userid = static::get_userid_or_global($user_or_id);
        $quiz_attempts = quiz_get_user_attempts([$cm->instance], $userid, 'all', false);
        return $quiz_attempts;
    }

    /**
     * Get time of assign submission by course-module (id) and user id
     *
     * @param int|\cm_info|object   $cm_or_id  - Id of course-module, or database object
     * @param numeric|object|null   $user_or_id - optional user, by whom submitted time need, uses global USER by default
     * @param bool                  $include_draft - optional, if true, also return time of unfinished work
     *
     * @return int|null
     */
    static public function assign_get_submitted_time_by_cm($cm_or_id, $user_or_id=null, $include_draft=false){
        $submission = static::assign_get_last_submission_by_cm($cm_or_id, $user_or_id);
        if (!$submission){
            return 0;
        }

        if ($submission->status == ASSIGN_SUBMISSION_STATUS_SUBMITTED || ($include_draft && $submission->status == ASSIGN_SUBMISSION_STATUS_DRAFT)){
            return $submission->timemodified ?? 0;
        }

        return 0;
    }

    /**
     * Get time of quiz submission by course-module (id) and user id
     *
     * @param int|\cm_info|object   $cm_or_id  - Id of course-module, or database object
     * @param numeric|object|null   $user_or_id - optional user, by whom submitted time need, uses global USER by default
     * @param bool                  $include_draft - optional, if true, also return time of unfinished work
     *
     * @return int|null
     */
    static public function quiz_get_submitted_time_by_cm($cm_or_id, $user_or_id=null, $include_draft=false){
        $quiz_attempts = static::quiz_get_quiz_attempts_by_cm($cm_or_id, $user_or_id);
        if (empty($quiz_attempts)){
            return 0;
        }

        $qa = end($quiz_attempts);
        if ($qa->state == \quiz_attempt::FINISHED || ($include_draft && $qa->state == \quiz_attempt::IN_PROGRESS)){
            return $qa->timefinish ?? 0;
        }

        return 0;
    }

    /**
     * Get time of submission by course-module (id) and user id
     * Really can get time only for assign and quiz, otherwise return false
     *
     * @param int|\cm_info|object   $cm_or_id  - Id of course-module, or database object
     * @param numeric|object|null   $user_or_id - optional user, by whom submitted time need, uses global USER by default
     * @param bool                  $include_draft - optional, if true, also return time of unfinished work
     *
     * @return int|false
     */
    static public function get_submitted_time_by_cm($cm_or_id, $user_or_id=null, $include_draft=false){
        $cmid = static::get_id($cm_or_id);
        $userid = static::get_userid_or_global($user_or_id);
        $res = static::g_get(__FUNCTION__, [$cmid, $userid]);

        if (is_null($res)){
            $res = false;
            $cm = static::get_cm_by_cmorid($cm_or_id);
            if ($cm){
                switch ($cm->modname){
                    case C::ASSIGN:
                        $res = static::assign_get_submitted_time_by_cm($cm, $user_or_id, $include_draft);
                        break;
                    case C::QUIZ:
                        $res = static::quiz_get_submitted_time_by_cm($cm, $user_or_id, $include_draft);
                        break;
                }
            }

            static::g_set(__FUNCTION__, [$cmid, $userid], $res);
        }

        return $res;
    }

    /**
     * Return number of last assign attempt by course-module (id) and user id
     * Starts from 0 (first attempt), if there are none attempts, return also 0
     *
     * @param int|\cm_info|object   $cm_or_id  - Id of course-module, or database object
     * @param numeric|object|null   $user_or_id - optional user, by whom submitted time need, uses global USER by default
     *
     * @return int
     */
    static public function assign_get_attempt_by_cm($cm_or_id, $user_or_id=null){
        $submission = static::assign_get_last_submission_by_cm($cm_or_id, $user_or_id);
        return $submission->attemptnumber ?? 0;
    }

    /**
     * Get attempt of last quiz submission by course-module (id) and user id
     * Starts from 0 (first attempt), if there are none attempts, return also 0
     *
     * @param int|\cm_info|object   $cm_or_id  - Id of course-module, or database object
     * @param numeric|object|null   $user_or_id - optional user, by whom submitted time need, uses global USER by default
     *
     * @return int
     */
    static public function quiz_get_attempt_by_cm($cm_or_id, $user_or_id=null){
        $quiz_attempts = static::quiz_get_quiz_attempts_by_cm($cm_or_id, $user_or_id);
        if (empty($quiz_attempts)){
            return 0;
        }

        $qa = end($quiz_attempts);
        return $qa->attempt - 1;
    }

    /**
     * Get attempt of last submission by course-module (id) and user id
     * Starts from 0 (first attempt), if there are none attempts, return also 0
     * Really can get attempt only for assign and quiz, forum & journal always return 0, otherwise return null
     *
     * @param int|\cm_info|object   $cm_or_id  - Id of course-module, or database object
     * @param numeric|object|null   $user_or_id - optional user, by whom submitted time need, uses global USER by default
     *
     * @return int|null
     */
    static public function get_attempt_by_cm($cm_or_id, $user_or_id=null){
        $cm = static::get_cm_by_cmorid($cm_or_id);
        if ($cm){
            switch ($cm->modname){
                case C::ASSIGN:
                    return static::assign_get_attempt_by_cm($cm, $user_or_id);
                case C::QUIZ:
                    return static::quiz_get_attempt_by_cm($cm, $user_or_id);
                case C::FORUM:
                case C::JOURNAL:
                    return 0;
            }
        }

        return null;
    }

    /**
     * Get assign active controller, by activity or its context
     *
     * @param \context|\assign|\cm_info $cm_or_assign_or_context
     *
     * @return \gradingform_controller|null
     */
    static public function assign_get_controller($cm_or_assign_or_context){
        if (is_a($cm_or_assign_or_context,'context')){
            $ctx = $cm_or_assign_or_context;
        } elseif (is_a($cm_or_assign_or_context, 'assign')){
            $ctx = $cm_or_assign_or_context->get_context();
        } elseif (is_a($cm_or_assign_or_context, 'cm_info')){
            $ctx = $cm_or_assign_or_context->context;
        }

        if (!empty($ctx)){
            $grading_manager = get_grading_manager($ctx, 'mod_assign', 'submissions');
            $controller = $grading_manager->get_active_controller();
            if ($controller){
                return $controller;
            }
        }

        return null;
    }

    /**
     * Get user start date (based on his groups) for some course
     *
     * @param object|numeric $course_or_id
     * @param object|numeric $user_or_id
     * @param bool           $least - if true, return the earliest value from the all groups, otherwise return the latest value
     *
     * @return int|null - UNIX time value, or null if its finds nothing
     */
    static public function get_user_start_date($course_or_id=null, $user_or_id=null, $least=true){
        $groups = static::get_all_user_course_groups($course_or_id, $user_or_id);
        if (empty($groups)) return null;

        $res = [];
        foreach ($groups as $group){
            if (empty($group->startdate)) continue;

            $res[] = $group->startdate;
        }

        if (empty($res)) return null;
        return $least ? static::min(...$res) : static::max(...$res);
    }

    /**
     * Get user end date (based on his groups) for some course
     *
     * @param object|numeric $course_or_id
     * @param object|numeric $user_or_id
     * @param bool           $most - if true, return the latest value from the all groups, otherwise return the earliest value
     *
     * @return int|null - UNIX time value, or null if its finds nothing
     */
    static public function get_user_end_date($course_or_id=null, $user_or_id=null, $most=true){
        $groups = static::get_all_user_course_groups($course_or_id, $user_or_id);
        if (empty($groups)) return null;

        $res = [];
        foreach ($groups as $group){
            if (empty($group->enddate)) continue;

            $res[] = $group->enddate;
        }

        if (empty($res)) return null;
        return $most ? static::max(...$res) : static::min(...$res);
    }

    /**
     * Remove data which depends on specific course
     * It can be useful, when you check many courses, and need to remove data from the previous (already checked) courses
     *
     * @param string|string[] $selected_keys - you can choose keys to deleting, when empty - choose all of them
     * @param bool            $kica - delete or not kica data too
     *
     * @return void
     */
    static public function purge_course_depended_caches($selected_keys=[], $kica=true){
        $default_keys = [
            'get_fast_modinfo',                 /** @see shared_lib::get_fast_modinfo() */
            'get_course_and_cm_from_cmid',      /** @see shared_lib::get_course_and_cm_from_cmid() */
            'get_course_activities',            /** @see shared_lib::get_course_activities() */
            'get_availability_info_module',     /** @see shared_lib::get_availability_info_module() */
            'get_block_config',                 /** @see shared_lib::get_block_config() */
            'get_site_and_course_block_config', /** @see shared_lib::get_site_and_course_block_config() */
            'get_course_groupids',              /** @see shared_lib::get_course_groupids() */
            'cm_get_tags',                      /** @see shared_lib::cm_get_tags() */
            'get_graderid_by_studentid',        /** @see shared_lib::get_graderid_by_studentid() */
            '_get_visibility_data_by_cm_user',  /** @see shared_lib::_get_visibility_data_by_cm_user() */
            'get_important_activities',         /** @see shared_lib::get_important_activities() */
            'get_module_instance',              /** @see shared_lib::get_module_instance() */

            'get_grade_item',                   /** @see shared_lib::get_grade_item() */
            'get_grade_item_by_id_or_params',   /** @see shared_lib::get_grade_item_by_id_or_params() */
            'get_course_grade_item',            /** @see shared_lib::get_course_grade_item() */
            'get_grade_grade',                  /** @see shared_lib::get_grade_grade() */
            'get_grading_area',                 /** @see shared_lib::get_grading_area() */
        ];

        if (empty($selected_keys)){
            $keys = $default_keys;
        } else {
            $keys = [];
            $selected_keys = static::val2arr($selected_keys);
            foreach ($default_keys as $def_key){
                if (($selected_keys[$def_key] ?? false) || in_array($def_key, $selected_keys)){
                    $keys[] = $def_key;
                }
            }
        }

        static::g_remove_functions_data($keys);
        if ($kica){
            static::kica_purge_cache();
        }
    }

    /**
     *  Removes all local cached data
     *  Note: Don't mix it up with the purging moodle cache, it's absolutely different functions!
     *
     * @param bool $kica - delete or not kica data too
     */
    static public function remove_all_used_local_cache($kica=true){
        /** TODO: function in developing */
        static::g_remove_all_ned_data();
        if ($kica){
            static::kica_purge_cache();
        }
    }
}
