<?php
/**
 * @package    local_ned_controller
 * @subpackage shared
 * @category   NED
 * @copyright  2021 NED {@link http://ned.ca}
 * @author     NED {@link http://ned.ca}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace local_ned_controller\shared;
use local_ned_controller\marking_manager as mm;
use local_ned_controller\ned_grade_grade as ned_grade_grade;

defined('MOODLE_INTERNAL') || die();

/**
 * Trait grade_util
 *
 * @package local_ned_controller\shared
 * @mixin \local_ned_controller\shared\base_trait
 */
trait grade_util {
    use util, data_util, kica_util;

    /**
     * Get grade_item itemnumber (gi.itemnumber) by course module
     *
     * @see \local_ned_controller\marking_manager\marking_manager_mod::SQL_GI_ITEMNUMBER
     *
     * @param \cm_info|object|numeric $cm_or_id
     *
     * @return int|null - int itemnumber, or null if it doesn't find the module
     */
    static public function grade_get_gi_itemnumber($cm_or_id){
        $cm = static::get_cm_by_cmorid($cm_or_id);
        if (!$cm) return null;

        if ($cm->modname == C::FORUM){
            $forum = static::get_module_instance_by_cm($cm);
            if (!$forum) return null;

            /**
             * Only if Rating is off, and Whole forum grading is ON – we use Whole forum grading,
             *  otherwise we use Rating grading (as default grading method)
             */
            if ($forum->assessed == 0 && $forum->grade_forum != 0){
                /** advanced method, @see \mod_forum\grades\gradeitems::get_advancedgrading_itemnames() */
                return C::GRADE_ITEMNUMBER_FORUM_WHOLE;
            } else {
                /** default one, @see \mod_forum\grades\gradeitems::get_itemname_mapping_for_component() */
                return C::GRADE_ITEMNUMBER_FORUM_RATING;
            }
        } else {
            return C::GRADE_ITEMNUMBER_USUAL;
        }
    }

    /**
     * Get grade item for a course module
     *
     * @param \cm_info|object|numeric $cm_or_id
     * @param \stdClass|int           $courseorid     - Optional course object (or its id) if already loaded
     * @param \stdClass|int           $userorid       - Optional user object (or its id; default = current)
     * @param string                  $modulename     - Optional modulename (improves security)
     * @param bool                    $load_by_course - Optional, load all grade items in course
     *
     * @return \grade_item|null
     */
    static public function get_grade_item($cm_or_id, $courseorid=null, $userorid=null, $modulename='', $load_by_course=false){
        $cmid = static::get_id($cm_or_id);
        $gradeitem_id = static::g_get(__FUNCTION__, [$cmid]);
        $gradeitem = null;
        if (is_null($gradeitem_id)){
            $cm = static::get_cm_by_cmorid($cm_or_id, $courseorid, $userorid, $modulename);
            if ($cm){
                $params = [
                    'itemtype' => 'mod',
                    'itemmodule' => $cm->modname,
                    'iteminstance' => $cm->instance,
                    'courseid' => $cm->course,
                    'itemnumber' => static::grade_get_gi_itemnumber($cm),
                ];
                $gradeitem = static::get_grade_item_by_id_or_params(null, $params, $load_by_course ? $cm->course : null);
            }

            static::g_set(__FUNCTION__, [$cmid], $gradeitem->id ?? 0);

            return $gradeitem;
        }

        if ($gradeitem_id){
            $gradeitem = static::get_grade_item_by_id_or_params($gradeitem_id);
        }

        return $gradeitem ?: null;
    }

    /**
     * Get grade item for a course module, and load other grade items in the course
     *
     * Alias @see grade_util::get_grade_item()
     *
     * @param \cm_info|int|string $cm_or_id
     * @param \stdClass|int       $courseorid - Optional course object (or its id) if already loaded
     *
     * @return \grade_item|null
     */
    static public function get_grade_item_by_cm_course($cm_or_id, $courseorid=null){
        return static::get_grade_item($cm_or_id, $courseorid, null, '', true);
    }

    /**
     * Get grade item for a course module
     *
     * WARNING: It can work incorrectly for loading by params without $id, but by for full course!
     *
     * @param numeric|null  $id - Optional, grade_item id
     * @param array         $params   - Optional, grade_items params to fetch
     * @param \stdClass|int $courseorid - Optional course (or its id) - if set, function will load all grade_items by courseid
     *
     * @return \grade_item|null
     */
    static public function get_grade_item_by_id_or_params($id=null, $params=[], $courseorid=null){
        if (empty($id) && empty($params)){
            static::debugging("You should set id or some params!");
            return null;
        }

        $courseid = static::get_id($courseorid);
        $gradeitem = null;
        $id = (int)$id;
        if (!$id){
            $id = $params['id'] ?? 0;
        }

        if ($id){
            $gradeitem = static::g_get(__FUNCTION__, [0, $id]);
        }

        if ($courseid && is_null($gradeitem)){
            $course_loaded = static::g_get(__FUNCTION__, [$courseid]);
            if ($course_loaded){
                $gradeitem = false;
            }
        }

        if (is_null($gradeitem)){
            $params = static::val2arr($params);
            $sql_params = $params;
            if ($courseid){
                $sql_params = [
                    'itemtype' => 'mod',
                    'courseid' => $courseid,
                ];
            } elseif ($id){
                $sql_params['id'] = $id;
            } else {
                if (!isset($sql_params['itemtype'])){
                    $sql_params['itemtype'] = 'mod';
                }
            }

            $gradeitems = \grade_item::fetch_all($sql_params);
            $gradeitems = static::val2arr($gradeitems);
            if ($courseid){
                static::g_set(__FUNCTION__, [$courseid], true);
                $modinfo = static::get_fast_modinfo($courseid);

                foreach ($gradeitems as $gradeitem){
                    /** @var \grade_item $gradeitem */
                    static::g_set(__FUNCTION__, [0, $gradeitem->id], $gradeitem);

                    if ($modinfo){
                        $cm = $modinfo->instances[$gradeitem->itemmodule][$gradeitem->iteminstance] ?? null;
                        if ($cm && static::grade_get_gi_itemnumber($cm) == $gradeitem->itemnumber){
                            static::g_set('get_grade_item', [$cm->id], $gradeitem->id);
                        }
                    }

                    if (!$id){
                        $checked = true;
                        foreach ($params as $param => $val){
                            if (!isset($gradeitem->$param) || strval($gradeitem->$param) != strval($val)){
                                $checked = false;
                                break;
                            }
                        }
                        if ($checked){
                            $id = $gradeitem->id;
                        }
                    }
                }
                $gradeitem = $gradeitems[$id] ?? null;
            } else {
                $gradeitem = reset($gradeitems);
                $id = $gradeitem->id ?? $id;
                if ($id){
                    static::g_set(__FUNCTION__, [0, $id], $gradeitem);
                }
            }
        }

        return $gradeitem ?: null;
    }

    /**
     *  Get grade item for a course
     *
     * @param object|numeric $courseorid
     *
     * @return \grade_item|null
     */
    static public function get_course_grade_item($courseorid){
        $courseid = static::get_id($courseorid);
        $course_item = static::g_get(__FUNCTION__, [$courseid]);
        if (is_null($course_item)){
            $course_item = \grade_item::fetch_course_item($courseid);
            static::g_set(__FUNCTION__, [$courseid], $course_item);
        }

        return $course_item ?: null;
    }

    /**
     * Get course module grade_grade object
     *
     * @param \cm_info|int|string $cm_or_id - course module object or its id (it's better to use object)
     * @param int|object          $user_or_id - user or id, global $USER by default
     * @param bool                $clone - to clone object for independence, highly recommended, if it will be using for writing,
     *                                 use 'false' if it will be read-only
     * @param bool                $create_if_none - create new grade_grade object, if their none yet
     * @param bool                $only_cached  - do not check DB, only local NED cache
     *
     * @return ned_grade_grade|\grade_grade|null - return grade_grade (gg) or null, if it finds nothing
     */
    static public function get_grade_grade($cm_or_id, $user_or_id=null, $clone=true, $create_if_none=false, $only_cached=false){
        $cmid = static::get_id($cm_or_id);
        $userid = static::get_userid_or_global($user_or_id);
        $gg = static::g_get(__FUNCTION__, [$cmid, $userid]);
        if (is_null($gg) && !$only_cached){
            $gi = static::get_grade_item($cm_or_id);
            $gg = ned_grade_grade::get_by_grade_item($gi, $userid, $create_if_none);
            static::g_set(__FUNCTION__, [$cmid, $userid], $gg ?: false);
        }

        if ($gg && $clone){
            return clone($gg);
        }

        return $gg ?: null;
    }

    /**
     * Get course module cached grade_grade object
     * Read-only, as it doesn't clone object
     *
     * Alias @see grade_util::get_grade_grade()
     *
     * @param $cm_or_id
     * @param $user_or_id
     *
     * @return \grade_grade|ned_grade_grade|null
     */
    static public function get_grade_grade_cached($cm_or_id, $user_or_id){
        return static::get_grade_grade($cm_or_id, $user_or_id, false, false, true);
    }

    /**
     * Get all grades by itemid(s) and/or userids
     * You can leave something empty, but you need set at least something
     * It saves result in the same cache, as get_grade_grade()
     *
     * If you wish not to cache result, you can use ned_grade_grade::fetch_all_itemid_userid function
     * @see \local_ned_controller\ned_grade_grade::fetch_all_itemid_userid()
     *
     * @param numeric|array $itemids - id(s) of grade items; if empty - load for all items
     * @param numeric|array $userids - user id(s); if empty - load for all users
     *
     * @return array|ned_grade_grade[]
     */
    static public function get_grade_grade_by_itemids_userids($itemids=[], $userids=[]){
        $ggs = ned_grade_grade::fetch_all_itemid_userid($itemids, $userids);
        foreach ($ggs as $gg){
            if (empty($gg->grade_item)){
                $gg->grade_item = static::get_grade_item_by_id_or_params($gg->itemid);
            }

            $cm = static::get_cm_by_kica_or_grade_item($gg->grade_item);
            if ($cm){
                static::g_set('get_grade_grade', [$cm->id, $gg->userid], $gg);
            }
        }

        return $ggs;
    }

    /**
     * Get grading areas DB record
     *
     * @param numeric $id
     *
     * @return object|null - return null, if it finds nothing
     */
    static public function get_grading_area($id){
        $area = static::g_get(__FUNCTION__, [$id]);
        if (is_null($area)){
            $area = static::db()->get_record('grading_areas', ['id' => $id]);
            static::g_set(__FUNCTION__, [$id], $area);
        }

        return $area ?: null;
    }

    /**
     * Get course grade_grade object
     *
     * @param object|numeric      $courseorid
     * @param int|object          $user_or_id
     * @param bool                $clone - to clone object for independence, highly recommended, if it will be using for writing,
     *                                 use 'false' if it will be read-only
     * @param bool                $create_if_none
     *
     * @return ned_grade_grade|\grade_grade|null - return null, if it finds nothing
     */
    static public function get_course_grade_grade($courseorid, $user_or_id, $clone=true, $create_if_none=false){
        $courseid = static::get_id($courseorid);
        $userid = static::get_id($user_or_id);
        $gg = static::g_get(__FUNCTION__, [$courseid, $userid]);
        if (is_null($gg)){
            $gi = static::get_course_grade_item($courseid);
            $gg = ned_grade_grade::get_by_grade_item($gi, $userid, $create_if_none);
            static::g_set(__FUNCTION__, [$courseid, $userid], $gg ?: false);
        }

        if ($gg && $clone){
            return clone($gg);
        }

        return $gg ?: null;
    }

    /**
     * Set grade_grade object
     *
     * @param \cm_info|int|string               $cm_or_id - course module object or its id (it's better to use object)
     * @param ned_grade_grade|\grade_grade|null $grade_grade - grade object to save
     */
    static protected function _set_grade_grade($cm_or_id, $grade_grade){
        $cmid = static::get_id($cm_or_id);
        static::g_set('get_grade_grade', [$cmid, $grade_grade->userid], $grade_grade ?: false);
    }

    /**
     * Check and do regrade for grade_item if need it
     *
     * @see \grade_item::update_final_grade()
     *
     * @param \cm_info|int|string $cm_or_id - course module object or its id (it's better to use object)
     * @param int|object          $user_or_id
     * @param \grade_item|null    $grade_item - optional grade_item object, if already get it
     *
     */
    static public function grade_item_check_regrade($cm_or_id, $user_or_id, $grade_item=null){
        $userid = static::get_id($user_or_id);
        $gi = $grade_item ?: static::get_grade_item($cm_or_id, null, $user_or_id);
        if (!$gi){
            return;
        }

        if ($gi->is_course_item() and !$gi->needsupdate) {
            if (grade_regrade_final_grades($gi->courseid, $userid, $gi) !== true) {
                $gi->force_regrading();
            }

        } elseif (!$gi->needsupdate) {
            $course_item = \grade_item::fetch_course_item($gi->courseid);
            if (!$course_item->needsupdate) {
                if (grade_regrade_final_grades($gi->courseid, $userid, $gi) !== true) {
                    $gi->force_regrading();
                }
            } else {
                $gi->force_regrading();
            }
        }

        if (grade_needs_regrade_final_grades($gi->courseid)) {
            grade_regrade_final_grades($gi->courseid);
        }
    }

    /**
     * Refetch grades from modules, plugins.
     * Also check grade_item for regrade, and update completion state
     *
     * @param \cm_info|object|numeric           $cm_or_id - course module object or its id
     * @param numeric|object                    $user_or_id - user or its id
     * @param ned_grade_grade|\grade_grade|null $grade_grade - optional grade_grade object, if already loaded
     *
     * @return bool
     */
    static public function grade_grade_refresh($cm_or_id=null, $user_or_id=null, $grade_grade=null){
        if (empty($grade_grade) && (empty($cm_or_id) || empty($user_or_id))){
            static::debugging("Can't refresh grade without grade object, or course-module and user id!");
            return false;
        }

        if (empty($grade_grade)){
            $grade_grade = static::get_grade_grade($cm_or_id, $user_or_id, false, true);
            if (empty($grade_grade)){
                return false;
            }
        }

        if ($grade_grade->is_overridden() || $grade_grade->is_excluded()){
            return false;
        }

        $grade_grade->load_grade_item();
        if (empty($cm_or_id)){
            $cm = static::get_cm_by_kica_or_grade_item($grade_grade->grade_item);
        } else {
            $cm = static::get_cm_by_cmorid($cm_or_id);
        }

        // By render assign grade, we can have additional refreshing from grading form (but we can't really check the result)
        if ($cm->modname == C::ASSIGN){
            $assign = static::$ned_assign::get_assign_by_cm($cm, $cm->course);
            $assign_grades = $assign->get_all_user_grades($grade_grade->userid);
            if (!empty($assign_grades)){
                $controller = static::assign_get_controller($assign);
                if ($controller){
                    $page = static::page();
                    $page->set_context(null); // avoid page warnings
                    $controller->set_grade_range(make_grades_menu($assign->get_instance()->grade), $assign->get_instance()->grade > 0);
                    foreach ($assign_grades as $assign_grade){
                        $controller->render_grade($page, $assign_grade->id, [], '', false);
                    }
                }
            }
        }

        // Refresh grades from modules, plugins.
        $res = $grade_grade->grade_item->refresh_grades($grade_grade->userid);
        if ($res){
            // check regrading possibility anyway
            static::grade_item_check_regrade($cm_or_id, $grade_grade->userid, $grade_grade->grade_item);
            // update grade_grade cache
            static::_set_grade_grade($cm_or_id, $grade_grade);
            // refresh completion
            $completion = new \completion_info($cm->get_course());
            $completion->update_state($cm, COMPLETION_UNKNOWN, $grade_grade->userid);
        }

       return $res;
    }

    /**
     * Updates final grade value for given user
     * Skipped many checks and sets - if you don't sure, what you do, use \grade_item::update_final_grade() better
     *
     * @see \grade_item::update_final_grade()
     *
     * @param \cm_info|int      $cm_or_id - course module object or its id (it's better to use object)
     * @param ned_grade_grade   $grade
     * @param bool              $check_changes - if true, try to check changes from the old grade
     * @param bool              $call_events - if true, triggers grade events
     * @param bool              $update_grade_history - if false, uses other insert/update methods to not change grade history table
     * @param bool              $force_regrading - if true, call $force_regrading without checks
     *
     * @return bool - success
     */
    static public function grade_grade_update($cm_or_id, $grade, $check_changes=true, $call_events=true, $update_grade_history=true,
        $force_regrading=false){
        if (empty($grade)) return false;

        $overridden_changed = false;
        $excluded_changed = false;
        $source = static::$PLUGIN_NAME;
        $oldgrade = static::get_grade_grade($cm_or_id, $grade->userid, false);
        $grade->load_grade_item();
        $grade->timecreated = $grade->timecreated ?? time();

        if ($call_events || $update_grade_history){
            $grade->timemodified = $grade->timemodified ?? time(); // Hack alert - date graded.
        }

        if (empty($grade->id) || empty($oldgrade)) {
            $result = (bool)($update_grade_history ? $grade->insert($source) : $grade->raw_insert(true));

            // If the grade insert was successful and the final grade was not null then trigger a user_graded event.
            if ($result && !is_null($grade->finalgrade) && $call_events) {
                \core\event\user_graded::create_from_grade($grade)->trigger();
            }
        } else {
            // Existing grade_grades.
            $overridden_changed = (bool)$oldgrade->overridden != (bool)$grade->overridden;
            $excluded_changed = (bool)$oldgrade->excluded != (bool)$grade->excluded;

            do {
                if (!$check_changes) break;

                $gradechanged = $overridden_changed || $excluded_changed;
                if ($gradechanged) break;

                $gradechanged = ($grade->feedback ?? '') != ($oldgrade->feedback ?? '') ||
                    ($grade->feedbackformat ?? 0) != ($oldgrade->feedbackformat ?? 0);
                if ($gradechanged) break;

                $gradechanged = grade_floats_different($grade->finalgrade, $oldgrade->finalgrade) ||
                    grade_floats_different($grade->rawgrade, $oldgrade->rawgrade) ||
                    grade_floats_different($grade->rawgrademin, $oldgrade->rawgrademin) ||
                    grade_floats_different($grade->rawgrademax, $oldgrade->rawgrademax);
                if ($gradechanged) break;

                // no grade changes, pass
                return false;
            } while (false);

            $result = $update_grade_history ? $grade->update($source) : $grade->raw_update();

            // If the grade update was successful and the actual grade has changed then trigger a user_graded event.
            if ($result && $call_events && (!$check_changes || grade_floats_different($grade->finalgrade, $oldgrade->finalgrade))) {
                \core\event\user_graded::create_from_grade($grade)->trigger();
            }
        }

        if ($result){
            // Refresh grades from modules, plugins.
            static::grade_grade_refresh($cm_or_id, $grade->userid, $grade);

            /**
             * If 'overridden' was changed - we need force regrading
             * @see \gradereport_singleview\local\ui\override::set()
             */
            if ($overridden_changed || $force_regrading){
                $grade->grade_item->force_regrading();
            }

            /**
             * If 'excluded' was changed - we need force regrading for parent
             * @see \gradereport_singleview\local\ui\exclude::set()
             */
            if ($excluded_changed){
                $grade->grade_item->get_parent_category()->force_regrading();
            }

            // check regrading possibility anyway
            static::grade_item_check_regrade($cm_or_id, $grade->userid, $grade->grade_item);
        }

        // update grade_grade cache
        static::_set_grade_grade($cm_or_id, $grade);

        return $result;
    }

    /**
     * Get average grade by course
     * If you use kica and will check all users by course,
     *  it will make sense to use @see kica_util::kg_get_grades_by_course() first
     *
     * @param object|numeric    $courseorid
     * @param int|object        $user_or_id
     * @param int|bool          $precision - (optional)
     * @param bool              $percentage_value - (optional) if true, return value as float percentage
     * @param string|mixed      $def_null_value - (optional) default value to return, if grade is null
     *
     * @return float|int|string
     */
    static public function get_course_grade($courseorid, $user_or_id, $precision=2, $percentage_value=false, $def_null_value='-'){
        $courseid = static::get_id($courseorid);
        $userid = static::get_id($user_or_id);
        $kica = static::get_kica_enabled($courseid);
        if ($kica){
            $finalgrade = static::kg_get_course_average($courseid, $userid, C::FINALGRADE, $precision);
        } else {
            $course_grade = static::get_course_grade_grade($courseid, $userid, false);
            $finalgrade = $course_grade->finalgrade ?? null;
            if ($finalgrade && $percentage_value){
                $finalgrade = $finalgrade / $course_grade->get_grade_max() * 100;
            }
        }

        if (is_null($finalgrade)){
            return $def_null_value;
        } else {
            if (!is_bool($precision) || $precision){
                $finalgrade = round($finalgrade, $precision);
            }
            return $finalgrade;
        }
    }

    /**
     * @param object|\local_ned_controller\marking_manager\mm_data_by_activity_user|\local_ned_controller\marking_manager\mm_data_by_user $mm_data
     *
     * @return string
     */
    static public function get_status_by_mm_data($mm_data){
        $MM = static::$MM;
        $NGC = static::$ned_grade_controller;
        $status = C::STATUS_NOTATTEMPTED;
        if (empty($mm_data)){
            return $status;
        }

        /** @var \local_ned_controller\stdClass2|object|\local_ned_controller\marking_manager\mm_data_by_activity_user|\local_ned_controller\marking_manager\mm_data_by_user $st_mod */
        $st_mod = static::stdClass2($mm_data);
        $get_usual_complete = function($completed=C::STATUS_COMPLETED, $incompleted=C::STATUS_INCOMPLETED, $notattempted=C::STATUS_NOTATTEMPTED)
        use (&$st_mod){
            if ($st_mod->shouldpass){
                return $st_mod->completed_grade_successfully ? $completed : $incompleted;
            } else {
                return $st_mod->activity_completion ? $notattempted : $completed;
            }
        };
        $get_tag_choice = function($if_summative, $if_formative, $other=null) use (&$st_mod){
            $other = $other ?? $if_formative;
            if ($st_mod->is_formative){
                return $if_formative;
            } elseif ($st_mod->is_summative){
                return $if_summative;
            } else {
                return $other;
            }
        };

        if ($st_mod->excluded){
            $status = C::STATUS_EXCLUDED;
        } elseif ($st_mod->ngc_record && $st_mod->ngc_status == $NGC::ST_DONE && $st_mod->ngc_grade_type == $NGC::GT_AWARD_ZERO){
            $status = C::STATUS_NGC_ZERO;
        } elseif ($st_mod->completed_grade_successfully || ($st_mod->require_only_submit && $st_mod->{$MM::ST_SUBMITTED})){
            if ($st_mod->kica_activity){
                $status = $get_tag_choice(C::STATUS_GRADED_KICA_SUMMATIVE, C::STATUS_GRADED_KICA_FORMATIVE, C::STATUS_GRADED_KICA);
            } else {
                $status = C::STATUS_COMPLETED;
            }
        } elseif ($st_mod->kica_activity){
            if ($st_mod->{$MM::ST_UNMARKED}){
                $status = $get_tag_choice(C::STATUS_UNGRADED_KICA_SUMMATIVE, C::STATUS_UNGRADED_KICA_FORMATIVE, C::STATUS_UNGRADED_KICA);
            } elseif ($st_mod->{$MM::ST_MARKED} && $st_mod->kica_zerograde){
                $status = C::STATUS_KICA_ZEROGRADE;
            } elseif ($st_mod->{$MM::ST_MARKED} || $st_mod->{$MM::ST_SUBMITTED}){
                $status = $get_usual_complete(
                    $get_tag_choice(C::STATUS_GRADED_KICA_SUMMATIVE, C::STATUS_GRADED_KICA_FORMATIVE, C::STATUS_GRADED_KICA),
                    $get_tag_choice(C::STATUS_INCOMPLETED_SUMMATIVE, C::STATUS_INCOMPLETED_FORMATIVE, C::STATUS_INCOMPLETED)
                );
            }
        } else {
            if ($st_mod->{$MM::ST_UNMARKED}){
                $status = C::STATUS_WAITINGFORGRADE;
            } elseif ($st_mod->{$MM::ST_MARKED} || $st_mod->{$MM::ST_SUBMITTED}){
                $status = $get_usual_complete();
            }
        }

        if ($status == C::STATUS_NOTATTEMPTED && $st_mod->{$MM::ST_DRAFT}){
            $status = C::STATUS_DRAFT;
        }

        return $status;
    }

    /**
     * Return image by activity status, empty string for unknowing status
     *
     * @param object|mm\mm_data_by_activity_user|mm\mm_data_by_user $mm_data
     * @param string                                                $status (optional) NED Grade status, if already counted
     * @param bool                                                  $notattempt_is_ungraded (optional)
     * @param bool                                                  $check_full_course (optional) set true, if you check data by full course
     * @param \cm_info|object                                       $cm (optional) course-module object, can optimize deadline checks
     *
     * @return array
     */
    static public function get_ned_grade_icon($mm_data=null, $status=null, $notattempt_is_ungraded=false, $check_full_course=false, $cm=null){
        $icon = [];
        $add = null;
        $status = $status ?? static::get_status_by_mm_data($mm_data);
        $deadline = $mm_data->duedate ?? 0;

        switch($status){
            case C::STATUS_NGC_ZERO:
                if ($mm_data){
                    $icon = static::$ned_grade_controller::get_grade_status($mm_data->ngc_grade_type, $mm_data->ngc_reason, $deadline,
                        C::MOD_VALS[$mm_data->modname] ?? 0);

                    $icon[C::ICON_URL_DATA] = [
                        C::PAR_ID => $mm_data->ngc_record ?? 0,
                        C::ICON_URL_AI_ID => $mm_data->ngc_relatedid ?? 0,
                    ];
                    $icon['title'] = static::$ned_grade_controller::get_human_type_reason_name($mm_data->ngc_grade_type, $mm_data->ngc_reason);
                } else {
                    $add = C::STATUS_KICA_ZEROGRADE;
                }
                break;
            case C::STATUS_GRADED_KICA:
            case C::STATUS_GRADED_KICA_SUMMATIVE:
                $add = C::STATUS_GRADED_KICA_SUMMATIVE; break;
            case C::STATUS_COMPLETED:
            case C::STATUS_GRADED_KICA_FORMATIVE:
            case C::STATUS_MARKED:
                $add = C::STATUS_GRADED_KICA_FORMATIVE; break;
            case C::STATUS_WAITINGFORGRADE:
            case C::STATUS_SUBMITTED:
            case C::STATUS_UNMARKED:
            case C::STATUS_UNGRADED_KICA_FORMATIVE:
                $add = C::STATUS_UNGRADED_KICA_FORMATIVE; break;
            case C::STATUS_UNGRADED_KICA:
            case C::STATUS_UNGRADED_KICA_SUMMATIVE:
                $add = C::STATUS_UNGRADED_KICA_SUMMATIVE; break;
            case C::STATUS_INCOMPLETED:
            case C::STATUS_INCOMPLETED_FORMATIVE:
                $add = C::STATUS_INCOMPLETED_FORMATIVE; break;
            case C::STATUS_INCOMPLETED_SUMMATIVE:
                $add = C::STATUS_INCOMPLETED_SUMMATIVE; break;
            case C::STATUS_NOTATTEMPTED:
            case C::STATUS_UNSUBMITTED:
                $add = $notattempt_is_ungraded ? C::STATUS_NONE : C::STATUS_UNSUBMITTED; break;
            case C::STATUS_KICA_ZEROGRADE:
            case C::STATUS_SOFT_ZERO:
            case C::STATUS_HARD_ZERO:
            case C::STATUS_EXCLUDED:
            case C::STATUS_DEADLINE_PAST:
            case C::STATUS_DRAFT:
            case C::STATUS_RESUBMISSION:
            case C::STATUS_EXTENSION:
                $add = $status; break;
            default:
            case C::STATUS_PASSED:
                $add = C::STATUS_NONE; break;
        }

        if (empty($icon) && empty($add)){
            $add = C::STATUS_NONE;
        }
        if ($add){
            $icon = [$add => $add] + $icon;
        }

        if (!$mm_data){
            return $icon;
        }

        if ($mm_data->count_missed_deadline ?? 0){
            $icon[C::STATUS_DEADLINE_PAST] = C::STATUS_DEADLINE_PAST;
        }
        if (($mm_data->is_summative ?? false) && ($mm_data->attempt ?? 0)){
            $icon[C::STATUS_RESUBMISSION] = C::STATUS_RESUBMISSION;
        }

        $icon = static::ned_grade_icon_check_dm_extension($icon, $cm ?: $mm_data->cmid, $mm_data->userid, $deadline, $mm_data, $check_full_course);

        return $icon;
    }

    /**
     * Add extension status to icon, if need it
     * About Ned grade icon {@see get_ned_grade_icon()}
     *
     * @param array                                                      $icon              Ned grade icon
     * @param \cm_info|object|numeric                                    $cm_or_id
     * @param numeric|object                                             $user_or_id
     * @param numeric|null                                               $deadline          - (optional) activity deadline for $user_or_id, UNIX time
     * @param null|object|mm\mm_data_by_activity_user|mm\mm_data_by_user $mm_data           - it can speed up some statuses checks
     * @param bool                                                       $check_full_course (optional) set true, if you check data by full course
     *
     * @return array
     */
    static public function ned_grade_icon_check_dm_extension($icon, $cm_or_id, $user_or_id, $deadline=null, $mm_data=null, $check_full_course=false){
        $icon = $icon ?: [];
        if (empty($icon) || isset($icon[C::STATUS_EXTENSION])) return $icon;

        if (static::can_add_dm_extension_to_ned_grade_icon($cm_or_id, $user_or_id, $deadline, $mm_data, $check_full_course)){
            $icon[C::STATUS_EXTENSION] = C::STATUS_EXTENSION;
        }

        return $icon;
    }

    /**
     * Return, can this icon to have "add extension" status or not
     * @see ned_grade_icon_check_dm_extension()
     *
     * For extension rules {@see \block_ned_teacher_tools\mod\deadline_manager_mod::get_extension_data()}
     * Exception from DM rules:
     *  don't show extension icon, if deadline haven't passed yet (even if user has capability to grant extension)
     *
     * @param \cm_info|object|numeric                                    $cm_or_id
     * @param numeric|object                                             $user_or_id
     * @param numeric|null                                               $deadline          - (optional) activity deadline for $user_or_id, UNIX time
     * @param null|object|mm\mm_data_by_activity_user|mm\mm_data_by_user $mm_data           - it can speed up some statuses checks
     * @param bool                                                       $check_full_course (optional) set true, if you check data by full course
     *
     * @return bool
     */
    static public function can_add_dm_extension_to_ned_grade_icon($cm_or_id, $user_or_id, $deadline=null, $mm_data=null, $check_full_course=false){
        // no DM or its module - no extension
        /** @var \block_ned_teacher_tools\deadline_manager $DM */
        $DM = static::get_DM();
        if (!$DM) return false;

        $check_submitted = true;
        // don't grant extension, if student has already submitted
        if (isset($mm_data->submitted)){
            if ($mm_data->submitted) return false;

            $check_submitted = false;
        }

        // we show icon only if deadline has passed
        if (empty($mm_data->count_missed_deadline)){
            $deadline = $deadline ?? $mm_data->duedate ?? static::get_deadline_by_cm($cm_or_id, $user_or_id, $mm_data->course ?? null);
            if ($deadline && $deadline > time()){
                return false;
            }
        }

        $cm = static::get_cm_by_cmorid($cm_or_id, $mm_data->course ?? null);
        if (!$cm) return false;

        $context = static::ctx($cm->course, null, null, IGNORE_MISSING);
        if (!$context) return false;

        if (!$DM::can_grader_add_extension_user_in_cm($cm, $user_or_id, null, $context,
            $deadline, $check_submitted, $check_full_course, $cm->course)){
            return false;
        }

        return true;
    }

    /**
     * Check some activities for completion status for some user
     * Some examples of rule using:
     * Get, that user has some incomplete activity - $wanted_status={@see COMPLETION_INCOMPLETE}, $logic_rule={@see C::LOGIC_ANY}
     * User well-passe all activities - $wanted_status=[{@see COMPLETION_COMPLETE}, {@see COMPLETION_COMPLETE_PASS}], $logic_rule={@see C::LOGIC_ALL}
     *
     * Note: function can return NULL, which means that it is impossible/meaningless to check the completion of the activities,
     *  but it can mean not the same as FALSE result
     *
     * @param object|numeric|null $course_or_id
     * @param object|numeric|null $user_or_id
     * @param int|object|array    $cms_or_ids       - activities to check for completion; if empty - load all cms from course
     * @param int|int[]           $wanted_status    - one or some of the {@see C::CM_COMPLETIONS} to check; default is {@see COMPLETION_INCOMPLETE}
     * @param int                 $logic_rule       - ANY|ALL activities should have $wanted_status; one of the {@see C::LOGIC_ANY_OR_ALL}
     * @param bool                $check_visibility - if true, skip check for the activities, which user can't see
     *
     * @return bool|null - null, if completion is off for course/user, true/false as result from checking activities completion
     */
    static public function check_cms_completion_by_course($course_or_id=null, $user_or_id=null, $cms_or_ids=[],
        $wanted_status=null, $logic_rule=C::LOGIC_ANY, $check_visibility=true){

        $course = static::get_chosen_course($course_or_id);
        $userid = static::get_userid_or_global($user_or_id);
        if (empty($cms_or_ids)){
            $cms_or_ids = static::get_course_cms($course, $user_or_id);
        } else {
            $cms_or_ids = static::val2arr($cms_or_ids);
        }

        $wanted_status = $wanted_status ?? COMPLETION_INCOMPLETE;
        $wanted_status = array_values(static::val2arr($wanted_status, false));
        $checked_statuses = array_combine($wanted_status, $wanted_status);

        $rule_any = ($logic_rule ?? C::LOGIC_ANY) == C::LOGIC_ANY;
        $rule_all = !$rule_any;

        $completion = new \completion_info($course);
        if (!$completion->is_enabled() || !$completion->is_tracked_user($userid)) return null;

        foreach ($cms_or_ids as $cm_or_id){
            $cm = static::get_cm_by_cmorid($cm_or_id, $course, $user_or_id);
            if (empty($cm) || $cm->completion == COMPLETION_TRACKING_NONE || $cm->deletioninprogress) continue;
            if ($check_visibility && !static::check_activity_visible_by_cm($cm)) continue;

            $completion_data = $completion->get_data($cm, true, $userid);
            $yes = isset($checked_statuses[$completion_data->completionstate]);
            if ($yes){
                if ($rule_any) return true;
            } elseif ($rule_all) {
                return false;
            }
        }

        return $rule_all;
    }

    /**
     * Check that any|all the resubmission timers more|less some time
     * Some examples:
     * User has some working resubmission timers - $compare_rule={@see C::COMPARE_MORE}, $logic_rule={@see C::LOGIC_ANY}
     * User has all finished resubmission timers - $compare_rule={@see C::COMPARE_LESS}, $logic_rule={@see C::LOGIC_ALL}
     *
     * Warning: Function doesn't any RM limits, as count of resubmissions per course or passing RM criteria.
     *  Here we check only availability RM on course, and value of RM timers, if they are exist.
     *  So, if there are some working RM timers (which you can check by this function), it doesn't mean that user really can resubmit.
     *
     * Note: function can return NULL, which means that it is impossible/meaningless to check the resubmission timers of the activities,
     *  but it can mean not the same as FALSE result
     *
     * @see /blocks/ned_student_menu/ask_to_resubmit.php - for resubmission logic
     * @see \block_ned_teacher_tools\render_activity_row() - here you also can find calculation of RM timers
     *
     * @param object|numeric|null $course_or_id
     * @param object|numeric|null $user_or_id
     * @param int                 $time             - timestamp for compare with, UNIX time; NOW by default
     * @param int                 $compare_rule     - check, that deadline more|less than $time; one of the {@see C::COMPARE_ML_OPTIONS}
     * @param int                 $logic_rule       - ANY|ALL activities should pass check; one of the {@see C::LOGIC_ANY_OR_ALL}
     * @param array               $cms_filter       - filter with cmids, if provided - check only them, otherwise check all activities from the RM
     * @param object|null         $tt_config        - TT config block, if already exists; if null - loads the new one
     * @param bool                $check_visibility - if true, skip check for the activities, which user can't see
     *
     * @return bool|null - null, if resubmission is off for course, true/false as result from checking activities resubmission timers
     */
    static public function resubmission_compare_dates($course_or_id=null, $user_or_id=null, $time=null,
        $compare_rule=C::COMPARE_MORE, $logic_rule=C::LOGIC_ANY, $cms_filter=[], $tt_config=null, $check_visibility=true){

        $course = static::get_chosen_course($course_or_id);
        $tt_config = $tt_config ?? static::get_site_and_course_block_config($course, C::TT_NAME);
        if (empty($tt_config->enableresubmissions)) return null;

        $resubmission_cmids = $tt_config->resubmission_assignments ??
            \local_ned_controller\tt_config_manager::get_enabled_resubmission_activities($course->id, false);
        if (empty($resubmission_cmids)) return null;

        $time = $time ?? time();
        $more = ($compare_rule ?? C::COMPARE_MORE) == C::COMPARE_MORE;
        $rule_any = ($logic_rule ?? C::LOGIC_ANY) == C::LOGIC_ANY;
        $rule_all = !$rule_any;

        if (empty($cms_filter)){
            $cms_or_ids = array_keys($resubmission_cmids);
        } else {
            $cms_or_ids = static::val2arr($cms_filter);
        }

        $userid = static::get_userid_or_global($user_or_id);
        if (min(count($cms_or_ids), count($resubmission_cmids)) > 10){
            // prepare whole data by course
            \local_ned_controller\grade_info::prepare_s($course->id, $userid);
            \local_ned_controller\mod_assign\assign_info::prepare_submission_info_records($course->id, $userid);
        }

        foreach ($cms_or_ids as $cm_or_id){
            $cm = static::get_cm_by_cmorid($cm_or_id, $course, $user_or_id);
            if (empty($cm) || empty($resubmission_cmids[$cm->id]) || $cm->deletioninprogress) continue;
            if ($check_visibility && !static::check_activity_visible_by_cm($cm)) continue;

            $submission_info = new \local_ned_controller\mod_assign\assign_info($cm, $userid);
            if (!$submission_info->exist || !$submission_info->first_last_grade_time) continue;

            $oldsubmission = $submission_info->get_needed_user_submission($userid);
            if (!$oldsubmission) continue;

            if (empty($tt_config->daysforresubmission)){
                // resubmission timers are unlimited -> all timers will be more than any time
                $yes = $more;
            } else {
                $resubmission_time_start = static::grade_get_midn_grade_time($submission_info->first_last_grade_time);
                $resubmission_duedate = $resubmission_time_start + ($tt_config->daysforresubmission * DAYSECS);
                $yes = $more ? ($resubmission_duedate > $time) : ($resubmission_duedate < $time);
            }

            if ($yes){
                if ($rule_any) return true;
            } elseif ($rule_all) {
                return false;
            }
        }

        return $rule_all;
    }

    /**
     * Check user for Final Evaluation completion in some course
     *
     * Rules:
     * • User should complete all activities with FE tags
     * • All DM deadlines should pass
     * • All RM timers should pass
     *
     * @param object|numeric|null $course_or_id
     * @param object|numeric|null $user_or_id
     * @param bool $ignore_cache - if true, don't check cached value
     *
     * @return bool
     */
    static public function is_course_final_evaluation_completed($course_or_id=null, $user_or_id=null, $ignore_cache=false){
        if (!static::is_tt_exists()) return false;

        $course = static::get_chosen_course($course_or_id);
        if (empty($course->id) || $course->id == SITEID) return false;

        $courseid = $course->id;
        $userid = static::get_userid_or_global($user_or_id);
        $now = time();
        $res = false;

        $cache = static::cache_get_shared_cache();
        $c_key = C::CACHE_SHARED_KEY_FE_COMPLETION.$courseid.'_'.$userid;
        $c_data = $ignore_cache ? null : $cache->get($c_key);
        if ($c_data && isset($c_data['timemodified']) && isset($c_data['res'])){
            if ($c_data['timemodified'] > $now){
                $res = $c_data['res'];
                // if completed saved in cache -> we still need to check TT settings, if not-completed -> return
                if (!$res) return false;
            }
        }

        do {
            $tt_config = static::get_site_and_course_block_config($course, C::TT_NAME);
            if (empty($tt_config->fe_coursecompletion)){
                $res = false;
                break;
            }


            if ($res) {
                // if we have cached completion - we can return now
                return true;
            }

            // Completion only for course students
            if (!static::user_is_student($userid, $courseid, null, null, null, false)) break;

            // Check that KICA is ready
            if (!static::is_kica_ready($course_or_id)) break;

            // Check Final Evaluation activities completion
            $fe_tag_names = $tt_config->fe_tags ?? [];
            if (!empty($fe_tag_names)){
                $cmids = static::cmids_get_by_tags($fe_tag_names, [], $course);
                if (!empty($cmids)){
                    $has_incomplete_cm = static::check_cms_completion_by_course($course, $userid, $cmids, COMPLETION_INCOMPLETE, C::LOGIC_ANY, false);
                    // some FE activities is incomplete -> course FE completion is failed
                    if ($has_incomplete_cm) break;
                }
            }

            // Check that all deadlines passed
            $DM = static::get_DM();
            if ($DM){
                $dm_entity = $DM::get_dm_entity($course, null, $userid);
                $has_upcoming_deadline = $dm_entity->compare_deadlines(null, C::COMPARE_MORE, C::LOGIC_ANY);
                // some deadlines have not passed yet -> no completion
                if ($has_upcoming_deadline) break;
            }

            // Check that all resubmissions timers passed
            $has_working_resubmission_timer = static::resubmission_compare_dates($course, $user_or_id, null,
                C::COMPARE_MORE, C::LOGIC_ANY, null, $tt_config);
            if ($has_working_resubmission_timer) break;

            // Final Evaluation complete!
            $res = true;
        } while (false);

        $c_data = ['timemodified' => ($now + HOURSECS), 'res' => $res];
        $cache->set($c_key, $c_data);

        return $res;
    }

    /**
     * Get count of student AIVs
     * Alias {@see \local_academic_integrity\infraction::get_user_aiv_count()}
     * If all time variables ($startdate, $enddate, $lastdays) are null, uses time period of the current NED school year
     *
     * @param object|numeric      $user_or_id   - student
     * @param object|numeric|null $course_or_id - filter by some course (otherwise count for all site)
     * @param numeric|null        $startdate    - count only after some date (UNIX time)
     * @param numeric|null        $enddate      - count only before some date (UNIX time)
     * @param numeric|null        $lastdays     - count only for some last days (num of days)
     * @param bool                $count_hidden - if true, count all AIVs, otherwise count only shown AIVs
     *
     * @return int|null - count of the AIVs, or null, if AI plugin doesn't exist
     */
    static public function ai_get_user_aiv_count($user_or_id, $course_or_id=null, $startdate=null, $enddate=null, $lastdays=null, $count_hidden=false){
        if (!static::is_ai_exists() || !method_exists('\local_academic_integrity\infraction', 'get_user_aiv_count')) return null;

        return \local_academic_integrity\infraction::get_user_aiv_count($user_or_id, $course_or_id, $startdate, $enddate, $lastdays, $count_hidden);
    }

    /**
     * Get student unapplied AIVs
     * Alias {@see \local_academic_integrity\infraction::get_user_unapplied_records()}
     *
     * @param object|numeric      $user_or_id   - student
     * @param object|numeric|null $course_or_id - filter by some course
     * @param object|numeric|null $cm_or_id     - filter by some activity
     * @param bool                $return_first - if true, return only one record (or null, if found nothing)
     *
     * @return array|object|null - AIV record(s)
     */
    static public function ai_get_user_unapplied_records($user_or_id, $course_or_id=null, $cm_or_id=null, $return_first=false){
        if (!static::is_ai_exists() || !method_exists('\local_academic_integrity\infraction', 'get_user_unapplied_records')) return null;

        return \local_academic_integrity\infraction::get_user_unapplied_records($user_or_id, $course_or_id, $cm_or_id, $return_first);
    }

    /**
     * Return, has user capability to view grades before midnight
     *
     * @param \context|object|numeric $course_or_id_or_context - course, or its id, or context
     *
     * @return bool
     */
    static public function cap_can_view_grades_before_midn($course_or_id_or_context=null){
        if (CLI_SCRIPT) return true;

        $ctx = static::course2ctx($course_or_id_or_context);
        $res = static::g_get(__FUNCTION__, [$ctx->id]);
        if (is_null($res)){
            $res = static::$C::has_capability('viewgradesbeforemidn', $ctx);
            static::g_set(__FUNCTION__, [$ctx->id], $res);
        }

        return $res;
    }

    /**
     * Get midnight grade time from the real grade time
     *
     * @param int $grade_time - it's better to provide first_last_grade_time here or gg->timemodified
     *
     * @return int|mixed
     */
    static public function grade_get_midn_grade_time($grade_time=0){
        if (empty($grade_time)) return 0;

        [, $end_grade_day] = static::get_day_start_end($grade_time, C::NED_TIMEZONE);
        return $end_grade_day;
    }

    /**
     * Return grade time shown to user, with checking capability of viewing grade before midnight
     * Note: capability is always checked only for global $USER, not for the $user_or_id from the parameters
     *
     * @param numeric|object      $cm_or_id       - activity to check, uses to get additional data when $grade_time or $has_capability is null
     * @param numeric|object|null $user_or_id     - user (or id) to check grade for, if null - uses global $USER
     * @param int $grade_time                     - (optional) original grade time,
     *                                            it's better to provide first_last_grade_time here or gg->timemodified,
     *                                            if $grade_time is null, then we load grade timemodified from the DB by $cm_or_id
     * @param null|bool $has_capability           - (optional) user has (or not) capability to view grade before midnight
     *                                            if null (default) - check capability by $cm_or_id; you can rewrite it by providing any other value
     *
     * @return int|null - return original grade time, or it's next midnight; null - if we can't find any information about activity or grade time
     */
    static public function grade_get_shown_midn_grade_time($cm_or_id=null, $user_or_id=null, $grade_time=null, $has_capability=null){
        $cm = null;
        $error_cm = function($param_name=''){
            static::debugging("[grade_get_shown_grade_time] You should provide \$cm_or_id or $param_name!");
            return null;
        };

        if (is_null($grade_time)){
            if (!$cm_or_id) return $error_cm('$grade_time');

            $cm = static::get_cm_by_cmorid($cm_or_id);
            if (!$cm) return null;

            $gg = static::get_grade_grade($cm_or_id, $user_or_id, false);
            if (!$gg) return null;

            /**
             * Note, that {@see \grade_grade::get_dategraded()} not quite grade time, but last grade modification time, see MDL-31379
             *
             * Maybe, it will be better to get first_last_grade_time based on the {@see static::sql_get_first_last_grade_join()}
             *  like {@see \local_ned_controller\mod_assign\assign::get_first_last_grade_time()},
             *  but it's too heavy to check by single activity for the lists of different users/activities
             */
            $grade_time = $gg->get_dategraded();
            if (empty($grade_time)) return null;
        }

        if (is_null($has_capability)){
            if (!$cm_or_id) return $error_cm('$has_capability');

            $cm = $cm ?? static::get_cm_by_cmorid($cm_or_id);
            /**
             * We can use $cm->context to check, but as we know, that function checks capability by course,
             *  then it will be better to provide course instead of cm context,
             *  especially in cases of checking several activities from the same course
             */
            $has_capability = static::cap_can_view_grades_before_midn($cm->course);
        }

        if ($has_capability){
            return $grade_time;
        } else {
            return static::grade_get_midn_grade_time($grade_time);
        }
    }

    /**
     * Check, that there is graded activity before midnight for checked user and viewer can't view grades before midnight
     * So, it returns false if there is no graded before midnight activity or viewer can view it
     *
     * Note: capability is always checked only for global $USER, not for the $user_or_id from the parameters
     *
     * @param numeric|object      $cm_or_id       - activity to check, uses to get additional data when $grade_time or $has_capability is null
     * @param numeric|object|null $user_or_id     - user (or id) to check grade for, if null - uses global $USER
     * @param int                 $grade_time     - (optional) original grade time,
     *                                            it's better to provide first_last_grade_time here or gg->timemodified,
     *                                            if $grade_time is null, then we load grade timemodified from the DB by $cm_or_id
     * @param null|bool           $has_capability - (optional) if null (default) - check capability by $cm_or_id;
     *                                            you can rewrite it by providing any other value
     *
     * @return bool|null - return true, if viewer can't view activity as graded before midn, otherwise return false,
     *                      or null if we can't find any information about activity or grade time
     */
    static public function grade_is_hidden_now_before_midn($cm_or_id=null, $user_or_id=null, $grade_time=null, $has_capability=null){
        $cm = null;
        $error_cm = function($param_name=''){
            static::debugging("[grade_is_hidden_before_midn] You should provide \$cm_or_id or $param_name!");
            return null;
        };

        /**
         * We can make all method based only on {@see static::grade_get_shown_midn_grade_time()],
         *  but it will be quicker to check capability first
         */
        if (is_null($has_capability)){
            if (!$cm_or_id) return $error_cm('$has_capability');

            $cm = static::get_cm_by_cmorid($cm_or_id);
            if (empty($cm)) return null;

            /**
             * We can use $cm->context to check, but as we know, that function checks capability by course,
             *  then it will be better to provide course instead of cm context,
             *  especially in cases of checking several activities from the same course
             */
            $has_capability = static::cap_can_view_grades_before_midn($cm->course);
        }

        // user have capability to view hidden grades – not need to check anything else
        if ($has_capability) return false;

        $grade_time = static::grade_get_shown_midn_grade_time($cm ?? $cm_or_id, $user_or_id, $grade_time, false);
        if (empty($grade_time)) return false;

        return time() < $grade_time;
    }

    /**
     * Get scale menu by scale ID
     *
     * @param numeric $scale_id
     *
     * @return array
     */
    static public function scale_get_menu_by_id($scale_id){
        if (empty($scale_id) || $scale_id < 0) return [];

        /** @var \grade_scale $scale */
        $scale = \grade_scale::fetch(['id' => $scale_id]);
        if (empty($scale)) return [];

        return static::scale_get_menu_by_scale($scale->scale);
    }

    /**
     * Return scale menu by scale record or scale->"scale" option
     * Based on {@see \make_menu_from_list()}, but without reverse
     *
     * @param object|string $scale - scale record or scale string option
     *
     * @return array
     */
    static public function scale_get_menu_by_scale($scale){
        if (is_object($scale)){
            $scale = $scale->scale ?? '';
        }
        if (empty($scale) || !is_string($scale)) return [];

        $array = explode(',', $scale);
        $menu = [];
        foreach ($array as $key => $item) {
            $menu[$key+1] = trim($item);
        }

        return $menu;
    }

    /**
     * Get scale item by grade value and scale menu
     *
     * @param array   $scale_menu - scale menu, you can get it by {@see scale_get_menu_by_scale()} or {@see \make_menu_from_list()}
     * @param numeric $grade_val  - numeric grade value for scaling
     *
     * @return string|null - return scale item from scale menu, or null if result can't be determined
     */
    static public function grade_get_scale_item_by_grade($scale_menu, $grade_val){
        if (empty($scale_menu) || is_null($grade_val)) return null;

        $grade = round($grade_val);
        if ($grade == -1) return null; // ungraded

        $min = array_key_first($scale_menu);
        $max = array_key_last($scale_menu);
        if ($min > $max){
            [$min, $max] = [$max, $min];
        }

        if ($grade <= $min) return $scale_menu[$min];
        if ($grade >= $max) return $scale_menu[$max];
        return $scale_menu[$grade] ?? null;
    }
}
