<?php
/**
 * @package    local_ned_controller
 * @subpackage shared
 * @category   NED
 * @copyright  2021 NED {@link http://ned.ca}
 * @author     NED {@link http://ned.ca}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace local_ned_controller\shared;
use local_kica\kica_item;
use local_kica\kica_grade;

defined('MOODLE_INTERNAL') || die();

/**
 * Trait kica_util
 *
 * To avoid recursive dependencies, don't use any others traits/classes/etc here
 *  - this trait only for using *by* something, not for depending on anything else.
 *
 * @package local_ned_controller\shared
 */
trait kica_util {
    use plugin_dependencies, moodle_util, data_util;

    /**
     * Get constant KICA keys
     *
     * @return string[]
     */
    static public function get_kica_keys(){
        return C::KICA_KEYS;
    }

    /**
     * Return KICA by courseid
     *
     * @param numeric|object $course_or_id
     *
     * @return null|\stdClass
     */
    static public function get_kica($course_or_id){
        global $DB;
        if (!static::is_kica_exists()) return null;

        $courseid = static::get_id($course_or_id);
        $res = static::g_get(__FUNCTION__, $courseid);

        if (is_null($res)){
            $res = $DB->get_record(C::KICA, ['courseid' => $courseid]) ?: false;
            static::g_set(__FUNCTION__, $courseid, $res);
        }

        return $res ?: null;
    }

    /**
     * Return KICA by courseid if it enabled
     *
     * @param numeric|object $course_or_id
     *
     * @return \stdClass|null|false
     */
    static public function get_kica_enabled($course_or_id){
        $kica = static::get_kica($course_or_id);
        if (!$kica){
            return null;
        } elseif ($kica->enabled ?? false){
            return $kica;
        }

        return false;
    }

    /**
     * Check kica calculation by course (id) or kica object
     *
     * @param numeric|object $course_or_id_or_kica
     *
     * @return bool|null - return null, if there is not kica or it hasn't calculation parameter
     */
    static public function is_kica_calculation_natural($course_or_id_or_kica){
        if (isset($course_or_id_or_kica->calculation)){
            $kica = $course_or_id_or_kica;
        } else {
            $kica = static::get_kica($course_or_id_or_kica);
            if (!isset($kica->calculation)){
                return null;
            }
        }

        return $kica->calculation == kica_grade::CALCULATION_NATURAL;
    }

    /**
     * Get kica groups (it's almost constant value, based on the KICA "gradesplit" parameter)
     *
     * @param numeric|object $course_or_id
     * @param bool           $only_keys - if true, return only keys
     *
     * @return array
     */
    static public function get_kica_groups($course_or_id, $only_keys=false){
        $kica = static::get_kica($course_or_id);
        $def = '-';

        switch ($kica->gradesplit ?? 0){
            default:
            case kica_grade::SPLIT_7030:
                $kica_groups = [
                    70 => $kica->nameofseventypercent ?? $def,
                    30 => $kica->nameofthirtypercent ?? $def,
                ];
                break;
            case kica_grade::SPLIT_702010:
                $kica_groups = [
                    70 => $kica->nameofseventypercent ?? $def,
                    20 => $kica->nameoftwentypercent ?? $def,
                    10 => $kica->nameoftenpercent ?? $def,
                ];
                break;
        }

        return $only_keys ? array_keys($kica_groups) : $kica_groups;
    }

    /**
     * Checked, that all kica items by course is completed
     *
     * @param numeric|object $course_or_id
     *
     * @return bool
     */
    static public function is_kica_ready($course_or_id){
        if (!static::is_kica_exists()) return false;

        $courseid = static::get_id($course_or_id);
        $res = static::g_get(__FUNCTION__, $courseid);
        if (is_null($res)){
            $kica_items = static::ki_get_all_by_course($courseid);
            $res = !empty($kica_items);
            foreach ($kica_items as $kica_item){
                if (empty($kica_item->id) || $kica_item->incomplete) {
                    $res = false;
                    break;
                }
            }

            static::g_set(__FUNCTION__, $courseid, $res);
        }

        return $res;
    }

    /**
     * Return KICA item by its id or full DB item
     * Alias @see kica_item::get_by_id()
     *
     * @param                $id
     * @param \stdClass|null $full_item
     *
     * @return kica_item|null
     */
    static public function ki_get_by_id($id, $full_item=null){
        if (!static::is_kica_exists()) return null;

        return kica_item::get_by_id($id, $full_item);
    }

    /**
     * Return KICA item by its id, if it exists
     * Alias @see kica_item::get_by_id_existing()
     *
     * @param numeric   $id
     * @param array     $params (optional) params for additional check
     *
     * @return kica_item|null
     */
    static public function ki_get_by_id_existing($id, $params=[]){
        if (!static::is_kica_exists()) return null;

        return kica_item::get_by_id_existing($id, $params);
    }

    /**
     * Get kica_item by grade_item or its id
     * Alias @see kica_item::get_by_grade_item()
     *
     * @param numeric|object|\grade_item $gi_item_or_id
     *
     * @return kica_item|null
     */
    static public function ki_get_by_grade_item($gi_item_or_id){
        if (!static::is_kica_exists()) return null;

        return kica_item::get_by_grade_item($gi_item_or_id);
    }

    /**
     * Get kica_item by course_module (cm) or its id
     * Alias @see kica_item::get_by_cm()
     *
     * @param numeric|object|\cm_info $cm_or_id
     * @param bool                    $only_loaded - check only from already loaded kica items
     *                                             useful, if you have already loaded all items by course, for example
     *
     * @return kica_item|null
     */
    static public function ki_get_by_cm($cm_or_id, $only_loaded=false){
        if (!static::is_kica_exists()) return null;

        return kica_item::get_by_cm($cm_or_id, $only_loaded);
    }

    /**
     * Return simple kica_item object, but load all of them for the course
     *  (so, if you will use all kica items by course, it will be quicker)
     *
     * Also, if you need filter by user visibility or KICA group, check static::get_all_items_by_course()
     * @see get_all_items_by_course()
     *
     * Alias @see kica_item::get_by_cm_and_course()
     *
     * @param numeric|object|\cm_info   $cm_or_id
     * @param numeric|object            $course_or_id (optional) course, if already loaded, otherwise it will use courseid from the cm
     *                                                it there is no $course_or_id, it will be better to sent course module (cm) object as $cm_or_id
     *
     * @return kica_item|null
     */
    static public function ki_get_by_cm_and_course($cm_or_id, $course_or_id=null){
        if (!static::is_kica_exists()) return null;

        return kica_item::get_by_cm_and_course($cm_or_id, $course_or_id);
    }

    /**
     * Return all kica_items by course at once
     * Alias @see kica_item::get_all_items_by_course()
     *
     * @param object|numeric   $course_or_id - course (or id) by which should be loaded kica items
     * @param numeric          $kica_group   - (optional) filter kica_items by kica group
     * @param array|int|object $users_or_ids - (optional) list of users or userids (or single user/id), who should see them,
     *                                       if null - check only global (default), otherwise check global AND these user(s)
     * @param bool             $only_load_cache - if true, only load exists items in the cache, and return all found KI keys
     *                                          NOTE: with this options in true, function ignore $kica_group and $users_or_ids parameters
     *
     * @return array|kica_item[]|int[] - array of kica_items or kica_items keys (if $only_load_cache is true)
     */
    static public function ki_get_all_by_course($course_or_id, $kica_group=null, $users_or_ids=[], $only_load_cache=false){
        if (!static::is_kica_exists()) return [];

        return kica_item::get_all_items_by_course($course_or_id, $kica_group, $users_or_ids, $only_load_cache);
    }

    /**
     * Return visibility data by kica items and users
     * Activity visibility checks much faster (around 10 times) when users - in outer circle, so,
     *  if in your case users should be in the inner circle, there will be more quickly to get visibility data in separate loop,
     *  for example by this function.
     * Alias @see kica_item::get_visibility_data()
     *
     * @param array|object[]|static[]   $kicaitems - list of kica items
     * @param array|object[]|numeric[]  $users_or_ids - list of users or users ids
     * @param bool                      $by_userid_itemid - (optional) if true, result array will be by userid and itemid
     * @param bool                      $unavailable_as_invisible - (optional) if true, than with false uservisible - return false
     * @param bool                      $check_global_visibility - (optional) check (or not), that global user can see it too
     *
     * @return array [$kicaitem_id => [$userid => true]] or [$userid => [$kicaitem_id => true]]
     */
    static public function ki_get_visibility_data($kicaitems, $users_or_ids, $by_userid_itemid=false,
        $unavailable_as_invisible=false, $check_global_visibility=false){
        if (!static::is_kica_exists()) return [];

        return kica_item::get_visibility_data($kicaitems, $users_or_ids, $by_userid_itemid, $unavailable_as_invisible, $check_global_visibility);
    }

    /**
     * Delete records from the kica grades table
     * Alias @see kica_grade::delete_records()
     *
     * @param array $params
     *
     * @return bool
     */
    static public function kg_delete_records($params=[]){
        if (!static::is_kica_exists()) return false;

        return kica_grade::delete_records($params);
    }

    /**
     * Get kica grade by id
     * Alias @see \local_kica\kica_grade::get_by_id()
     *
     * @param numeric $id
     *
     * @return kica_grade|null
     */
    static public function kg_get_by_id($id){
        if (!static::is_kica_exists()) return null;

        return kica_grade::get_by_id($id);
    }

    /**
     * Get kica grade by id
     * Alias @see kica_grade::get_by_userid_itemid()
     *
     * Get kica grade by userid and itemid
     *
     * @param numeric|object                    $user_or_id
     * @param kica_item|numeric|object|array    $item_or_id
     *
     * @return kica_grade|null
     */
    static public function kg_get_by_userid_itemid($user_or_id, $item_or_id){
        if (!static::is_kica_exists()) return null;

        return kica_grade::get_by_userid_itemid($user_or_id, $item_or_id);
    }

    /**
     * Get kica grade by userid and cmid
     * Alias @see kica_grade::get_by_userid_cmid()
     *
     * @param numeric|object            $user_or_id
     * @param \cm_info|object|numeric   $cm_or_id
     *
     * @return kica_grade|null
     */
    static public function kg_get_by_userid_cmid($user_or_id, $cm_or_id){
        if (!static::is_kica_exists()) return null;

        return kica_grade::get_by_userid_cmid($user_or_id, $cm_or_id);
    }

    /**
     * Get kica grade by userid and itemid
     * Alias @see kica_grade::get_grades_by_course()
     *
     * @param numeric|object    $course_or_id
     * @param array|int[]|int   $userids
     * @param bool              $by_userid_itemid - (optional) if true, result array will be by userid and itemid
     * @param bool              $only_load - (optional) if true, return empty array, if there no need to load more users grades
     *
     * @return array|object[][]|kica_grade[][] - [itemid => [userid => kica_grade]] or [userid => [itemid => kica_grade]]
     */
    public static function kg_get_grades_by_course($course_or_id, $userids, $by_userid_itemid=false, $only_load=false){
        if (!static::is_kica_exists()) return [];

        return kica_grade::get_grades_by_course($course_or_id, $userids, $by_userid_itemid, $only_load);
    }

    /**
     * Get average grade by KICA section, or array with averages result
     * Alias @see kica_grade::get_kica_average()
     *
     * @param numeric|object        $course_or_id
     * @param numeric|object        $user_or_id
     * @param null|string           $section - KICA section or 'finalgrade', if null|false - return all averages in array
     * @param null|numeric|string   $kicagroup - KICA group, if null|false - return all course kica groups in array
     * @param bool|int              $round - int precision or false, if not need it; NOTE: 0 (zero) $round is interpreted as TRUE
     * @param array|kica_grade[]    $kica_grades - kica grades to average calculate from, if null - load it
     *
     * @return array [kica_group => [key => average(or max)]],
     *                  where kica_group is int group or string "max" with max list
     *                  and keys - one of the KICAF keys (KICA keys + "finalgrade")
     */
    public static function kg_get_kica_average($course_or_id, $user_or_id, $section=null, $kicagroup=null, $round=2, $kica_grades=null){
        if (!static::is_kica_exists()) return [];

        return kica_grade::get_kica_average($course_or_id, $user_or_id, $section, $kicagroup, $round, $kica_grades);
    }

    /**
     * Get average grade by KICA section, form of result related of params
     * Alias @see kica_grade::get_kica_average_by_section() - you also can get raw data by this function
     *
     * @param numeric|object        $course_or_id
     * @param numeric|object        $user_or_id
     * @param null|string           $section - KICA section or 'finalgrade', if null|false - return all averages in array
     * @param null|numeric|string   $kicagroup
     * @param bool                  $get_max - if true, also return max values
     * @param bool|int              $round - int precision or false, if not need it; NOTE: 0 (zero) $round is interpreted as TRUE
     * @param array|kica_grade[]    $kica_grades - kica grades to average calculate from, if null - load it
     *
     * @return array|float|null :
     *          • list($res, $max) - if $get_max is true, otherwise only $res, where $res and $max are:
     *              • float|null - if $section and $kicagroup are provided;
     *              • array(key => average(or max)) - if $kicagroup is provided and $section is not;
     *              • array(kica_group => average(or max)) - if $section is provided and $kicagroup is not;
     *              • array(kica_group => [key => average(or max)]) - if $section and $kicagroup are not provided;
     *          Where key - one of the KICAF keys (KICA keys + "finalgrade")
     */
    public static function kg_get_kica_average_by_section($course_or_id, $user_or_id, $section=null, $kicagroup=null,
        $get_max=false, $round=false, $kica_grades=null){
        if (!static::is_kica_exists()){
            $res = ($section && $kicagroup) ? null : [];
            return $get_max ? [$res, $res] : $res;
        }

        return kica_grade::get_kica_average_by_section($course_or_id, $user_or_id, $section, $kicagroup, $get_max, $round, $kica_grades);
    }

    /**
     * Get course average grade by KICA section, or array with averages result
     * Alias @see kica_grade::get_course_average()
     *
     * @param numeric|object            $course_or_id
     * @param numeric|object            $user_or_id
     * @param null|string               $section - KICA section or 'finalgrade', if null|false - return all averages in array
     * @param bool|int                  $round - int precision or false, if not need it; NOTE: 0 (zero) $round is interpreted as TRUE
     * @param array|float[][]|null      $average_data - average kica grades [kica_group => [key => average(or max)]],
     *                                          you can get them from @see kica_util::kg_get_kica_average()
     * @param array|kica_grade[]|null   $kica_grades  - kica grades to average calculate from,
     *                                          make sense to send them only if $average_data is null, if $kica_grades null - load it
     *
     * @return array|float|null - float|null value for the provided $section, otherwise array [key => value],
     *                              where keys - one of the KICAF keys (KICA keys + "finalgrade")
     */
    public static function kg_get_course_average($course_or_id, $user_or_id, $section=null, $round=false, $average_data=null, $kica_grades=null){
        if (!static::is_kica_exists()) return $section ? null : [];

        return kica_grade::get_course_average($course_or_id, $user_or_id, $section, $round, $average_data, $kica_grades);
    }

    /**
     * Get course average grade by KICA section, or array with averages result
     * Alias @see kica_grade::get_course_average_calc()
     *
     * @param numeric|object  $course_or_id
     * @param array|float[][] $average_data - average kica grades [kica_group => [key => average(or max)]], from @see kica_grade::get_kica_average()
     * @param array           $calculated_average  - calculated grades from @see kica_grade::get_course_average()
     * @param bool|int        $round - int precision or false, if not need it; NOTE: 0 (zero) $round is interpreted as TRUE
     *
     * @return array|string[][] - data for grade-calculation template
     */
    public static function kg_get_course_average_calculation($course_or_id, $average_data, $calculated_average, $round=2){
        if (!static::is_kica_exists()) return [];

        return kica_grade::get_course_average_calculation($course_or_id, $average_data, $calculated_average, $round);
    }

    /**
     * Get link to the local_kicaquizgrading grading page (KICA Quiz)
     *
     * @param object|numeric $cm_or_id
     * @param object|numeric $user_or_id
     * @param object|numeric $course_or_id - (optional) course to load CM
     *
     * @return null|\moodle_url - URL to the local_kicaquizgrading page
     */
    public static function kq_get_kica_quiz_grader_link($cm_or_id=null, $user_or_id=null, $course_or_id=null){
        if (!class_exists('\\local_kicaquizgrading\\helper')) return null;

        $cm = static::get_cm_by_cmorid($cm_or_id, $course_or_id);
        if (empty($cm)) return null;

        $userid = static::get_id($user_or_id);
        if (empty($userid)) return null;

        return \local_kicaquizgrading\helper::get_kica_grader_link_from_cm($cm, $userid) ?: null;
    }

    /**
     * Clear kica cache from NED functions, kica_item and kica_grade
     */
    static public function kica_purge_cache(){
        if (!static::is_kica_exists()) return;

        $default_keys = [
            'get_kica',                 /** @see shared_lib::get_kica() */
            'is_kica_ready',            /** @see shared_lib::is_kica_ready() */
            'get_course_activities',    /** @see shared_lib::get_course_activities() */
        ];
        static::g_remove_functions_data($default_keys);

        kica_grade::clear_grades_cache();
        kica_item::clear_items_cache();
    }
}
