<?php
/**
 * @package    local_ned_controller
 * @subpackage shared
 * @category   NED
 * @copyright  2021 NED {@link http://ned.ca}
 * @author     NED {@link http://ned.ca}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 * @noinspection DuplicatedCode
 */

namespace local_ned_controller\shared;

use local_ned_controller\marking_manager as mm;

defined('MOODLE_INTERNAL') || die();

/**
 * Trait output
 *
 * @package local_ned_controller\shared
 * @mixin base_trait
 */
trait output {
    use plugin_dependencies, util;

    /**
     * Get global $OUTPUT
     *
     * @return \theme_boost\output\core_renderer|\theme_ned_clean_core_renderer|\bootstrap_renderer|\core_renderer|object
     */
    static public function O(){
        global $OUTPUT;
        return $OUTPUT;
    }

    /**
     * alias for O()
     * @see O()
     *
     * @return \theme_boost\output\core_renderer|\theme_ned_clean_core_renderer|\bootstrap_renderer|\core_renderer|object
     */
    static public function output(){
        return static::O();
    }

    /**
     * Get global $PAGE
     *
     * @return \moodle_page|object
     */
    static public function page(){
        global $PAGE;
        return $PAGE;
    }

    /**
     * Get $PAGE title and url
     * You need to call require_login() or $PAGE->set_context() before calling this method
     *
     * @param string                $title - string title or lang key of the current plugin
     * @param \moodle_url|string    $url - URL relative to $CFG->wwwroot or {@link moodle_url} instance
     *
     * @return void
     */
    static public function page_set_title($title='', $url=null){
        global $FULLME;
        $P = static::page();
        $title = static::str_check($title);
        $url = $url ?? $FULLME;

        $P->set_url($url);
        $P->set_title($title);
        $P->set_heading($title);

        $active_node = $P->settingsnav->find_active_node();
        if (!$active_node || $active_node->text != $title){
            $P->navbar->add($title, $url);
        }
    }

    /**
     * Render something through the global $OUTPUT
     *
     * @param mixed ...$args
     *
     * @return string
     */
    static public function render(...$args){
        return static::O()->render(...$args);
    }

    /**
     * Returns instance of page renderer
     *
     * @noinspection PhpReturnDocTypeMismatchInspection
     *
     * @param string $component name such as 'core', 'mod_forum' or 'qtype_multichoice'.
     * @param string $subtype optional subtype such as 'news' resulting to 'mod_forum_news'
     * @param string $target one of rendering target constants
     *
     * @return \renderer_base|\local_ned_controller\output\renderer|\block_ned_teacher_tools\output\renderer|\local_kica\output\renderer
     */
    static public function get_renderer($component=null, $subtype=null, $target=null){
        return static::page()->get_renderer($component ?: static::$PLUGIN_NAME, $subtype, $target);
    }

    /**
     * Get new top-parent custom_menu_
     * @see \local_ned_controller\output\custom_ned_menu\custom_ned_menu::new_menu()
     *
     * @param string         $raw_settings - initial raw text settings
     * @param bool           $use_units - set option of using units
     * @param numeric|object $course_or_id - course or its id (global by default)
     * @param numeric|object $user_or_id - user or its id (global by default)
     * @param bool   $init - if true (default), init all right now
     *
     * @return \local_ned_controller\output\custom_ned_menu\custom_ned_menu
     */
    public static function new_custom_menu_item($raw_settings='', $use_units=false, $course_or_id=null, $user_or_id=null, $init=true){
        return \local_ned_controller\output\custom_ned_menu\custom_ned_menu::new_menu($raw_settings, $use_units, $course_or_id, $user_or_id, $init);
    }

    /**
     * Return html fa (<i>) element with fa (and $class) class
     *
     * @param string|array $class
     * @param string $content
     * @param string $title
     * @param array  $attr
     *
     * @return string
     */
    static public function fa($class='', $content='', $title='', $attr=[]){
        $attr = array_merge(['class' => static::arr2str($class, 'icon fa'), 'aria-hidden' => 'true'], $attr);
        if (!empty($title)){
            $attr['title'] = $title;
        }
        return \html_writer::tag('i', $content, $attr);
    }

    /**
     * Return html link
     *
     * @param string|\moodle_url|array  $url_params - if it's array, that used [$url_text='', $params=null, $anchor=null]
     * @param string       $text
     * @param string|array $class
     * @param array        $attr
     * @param bool         $from_plugin
     *
     * @return string
     */
    static public function link($url_params='', $text='', $class='', $attr=[], $from_plugin=false){
        if ($url_params instanceof \moodle_url){
            $m_url = $url_params;
        } else {
            if (is_string($url_params)){
                [$t_url, $params, $anchor] = [$url_params, null, null];
            } else {
                [$t_url, $params, $anchor] = $url_params + ['', null, null];
            }
            if (!empty($t_url) && is_string($t_url)){
                $t_url = static::url($t_url, null, null, $from_plugin);
            }
            $m_url = new \moodle_url($t_url, $params, $anchor);
        }
        $attr['class'] = static::arr2str($class, $attr['class'] ?? '');

        return \html_writer::link($m_url, static::str_check($text), $attr);
    }

    /**
     * Return external html link
     *
     * @param string|\moodle_url|array  $url_params - if it's array, that used [$url_text='', $params=null, $anchor=null]
     * @param string       $text
     * @param string|array $class
     * @param array        $attr
     * @param bool         $from_plugin
     *
     * @return string
     */
    static public function ext_link($url_params='', $text='', $class='', $attr=[], $from_plugin=false){
        $attr = array_merge($attr, ['target' => '_blank']);
        return static::link($url_params, $text, $class, $attr, $from_plugin);
    }

    /**
     * Return html link looks like button
     *
     * @param string|\moodle_url|array  $url_params - if it's array, that used [$url_text='', $params=null, $anchor=null]
     * @param string       $text
     * @param string|array $class
     * @param bool         $primary
     * @param array        $attr
     * @param bool         $from_plugin
     *
     * @return string
     */
    static public function button_link($url_params='', $text='', $class='', $primary=false, $attr=[], $from_plugin=false){
        $class = static::val2arr($class);
        $class[] = 'btn';
        $class[] = $primary ? 'btn-primary' : 'btn-secondary';

        return static::link($url_params, $text, $class, $attr, $from_plugin);
    }

    /**
     * Return html row
     *
     * @param array  $cells
     * @param array|string $class
     * @param array  $attr
     *
     * @return \html_table_row
     */
    static public function row($cells=null, $class='', $attr=null){
        if (!is_null($cells) && !is_array($cells)){
            $cells = [$cells];
        }
        $row = new \html_table_row($cells);
        $row->attributes['class'] = static::arr2str($class);
        if ($attr){
            $row->attributes = array_merge($row->attributes, $attr);
        }
        return $row;
    }

    /**
     * Return html cell
     *
     * @param string|array $text
     * @param string|array $class
     * @param array  $attr
     *
     * @return \html_table_cell
     */
    static public function cell($text=null, $class='', $attr=null){
        if (is_array($text)){
            $text = static::arr2str($text, '', '');
        } elseif (!is_null($text) && !is_string($text)){
            $text = strval($text);
        }

        $cell = new \html_table_cell($text);
        $cell->attributes['class'] = static::arr2str($class);
        if ($attr){
            $cell->attributes = array_merge($cell->attributes, $attr);
        }

        return $cell;
    }

    /**
     * Return html_table table
     *
     * @param string|array  $class
     * @param string        $id
     * @param array         $head
     *
     * @return \html_table
     */
    static public function html_table($class='', $id=null, $head=[]){
        $table = new \html_table();
        if (!empty($class)){
            $table->attributes['class'] = static::arr2str($class);
        }
        if (!is_null($id)){
            $table->id = $id;
        }

        $table->head = $head ?: [];

        return $table;
    }

    /**
     * Render html_table table as html string
     *
     * @param \html_table   $table data to be rendered
     * @param string|array  $wrapper_class - if not empty, wrap table in element (div by default) with this class
     * @param bool|string   $add_wrapper - if true, wrap it in div; if string - wrap in it as html tag
     *
     * @return string HTML code
     */
    static public function render_table($table, $wrapper_class='', $add_wrapper=false){
        $add_wrapper = $add_wrapper || !empty($wrapper_class);
        $table->responsive = $add_wrapper === true && empty($wrapper_class);

        $t = \html_writer::table($table);
        if ($add_wrapper && !$table->responsive){
            /** @noinspection PhpConditionAlreadyCheckedInspection */
            if (is_string($add_wrapper)){
                return static::tag($add_wrapper, $t, $wrapper_class);
            } else {
                // if (!empty($wrapper_class))
                return static::div($t, $wrapper_class);
            }
            // else uses default 'table-responsive'
        }

        return $t;
    }

    /**
     * Outputs a tag with class, attributes and contents
     * @see \html_writer::tag()
     *
     * @param string       $tagname    The name of tag ('a', 'img', 'span' etc.)
     * @param string|array $content    What goes between the opening and closing tags
     * @param string|array $class
     * @param array        $attributes The tag attributes (array('src' => $url, 'class' => 'class1') etc.)
     *
     * @return string HTML fragment
     */
    public static function tag($tagname, $content='', $class='', $attributes=null){
        $content = static::arr2str($content, '', " ");
        $attributes = static::val2arr($attributes);
        $attributes['class'] = static::arr2str($class);
        return \html_writer::tag($tagname, $content, $attributes);
    }

    /**
     * Outputs an empty tag with class and attributes
     * @see \html_writer::empty_tag()
     *
     * @param string       $tagname    The name of tag ('input', 'img', 'br' etc.)
     * @param array        $attributes The tag attributes (array('src' => $url, 'class' => 'class1') etc.)
     * @param string|array $class
     *
     * @return string HTML fragment
     */
    public static function tag_empty($tagname, $attributes=null, $class=''){
        $attributes = static::val2arr($attributes);
        $attributes['class'] = static::arr2str($class);
        return \html_writer::empty_tag($tagname, $attributes);
    }

    /**
     * Outputs an opening tag with class and attributes
     * @see \html_writer::start_tag()
     *
     * @param string       $tagname    The name of tag ('a', 'img', 'span' etc.)
     * @param string|array $class
     * @param array        $attributes The tag attributes (array('src' => $url, 'class' => 'class1') etc.)
     *
     * @return string HTML fragment
     */
    public static function tag_start($tagname, $class='', $attributes=null){
        $attributes = static::val2arr($attributes);
        $attributes['class'] = static::arr2str($class);
        return \html_writer::start_tag($tagname, $attributes);
    }

    /**
     * Outputs an opening div with class and attributes
     * @see \local_ned_controller\shared\output::tag_start()
     *
     * @param string|array $class
     * @param array        $attributes The tag attributes (array('src' => $url, 'class' => 'class1') etc.)
     *
     * @return string HTML fragment
     */
    public static function div_start($class='', $attributes=null){
        return static::tag_start('div', $class, $attributes);
    }

    /**
     * Outputs a closing tag
     * Alias @see \html_writer::end_tag()
     *
     * @param string $tagname The name of tag ('a', 'img', 'span' etc.)
     *
     * @return string HTML fragment
     */
    public static function tag_end($tagname){
        return \html_writer::end_tag($tagname);
    }

    /**
     * Outputs a closing div
     * Alias @see \html_writer::end_tag()
     *
     * @return string HTML fragment
     */
    public static function div_end(){
        return \html_writer::end_tag('div');
    }

    /**
     * Outputs <p> with attributes and contents
     * @see tag()
     *
     * @param string|array $content What goes between the opening and closing tags
     * @param string|array $class
     * @param array $attributes The tag attributes (array('src' => $url, 'class' => 'class1') etc.)
     *
     * @return string HTML fragment
     */
    public static function html_p($content, $class='', $attributes=null){
        return static::tag('p', $content, $class, $attributes);
    }

    /**
     * Outputs <i> with attributes and contents
     * @see tag()
     *
     * @param string|array $content What goes between the opening and closing tags
     * @param string|array $class
     * @param array $attributes The tag attributes (array('src' => $url, 'class' => 'class1') etc.)
     *
     * @return string HTML fragment
     */
    public static function html_i($content, $class='', $attributes=null){
        return static::tag('i', $content, $class, $attributes);
    }


    /**
     *  Outputs <input> with attributes
     *
     * @param string $type       - type of input
     * @param string $value      - default value of input
     * @param array  $attributes - tag attributes (array('src' => $url, 'class' => 'class1') etc.)
     * @param string $class
     * @param string $id
     *
     * @return string
     */
    public static function html_input($type='text', $value='', $attributes=null, $class='', $id=''){
        $attributes = static::val2arr($attributes);
        $attributes['class'] = static::arr2str($class);
        $attributes['type'] = $type;
        $attributes['value'] = $value;
        if (!empty($id)){
            $attributes['id'] = $id;
        }

        return \html_writer::empty_tag('input', $attributes);
    }

    /**
     * Creates a <div> tag. (Shortcut function.)
     *
     * @param string|array  $content HTML content of tag
     * @param string|array  $class Optional CSS class (or classes as space-separated list)
     * @param array         $attributes Optional other attributes as array
     *
     * @return string HTML code for div
     */
    public static function div($content='', $class='', $attributes=null){
        return \html_writer::div(static::arr2str($content, '', "\n"), static::arr2str($class), $attributes);
    }

    /**
     * Creates a <div> tag (Shortcut function.) and echo it
     * Alias for @see \local_ned_controller\shared\output::div()
     *
     * @param string|array $content    HTML content of tag
     * @param string|array $class      Optional CSS class (or classes as space-separated list)
     * @param array        $attributes Optional other attributes as array
     *
     * @return void - echo string HTML code for div
     */
    public static function ediv($content='', $class='', $attributes=null){
        echo static::div($content, $class, $attributes);
    }

    /**
     * Creates a <br> tag. (Shortcut function.)
     *
     * @return string HTML code for br
     */
    public static function br(){
        return static::tag_start('br');
    }

    /**
     * Creates a <span> tag. (Shortcut function.)
     *
     * @param string|array  $content HTML content of tag
     * @param string|array  $class Optional CSS class (or classes as space-separated list)
     * @param array         $attributes Optional other attributes as array
     * @return string HTML code for span
     */
    public static function span($content, $class='', $attributes=null){
        return \html_writer::span(static::arr2str($content, '', "\n"), static::arr2str($class), $attributes);
    }

    /**
     * @param string $filename
     * @param string|array $class
     * @param string $plugin
     * @param array  $attr
     * @param string|null $alt
     *
     * @return string
     */
    static public function img($filename, $class='icon', $plugin='moodle', $attr=[], $alt=null){
        $plugin = $plugin ?? static::$PLUGIN_NAME;
        if ($filename instanceof \moodle_url){
            $url = $filename;
        } elseif (is_string($filename) && static::str_starts_with($filename,['http://', 'https://'])){
            $url = new \moodle_url($filename);
        } else {
            $url = static::$PLUGIN_URL.'/pix/'.$filename;
            if (!file_exists(static::$DIRROOT . $url)){
                if (file_exists(static::$DIRROOT . $filename)){
                    $url = $filename;
                } else {
                    $url = static::O()->image_url($filename, $plugin);
                }
            }
            $url = new \moodle_url($url);
        }

        $attr['class'] = static::arr2str($class, $attr['class'] ?? '');
        $alt = $alt ?? ($attr['alt'] ?? ($attr['title'] ?? ''));
        if ($alt && !isset($attr['title'])){
            $attr['title'] = $alt;
        }

        return \html_writer::img($url, $alt, $attr);
    }

    /**
     * Return HTML for a pix_icon.
     *
     * @param string $pix        short pix name
     * @param string $alt        alt attribute, can be string key for $plugin string
     * @param string $plugin     standard component name like 'moodle', 'mod_forum', etc., by default uses current plugin
     * @param array  $attributes html attributes
     *
     * @return string HTML fragment
     */
    static public function pix_icon($pix, $alt='', $plugin=null, $attributes=null){
        $plugin = $plugin ?? static::$PLUGIN_NAME;
        return static::O()->pix_icon($pix, static::str_check($alt, null, $alt, $plugin), $plugin, $attributes);
    }

    /**
     * Return ned_grade_element by status or ready icons
     * You should provide $icons or $status
     * You can provide also additional data like "title", "prefix" or "postfix" in the $icons array to add such information at the final element
     *
     * @param array|string $icons       (optional) data icons for the html element,
     *                                      if you didn't provide $icons, you should provide $status
     * @param string        $status     (optional) - it will not use, if you provided icons
     * @param array         $attributes (optional) more attributes for html element
     * @param array         $add_class  (optional) more classes for html element
     *
     * @return string - html element
     */
    static public function get_ned_grade_element($icons=[], $status='', $attributes=[], $add_class=[]){
        $attributes = static::val2arr($attributes);
        $add_class = static::val2arr($add_class);
        if (empty($icons)){
            $icons = static::get_ned_grade_icon(null, $status);
        } else {
            $icons = static::val2arr($icons);
        }

        $title = $icons['title'] ?? '';
        unset($icons['title']);
        $prefix = $icons['prefix'] ?? '';
        unset($icons['prefix']);
        $postfix = $icons['postfix'] ?? '';
        unset($icons['postfix']);

        $url_params = $icons[C::ICON_URL_DATA] ?? null;
        unset($icons[C::ICON_URL_DATA]);

        $main_st = reset($icons);
        $add_class[] = 'add-ned-grade-icon';
        $attributes['title'] = $title ?: ($main_st == C::STATUS_NONE ? '' : static::$C::str($main_st));
        $attributes['data-icons'] = static::arr2str($icons);

        $url = null;
        $span = static::span('', $add_class, $attributes);
        $res = $prefix.$span.$postfix;

        $show_ai_link = $show_ngc_link = false;
        if (static::$ned_grade_controller::has_any_see_capability()){
            if (!empty($url_params) && (isset($icons[C::ICON_TYPE_DEDUCTION]) || isset($icons[C::ICON_TYPE_ZERO]))){
                $show_ai_link = isset($icons[C::ICON_REASON_AI]) && !empty($url_params[C::ICON_URL_AI_ID]);
                $show_ngc_link = !empty($url_params[C::PAR_ID]);
            }
        }

        if ($show_ai_link || $show_ngc_link){
            $course_view = $url_params[C::PAR_COURSE_VIEW] ?? null;

            if ($show_ai_link){
                $url = static::ai_record_view_url($url_params[C::ICON_URL_AI_ID], $course_view);
            } elseif ($show_ngc_link){
                $url = static::ngc_record_view_url($url_params[C::PAR_ID], $course_view);
            }
        }

        if ($url){
            return static::link($url, $res, 'ned-grade-icon-link d-inline-flex align-items-center');
        }

        return $res;
    }

    /**
     * Get NED Grade Element by MM data
     * Alias {@see \local_ned_controller\shared\output::get_ned_grade_element()}
     *
     * @param object|mm\mm_data_by_activity_user|mm\mm_data_by_user $mm_data
     * @param bool                                                  $notattempt_is_ungraded (optional)
     * @param array                                                 $attributes             (optional) more attributes for html element
     * @param array                                                 $add_class              (optional) more classes for html element
     *
     * @return string - html element
     */
    static public function icon_get_nge_by_mm($mm_data, $notattempt_is_ungraded=false, $attributes=[], $add_class=[]){
        $icons = static::get_ned_grade_icon($mm_data, null, $notattempt_is_ungraded);
        return static::get_ned_grade_element($icons, '', $attributes, $add_class);
    }

    /**
     * Get NED Grade Element by MM data or NED grade status
     * You can provide both (then status rewrite status from MM data, but uses MM data as additional source of info)
     * But you need provide at leas MM data or NED grade status
     * Alias {@see \local_ned_controller\shared\output::get_ned_grade_icon()}
     * Alias {@see \local_ned_controller\shared\output::get_ned_grade_element()}
     *
     * @param object|null $mm_data                MM data
     * @param string|null $status             (optional) NED Grade status, if you wish rewrite status from MM data
     * @param bool   $notattempt_is_ungraded (optional)
     * @param array  $attributes             (optional) more attributes for html element
     * @param array  $add_class              (optional) more classes for html element
     *
     * @return string - html element
     */
    static public function icon_get_nge($mm_data=null, $status=null, $notattempt_is_ungraded=false, $attributes=[], $add_class=[]){
        $icons = static::get_ned_grade_icon($mm_data, $status, $notattempt_is_ungraded);
        return static::get_ned_grade_element($icons, $status, $attributes, $add_class);
    }

    /**
     * Get NED Grade Element by NED grade status
     * Alias {@see \local_ned_controller\shared\output::get_ned_grade_element()}
     *
     * @param string $status                 NED Grade status
     * @param bool   $notattempt_is_ungraded (optional)
     * @param array  $attributes             (optional) more attributes for html element
     * @param array  $add_class              (optional) more classes for html element
     *
     * @return string - html element
     */
    static public function icon_get_nge_by_status($status, $notattempt_is_ungraded=false, $attributes=[], $add_class=[]){
        $icons = static::get_ned_grade_icon(null, $status, $notattempt_is_ungraded);
        return static::get_ned_grade_element($icons, $status, $attributes, $add_class);
    }

    /**
     * Get icons and render element by ngc_record
     *
     * @param object|\local_ned_controller\support\ned_grade_controller_record $ngc_record
     * @param array $icons                - optional, icons data, if already counted
     * @param array $additional_icon_data - optional, additional icon data, if you need to send some data,
     *                                       but also need to get fresh $icons data
     *
     * @return string - html (or empty)
     */
    static public function get_ned_grade_icon_by_ngc($ngc_record, $icons=[], $additional_icon_data=[]){
        if (empty($ngc_record)){
            return '';
        }

        if (empty($icons)){
            $icons = static::$ned_grade_controller::get_grade_status($ngc_record->grade_type, $ngc_record->reason, $ngc_record->deadline, $ngc_record->cm_type);
            $icons = static::ned_grade_icon_check_dm_extension($icons, $ngc_record->cmid, $ngc_record->userid, $ngc_record->deadline);
        }
        $icons = array_merge($icons, $additional_icon_data);

        if (static::$ned_grade_controller::can_see_record($ngc_record)){
            if (empty($icons[C::ICON_URL_DATA])){
                $icons[C::ICON_URL_DATA] = [
                    C::PAR_ID => $ngc_record->id,
                    C::ICON_URL_AI_ID => $ngc_record->relatedid,
                ];
            }

            if (empty($icons['title'])){
                $icons['title'] = static::$ned_grade_controller::get_human_record_name($ngc_record);
            }
        }

        return static::get_ned_grade_element($icons);
    }

    /**
     * Get icon deduction postfix by ngc_record or its reason
     * Alias @see get_ned_grade_element()
     *
     * @param object|\local_ned_controller\support\ned_grade_controller_record|numeric $ngc_record_or_reason
     *
     * @return string - html (or empty)
     */
    static public function get_grade_deduction_postfix_by_ngc_or_reason($ngc_record_or_reason){
        if (empty($ngc_record_or_reason)){
            return '';
        }

        $NGC = static::$ned_grade_controller;
        $ngc_record = null;
        if (is_object($ngc_record_or_reason)){
            $ngc_record = $ngc_record_or_reason;
            if (empty($ngc_record->grade_change) || $ngc_record->grade_type != $NGC::GT_DEDUCTION){
                return '';
            }

            $reason = $ngc_record->reason;
        } else {
            $reason = $ngc_record_or_reason;
        }

        $icons = $NGC::get_grade_status($NGC::GT_DEDUCTION, $reason);
        if ($ngc_record){
            return static::get_ned_grade_icon_by_ngc($ngc_record, $icons);
        } else {
            return static::get_ned_grade_element($icons);
        }
    }

    /**
     * Generate url to view single NGC (or AI) record
     *
     * @param int                $id
     * @param bool               $course_view - display record in course context or not
     * @param \moodle_url|string $returnurl
     * @param bool               $for_ai - if true, return link for the AI record page
     *
     * @return \moodle_url|string
     */
    static public function ngc_record_view_url($id, $course_view=true, $returnurl=null, $for_ai=false){
        if (empty($id)) return '';

        $returnurl = $returnurl ?? static::get_current_url();
        $params = [static::PAR_ID => $id];
        if ($course_view === false){
            $params[static::PAR_COURSE_VIEW] = false;
        }
        if (!empty($returnurl)){
            $params[static::PAR_RETURN_URL] = $returnurl;
        }

        if ($for_ai && static::is_ai_exists()){
            return static::url(static::PAGE_AI_VIEW, $params);
        }

        return static::$ned_grade_controller::get_render()::get_url($params);
    }

    /**
     * Generate url to AI view page
     *
     * @param int $id - id record from {local_academic_integrity_inf} table
     * @param bool $course_view - display record in course context or not
     * @param \moodle_url|string $returnurl
     *
     * @return \moodle_url|string
     */
    static public function ai_record_view_url($id, $course_view=null, $returnurl=null){
        return static::ngc_record_view_url($id, $course_view, $returnurl, true);
    }

    /**
     * Get current requested url (in address line) if PAGE->url was set
     *
     * @return string
     */
    static public function get_current_url(){
        static $url;
        if (is_null($url)){
            if (static::page()->has_set_url()){
                $url = static::page()->url->out(false);
            } else {
                $url = '';
            }
        }

        return $url;
    }

    /**
     * Return name/icon for activity
     *
     * @param \cm_info|int|string $activity   - Id of course-module, database object or cm-info
     * @param numeric|string      $icon_size  - icon size for html attributes
     * @param bool                $only_icon  - return only activity icon, without text
     * @param array|string        $add_class  - additional classes, if result will be a link or has html wrapper
     * @param array               $add_params - additional html attributes, if result will be a link or has html wrapper
     *
     * @return string
     */
    static public function mod_link($activity, $icon_size=20, $only_icon=false, $add_class='', $add_params=[]){
        return static::cm_get_text_icon_link($activity, null, null, false, $only_icon, false, false, false,
            null, $icon_size, $add_class, $add_params);
    }

    /**
     * Return html checkbox as checkbox & link (the same to single selector)
     *
     * @param \moodle_url   $url
     * @param mixed|bool    $value
     * @param string        $name
     * @param string|null   $text - if null - check plugin language strings by $name
     * @param string|array  $class
     * @param array         $attributes
     *
     * @return string
     */
    static public function single_checkbox(\moodle_url $url, $value, $name, $text=null, $class='', $attributes=[]){
        if (!isset($attributes['id'])){
            $attributes['id'] = 'ned-single-checkbox-' . $name;
        }
        $url = new \moodle_url($url);
        $url->param($name, (int)(!$value));
        $attributes['onclick'] = "window.location.href = '{$url->out(false)}';";

        if (is_null($text)){
            $text = static::str_check($name);
        }
        $class = static::arr2str($class, $value ? 'checked' : 'unchecked');

        $checkbox =  \html_writer::checkbox($name, $value, $value, $text, $attributes);
        return \html_writer::div($checkbox, 'ned-single-checkbox ' . $class, ['id' => 'ned-single-checkbox']);
    }

    /**
     * Return html checkbox as checkbox & link (the same to single selector)
     *  use two labels (and two titles), looks like a changeable button
     *
     * @param \moodle_url $url
     * @param             $value
     * @param             $name
     * @param string      $text_on
     * @param string      $text_off
     * @param string      $class
     * @param string      $title_on
     * @param string      $title_off
     * @param array       $attributes
     *
     * @return string
     */
    static public function single_checkbox2(\moodle_url $url, $value, $name, $text_on='', $text_off='', $class='',
        $title_on='', $title_off='', $attributes=[]){

        $label = \html_writer::span($text_on, 'label-checked', ['title' => $title_on]) .
            \html_writer::span($text_off, 'label-unchecked', ['title' => $title_off]);
        if ($attributes && (!empty($title_on) || !empty($title_off))){
            unset($attributes['title']);
        }

        if (!isset($attributes['id'])){
            $attributes['id'] = 'ned-single-checkbox2-' . $name;
        }

        $url = new \moodle_url($url);
        $url->param($name, (int)(!$value));
        $attributes['onclick'] = "window.location.href = '{$url->out(false)}';";

        $class = static::arr2str($class, $value ? 'checked' : 'unchecked');
        $checkbox =  \html_writer::checkbox($name, $value, $value, $label, $attributes);
        return \html_writer::div($checkbox, 'ned-single-checkbox2 ' . $class, ['id' => 'ned-single-checkbox2']);
    }

    /**
     * Return string selector if $render=true, \single_select otherwise
     *
     * @param        $url
     * @param        $name
     * @param array  $options
     * @param string $selected
     * @param null   $label
     * @param array  $attributes
     * @param null   $formid
     * @param null   $nothing
     * @param bool   $render
     *
     * @return \single_select | string
     */
    static public function single_select($url, $name, $options=[], $selected='', $label=null, $attributes=[], $formid=null, $nothing=null, $render=true){
        $url = new \moodle_url($url);
        $url->remove_params($name);
        $select = new \single_select($url, $name, $options, $selected, $nothing, $formid);

        if ($label){
            $select->set_label($label);
        } elseif (array_key_exists('label', $attributes)){
            $select->set_label($attributes['label']);
            unset($attributes['label']);
        }

        if (isset($attributes['disabled'])){
            $select->disabled = true;
        }

        $select->class = $attributes['class'] ?? 'single-select2';
        if (isset($options[$selected])){
            $select->class .= " value-$selected";
        }

        $select->attributes = $attributes ?: [];

        return $render ? static::render($select) : $select;
    }

    /**
     * Return autocomplete element working as link (the same to single selector)
     *
     * @param \moodle_url|string $url
     * @param string             $name
     * @param array              $options
     * @param int|mixed          $value
     * @param string             $label
     * @param string|array       $class
     * @param array              $attributes
     *
     * @return string
     */
    static public function single_autocomplete($url, $name, $options, $value, $label, $class='', $attributes=[]){
        $label = static::span($label, 'single-autocomplete-label');
        if (!isset($attributes['id'])){
            $attributes['id'] = 'ned-single-autocomplete-' . $name;
        }
        $value = static::isset_key($options, $value);
        $attributes['data-value'] = $value;
        $editable = !($attributes['disabled'] ?? false);
        $editable = new \core\output\inplace_editable(static::CTRL, 'single_autocomplete',0, $editable,
            $options[$value] ?? null, $value);
        $editable->set_type_autocomplete($options, []);

        $url = new \moodle_url($url);
        $url->remove_params($name);
        static::js_call_amd(static::CTRL.'/single_autocomplete', 'init',
            ['name' => $name, 'url' => $url->out(false)]);

        $class = static::val2arr($class);
        $class[] = 'value-'.$value;
        $class[] = 'ned-single-autocomplete-'.$name;

        return static::div($label . static::render($editable), static::arr2str($class, 'ned-single-autocomplete'), $attributes);
    }

    /**
     * Return autocomplete course element working as link, to select course
     *
     * @param \moodle_url   $url
     * @param int           $courseid (optional)
     * @param array         $options (optional) options for selector, or course array
     * @param string        $name (optional) - name of url param, which result will have
     * @param string        $label (optional)
     *
     * @return string
     */
    static public function course_autocomplete($url, $courseid=0, $options=null, $name=null, $label=null){
        $name = $name ?? C::PAR_COURSE;
        $label = $label ?? static::$C::str('course');
        if (is_null($options)){
            $courses = static::get_grader_courses();
            $options = static::records2menu($courses, 'shortname');
        } else {
            $options = static::records2menu($options, 'shortname');
        }

        return static::single_autocomplete($url, $name, $options, $courseid, $label);
    }

    /**
     * Create html element to quick edit a title inline.
     * It is used for displaying an element that can be in-place edited by the user.
     * @see  \core\output\inplace_editable class for additional info
     *
     * Important: Callback {$component}_inplace_editable($itemtype, $itemid, $newvalue) must be present in the lib.php file of
     * the component or plugin. It must return instance of this class.
     *
     * Moodle example:
     * @link https://docs.moodle.org/dev/Inplace_editable
     *
     * @param string              $itemtype     type of the item inside the component - each component/plugin may implement multiple inplace-editable elements
     * @param int                 $itemid       identifier of the item that can be edited in-place
     * @param bool                $editable     whether this value is editable (check capabilities and editing mode), if false, only "displayvalue"
     *                                          will be displayed without anything else
     * @param string              $displayvalue what needs to be displayed to the user, it must be cleaned, with applied filters (call
     *                                          {@link format_string()}). It may be wrapped in a html link, contain icons or other decorations
     * @param string              $value        what needs to be edited - usually raw value from the database, it may contain multilang tags
     * @param \lang_string|string $edithint     hint (title) that will be displayed under the edit link
     * @param \lang_string|string $editlabel    label for the input element in the editing mode (for screenreaders)
     * @param bool                $render       - if true, return rendered html text, inplace_editable object otherwise
     *
     * @return \core\output\inplace_editable|string
     */
    static public function inplace_editable($itemtype, $itemid, $editable=true, $displayvalue=null, $value=null, $edithint=null, $editlabel=null,
        $render=false){
        $tmpl = new \core\output\inplace_editable(static::$PLUGIN_NAME, $itemtype, $itemid, $editable, $displayvalue, $value, $edithint, $editlabel);
        return $render ? static::render($tmpl) : $tmpl;
    }

    /**
     * Create html element to quick edit a title inline
     * Displaying an element that can be in-place edited by the user. Sets the element type to be a dropdown.
     *
     * For select element specifying $displayvalue is optional, if null it will
     * be assumed that $displayvalue = $options[$value].
     * However displayvalue can still be specified if it needs icons and/or
     * html links.
     *
     * Important: Callback {$component}_inplace_editable($itemtype, $itemid, $newvalue) must be present in the lib.php file of
     * the component or plugin. It must return instance of this class.
     *
     * If only one option specified, the element will not be editable.
     * @see \local_ned_controller\shared\output::inplace_editable - base inplace_editable element
     * @see \core\output\inplace_editable class for additional info
     * @see \core\output\inplace_editable::set_type_select()
     *
     * @param string              $itemtype     type of the item inside the component - each component/plugin may implement multiple inplace-editable elements
     * @param int                 $itemid       identifier of the item that can be edited in-place
     * @param bool                $options      associative array with dropdown options
     * @param bool                $editable     whether this value is editable (check capabilities and editing mode), if false, only "displayvalue"
     *                                          will be displayed without anything else
     * @param string              $value        what needs to be edited - usually raw value from the database, it may contain multilang tags
     * @param \lang_string|string $edithint     hint (title) that will be displayed under the edit link
     * @param \lang_string|string $editlabel    label for the input element in the editing mode (for screenreaders)
     * @param string              $displayvalue what needs to be displayed to the user, it must be cleaned, with applied filters (call
     *                                          {@link format_string()}). It may be wrapped in a html link, contain icons or other decorations
     * @param bool                $render       - if true, return rendered html text, inplace_editable object otherwise
     *
     * @return \core\output\inplace_editable|string
     */
    static public function inplace_editable_select($itemtype, $itemid, $options=[], $editable=true, $value=null, $edithint=null, $editlabel=null,
        $displayvalue=null, $render=false){
        $tmpl = static::inplace_editable($itemtype, $itemid, $editable, $displayvalue, $value, $edithint, $editlabel, false);
        $tmpl->set_type_select($options);
        return $render ? static::render($tmpl) : $tmpl;
    }

    /**
     * Create html element to quick edit a title inline
     * Displaying an element that can be in-place edited by the user. Sets the element type to be an autocomplete field
     *
     * Important: Callback {$component}_inplace_editable($itemtype, $itemid, $newvalue) must be present in the lib.php file of
     * the component or plugin. It must return instance of this class.
     *
     * @see \local_ned_controller\shared\output::inplace_editable - base inplace_editable element
     * @see \core\output\inplace_editable class for additional info
     * @see \core\output\inplace_editable::set_type_autocomplete()
     *
     * @param string              $itemtype     type of the item inside the component - each component/plugin may implement multiple inplace-editable elements
     * @param int                 $itemid       identifier of the item that can be edited in-place
     * @param bool                $options      associative array with dropdown options
     * @param bool                $editable     whether this value is editable (check capabilities and editing mode), if false, only "displayvalue"
     *                                          will be displayed without anything else
     * @param string              $value        what needs to be edited - usually raw value from the database, it may contain multilang tags
     * @param \lang_string|string $edithint     hint (title) that will be displayed under the edit link
     * @param \lang_string|string $editlabel    label for the input element in the editing mode (for screenreaders)
     * @param string              $displayvalue what needs to be displayed to the user, it must be cleaned, with applied filters (call
     *                                          {@link format_string()}). It may be wrapped in a html link, contain icons or other decorations
     * @param array               $attributes   associative array with attributes for autoselect field. See AMD module core/form-autocomplete.
     * @param bool                $render       - if true, return rendered html text, inplace_editable object otherwise
     *
     * @return \core\output\inplace_editable|string
     */
    static public function inplace_editable_autocomplete($itemtype, $itemid, $options=[], $editable=true, $value=null, $edithint=null, $editlabel=null,
        $displayvalue=null, $attributes=[], $render=false){
        $tmpl = static::inplace_editable($itemtype, $itemid, $editable, $displayvalue, $value, $edithint, $editlabel, false);
        $tmpl->set_type_autocomplete($options, $attributes);
        return $render ? static::render($tmpl) : $tmpl;
    }

    /**
     * Create html element to quick edit a title inline
     * Displaying an element that can be in-place edited by the user. Sets the element type to be a toggle
     *
     * For toggle element $editlabel is not used.
     * $displayvalue must be specified, it can have text or icons but can not contain html links.
     *
     * Important: Callback {$component}_inplace_editable($itemtype, $itemid, $newvalue) must be present in the lib.php file of
     * the component or plugin. It must return instance of this class.
     *
     * Toggle element can have two or more options.
     * @see \local_ned_controller\shared\output::inplace_editable - base inplace_editable element
     * @see \core\output\inplace_editable class for additional info
     * @see \core\output\inplace_editable::set_type_toggle()
     *
     * @param string              $itemtype     type of the item inside the component - each component/plugin may implement multiple inplace-editable elements
     * @param int                 $itemid       identifier of the item that can be edited in-place
     * @param bool                $options      toggle options as simple, non-associative array; defaults to array(0,1)
     * @param bool                $editable     whether this value is editable (check capabilities and editing mode), if false, only "displayvalue"
     *                                          will be displayed without anything else
     * @param string              $displayvalue what needs to be displayed to the user, it must be cleaned, with applied filters (call
     *                                          {@link format_string()}). It may be wrapped in a html link, contain icons or other decorations
     * @param string              $value        what needs to be edited - usually raw value from the database, it may contain multilang tags
     * @param \lang_string|string $edithint     hint (title) that will be displayed under the edit link
     * @param bool                $render       - if true, return rendered html text, inplace_editable object otherwise
     *
     * @return \core\output\inplace_editable|string
     */
    static public function inplace_editable_toggle($itemtype, $itemid, $options=[], $editable=true, $displayvalue=null, $value=null, $edithint=null,
        $render=false){
        $tmpl = static::inplace_editable($itemtype, $itemid, $editable, $displayvalue, $value, $edithint, null, false);
        $tmpl->set_type_toggle($options);
        return $render ? static::render($tmpl) : $tmpl;
    }

    /**
     * Create html element to quick edit a title inline.
     * Alias for @see \local_ned_controller\shared\output::inplace_editable() with reder = true
     * For more documentation, see the source function
     *
     * @param string              $itemtype     type of the item inside the component - each component/plugin may implement multiple inplace-editable elements
     * @param int                 $itemid       identifier of the item that can be edited in-place
     * @param bool                $editable     whether this value is editable (check capabilities and editing mode), if false, only "displayvalue"
     *                                          will be displayed without anything else
     * @param string              $displayvalue what needs to be displayed to the user, it must be cleaned, with applied filters (call
     *                                          {@link format_string()}). It may be wrapped in a html link, contain icons or other decorations
     * @param string              $value        what needs to be edited - usually raw value from the database, it may contain multilang tags
     * @param \lang_string|string $edithint     hint (title) that will be displayed under the edit link
     * @param \lang_string|string $editlabel    label for the input element in the editing mode (for screenreaders)
     *
     * @return string - rendered inplace_editable element
     */
    static public function inplace_editable_r($itemtype, $itemid, $editable=true, $displayvalue=null, $value=null, $edithint=null, $editlabel=null){
        return static::inplace_editable($itemtype, $itemid, $editable, $displayvalue, $value, $edithint, $editlabel, true);
    }

    /**
     * Create html element to quick edit a title inline
     * Displaying an element that can be in-place edited by the user. Sets the element type to be a dropdown.
     * Alias for @see \local_ned_controller\shared\output::inplace_editable_select() with reder = true
     * For more documentation, see the source function
     *
     * @param string              $itemtype     type of the item inside the component - each component/plugin may implement multiple inplace-editable elements
     * @param int                 $itemid       identifier of the item that can be edited in-place
     * @param bool                $options      associative array with dropdown options
     * @param bool                $editable     whether this value is editable (check capabilities and editing mode), if false, only "displayvalue"
     *                                          will be displayed without anything else
     * @param string              $value        what needs to be edited - usually raw value from the database, it may contain multilang tags
     * @param \lang_string|string $edithint     hint (title) that will be displayed under the edit link
     * @param \lang_string|string $editlabel    label for the input element in the editing mode (for screenreaders)
     * @param string              $displayvalue what needs to be displayed to the user, it must be cleaned, with applied filters (call
     *                                          {@link format_string()}). It may be wrapped in a html link, contain icons or other decorations
     *
     * @return string  - rendered inplace_editable element with select type
     */
    static public function inplace_editable_select_r($itemtype, $itemid, $options=[], $editable=true, $value=null, $edithint=null, $editlabel=null,
        $displayvalue=null){
        return static::inplace_editable_select($itemtype, $itemid, $options, $editable, $value, $edithint, $editlabel, $displayvalue, true);
    }

    /**
     * Create html element to quick edit a title inline
     * Displaying an element that can be in-place edited by the user. Sets the element type to be an autocomplete field
     * Alias for @see \local_ned_controller\shared\output::inplace_editable_autocomplete() with reder = true
     * For more documentation, see the source function
     *
     * @param string              $itemtype     type of the item inside the component - each component/plugin may implement multiple inplace-editable elements
     * @param int                 $itemid       identifier of the item that can be edited in-place
     * @param bool                $options      associative array with dropdown options
     * @param bool                $editable     whether this value is editable (check capabilities and editing mode), if false, only "displayvalue"
     *                                          will be displayed without anything else
     * @param string              $value        what needs to be edited - usually raw value from the database, it may contain multilang tags
     * @param \lang_string|string $edithint     hint (title) that will be displayed under the edit link
     * @param \lang_string|string $editlabel    label for the input element in the editing mode (for screenreaders)
     * @param string              $displayvalue what needs to be displayed to the user, it must be cleaned, with applied filters (call
     *                                          {@link format_string()}). It may be wrapped in a html link, contain icons or other decorations
     * @param array               $attributes   associative array with attributes for autoselect field. See AMD module core/form-autocomplete.
     *
     * @return string - rendered inplace_editable element with autocomplete type
     */
    static public function inplace_editable_autocomplete_r($itemtype, $itemid, $options=[], $editable=true, $value=null, $edithint=null,
        $editlabel=null, $displayvalue=null, $attributes=[]){
        return static::inplace_editable_autocomplete($itemtype, $itemid, $options, $editable, $value,
            $edithint, $editlabel, $displayvalue, $attributes, true);
    }

    /**
     * Create html element to quick edit a title inline
     * Displaying an element that can be in-place edited by the user. Sets the element type to be a toggle
     * Alias for @see \local_ned_controller\shared\output::inplace_editable_toggle() with reder = true
     * For more documentation, see the source function
     *
     * @param string              $itemtype     type of the item inside the component - each component/plugin may implement multiple inplace-editable elements
     * @param int                 $itemid       identifier of the item that can be edited in-place
     * @param bool                $options      toggle options as simple, non-associative array; defaults to array(0,1)
     * @param bool                $editable     whether this value is editable (check capabilities and editing mode), if false, only "displayvalue"
     *                                          will be displayed without anything else
     * @param string              $displayvalue what needs to be displayed to the user, it must be cleaned, with applied filters (call
     *                                          {@link format_string()}). It may be wrapped in a html link, contain icons or other decorations
     * @param string              $value        what needs to be edited - usually raw value from the database, it may contain multilang tags
     * @param \lang_string|string $edithint     hint (title) that will be displayed under the edit link
     *
     * @return string - rendered inplace_editable element with toggle type
     */
    static public function inplace_editable_toggle_r($itemtype, $itemid, $options=[], $editable=true, $displayvalue=null, $value=null, $edithint=null){
        return static::inplace_editable_toggle($itemtype, $itemid, $options, $editable, $displayvalue, $value, $edithint, true);
    }

    /**
     * Print (echo) element or add it to the page heading menu (if possible)
     *
     * @param string $elem
     */
    static public function print_or_add_headingmenu($elem){
        global $PAGE;
        if ($PAGE->headerprinted){
            echo $elem;
        } else {
            $headingmenu = $PAGE->headingmenu;
            if (is_null($headingmenu)){
                $headingmenu = '';
            }

            $PAGE->set_headingmenu($headingmenu . $elem);
        }
    }

    /**
     * Shortcut for \local_academic_integrity\ai_flag::flag
     * @see \local_academic_integrity\ai_flag::flag()
     *
     * @param object|numeric|null $user_or_id The user to get the academic integrity for, uses current $USER if null
     *
     * @return string - the HTML fragment
     */
    static public function get_ai_flag($user_or_id){
        if (!static::is_ai_flag_exists()) return '';

        return \local_academic_integrity\ai_flag::flag($user_or_id).
            \report_ghs\helper::grade_discrepancy_flag($user_or_id);
    }

    /**
     * Return translated AI user status
     * Alias {@see \local_academic_integrity\ai_flag::status()}
     *
     * @param object|numeric|null $user_or_id The user to get the academic integrity for, uses current $USER if null
     *
     * @return string
     */
    static public function get_ai_flag_status($user_or_id){
        if (!static::is_ai_flag_exists()) return '';

        return \local_academic_integrity\ai_flag::status($user_or_id, true);
    }

    /**
     * Profile context menu
     * @see \local_ned_controller\output\core_user\myprofile_renderer::get_profile_context_menu()
     *
     * @param numeric|object               $user_or_id   User or its ID
     * @param numeric|object|null          $course_or_id Course or its ID
     * @param numeric|object|\cm_info|null $cm_or_id     Course Module object or ID
     *
     * @return string
     */
    static public function get_profile_context_menu($user_or_id, $course_or_id=null, $cm_or_id=null){
        [$userid, $courseid, $cmid] = static::get_ids($user_or_id, $course_or_id, $cm_or_id);
        $courseid = $courseid ?: SITEID;

        $res = static::g_get(__FUNCTION__, [$userid, $courseid, $cmid]);
        if (is_null($res)){
            $res = \local_ned_controller\output\core_user\myprofile_renderer::get_profile_context_menu($userid, $courseid, $cm_or_id);
            static::g_set(__FUNCTION__, [$userid, $courseid, $cmid], $res);
        }

        return $res;
    }

    /**
     * Return profile context menu with academic integrity flag and some string (username, for example) between
     *
     * @param numeric|object $user_or_id
     * @param numeric|object $course_or_id
     * @param string         $insert_in        - string, which you can insert between ai_flag and profile_menu
     * @param bool           $use_ai_flag      - add ai_flag
     * @param bool|string    $use_container    - if true, insert result into the container, if not empty string - uses it as container class
     * @param bool           $use_profile_menu - if true, add user menu to the result inside container
     *
     * @return string
     */
    static public function get_profile_with_menu_flag($user_or_id, $course_or_id, $insert_in='',
            $use_ai_flag=true, $use_container=false, $use_profile_menu=true){
        $res = '';
        if ($use_ai_flag){
            $res .= static::get_ai_flag($user_or_id);
        }
        if (!empty($insert_in)){
            $res .= $insert_in;
        }

        if ($use_profile_menu){
            $res .= static::get_profile_context_menu($user_or_id, $course_or_id);
        }

        if ($use_container){
            $class = is_string($use_container) ? $use_container : '';
            $res = static::span($res, $class);
        }

        return $res;
    }

    /**
     * Return html link to the course
     *
     * @param numeric|object $courseorid - Course (or its id)
     * @param bool           $shortname - use course shortname (otherwise - fullname)
     * @param bool           $simple - return only text name
     *
     * @return string html link or text name
     */
    static public function q_course_link($courseorid, $shortname=true, $simple=false){
        $course = static::get_chosen_course($courseorid, false);
        $name = $shortname ? $course->shortname : $course->fullname;
        if ($simple || !static::course_can_view_course_info($course)){
            return $name;
        }
        return static::link(['/course/view.php', ['id' => $course->id]], $name, '', ['title' => $course->fullname]);
    }

    /**
     * Return html icon+name of course module
     *
     * @param \cm_info|int|string   $cm_or_id   ID of course-module, or database object
     * @param numeric|object        $courseorid Optional course object (or its id) if already loaded
     * @param numeric|object        $userorid   Optional user object (or its id; default = current)
     * @param bool                  $only_name  Optional return only text name
     *
     * @return string html
     */
    static public function q_cm_name($cm_or_id, $courseorid=null, $userorid=null, $only_name=false){
        return static::cm_get_text_icon_link($cm_or_id, $courseorid, $userorid, $only_name, false, true, true);
    }

    /**
     * Return html link to the course module
     *
     * @param \cm_info|int|string   $cm_or_id   ID of course-module, or database object
     * @param numeric|object        $courseorid Optional course object (or its id) if already loaded
     * @param numeric|object        $userorid   Optional user object (or its id; default = current)
     * @param bool                  $simple     Optional return only text name
     *
     * @return string
     */
    static public function q_cm_link($cm_or_id, $courseorid=null, $userorid=null, $simple=false){
        return static::cm_get_text_icon_link($cm_or_id, $courseorid, $userorid, $simple, false, $simple, $simple);
    }

    /**
     * Return activity (cm) link for student view
     * If student can't access activity, the special icon will be added (also for teacher view)
     *
     * @param \cm_info|int|string   $cm_or_id  ID of course-module, or database object
     * @param numeric|object        $userorid_for_grade Student, who should be graded
     * @param numeric|object        $course_or_id Course if already loaded, improves optimization if $cm_or_id is represented as ID
     *
     * @return string
     */
    static public function q_cm_student_link($cm_or_id, $userorid_for_grade=null, $course_or_id=null){
        $cm = static::get_cm_by_cmorid($cm_or_id, $course_or_id);
        if (!$cm) return '';

        $add_text = static::render_cm_unavailable_info($cm, $userorid_for_grade, true, false);
        return static::cm_get_text_icon_link($cm, null, null, false, false, false, false, $add_text);
    }

    /**
     * Return html link to the MM or Moodle-grader page
     *
     * @param \cm_info|int|string $cm_or_id           - ID of course-module, or database object
     * @param numeric|object      $userorid_for_grade - Student, who should be graded
     * @param object|numeric      $courseorid         - optional course (or its id) if loaded, improves optimization if $cm_or_id is represented as ID
     * @param bool                $use_moodle_grading - if true, create link to the Moodle-grader (link to the MM-page otherwise)
     * @param bool                $new_tab            - if true, link will be opened in the new tab
     *
     * @return string - HTML link (<a>)
     */
    static public function q_cm_grade_link($cm_or_id, $userorid_for_grade=null, $courseorid=null, $use_moodle_grading=false, $new_tab=false){
        $cm = static::get_cm_by_cmorid($cm_or_id, $courseorid);
        if (!$cm) return '';

        $studentid = static::get_id($userorid_for_grade);
        if ($use_moodle_grading){
            $grader_url = static::cm_get_grader_url($cm, $userorid_for_grade, $cm->course, [C::PAR_USE_KICA_QUIZ => 1]);
        } else {
            $grader_url = new \moodle_url(C::PAGE_MM,
                [C::PAR_COURSE => $cm->course, C::PAR_SETUSER => $studentid, static::$MM::P_GETSHOW => 1, static::$MM::P_MID => $cm->id]);
        }

        $html_attributes = [];
        if ($new_tab){
            $html_attributes['target'] = '_blank';
        }

        return static::cm_get_text_icon_link($cm, null, null, false, false, false,
            false, false, $grader_url, null, [], $html_attributes);
    }

    /**
     * Return html link to the user profile, without any additional supplements
     * For the student user link {@see static::q_user_any_link()}
     *
     * @param numeric|object      $user_or_id
     * @param numeric|object|null $course_or_id
     * @param bool|string         $container - if true, insert result into the container, if not empty string - uses it as container class
     * @param bool                $simple    - if true, return only text name
     * @param string              $link_class - class for the link (<a>)
     *
     * @return string
     */
    static public function q_user_any_link($user_or_id, $course_or_id=null, $container='inline-block', $simple=false, $link_class='username'){
        if (is_object($user_or_id)){
            $user = $user_or_id;
        } else {
            $user = static::get_user($user_or_id);
        }
        if (!$user){
            return static::$C::str('none');
        }

        if ($simple){
            return fullname($user);
        }

        $courseid = static::get_id($course_or_id ?: SITEID);
        $user_params = ['id' => $user->id];
        if (static::course_can_view_course_info($courseid)){
            $user_params['course'] = $courseid;
        }
        $res = static::link(['/user/view.php', $user_params], fullname($user), $link_class);

        if ($container){
            $class = is_string($container) ? $container : '';
            $res = static::span($res, $class);
        }

        return $res;
    }

    /**
     * Return html link to the user profile, with additional student features: NED action menu and AIV flag
     * For clear user link {@see static::q_user_any_link()}
     *
     * @param numeric|object      $user_or_id
     * @param numeric|object|null $course_or_id
     * @param bool|string         $container        - if true, insert result into the container, if not empty string - uses it as container class
     * @param bool                $simple           - if true, return only text name
     * @param bool                $use_ai_flag      - add AIV flag to the username
     * @param bool                $use_profile_menu - if true, add user menu to the username
     *
     * @return string
     */
    static public function q_user_link($user_or_id, $course_or_id=null, $container='inline-block', $simple=false,
            $use_ai_flag=true, $use_profile_menu=true){
        $user_link = static::q_user_any_link($user_or_id, $course_or_id, false, $simple);
        if ($simple){
            return $user_link;
        }

        return static::get_profile_with_menu_flag($user_or_id, $course_or_id, $user_link, $use_ai_flag, $container, $use_profile_menu);
    }

    /**
     * Return html link to the user profile, with additional student features: NED action menu and AIV flag
     * Alias for the {@see static::q_user_link()}
     * For clear user link {@see static::q_user_any_link()}
     *
     * @param numeric|object      $user_or_id
     * @param numeric|object|null $course_or_id
     * @param bool|string         $container        - if true, insert result into the container, if not empty string - uses it as container class
     * @param bool                $simple           - if true, return only text name
     * @param bool                $use_ai_flag      - add AIV flag to the username
     * @param bool                $use_profile_menu - if true, add user menu to the username
     *
     *
     * @return string
     */
    static public function q_student_link($user_or_id, $course_or_id=null, $container='inline-block', $simple=false,
            $use_ai_flag=true, $use_profile_menu=true){
        return static::q_user_link($user_or_id, $course_or_id, $container, $simple, $use_ai_flag, $use_profile_menu);
    }

    /**
     * Return user fullnames by role as array or join in the one string
     *
     * @param string        $rolename - short role name
     * @param numeric       $courseid
     * @param numeric       $groupid
     * @param numeric       $cmid
     * @param bool          $linkable - if true, user fullnames will be strings
     * @param bool|string   $join - if non-empty string, join result by it, otherwise if true - join by ', '
     *
     * @return array|string
     */
    public static function get_role_users_fullnames($rolename, $courseid=0, $groupid=0, $cmid=0, $linkable=false, $join=false){
        $user_names = array();
        $users = static::get_users_by_role($rolename, $courseid, $cmid, $groupid, false, 'u.*');
        foreach ($users as $user){
            $user_names[$user->id] = static::q_user_link($user, $courseid, false, !$linkable);
        }

        if ($join){
            $join = is_string($join) ? $join : ', ';
            return join($join, $user_names);
        }

        return $user_names;
    }

    /**
     * Get secondary menu for the activity list
     *
     * @param \cm_info|\stdClass    $cm
     * @param int|string            $userid
     * @param array                 $add2menu
     * @param bool                  $by_user - if false, it will be counted by activity
     * @param bool                  $render - if true, return string menu, else \action_menu
     *
     * @return \action_menu|string
     */
    static public function get_activity_menu($cm, $userid, $add2menu=[], $by_user=true, $render=true){
        static $_ngc_data = [];
        static $_course_data = [];
        $NGC = static::$ned_grade_controller;
        $NGC_render = $NGC::get_render();

        $courseid = $cm->course;
        $cmid = $cm->id;
        $by_activity = !$by_user;
        if (!isset($_course_data[$courseid]['cap_manage_activities'])){
            $_course_data[$courseid]['cap_manage_activities'] = static::has_capability('moodle/course:manageactivities', static::ctx($courseid));
        }
        if (!isset($_ngc_data['cap'])){
            $_ngc_data['cap'] = $NGC::get_see_capability();
            $_ngc_data['cap_edit'] = $NGC::has_edit_capability();
        }

        if ($_ngc_data['cap'] >= $NGC::CAP_SEE_OWN_ANY){
            if ($by_user){
                if (!isset($_ngc_data[$courseid]['user'][$userid])){
                    $_ngc_data[$courseid]['user'][$userid] = [];
                    $records = $NGC::get_records_by_params(0, $userid);
                    foreach ($records as $record){
                        $_ngc_data[$courseid]['user'][$userid][$record->cmid] = $record;
                    }
                }
                $ngc_data = $_ngc_data[$courseid]['user'][$userid][$cmid] ?? null;
            } else {
                if (!isset($_ngc_data[$courseid]['cm'][$cmid])){
                    $_ngc_data[$courseid]['cm'][$cmid] = [];
                    $records = $NGC::get_records_by_params($cmid);
                    foreach ($records as $record){
                        $_ngc_data[$courseid]['cm'][$cmid][$record->userid] = $record;
                    }
                }
                $ngc_data = $_ngc_data[$courseid]['cm'][$cmid][$userid] ?? null;
            }
        } else {
            $ngc_data = null;
        }

        $menu = new \action_menu();
        $menu->set_menu_trigger(static::fa('fa-caret-down'));
        $menu->attributes['class'] .= ' ned-actionmenu teacher-menu';
        $m_add = function($url, $url_params=[], $class='', $fa_icon=null, $text=null, $link_attr=[]) use (&$menu){
            $attr = $link_attr;

            $class = static::str2arr($class);
            if (in_array('target-blank', $class)){
                $attr = array_merge(['target' => '_blank'], $attr);
            }

            if (is_null($text)){
                $text = static::str_check(reset($class));
            }

            if ($fa_icon){
                if (static::str_starts_with_s($fa_icon, '<')){
                    $text = $fa_icon.$text;
                } else {
                    $text = static::fa($fa_icon).$text;
                }
            }

            $menu->add(static::link([$url, $url_params], $text, $class, $attr));
        };

        foreach ($add2menu as $item){
            $m_add(...$item);
        }

        if (static::cm_can_grade_cm($cm)){
            $m_add(static::cm_get_grader_url($cm, $userid, $cm->course), [],
                'moodlegradingpage target-blank', 'fa-table');

            $m_add(static::MM_PAGE,
                [static::$MM::P_COURSEID => $cm->course, static::$MM::P_GETSHOW => 1, static::$MM::P_GROUPID => 0, 'setuser' => $userid, static::$MM::P_MID => $cm->id],
                'nedgradingpage target-blank', 'fa-table');
        }

        if (!$by_activity && $cm->uservisible){
            $m_add("/mod/$cm->modname/view.php", ['id' => $cm->id], 'viewactivity target-blank', 'fa-external-link');

            if (!empty($_course_data[$courseid]['cap_manage_activities'])){
                $m_add("/course/modedit.php", ['update' => $cm->id], 'activitysettings target-blank', 'fa-cog');
            }
        }

        $params = [static::PAR_COURSE => $courseid, static::PAR_CM => $cmid, static::PAR_USER => $userid, $NGC_render::PAR_COURSE_VIEW => 1];
        /** @var \local_ned_controller\support\ned_grade_controller_record|object|null $ngc_data */
        if ($ngc_data){
            $text = $NGC::get_see_text($ngc_data);
            if (!empty($text)){
                $m_add(static::PAGE_GRADE_CONTROLLER, [static::PAR_ID => $ngc_data->id, static::PAR_RETURN_URL => static::get_current_url()],
                    'ngc-link target-blank', 'fa-bullseye', $text);
            }
        } elseif ($_ngc_data['cap_edit'] && $NGC::check_cmid($cm, $courseid)){
            if (static::is_ai_exists()){
                // Report AI Violation
                $icon = static::get_ned_grade_element($NGC::get_grade_status($NGC::GT_AWARD_ZERO, $NGC::REASON_AI));
                $m_add(static::PAGE_AI_ADD_INFRACTION, $params, 'ngc-link target-blank', $icon,
                    static::$C::str('reportaiviolation'));
            }

            // Report Wrong File Submission
            $params[static::PAR_ACTION] = $NGC_render::ACTION_ADD_WRONG_SUBMISSION;
            $icon = static::get_ned_grade_element($NGC::get_grade_status($NGC::GT_AWARD_ZERO, $NGC::REASON_FILE));
            $m_add(static::PAGE_GRADE_CONTROLLER, $params, 'ngc-link target-blank', $icon,
                static::$C::str('reportwrongfilesubmission'));

            // Award Zero Grade (Other)
            $params[static::PAR_ACTION] = $NGC_render::ACTION_ADD;
            $params[$NGC_render::PAR_GRADE_TYPE] = $NGC::GT_AWARD_ZERO;
            $params[$NGC_render::PAR_REASON] = $NGC::REASON_OTHER;
            $icon = static::get_ned_grade_element($NGC::get_grade_status($NGC::GT_AWARD_ZERO, $NGC::REASON_OTHER));
            $m_add(static::PAGE_GRADE_CONTROLLER, $params, 'ngc-link target-blank', $icon,
                static::$C::str('award_smth', static::$C::str($NGC::S_GT_PREFIX.$NGC::GT_AWARD_ZERO)));
        }

        if ($render){
            if ($menu->is_empty()) return '';

            return static::render($menu);
        }
        return $menu;
    }

    /**
     * Output a notification (that is, a status message about something that has just happened).
     *
     * Note: static::notification_add() may be more suitable for your usage.
     * @see notification_add()
     *
     * @param string $message The message to print out, can be key to string translate
     * @param string $type    The type of notification. See constants as static::NOTIFY_*.
     * @param bool   $close_button - optional, whether to show a close icon to remove the notification (default true).
     *
     * @return string the HTML to output.
     */
    static public function notification($message, $type=C::NOTIFY_INFO, $close_button=true){
        return static::O()->notification(static::str_check($message), $type, $close_button);
    }

    /**
     * Add a message to the session notification stack.
     *
     * @param string $message The message to print out, can be key to string translate
     * @param string $type    The type of notification. See constants as static::NOTIFY_*.
     *
     * @return void
     */
    static public function notification_add($message, $type=C::NOTIFY_INFO){
        \core\notification::add(static::str_check($message), $type);
    }

    /**
     * Returns a dataformat selection and download form
     *
     * @see base_trait::download_data() to download file by this element
     *
     * @param string $label A text label
     * @param \moodle_url|string $base The download page url
     * @param string $name The query param which will hold the type of the download
     * @param array $params Extra params sent to the download page; if null - will try to get it from url, to avoid it - send empty array
     *
     * @return string HTML fragment
     */
    static public function download_dataformat_selector($label, $base=null, $name=C::PAR_DOWNLOAD, $params=null){
        $label = static::str_check($label);
        $base = $base ?? static::page()->url;
        if ($base instanceof \moodle_url && is_null($params)){
            $params = $base->params();
        } else {
            $params = $params ?: [];
        }

        return static::O()->download_dataformat_selector($label, $base, $name, $params);
    }

    /**
     * Return the site's logo, if any (empty string otherwise)
     *
     * @param int $maxwidth The maximum width, or null when the maximum width does not matter.
     * @param int $maxheight The maximum height, or null when the maximum height does not matter.
     *
     * @return string html
     */
    static public function get_site_logo($maxwidth=null, $maxheight=150){
        global $SITE;
        $logo_url = static::O()->get_logo_url($maxwidth, $maxheight);
        if (empty($logo_url)){
            return '';
        }

        $sitename = format_string($SITE->fullname, true, array('context' => \context_course::instance(SITEID)));
        return static::div(\html_writer::img($logo_url, $sitename, ['class' => 'img-fluid', 'style' => 'background: #d01e2b;']), 'logo');
    }

    /**
     * Return html link for the stored_file
     *
     * @param \stored_file $file
     * @param bool         $include_itemid - should include itemid in the url or not
     * @param bool         $forcedownload - add force download param to the url params
     * @param int          $icon_size The size of the icon. Defaults to 16 can also be 24, 32, 64, 128, 256;
     *                          if false - do not include icon
     * @param bool         $iconlink Display icon link without text
     *
     * @return string - html link for the file
     */
    static public function file_get_link($file, $include_itemid=true, $forcedownload=false, $icon_size=16, $iconlink=false){
        $filename = $file->get_filename();
        $url = static::file_get_url($file, $include_itemid, $forcedownload);
        $icon = '';
        if ($icon_size){
            $icon = static::O()->pix_icon(file_file_icon($file, $icon_size), get_mimetype_description($file));
        }
        if ($iconlink){
            return \html_writer::link($url, $icon);
        }
        return \html_writer::link($url, $icon.$filename);
    }

    /**
     * Get icon for the unavailable activity with help title
     *
     * @param \cm_info|numeric $cm_or_id
     * @param object|numeric   $user_or_id               - if null, use current(global) $USER by default
     * @param bool             $unavailable_as_invisible - if true, then with false uservisible - return false, despite available info
     * @param bool             $check_global_visibility  - if true, return false, if loaded cm (or $USER) can't see this activity, despite $userid
     *
     * @return string - icon for unavailable activity, or empty string, if it's available
     */
    static public function render_cm_unavailable_info($cm_or_id, $user_or_id=null, $unavailable_as_invisible=false, $check_global_visibility=true){
        if (!static::get_cm_visibility_by_user($cm_or_id, $user_or_id, $unavailable_as_invisible, $check_global_visibility)){
            $info = static::cm_get_cm_unavailable_info($cm_or_id, $user_or_id, true) ?: static::$C::str('unavailable');
            return static::$C::render_from_template('cm_unavailable_info', ['info' => $info]);
        }
        return '';
    }

    /**
     * Check data from $source and save result data in the $res
     * From $source checked fields: timecreated, timemodified, authorid, editorid
     * To $res save fields: timecreated, timemodified, authorid, editorid,
     *  timecreated_str, timemodified_str, author, editor, show_changed_by
     *
     * @param object $source - source object to get information
     * @param object $res - modified object for result
     *
     * @return object
     */
    static public function save_author_and_editor_to_obj($source, &$res=null){
        $res = $res ?? $source ?? new \stdClass();
        if ($source->timecreated ?? false){
            $user_system = strtoupper(static::$C::str('system'));
            $res->authorid = $source->authorid ?? 0;
            $res->editorid = $source->editorid ?? 0;
            $res->show_changed_by = !empty($source->timemodified) && !empty($source->timecreated) &&
                (abs($source->timemodified - $source->timecreated) > MINSECS || $res->editorid != $res->authorid);
            if ($res->show_changed_by){
                $res->timemodified = $source->timemodified;
                $res->timemodified_str = static::ned_date($source->timemodified);
                $res->editor = $res->editorid ? static::q_user_link($res->editorid) : $user_system;
            }
            $res->timecreated = $source->timecreated;
            $res->timecreated_str = static::ned_date($source->timecreated);
            if (isset($res->editor) && $res->editorid == $res->authorid){
                $res->author = $res->editor;
            } else {
                $res->author = $res->authorid ? static::q_user_link($res->authorid) : $user_system;
            }
        } else {
            $res->timecreated = $res->timemodified = 0;
            $res->authorid = $res->editorid = 0;
            $res->show_changed_by = false;
        }

        return $res;
    }
}
