<?php
/**
 * @package    local_ned_controller
 * @subpackage shared
 * @category   NED
 * @copyright  2021 NED {@link http://ned.ca}
 * @author     NED {@link http://ned.ca}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace local_ned_controller\shared;
use local_ned_controller\tt_config_manager as CM;

defined('MOODLE_INTERNAL') || die();

/**
 * Trait base_trait
 *
 * @package local_ned_controller\shared
 *
 * Resolve trait conflicts here.
 * WARNING: When create class with this trait, don't forget use init() method
 *
 * @example of using
 *         class CLASSNAME extends base_class {
 *              use base_trait;
 *         }
 *
 */
trait tt_and_sm {
    use base_trait
    {
        base_trait::_real_init as protected _base_trait_real_init;
    }

    /**
     * @var \block_ned_teacher_tools\activity_status @readonly - activity_status class
     */
    static public $activity_status = '\block_ned_teacher_tools\activity_status';

    static protected function _real_init(){
        static::_base_trait_real_init();
        static::_tt_and_sm_init();
    }

    static protected function _tt_and_sm_init(){

    }

    /**
     * @param string $identifier
     * @param array  $params
     * @param string $plugin
     *
     * @return string
     */
    static public function str($identifier, $params=null, $plugin=null){
        $plugin = $plugin ?? static::$PLUGIN_NAME;
        return static::$C::str2($identifier, $params, $plugin, C::CTRL);
    }

    /**
     * Check and return moodle get_string for this plugin, or $def if didn't find
     *
     * @param string $identifier
     * @param array  $params
     * @param string $def
     * @param string $plugin
     *
     * @return string
     */
    static public function str_check($identifier, $params=null, $def=null, $plugin=null){
        $plugin = $plugin ?? static::$PLUGIN_NAME;
        return static::$C::str_check2($identifier, $params, $def, $plugin, C::CTRL);
    }

    /**
     * Get moodle get_strings for this plugin by array
     *
     * @param array  $identifiers
     * @param array  $params
     * @param string $plugin
     * @param bool   $check
     * @param string $def
     *
     * @return array
     */
    static public function str_arr($identifiers, $params=null, $plugin=null, $check=false, $def=null){
        $plugin = $plugin ?? static::$PLUGIN_NAME;
        return static::$C::str_arr2($identifiers, $params, $plugin, $check, $def, C::CTRL);
    }

    /**
     * Get options for RM from block and site config by course id
     *
     * @param int    $courseid
     * @param object $tt_config TT config
     *
     * @return object|null
     */
    static public function get_rm_config_by_courseid($courseid, $tt_config=null){
        $resubmission_assignments = CM::check_and_get_resubmission_activities($courseid);
        if (empty($resubmission_assignments)) return null;

        // if resubmission enabled and there are resubmission assignments, load config
        $tt_config = $tt_config ?? static::get_site_and_course_block_config($courseid, C::TT_NAME);

        // need to set up this values because they are used in get_rm_config() function
        $tt_config->resubmission_assignments = $resubmission_assignments;
        $tt_config->enableresubmissions = true;

        return static::get_rm_config($tt_config);
    }

    /**
     * Get options for RM from block and site config
     *
     * @param object $tt_config
     *
     * @return object|null
     */
    static public function get_rm_config($tt_config=null){
        if (empty($tt_config->enableresubmissions) || empty($tt_config->resubmission_assignments)){
            return null;
        }

        $rm_config = new \stdClass();
        $rm_config->max_attempts = $tt_config->maximumattempts;
        $rm_config->max_course_attempts = $tt_config->maximumattemptsincourse;
        $rm_config->max_time = $tt_config->daysforresubmission * 24 * 3600;
        $rm_config->assignments = $tt_config->resubmission_assignments;
        $rm_config->pp_error = [];

        $rm_config->show_pp = false;
        $rm_config->can_ressubmit = false;
        // resubmissioncriteria: 0 - if none, 1 - if it needs check participation_power
        $rm_config->check_pp = (bool) $tt_config->resubmissioncriteria;
        $rm_config->need_pp = $tt_config->participationpowerrequired;

        return $rm_config;
    }

    /**
     * Update RM options for user by its submissions count & participation power
     *
     * @param object   $rm_config
     * @param int      $submissions_count
     * @param int|null $participation_power
     *
     * @return object|null
     */
    static public function update_rm_config($rm_config, $submissions_count=0, $participation_power=null){
        if (empty($rm_config)) return null;

        $rm_config->can_ressubmit = !($rm_config->max_course_attempts && $rm_config->max_course_attempts <= $submissions_count);
        if ($rm_config->check_pp && !is_null($participation_power)){
            $rm_config->show_pp = true;
            if ($participation_power < $rm_config->need_pp){
                $rm_config->pp_error[] = static::str('resubmit_pp_error', [$participation_power, $rm_config->need_pp]);
            }
        }

        return $rm_config;
    }

    /**
     * Get max Final deadline for user in the course, bases on max deadline of Final activities
     * If course-module provided, and it's final activity - return max group end of user
     *
     * @param numeric|object $course_or_id
     * @param numeric|object $user_or_id
     * @param \cm_info|object|numeric $cm_or_id - optional course-module, for which need deadline,
     *                                          if provided, checks it for been "final activity", as it has different rules
     *
     * @return int|null
     */
    static public function get_final_deadline($course_or_id, $user_or_id, $cm_or_id=null){
        $DM = static::get_DM();
        if (!$DM){
            return null;
        }

        list($courseid, $userid) = static::get_ids($course_or_id, $user_or_id);

        if ($cm_or_id){
            if (static::cm_is_final($cm_or_id, $courseid)){
                return static::get_user_max_group_end($courseid, $userid);
            }
        }

        $deadlines = [];
        $cmids = static::cmids_get_final($courseid);
        foreach ($cmids as $cmid){
            $deadline = static::get_deadline_by_cm($cmid, $userid, $courseid);
            if ($deadline){
                $deadlines[] = $deadline;
            }
        }

        return static::max($deadlines);
    }

    /**
     * Return user max group end date
     *
     * @param numeric $courseid
     * @param numeric $userid
     *
     * @return int
     */
    static public function get_user_max_group_end($courseid, $userid) {
        return static::db()->get_field_sql("SELECT MAX(g.enddate) 
                                     FROM {groups} g
                                     JOIN {groups_members} gm ON g.id = gm.groupid
                                    WHERE g.courseid = ? AND gm.userid = ?", [$courseid, $userid]) ?: 0;
    }
}
