<?php
/**
 * @package    local_ned_controller
 * @category   NED
 * @copyright  2021 NED {@link http://ned.ca}
 * @author     NED {@link http://ned.ca}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace local_ned_controller;

use local_ned_controller\mod_assign\assign as ned_assign;

defined('MOODLE_INTERNAL') || die();

/**
 * Class shared_lib
 *
 * @package local_ned_controller
 */
class shared_lib extends \local_ned_controller\shared\base_class {
    use \local_ned_controller\shared\base_trait;

    //  You really must have these plugins at current version (or upper)
    const PLUGIN_DEPENDENCIES = [

    ];
    //  If you have these plugins, you should update them to current version (or upper)
    const PLUGIN_REQUIREMENTS = [
        self::TT   => 2023071200,
        self::SM   => 0,
        self::KICA => 2021120300,
        self::SchM => 0,
        self::AI   => 2023042200,
    ];

    /**
     * Triggered as soon as practical on every moodle bootstrap after config has been loaded.
     * The $USER object is available at this point too.
     *
     * Originally called from the {@see local_ned_controller_after_config()}
     *
     * @return void
     */
    static public function ctrl_after_config(){
        if (static::during_update()) return;
    }

    /**
     * Give an opportunity touch things before the http headers are sent.
     * Originally called from the {@see local_ned_controller_before_http_headers()}
     */
    static public function ctrl_before_http_headers(){
        global $CFG, $SCRIPT;

        if (!empty($CFG->maintenance_enabled) || WS_SERVER || AJAX_SCRIPT) return;
        if (!isloggedin() || isguestuser() || is_siteadmin()) return;
        if (!empty(static::PAGES_MOODLE_LOGIN[$SCRIPT])) return;

        $unredirected_pages = [static::PAGE_MOODLE_FILE_DOWNLOAD, static::PAGE_COPYLEAKS_REPORT, static::PAGE_COPYLEAKS_DOWNLOAD];
        if (in_array($SCRIPT, $unredirected_pages)){
            return;
        }

        /**
         * Checks drafts redirects first
         */
        static::check_user_drafts_and_redirect();
        static::check_user_final_evaluation_completion_and_redirect();
    }

    /**
     * If user (not admin) has draft - redirect him to this assign
     * Don't work on assign pages for avoid loop redirects or breaking user possibility to submit
     */
    static public function check_user_drafts_and_redirect(){
        global $DB, $USER, $SCRIPT, $assign;
        if (!static::get_config('redirecttodraft')){
            return;
        }

        // don't redirect from quiz process pages
        $unredirected_pages = [static::PAGE_QUIZ_ATTEMPT, static::PAGE_QUIZ_PROCESS, static::PAGE_QUIZ_SUMMARY];
        if (in_array($SCRIPT, $unredirected_pages)){
            return;
        }

        $cache = static::get_user_cache();
        $c_key_draft = static::CACHE_USERS_KEY_CHECK_DRAFT;
        $c_last_draft = static::CACHE_USERS_KEY_LAST_DRAFT;
        $c_key_close = static::CACHE_USERS_KEY_ASSIGN_CLOSE;
        $redirect_to_draft = function($cmid, $show_text=false){
            static::redirect(new \moodle_url(ned_assign::URL, ['id' => $cmid]),
                $show_text ? static::str('draft_warning') : '', null, static::NOTIFY_WARNING);
            die;
        };

        $delay_check = 10*MINSECS;
        $is_time_to_check = function($key) use (&$cache, &$delay_check){
            $check = $cache->get($key) ?: 0;
            if (($check + $delay_check) > time()){
                return false;
            }
            return true;
        };

        /** @var \assign|\local_ned_controller\mod_assign\assign $assign */
        if (isset($assign)){
            if ($assign->get_instance()->teamsubmission) {
                $submission = $assign->get_group_submission($USER->id, 0, false);
            } else {
                $submission = $assign->get_user_submission($USER->id, false);
            }

            if ($submission){
                if ($submission->status == ASSIGN_SUBMISSION_STATUS_DRAFT){
                    $cmid = $assign->get_course_module()->id;
                    if ($SCRIPT != ned_assign::URL){
                        $redirect_to_draft($cmid);
                        die;
                    }

                    /** @var \local_ned_controller\mod_assign\assign $assign */
                    $assign_id = $assign->get_instance()->id;
                    $gg = static::get_grade_grade($cmid, $USER->id, false);
                    if ($assign->submissions_open($USER->id, false, $submission) && !isset($gg->finalgrade)){
                        $cache->set($c_last_draft, $cmid);
                        // there is some open draft, remove flags
                        $cache->delete($c_key_close.$assign_id);
                    } else {
                        // there is closed draft, set flag to ignore it
                        $cache->set($c_key_close.$assign_id, time());
                        $cache->delete($c_last_draft);
                    }
                } else {
                    $cache->delete($c_last_draft);
                }
            }

            // don't redirect from assign pages
            $cache->delete($c_key_draft);
            return;
        }

        if ($SCRIPT == ned_assign::URL || $SCRIPT == ned_assign::MOODLE_URL){
            // maybe there was some error on the assign page, so $assign is unset - don't try to redirect
            return;
        }

        $last_draft = $cache->get($c_last_draft);
        if ($last_draft){
            $show_text = true;
            $ned_assign = static::$ned_assign::get_assign_by_cm($last_draft);
            if ($ned_assign){
                $show_text = !$ned_assign->have_wrong_file_warning($USER);
            }

            $redirect_to_draft($last_draft, $show_text);
            die;
        }

        if (!$is_time_to_check($c_key_draft)){
            return;
        }

        $drafts = $DB->get_records(ned_assign::TABLE_SUBMISSION,
            ['userid' => $USER->id, 'status' => ASSIGN_SUBMISSION_STATUS_DRAFT, 'latest' => 1]) ?: [];
        foreach ($drafts as $draft){
            if (!$is_time_to_check($c_key_close.$draft->assignment)){
                continue;
            }

            $skip = false;
            try{
                /** @var \cm_info $cm */
                /** @var \stdClass $course */
                [$course, $cm] = get_course_and_cm_from_instance($draft->assignment, static::ASSIGN);
                $cmid = $cm->id;
                $ned_assign = static::ned_assign_by_cm($cm, $course);
                if (!$ned_assign->submissions_open($USER->id)){
                    $skip = true;
                }
            } catch (\Throwable){
                $skip = true;
            }

            if (!$skip){
                if (!static::has_user_access_cm($cm, $course, $USER)){
                    $skip = true;
                }
            }

            if (!$skip){
                if (!has_capability(static::CAPABILITY_STUDENT, $cm->context)){
                    $skip = true;
                }
            }

            if ($skip){
                $cache->set($c_key_close.$draft->assignment, time());
            } else {
                $show_text = true;
                if (isset($ned_assign)){
                    $show_text = !$ned_assign->have_wrong_file_warning($USER);
                }

                $redirect_to_draft($cmid ?? 0, $show_text);
                die;
            }
        }

        // we have checked all for now, don't recheck without necessary for now
        $cache->set($c_key_draft, time());
    }

    /**
     * If user - student and has Final Evaluation completion on course page - redirect him to his KICA gradebook page
     * Don't work on assign and KICA-quiz pages
     */
    static public function check_user_final_evaluation_completion_and_redirect(){
        global $SCRIPT;
        // there is no FE-completion without TT, and there will be nowhere redirect without KICA
        if (!static::is_tt_exists() || !static::is_kica_exists()) return;

        // we redirect only from course pages
        $courseid = static::page()->course->id;
        if ($courseid == SITEID || static::page()->context->contextlevel < CONTEXT_COURSE) return;

        // Do not redirect from KICA, assigns and KICA-quiz-grading
        $unredirected_pages = [ned_assign::MOODLE_URL, ned_assign::URL, static::PAGE_KICA, static::PAGE_KICA_QUIZGRADING];
        if (in_array($SCRIPT, $unredirected_pages)) return;

        // No completion - no redirection
        if (!static::is_course_final_evaluation_completed($courseid)) return;

        static::redirect(new \moodle_url(static::PAGE_KICA, ['courseid' => $courseid]),
            static::str('completedcourse:nosubmissionsallowed'));
        die;
    }
}

shared_lib::init();
