<?php
/**
 * @package    local_ned_controller
 * @subpackage support
 * @category   NED
 * @copyright  2021 NED {@link http://ned.ca}
 * @author     NED {@link http://ned.ca}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace local_ned_controller\support;
use local_ned_controller\shared_lib as NED;

defined('MOODLE_INTERNAL') || die();

/**
 * Class ned_notify
 *
 * @package local_ned_controller\support
 */
class ned_notify_simple {
    use \local_ned_controller\base_empty_class;

    const TYPE_MESSAGE = 0;
    const TYPE_INFO = 1;
    const TYPE_REMINDER = 2;
    const TYPE_NOTICE = 3;

    public $notification_type = 0;
    public $message = '';
    public $icon = '';
    public $title = '';
    public $type = '';
    public $dontshow = false;

    public $ok_text = '';
    public $cancel_button = false;
    public $cancel_text = '';

    public $template = false;
    public $template_context = false;

    /**
     * ned_notify constructor.
     *
     * @param array|\stdClass $record - record from DB
     */
    public function __construct($record=[]){
        NED::import_array_to_object($this, $record);
    }

    /**
     * @param string $name
     * @param mixed  $value
     *
     * @return mixed
     */
    public function __set($name, $value){
        $this->$name = $value;
        return $value;
    }

    /**
     * Get only data, which need to show message on js
     *
     * @param null|object $obj
     *
     * @return \stdClass
     */
    public function js_export($obj=null){
        $obj = $obj ?? $this->export();

        $obj->message = format_text($this->message, FORMAT_HTML);
        $obj->icon = ($obj->icon ?? 0) ?: static::get_type_icon($this->notification_type);
        $obj->type = ($obj->type ?? 0) ?: static::get_type_title($this->notification_type);
        return $obj;
    }

    /**
     * Set type of notification
     *
     * @param $type
     *
     * @return string|null
     */
    public function set_type($type){
        switch ($type){
            case static::TYPE_MESSAGE:
            case static::TYPE_INFO:
            case static::TYPE_REMINDER:
            case static::TYPE_NOTICE:
                $this->notification_type = $type;
                return $type;
            default:
                trigger_error ("Unknown notification type \"$type\".", E_USER_NOTICE);
        }

        return null;
    }

    /**
     * Get name for fa-icon by type of notification
     *
     * @param $type
     *
     * @return string
     */
    static public function get_type_icon($type){
        switch ($type){
            /** @noinspection PhpMissingBreakStatementInspection */
            default:
                trigger_error ("Unknown message type \"$type\".", E_USER_NOTICE);
            case static::TYPE_MESSAGE:
                $name = 'fa-bullhorn';
                break;
            case static::TYPE_INFO:
                $name = 'fa-lightbulb-o';
                break;
            case static::TYPE_REMINDER:
                $name = 'fa-hand-o-up';
                break;
            case static::TYPE_NOTICE:
                $name = 'fa-exclamation-triangle';
                break;
        }

        return $name;
    }

    /**
     * Get title for notification by type
     *
     * @param $type
     *
     * @return string
     */
    static public function get_type_title($type){
        switch ($type){
            /** @noinspection PhpMissingBreakStatementInspection */
            default:
                trigger_error ("Unknown message type \"$type\".", E_USER_NOTICE);
            case static::TYPE_MESSAGE:
                $name = 'message';
                break;
            case static::TYPE_INFO:
                $name = 'info';
                break;
            case static::TYPE_REMINDER:
                $name = 'reminder';
                break;
            case static::TYPE_NOTICE:
                $name = 'notice';
                break;
        }

        return NED::str('notifications:type:' . $name);
    }

    /**
     * Show notification on page by notification list
     *
     * @param        $notes
     * @param array  $args - first element should be element id fot html (or empty)
     * @param string $amd
     */
    static public function show_notifications($notes, $args=null, $amd='show_ned_notifications'){
        static $i=0;
        if (empty($notes)){
            return;
        }

        $args = NED::val2arr($args);
        $notes = NED::val2arr($notes);
        $elem_id = ($args[0] ?? 0) ?: ('ned_notification_data_'.$i);
        $args[0] = $elem_id;
        $i++;

        $elem = \html_writer::empty_tag('input', ['id' => $elem_id, 'type' => 'hidden', 'value' => json_encode($notes)]);
        NED::print_or_add_headingmenu($elem);
        NED::js_call_amd($amd, 'init', $args);
    }
}
