<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * @package    local_ned_controller
 * @subpackage NED
 * @copyright  2021 NED {@link http://ned.ca}
 * @author     NED {@link http://ned.ca}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace local_ned_controller\task;

use local_ned_controller\shared_lib as NED;

defined('MOODLE_INTERNAL') || die;
require_once(__DIR__. '/../../lib.php');

/**
 * Trait adhoc_base_crongt
 * It's base trait for adhoc ned grade controller crones
 *
 * @package local_ned_controller\task
 */
class adhoc_base_cron_ned_grade_controller extends base_adhoc_task {
    use base_task;

    /**
     * If return false, task will not execute
     *
     * @return bool
     */
    public function can_execute(){
        $config = NED::get_config();
        if (!$config->cron_ngc_enable){
            static::print('cron_ngc is not enable, pass.');
            return false;
        }

        return true;
    }

    /**
     * Process custom data before saving it into the task
     *
     * @param array|mixed $custom_data
     *
     * @return array|mixed
     */
    static protected function _process_adding_custom_data($custom_data=[]){
        if (is_null($custom_data)){
            $custom_data = ['course_ids' => array_keys(NED::get_all_courses())];
        }
        return $custom_data;
    }


    /**
     * Do task job without any checks
     *
     * @param array|object|adhoc_base_cron_ned_grade_controller|null $task_or_data
     *
     * @return void
     */
    static public function do_job($task_or_data=[]){
        $data = static::get_data_from_task_or_data($task_or_data);
        $course_ids = $data['course_ids'] ?? [];
        if (empty($course_ids)){
            static::print("There are no courses to check, pass.");
            return;
        } else {
            $course_ids = NED::val2arr($course_ids);
        }

        $config = NED::get_config();
        $count_all = 0;
        $checked_courses = 0;
        $max_delay = max($config->cron_ngc_time, 1)*MINSECS;
        $timeout = time() + $max_delay;

        while(time() < $timeout && !empty($course_ids)){
            $courseid = array_pop($course_ids);
            if (empty($courseid)){
                static::print("There are no courses to check, pass.");
                return;
            }

            static::print("Begin course check with id '$courseid'.");
            $count = static::do_course_job($courseid);
            if ($count){
                static::print("Successfully added/updated $count records from course with id '$courseid'.");
            } else {
                static::print("There is no new records or other changes from course with id '$courseid'.");
            }
            $count_all += $count;
            $checked_courses++;
        }

        if (!empty($course_ids)){
            static::print("Time is up for this task, create new one for the other courses");
            static::add_new_job(0, ['course_ids' => $course_ids]);
        }

        static::print("\n\tSuccessfully added $count_all records from $checked_courses courses so far.");
    }

    /**
     * Return count of added or changed records
     *
     * @param numeric $courseid
     *
     * @return int
     *
     * @noinspection PhpUnusedParameterInspection
     */
    static public function do_course_job($courseid){
        return 0;
    }

    /**
     * Base adding NGC cron task
     *
     * @param int|numeric $multiplier
     * @param array|mixed $custom_data
     * @param bool        $check_same_tasks
     * @param null|static $task
     *
     * @return mixed|static|null
     */
    static protected function _cron_ngc_add_new_job($multiplier=1, $custom_data=[], $check_same_tasks=false, $task=null){
        if ($check_same_tasks){
            if (static::has_same_tasks_warning()){
                return null;
            }
        }

        $task = $task ?? new static();
        $time_delay = 0;
        if ($multiplier > 0){
            $config = NED::get_config();
            $time_delay = $multiplier*$config->cron_ngc_delay;
        }

        $custom_data = static::_process_adding_custom_data($custom_data);
        if (!empty($custom_data)){
            $task->set_custom_data($custom_data);
        }
        static::_add_new_job($task, $time_delay*MINSECS, !$check_same_tasks);
        static::_print_adding_result($task->get_name(), $time_delay, $custom_data, $task);

        return $task;
    }

    /**
     * Queue new task
     * Use it instead of just_add_new_job
     *
     * @param int|numeric $multiplier
     * @param array|mixed $custom_data
     * @param bool        $check_same_tasks
     * @param null|static $task
     *
     * @return mixed|static|null
     */
    static public function add_new_job($multiplier=1, $custom_data=[], $check_same_tasks=false, $task=null){
        return static::_cron_ngc_add_new_job($multiplier, $custom_data, $check_same_tasks, $task);
    }

    /**
     * Print log text about adding task
     *
     * @param string      $name - name of task
     * @param int|numeric $time_delay - time delay of task starting, in minutes
     * @param array|mixed $custom_data - some other data
     * @param static|null $task - added task
     *
     * @noinspection PhpUnusedParameterInspection
     */
    static protected function _print_adding_result($name, $time_delay, $custom_data, $task){
        $course_count = count($custom_data['course_ids'] ?? []);
        if ($time_delay){
            static::print("Next run $name through $time_delay minutes for $course_count course(s).");
        } else {
            static::print("Queue $name task for $course_count course(s) (zero delay).");
        }
    }
}
