<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Update Grade Controller task - Adhoc
 *
 * @package    local_ned_controller
 * @subpackage task
 * @category   NED
 * @copyright  2021 NED {@link http://ned.ca}
 * @author     NED {@link http://ned.ca}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace local_ned_controller\task;
use local_ned_controller\shared_lib as NED;
use local_ned_controller\ned_grade_controller as NGC;

defined('MOODLE_INTERNAL') || die();

/**
 * Class adhoc_ned_grade_controller_update
 *
 * @package local_ned_controller\task
 */
class adhoc_ned_grade_controller_update extends base_adhoc_task {
    use base_task;

    const MAX_RECORDS = 100000;
    const MUCH_RECORDS = 2000;
    const TASK_DELAY = MINSECS;

    /**
     * Do task job without any checks
     *
     * @param base_task|array|null task
     *
     * @return void
     */
    static public function do_job($task_or_data=[]){
        $data = static::get_data_from_task_or_data($task_or_data);
        $start = $data['start'] ?? 0;
        if ($start > 0){
            static::print("Continue task with start '$start'");
        }

        $checked_records = static::check_and_fix_ngc_records($start);
        if ($checked_records >= static::MAX_RECORDS){
            $start += $checked_records;
            static::add_new_job($start, static::TASK_DELAY);
        } else {
            static::print("Task have checked all NGC records, finish...");
        }
    }

    /**
     * Check and fix graders and deadlines
     * Update and delete records, if need
     *
     * @param int $start - check records after this number
     *
     * @return int number of checked records
     */
    static public function check_and_fix_ngc_records($start=0){
        $records = NGC::get_records([], '*', $start, static::MAX_RECORDS);
        $c_records = count($records);
        if ($c_records >= static::MUCH_RECORDS){
            NED::load_all_user_groups();
        }
        static::print("Get $c_records records, will check...");
        list($c_upd, $c_del) = NGC::check_and_fix_records($records);

        if ($c_del > 0){
            static::print("Task have checked $c_records records, $c_del from them need to delete.");
        }
        static::print("Task have checked $c_records records, $c_upd from them need to update.");

        return $c_records;
    }

    /**
     * Queue new task with some start option
     *
     * @param int  $start
     * @param int  $time_delay
     * @param static|null $task
     *
     * @return static|base_adhoc_task
     */
    static public function add_new_job($start=0, $time_delay=0, $task=null){
        if (!$start){
            return static::just_add_new_job_without_doubles($time_delay, $task);
        }

        $task = $task ?? new static();
        $task->set_custom_data(['start' => $start]);
        static::_add_new_job($task, $time_delay);
        $name = $task->get_name();

        if ($time_delay){
            static::print("Next run $name through $time_delay seconds with start '$start'.");
        } else {
            static::print("Queue $name task with zero delay with start '$start'.");
        }

        return $task;
    }
}
