<?php
/**
 * @package    local_ned_controller
 * @subpackage NED
 * @copyright  2022 NED {@link http://ned.ca}
 * @author     NED {@link http://ned.ca}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace local_ned_controller\task;

use local_ned_controller\shared_lib as NED;
use local_ned_controller\marking_manager\marking_manager as MM;

defined('MOODLE_INTERNAL') || die();
require_once(__DIR__. '/../../lib.php');

/**
 * Class grade_empty_resubmissions_task
 *
 * @package local_ned_controller\task
 */
class grade_empty_resubmissions_task extends \core\task\scheduled_task {
    use base_task;

    /**
     * Do the job.
     *
     * @param array|object|base_task|null $task_or_data
     *
     * @return void
     */
    static public function do_job($task_or_data=[]) {
        $courses = NED::get_all_courses();
        $i = 0;
        $i_max = count($courses);
        static::print("Get $i_max courses to check");
        foreach ($courses as $courseid => $course){
            $i++;
            static::print("$i/$i_max) Checking course '$courseid'...");
            static::check_course($course);
        }
    }

    /**
     * Check course for ungraded, unmarked and passed deadline resubmissions -
     *  and copy previous grade for them if need
     *
     * @see copy_last_grade_and_lock_submission
     *
     * @param object|numeric  $course_or_id
     * @param numeric         $userid - (optional) check only user with this id
     *
     * @return int - count of updated grades
     */
    static public function check_course($course_or_id, $userid=null){
        $course = NED::get_chosen_course($course_or_id, false);
        $courseid = $course->id ?? 0;
        $context = NED::ctx($courseid, null, null, IGNORE_MISSING);
        if (!$context || !$courseid){
            static::print("There is no course '$courseid', pass it", false);
            return 0;
        }

        $filter = [MM::ST_REOPENED, MM::ST_UNMARKED_MISSED_DEADLINE, MM::BY_ACTIVITY_USER];

        $mm_params = [
            'course' => $course, 'context' => $context, 'type' => MM::MOD_ASSIGN,
            'groupid' => 0, 'set_students' => '*',
        ];
        if ($userid){
            $user = NED::get_user($userid);
            if ($user){
                $mm_params['set_students'] = [$userid => $user];
            }
        }

        $MM = MM::get_MM_by_params($mm_params);
        $mm_data_list = $MM->get_raw_data($filter);
        if (empty($mm_data_list)){
            static::print("There is no unmarked missed resubmissions, pass course", false);
            return 0;
        }

        $c = count($mm_data_list);
        static::print("+ Found $c records, process", false);
        $res = 0;

        foreach ($mm_data_list as $mm_data){
            try{
                if (static::copy_last_grade_and_lock_submission($mm_data->cmid, $mm_data->userid)){
                    $res++;
                } else {
                    static::print("\t!Copy assign (cmid: {$mm_data->cmid}) grade for the user {$mm_data->userid} is unnecessary.", false);
                }
            } catch (\Throwable $ex) {
                static::print("\tCopy assign (cmid: {$mm_data->cmid}) grade for the user {$mm_data->userid} failed!!!", false);
                NED::cron_print_error($ex);
            }
        }

        if ($res != $c){
            static::print("\n!!! $res from $c were processed fine.\n", false);
        } else {
            static::print("All of them were processed fine.\n", false);
        }

        return $res;
    }

    /**
     * If user has ungraded and unmarked resubmission - copy grade from the previous attempt
     *  and lock submission
     *
     * @param \cm_info|int|string $cm_or_id
     * @param object|numeric      $userorid   - Optional user object (or its id)
     * @param object|numeric      $courseorid - Optional course object (or its id) if loaded, improves optimization if $cm_or_id is represented as ID
     *
     * @return bool
     */
    static public function copy_last_grade_and_lock_submission($cm_or_id, $userorid, $courseorid=null){
        if (!$cm_or_id || !$userorid) return false;

        $userid = NED::get_id($userorid);
        $ned_assign = NED::$ned_assign::get_assign_by_cm($cm_or_id, $courseorid);
        $ned_assign->lock_submission($userid);
        return $ned_assign->copy_last_assign_grade($userid, ASSIGN_SUBMISSION_STATUS_REOPENED, true, true);
    }
}
