<?php
/**
 * @package    local_ned_controller
 * @category   NED
 * @copyright  2022 NED {@link http://ned.ca}
 * @author     NED {@link http://ned.ca}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace local_ned_controller;

use local_ned_controller\shared_lib as NED;

/**
 * Class for easy use of TT DM config.
 * Config data store in CONFIG_TABLE in db.
 * @package local_ned_controller
 *
 * If all cfg fields record equals {@see tt_config_manager::VALUE_INACTIVE}, record will be deleted from that table
 */
class tt_config_manager {
    use \local_ned_controller\shared\global_util;

    protected const _CONFIG_COURSE_DATA = '__config_course_data';

    const CONFIG_TABLE = 'block_ned_teacher_tools_cfg';

    // Config fields
    const CONFIG_RM_FIELD = 'resubmission';
    const CONFIG_DM_FIELD = 'deadline_manager';
    const CONFIG_EXT_FIELD = 'extensions';

    const BASE_CONFIG_FIELDS = [
        self::CONFIG_RM_FIELD,
        self::CONFIG_DM_FIELD,
        self::CONFIG_EXT_FIELD,
    ];

    // All config fields. If you add field here, this field must be in config table
    const CONFIG_FIELDS = [
        self::CONFIG_RM_FIELD,
        self::CONFIG_DM_FIELD,
        self::CONFIG_EXT_FIELD,
    ];

    /** Inactive field value */
    const VALUE_INACTIVE = 0;

    /** Active field value */
    const VALUE_ACTIVE = 1;

    /** @var array static store */
    static protected $_global_data = [];

    /**
     * Updated or remove saved value from local cache
     *
     * @param numeric     $courseid
     * @param numeric|null $cmid - if set, edit config for some $courseid and $cmid
     * @param object|null $data - if object - updated saved value, otherwise deleted it
     */
    protected static function _update_config_course_data_cache($courseid, $cmid=null, $data=null){
        if (empty($courseid)) return;

        $keys = [$courseid];
        if ($cmid){
            $keys[] = $cmid;
        }
        if ($data){
            static::g_set(static::_CONFIG_COURSE_DATA, $keys, $data);
        } else {
            static::g_remove(static::_CONFIG_COURSE_DATA, $keys);
        }
    }

    /**
     * Function filter course config records by active fields
     *
     * @param int   $courseid
     * @param array $fields    values from {@see tt_config_manager::CONFIG_FIELDS}
     * @param bool  $only_ids  return array of ids or config records
     *
     * @return array
     */
    protected static function _filter_config_records($courseid, $fields=[], $only_ids=false){
        $course_data = static::get_config_course_data($courseid);

        $result = [];
        foreach ($course_data as $cmid => $config){
            if (!static::check_required_config_fields($config, $fields)) continue;

            $result[$cmid] = $cmid;
        }

        return ($only_ids) ? array_keys($result) : $result;
    }

    /**
     * Save/update data to the DB and local cache
     *
     * @param array|object $data     data to save
     * @param int          $cmid     no need if $data object contains id field
     * @param int          $courseid no need if $data object contains id field
     *
     * @return bool
     */
    protected static function _save_config_record($data, $cmid=null, $courseid=null){
        $data = (object)$data;
        if (empty($data->id)){
            if (empty($cmid) || empty($courseid)) return false;

            $data->cmid = $cmid;
            $data->courseid = $courseid;
            $res = $data->id = NED::db()->insert_record(static::CONFIG_TABLE, $data);
        } else {
            $cmid = $data->cmid;
            $courseid = $data->courseid;
            $res = NED::db()->update_record(static::CONFIG_TABLE, $data);
        }

        if ($res){
            static::_update_config_course_data_cache($courseid, $cmid, $data);
        }
        return $res;
    }

    /**
     * Check is  resubmission enable in block and return resubmission activities
     *
     * @param int  $courseid
     * @param bool $only_ids return array of ids or config records
     *
     * @return array|false - array if rm enabled in course
     */
    public static function check_and_get_resubmission_activities($courseid, $only_ids=false){
        return static::is_resubmission_enabled($courseid) ? static::get_enabled_resubmission_activities($courseid, $only_ids) : false;
    }

    /**
     * Function returns only enabled resubmission config activities.
     *
     * @param int $courseid
     * @param bool $only_ids return array of ids or config records
     *
     * @return array
     */
    public static function get_enabled_resubmission_activities($courseid, $only_ids=true){
        return static::_filter_config_records($courseid, [static::CONFIG_RM_FIELD], $only_ids);
    }

    /**
     * Function returns only enabled deadline manager config activities.
     *
     * @param int  $courseid
     * @param bool $only_ids    return array of ids or config records
     *
     * @return array
     */
    public static function get_enabled_dm_activities($courseid, $only_ids=true){
        return static::_filter_config_records($courseid, [static::CONFIG_DM_FIELD], $only_ids);
    }

    /**
     * Function returns only enabled extension config activities.
     *
     * @param int  $courseid
     * @param bool $only_ids return array of ids or config records
     * @param bool $use_dm   need to check active DM field or not.
     *
     * @return array
     */
    public static function get_enabled_extension_activities($courseid, $only_ids=true, $use_dm=true){
        $fields = [static::CONFIG_EXT_FIELD];
        if ($use_dm){
            $fields[] = static::CONFIG_DM_FIELD;
        }

        return static::_filter_config_records($courseid, $fields, $only_ids);
    }

    /**
     * Check is resubmission config enable in activity
     *
     * @param int $cmid
     * @param int $courseid
     *
     * @return bool true if enabled
     */
    public static function is_resubmission_enable_for_activity($cmid, $courseid=null){
        if (empty($courseid)){
            $courseid = NED::get_courseid_by_cmorid($cmid);
        }
        $activities = static::get_config_course_data($courseid);
        return !empty($activities[$cmid]->{static::CONFIG_RM_FIELD});
    }

    /**
     * Check is dm config enable in activity
     *
     *
     * @param int $cmid
     * @param int $courseid
     *
     * @return bool true if enabled
     */
    public static function is_dm_enable_for_activity($cmid, $courseid=null){
        if (empty($courseid)){
            $courseid = NED::get_courseid_by_cmorid($cmid);
        }
        $activities = static::get_config_course_data($courseid);
        return !empty($activities[$cmid]->{static::CONFIG_DM_FIELD});
    }

    /**
     * Check is extension config enable in activity
     *
     * @param int $cmid
     * @param int $courseid
     *
     * @return bool true if enabled
     */
    public static function is_extension_enable_for_activity($cmid, $courseid=null){
        if (empty($courseid)){
            $courseid = NED::get_courseid_by_cmorid($cmid);
        }
        $activities = static::get_config_course_data($courseid);
        return !empty($activities[$cmid]->{static::CONFIG_EXT_FIELD});
    }

    /**
     * Get config records by course
     *
     * @param int $courseid
     *
     * @return object[] data from {@see shared_lib::CONFIG_TABLE}, with cmids as keys
     */
    public static function get_config_course_data($courseid){
        $course_data = static::g_get(static::_CONFIG_COURSE_DATA, $courseid);
        if (is_null($course_data)){
            //get the cmid field first
            $select = ['cmid', 'cfg.*'];
            $where = ['courseid = ?'];
            $params = [$courseid];
            $sql = NED::sql_generate($select, [], static::CONFIG_TABLE, 'cfg', $where);

            $course_data = NED::db()->get_records_sql($sql, $params);
            static::_update_config_course_data_cache($courseid, null, $course_data);
        }

        return $course_data;
    }

    /**
     * Get single config record by cmid and courseid
     *
     * @param int $cmid
     * @param int $courseid
     *
     * @return null|object
     */
    public static function get_config_record($cmid, $courseid){
        $course_configs = static::get_config_course_data($courseid);
        return $course_configs[$cmid] ?? null;
    }

    /**
     * Function check each data record for duplicate config values.
     * Save or delete (if all configs eq. {@see tt_config_manager::VALUE_INACTIVE}) records
     *
     * @param array $data     - associative array ['cmid' => [data too update or create] ]
     * @param int   $courseid - courseid
     */
    public static function save_config_data_in_course($data, $courseid){
        if (empty($data)) return;

        foreach ($data as $cmid => $params){
            static::check_and_save_record($params, $cmid, $courseid);
        }
    }

    /**
     * Function check data in $config variable. If data is empty, then delete record from database.
     * Else insert|update record
     *
     * @param array|object $config   config data. It can be a record from database or object with params, needed to save
     * @param int|null     $cmid     Unnecessary if data contains cmid field
     * @param int|null     $courseid Unnecessary if data contains courseid field
     *
     * @return bool
     */
    public static function check_and_save_record($config, $cmid=null, $courseid=null){
        $config = (object)$config;
        if (!empty($config->cmid)){
            $cmid = $config->cmid;
        }
        if (!empty($config->courseid)){
            $courseid = $config->courseid;
        }
        if (empty($cmid) || empty($courseid)){
            return false;
        }

        $record = static::get_config_record($cmid, $courseid);
        if ($record){
            // merge database values into new config object without rewriting existing fields in first object
            $config = (object)((array)$config + (array)$record);
        }

        // check result fields. If need - delete record
        if (static::check_empty_config_fields($config)){
            return static::delete_config_by_cmids([$cmid]);
        }

        // save record
        return static::_save_config_record($config, $cmid, $courseid);
    }

    /**
     * Function check the value of the $config object's fields for emptiness
     *
     * @param array|object $config_record data
     * @param array        $fields        checking fields
     *
     * @return bool false if data has one non-empty config field
     */
    public static function check_empty_config_fields($config_record, $fields=[]){
        $fields = empty($fields) ? static::CONFIG_FIELDS : $fields;
        foreach ($fields as $field){
            if (!empty($config_record->$field)) return false;
        }
        return true;
    }

    /**
     * Function check the value of the $config object's fields for emptiness
     *
     * @param array|object $config_record data
     * @param array        $fields        necessary active fields
     *
     * @return bool false if data has one empty config field
     */
    public static function check_required_config_fields($config_record, $fields){
        foreach ($fields as $field){
            if (empty($config_record->$field)) return false;
        }
        return true;
    }

    /**
     * Delete all records by courseid
     *
     * @param int $courseid
     *
     * @return bool
     */
    public static function delete_config_by_course($courseid){
        if (empty($courseid)) return false;

        $res = NED::db()->delete_records(static::CONFIG_TABLE, ['courseid' => $courseid]);
        if ($res){
            static::_update_config_course_data_cache($courseid, null, null);
        }
        return $res;
    }

    /**
     * Delete records by cmids
     *
     * @param int[] $cmids cmids, which config data need to delete
     * @param numeric $courseid - optional, if we have already known it, and it's the same for all $cmids
     *
     * @return bool
     */
    public static function delete_config_by_cmids($cmids, $courseid=null){
        if (empty($cmids)) return false;

        [$query, $cm_params] = NED::db()->get_in_or_equal($cmids);
        $res = NED::db()->delete_records_select(static::CONFIG_TABLE, 'cmid '.$query, $cm_params);

        if ($res){
            foreach ($cmids as $cmid){
                $cm_courseid = $courseid ?: NED::get_courseid_by_cmorid($cmid);
                static::_update_config_course_data_cache($cm_courseid, $cmid, null);
            }
        }

        return $res;
    }

    /**
     * @param int $courseid
     *
     * @return bool are resubmissions enabled in course
     */
    public static function is_resubmission_enabled($courseid){
        $result = static::g_get(__FUNCTION__, $courseid);
        if (is_null($result)){
            $tt_config = NED::get_site_and_course_block_config($courseid, NED::TT_NAME);
            $result = $tt_config->enableresubmissions ?? false;
            static::g_set(__FUNCTION__, $courseid, $result);
        }
        return $result;
    }

    /**
     * Function check module for Deadline Manager by its mod type
     *
     * @param \cm_info|object|numeric $cm_or_id
     * @param object|numeric          $course_or_id (optional) course to load course-module (if it was provided as cmid) more quickly
     *
     * @return bool true if module type is enabled {@see deadline_manager::ENABLED_MODULES}
     */
    public static function dm_is_cm_enabled_by_type($cm_or_id, $course_or_id=null){
        if (!$DM = NED::get_DM()) return false;
        return $DM::is_cm_enabled_by_type($cm_or_id, $course_or_id);
    }

    /**
     * Function check course module for Resubmission Manager by its mod type
     *
     * @param \cm_info|object|numeric $cm_or_id
     *
     * @return bool true if module type is enabled for resubmissions
     */
    public static function rm_is_cm_enabled_by_type($cm_or_id){
        $cmid = NED::get_id($cm_or_id);
        $result = static::g_get(__FUNCTION__, $cmid);
        if (is_null($result)){
            $cm = NED::get_cm_by_cmorid($cm_or_id);
            $result = $cm && $cm->modname == NED::ASSIGN;
            static::g_set(__FUNCTION__, $cmid, $result);
        }
        return $result;
    }
}
