<?php
/**
 * Plugins overview
 *
 * @package     local_ned_controller
 * @license     http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 * @noinspection PhpUnused
 */

require_once("$CFG->libdir/externallib.php");
require_once("$CFG->dirroot/local/ned_controller/lib.php");

use local_ned_controller\shared_lib as NED;
use local_ned_controller\ned_grade_controller as NGC;
use local_ned_controller\output\ned_grade_controller_render as NGC_render;

/**
 * update_user_notifications functionality include
 * update_user_notifications(...)
 * update_user_notifications_parameters()
 * update_user_notifications_returns()

 * ned_badge_revoke functionality include
 * badge_revoke(...)
 * badge_revoke_parameters()
 * badge_revokes_returns()
 *
 * preview_ned_notification functionality include
 * preview_ned_notification(...)
 * preview_ned_notification_parameters()
 * preview_ned_notification_returns()
 */
class local_ned_controller_external extends external_api {
    /**
     * update_user_notifications:
     */

    /**
     * Returns description of method parameters
     * @return external_function_parameters
     */
    public static function update_user_notifications_parameters() {
        return new external_function_parameters(
            [
                'notify' => new external_single_structure( // it's easier validate one structure variable, then many raw values
                    [
                        'notifyid' => new external_value(PARAM_INT, 'id of notification', VALUE_DEFAULT, 0),
                        'courseid' => new external_value(PARAM_INT, 'id of course', VALUE_DEFAULT, 0),
                        'close' => new external_value(PARAM_INT, 'notification was closed', VALUE_DEFAULT, 0),
                        'dontshow' => new external_value(PARAM_INT, 'don\'t show was pressed', VALUE_DEFAULT, 0),
                    ]
                )
            ]
        );
    }

    /**
     * Returns description of return parameters
     * @return external_function_parameters
     */
    public static function update_user_notifications_returns() {
        return new external_function_parameters(
            [
                'data' => new external_single_structure( // it's easier validate one structure variable, then many raw values
                    [
                        'redirect_url' => new external_value(PARAM_TEXT, 'url to redirect', VALUE_OPTIONAL),
                    ]
                )
            ]
        );
    }

    /**
     * @param $data
     *
     * @return array
     */
    public static function update_user_notifications($data){
        global $CFG, $USER;
        require_once($CFG->dirroot . '/local/ned_controller/classes/ned_notifications.php');
        $params = self::validate_parameters(self::update_user_notifications_parameters(), ['notify' => $data]);
        $notify = (object)$params['notify'];
        $resp_data = [];

        if ($USER->id < 1){
            throw new invalid_parameter_exception('Invalid user id');
        }

        if ($notify->notifyid < 1){
            throw new invalid_parameter_exception('Invalid notification id');
        }

        if ($notify->close != 1){
            throw new invalid_parameter_exception('Invalid data');
        }

        $NN = new local_ned_controller\ned_notifications($USER->id, $notify->courseid);
        $actual_note = $NN->get_actual_note($notify->notifyid);
        if (is_null($actual_note) || !$actual_note->actual){
            throw new invalid_parameter_exception('There is no such notification for this user');
        }

        if (!$NN->update_user_notifications($notify->notifyid, $notify->dontshow)){
            throw new coding_exception('Some unexpected error');
        }

        $ned_notify = $NN::get_notify_from_db($notify->notifyid);
        $url = $ned_notify->get_redirect_url($notify->courseid);
        if (!empty($url)){
            $resp_data['redirect_url'] = $url->out(false);
        }

        // it can't be empty return, and it should be an array or object
        return ['data' => $resp_data];
    }

    /**
     * preview_ned_notification:
     */

    /**
     * Returns description of method parameters
     * @return external_function_parameters
     */
    public static function preview_ned_notification_parameters() {
        return new external_function_parameters(
            [
                'notifyid' => new external_value(PARAM_INT, 'id of notification', VALUE_DEFAULT, 0)
            ]
        );
    }

    /**
     * Returns description of return parameters
     * @return external_function_parameters
     */
    public static function preview_ned_notification_returns() {
        return new external_function_parameters(
            [
                'notify' => new external_single_structure( // it's easier validate one structure variable, then many raw values
                    [
                        'id' => new external_value(PARAM_INT, 'id of notification', VALUE_DEFAULT, 0),
                        'message' => new external_value(PARAM_RAW, 'message of notification', VALUE_DEFAULT, ''),
                        'icon' => new external_value(PARAM_TEXT, 'fa-icon of notification', VALUE_DEFAULT, ''),
                        'type' => new external_value(PARAM_TEXT, 'type of notification', VALUE_DEFAULT, ''),
                        'dontshow' => new external_value(PARAM_BOOL, '"don\'t show" can be pressed', VALUE_DEFAULT, false),
                    ]
                )
            ]
        );
    }

    /**
     * @param $data
     *
     * @return array
     */
    public static function preview_ned_notification($data){
        global $CFG, $PAGE;
        require_once($CFG->dirroot . '/local/ned_controller/classes/ned_notifications.php');
        $notifyid = self::validate_parameters(self::preview_ned_notification_parameters(), ['notifyid' => $data])['notifyid'];

        if ($notifyid < 1){
            throw new invalid_parameter_exception('Invalid notification id');
        }

        $notify = local_ned_controller\ned_notifications::get_notify_from_db($notifyid);
        if (!$notify){
            throw new invalid_parameter_exception('No notification with this id');
        }

        $context = context_system::instance();
        $PAGE->set_context($context);

        return ['notify' => $notify->js_export()];
    }

    /**
     * update_deadline_notification:
     */

    /**
     * Returns description of method parameters
     * @return external_function_parameters
     */
    public static function update_deadline_notification_parameters() {
        return new external_function_parameters([]);
    }

    /**
     * Returns description of return parameters
     * @return external_function_parameters
     */
    public static function update_deadline_notification_returns() {
        return new external_function_parameters([]);
    }

    /**
     * @return array
     */
    public static function update_deadline_notification(){
        $DN = new local_ned_controller\deadline_notification();
        $DN->set_saw_time();

        // it can't be empty return, and it should be an array or object
        return [];
    }

    /**
     * ned_badge_revoke:
     */

    /**
     * Returns description of method parameters
     * @return external_function_parameters
     */
    public static function badge_revoke_parameters() {
        return new external_function_parameters(
            [
                'badge_params' => new external_single_structure( // it's easier validate one structure variable, then many raw values
                    [
                        'hash' => new external_value(PARAM_TEXT, 'unique hash of bage', VALUE_DEFAULT, ''),
                        'userid' => new external_value(PARAM_INT, 'id of user', VALUE_DEFAULT, 0),
                        'courseid' => new external_value(PARAM_INT, 'id of course', VALUE_DEFAULT, 0),
                    ]
                )
            ]
        );
    }

    /**
     * Returns description of return parameters
     * @return external_function_parameters
     */
    public static function badge_revoke_returns() {
        return new external_function_parameters(
            []
        );
    }

    /**
     * @param $data
     *
     * @return array
     */
    public static function badge_revoke($data){
        global $CFG, $USER, $DB;
        require_once($CFG->dirroot . '/badges/classes/output/issued_badge.php');
        require_once($CFG->dirroot . '/lib/classes/event/badge_revoked.php');
        $params = self::validate_parameters(self::badge_revoke_parameters(), ['badge_params' => $data]);
        $badge_params = (object)$params['badge_params'];

        if ($USER->id < 1){
            throw new invalid_parameter_exception('Invalid user');
        }

        if (!local_ned_controller_has_capability_revokebadge($badge_params->courseid)){
            throw new invalid_parameter_exception('User has no capability');
        }

        if ($badge_params->userid < 1){
            throw new invalid_parameter_exception('Invalid user id');
        }

        if (empty($badge_params->hash)){
            throw new invalid_parameter_exception('Invalid badge hash');
        }

        /**
         * @var \stdClass $badge
         * @see \core_badges\output\issued_badge
         */
        $badge = $DB->get_record_sql('SELECT userid, badgeid
                FROM {badge_issued}
                WHERE ' . $DB->sql_compare_text('uniquehash', 40) . ' = ' . $DB->sql_compare_text(':hash', 40),
            ['hash' => $badge_params->hash], IGNORE_MISSING);

        if (empty($badge) || $badge->userid != $badge_params->userid){
            throw new invalid_parameter_exception('Invalid user id or its badge');
        }

        $DB->delete_records('badge_manual_award',
            ['badgeid' =>  $badge->badgeid, 'recipientid' => $badge->userid]);
        $DB->delete_records('badge_issued',
            ['badgeid' => $badge->badgeid, 'userid' => $badge->userid]);
        // Core badge system doesn't delete records from badge_criteria_met table, so we don't too

        // Trigger event, badge revoked.
        $eventparams = array(
            'objectid' => $badge->badgeid,
            'relateduserid' => $badge->userid,
            'context' => \context_system::instance(),
        );
        $event = \core\event\badge_revoked::create($eventparams);
        $event->trigger();

        return [];
    }

    /**
     * Returns description of method parameters
     * @return external_function_parameters
     */
    public static function ngc_info_parameters() {
        return new external_function_parameters(
            [
                'ngc_info_params' => new external_single_structure( // it's easier validate one structure variable, then many raw values
                    [
                        'id' => new external_value(PARAM_INT, 'id of the record', VALUE_DEFAULT, 0),
                        'course_view' => new external_value(PARAM_INT, 'set pop up links in course context', VALUE_DEFAULT, 0),
                    ]
                )
            ]
        );
    }

    /**
     * Returns description of return parameters
     * @return external_description
     */
    public static function ngc_info_returns() {
        return new external_value(PARAM_RAW, 'Info about record');
    }

    /**
     * @param $data
     *
     * @return string
     */
    public static function ngc_info($data){
        global $USER, $PAGE;
        $params = self::validate_parameters(self::ngc_info_parameters(), ['ngc_info_params' => $data]);
        $info_params = (object)$params['ngc_info_params'];

        if ($USER->id < 1){
            throw new invalid_parameter_exception('Invalid user');
        }

        if (empty($info_params->id)){
            throw new invalid_parameter_exception('Invalid record id');
        }
        $info_params->course_view = (bool)$info_params->course_view;

        $context = context_system::instance();
        $PAGE->set_context($context);

        $cap = NGC::get_see_capability($context);
        $options = [];
        if ($cap <= NGC::CAP_CANT_VIEW){
            throw new invalid_parameter_exception('No capability!');
        } elseif ($cap == NGC::CAP_SEE_ME){
            $options['userid'] = $USER->id;
        } elseif ($cap == NGC::CAP_SEE_OWN_GRADER){
            $options['graderid'] = $USER->id;
        }

        $record = NGC::get_record_by_id($info_params->id, $options);
        if (!$record) throw new invalid_parameter_exception('Invalid record id [2]');
        if (!NGC::can_see_record($record)) throw new invalid_parameter_exception('No capability!');

        $record->course_view = $info_params->course_view;
        $desc = NGC_render::get_description_object($record, $record, null, '', $cap);

        return NGC_render::render_description_object($desc);
    }
}
