<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Helper
 *
 * @package    local_proxy
 * @copyright  2020 Michael Gardener <mgardener@cissq.com>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace local_proxy;

use local_ned_controller\ned_grade_controller as NGC;
use local_proxy\shared_lib as NED;

defined('MOODLE_INTERNAL') || die();

/** @var object $CFG */
require_once($CFG->dirroot . '/mod/assign/locallib.php');

/**
 * Class helper
 * @package local_proxy
 */
class helper {
    /**
     * @param $courseid
     * @return array
     * @throws \dml_exception
     */
    public static function get_proxy_assignments($courseid){
        global $DB;
        $sql = "SELECT apc.id,
                       apc.assignment, 
                       a.course
                  FROM {assign_plugin_config} apc 
                  JOIN {assign} a 
                    ON apc.assignment = a.id
                 WHERE apc.plugin = 'proxy' 
                   AND apc.subtype = 'assignsubmission' 
                   AND apc.name = 'enabled'
                   AND apc.value = 1 
                   AND a.course = ?
              ORDER BY a.name ASC";

        return $DB->get_records_sql($sql, [$courseid]);
    }

    /**
     * Return list or users by there role name in group, as one string
     *
     * @param string    $role_name
     * @param object    $group
     * @param bool      $linkable
     *
     * @return string
     */
    public static function get_users_by_role($role_name, $group, $linkable=false){
        return NED::get_role_users_fullnames($role_name, $group->courseid, $group->id, 0, $linkable, true);
    }

    /**
     * @param int $cmid
     * @param int $userid
     *
     * @return bool
     */
    public static function is_mod_completed($cmid, $userid){
        global $DB;

        $cmcompletion = $DB->get_record('course_modules_completion', ['userid' => $userid, 'coursemoduleid' => $cmid]);
        if ($cmcompletion && $cmcompletion->completionstate){
            return true;
        }
        return false;
    }

    /**
     * Return total useful users and submitted
     *
     * @param \cm_info|object|numeric $cm_or_id
     * @param object[] $users
     *
     * @return int[] - list($userstotal, $userssubmitted)
     */
    public static function get_counted_users_submissions($cm_or_id, $users){
        $def = [0, 0];
        if (empty($users)) return $def;

        $cm = NED::get_cm_by_cmorid($cm_or_id, null, null, NED::ASSIGN);
        if (!$cm) return $def;

        $assign = NED::ned_assign_by_cm($cm);
        $completion = new \completion_info($cm->get_course());
        $completionenabled = $completion->is_enabled($cm);
        $userstotal = 0;
        $userssubmitted = 0;

        foreach ($users as $user){
            if (!NED::get_cm_visibility_by_user($cm, $user, false, false)) continue;

            ++$userstotal;
            $submission = $assign->get_user_submission($user->id, false);
            if (isset($submission->timemodified) && $submission->status == ASSIGN_SUBMISSION_STATUS_SUBMITTED){
                ++$userssubmitted;
            }/* elseif ($completionenabled && helper::is_mod_completed($cm->id, $user->id)){
                ++$userssubmitted;
            }*/
        }

        return [$userstotal, $userssubmitted];
    }

    /**
     * Checks NGC penalty for the current activity/user, and provides HTML result
     *  with links to see existing penalty, or links to add a new penalty
     *
     * @param \cm_info|object|numeric       $cm_or_id     - checked activity
     * @param object|numeric                $user_or_id   - checked user
     * @param object|numeric|null           $course_or_id - optional, course of the checked activity
     * @param \context_module|\context|null $context      - optional, context of the checked activity
     *
     * @return string - HTML
     */
    public static function render_ngc_links($cm_or_id, $user_or_id, $course_or_id=null, $context=null){
        $cmid = NED::get_id($cm_or_id);
        $userid = NED::get_id($user_or_id);
        if (empty($course_or_id) || empty($context)){
            $cm = NED::get_cm_by_cmorid($cm_or_id, $course_or_id);
            $courseid = $cm->course;
            $context = $cm->context;
        } else {
            $courseid = NED::get_id($course_or_id);
        }

        $o = '';
        $NGC =  NED::$ned_grade_controller;
        $NGC_render = $NGC::$NGC_RENDER;
        $records = $NGC::get_records_by_params($cmid, $userid);

        if (!empty($records)) {
            foreach ($records as $ngc_record) {
                //if (!$NGC::can_see_record($ngc_record)) continue;

                $ngc_name = NED::span($NGC::get_human_record_name($ngc_record), 'h5 mx-2 my-0');
                $ngc_icon = NED::get_ned_grade_icon_by_ngc($ngc_record, [], ['postfix' => $ngc_name]);
                if (empty($ngc_icon)) continue;

                $o .= NED::div($ngc_icon, 'ngc-penalty d-flex mt-2');
            }
        }

        return $o;
    }

    /**
     * @param $cm_or_id
     * @param $users
     * @return int
     * @throws \coding_exception
     * @throws \dml_exception
     */
    public static function count_wrong_file_submissions($cm_or_id, $users){
        global $DB;

        $def = 0;
        if (empty($users)) return $def;

        $cm = NED::get_cm_by_cmorid($cm_or_id, null, null, NED::ASSIGN);
        if (!$cm) return $def;

        list($isql, $params) = $DB->get_in_or_equal(array_keys($users));
        $params[] = NGC::REASON_FILE;
        $params[] = $cm->id;

        $sql = "SELECT COUNT(1)
                  FROM {local_ned_controller_grades} cg
                 WHERE userid {$isql} 
                   AND reason = ? 
                   AND cmid = ?";

        return $DB->count_records_sql($sql, $params);
    }
}