<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Filter
 *
 * @package    local_proxy
 * @copyright  2020 Michael Gardener <mgardener@cissq.com>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace local_proxy\output;

use local_proxy\helper;
use local_proxy\shared_lib as NED;
use block_ned_teacher_tools\shared_lib as TT;

defined('MOODLE_INTERNAL') || die();
/** @var object $CFG */
require_once($CFG->dirroot . '/mod/assign/locallib.php');

/**
 * Class proxy_overview
 * @package local_proxy\output
 */
class proxy_notification implements \renderable, \templatable {
    /**
     * @var \stdClass $course
     */
    protected $course;

    /**
     * @var int $_groupid
     */
    protected $_groupid;

    /**
     * @var object $_group
     */
    protected $_group;


    /**
     * assign_summary constructor.
     *
     * @param object               $course
     * @param numeric|object|null  $group_or_id
     *
     */
    public function __construct($course, $group_or_id=null){
        $this->course = $course;
        $this->_groupid = NED::get_id($group_or_id);
    }

    /**
     * @return false|object
     */
    public function get_group(){
        if (is_null($this->_group)){
            $this->_group = NED::get_chosen_group($this->course->id, $this->_groupid) ?: false;
            $this->_groupid = $this->_group->id ?? 0;
        }

        return $this->_group;
    }

    /**
     * Export page contents for template
     * @param \renderer_base $output
     * @return \stdClass
     */
    public function export_for_template(\renderer_base $output){
        $data = new \stdClass();

        // Assignments.
        if (!$assigns = helper::get_proxy_assignments($this->course->id)){
            return $data;
        }

        $group = $this->get_group();
        if (empty($group)){
            return $data;
        }

        $proxyassignments = [];
        foreach ($assigns as $assign){
            $cm = NED::get_cm_by_params($this->course->id, NED::ASSIGN, $assign->assignment);
            if (!NED::check_activity_visible_by_cm($cm)) continue;

            if (!has_capability('assignsubmission/proxy:submit', $cm->context)) continue;

            list($window_start_str, $window_end_str, $window_start, $window_end) =
                NED::dm_get_submission_window($cm, NED::PROXY, null, $group);

            if (!$window_start || $window_start > time()) continue;

            $users = NED::get_course_students_by_role($this->course, 0, $group->id, true, false);
            if (empty($users)) continue;

            $proxyassignment = new \stdClass();
            $proxyassignment->classname = $group->name;
            $proxyassignment->classroomteacher = helper::get_users_by_role(NED::ROLE_CT, $group, true);
            $proxyassignment->name = $cm->name;
            $proxyassignment->url = NED::url('~/submission.php', ['id' => $cm->id]);
            $proxyassignment->studentdeadline = $window_start_str;
            $proxyassignment->proxysubmissiondeadline = $window_end_str;

            list($userstotal, $userssubmitted) = helper::get_counted_users_submissions($cm, $users);
            if (empty($userstotal) || ($userstotal == $userssubmitted)){
                continue;
            }

            $proxyassignment->studentsubmissionsreceived = $userssubmitted;
            $proxyassignment->studentsubmissionsmissing = $userstotal - $userssubmitted;
            $proxyassignment->wrongfilesubmissions = helper::count_wrong_file_submissions($cm, $users);

            // Countdown.
            $remainingtime = $window_end - time();
            $proxyassignment->timeremaining = NED::time_counter_to_str_max($remainingtime, 2, true);

            $proxyassignments[] = $proxyassignment;
        }

        $data->proxyassignments = $proxyassignments;

        return $data;
    }
}
