<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Filter
 *
 * @package    local_proxy
 * @copyright  2020 Michael Gardener <mgardener@cissq.com>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace local_proxy\output;

use local_proxy\helper;
use local_proxy\shared_lib as NED;
use block_ned_teacher_tools\shared_lib as TT;

defined('MOODLE_INTERNAL') || die();
/** @var object $CFG */
require_once($CFG->dirroot . '/mod/assign/locallib.php');

/**
 * Class filter
 * @package local_proxy\output
 */
class proxy_submission implements \renderable, \templatable {
    /**
     * @var \assign
     */
    protected $assign;
    protected $groupid;


    /**
     * proxy_submission constructor.
     *
     * @param $assign
     * @param $groupid
     */
    public function __construct($assign, $groupid){
        $this->assign = $assign;
        $this->groupid = $groupid;
    }

    /**
     * Export page contents for template
     * @param \renderer_base $output
     * @return \stdClass
     */
    public function export_for_template(\renderer_base $output){
        $data = new \stdClass();

        $course = $this->assign->get_course();
        $group = NED::get_chosen_group($course, $this->groupid);
        if (empty($group)){
            return $data;
        }

        $cm = $this->assign->get_course_module();
        if (!NED::check_activity_visible_by_cm($cm)){
            return $data;
        }

        $data->activity = NED::q_cm_link($cm);
        $data->class = $group->name;
        $data->ct = helper::get_users_by_role(NED::ROLE_CT, $group, true);

        $br = \html_writer::empty_tag('br');

        // Users.
        $users = NED::get_course_students_by_role($course, 0, $group->id, true, false);
        $users_processed = [];
        if (empty($users)){
            return $data;
        }

        foreach ($users as $user){
            if (!NED::get_cm_visibility_by_user($cm, $user, false, false)) continue;

            $user->fullname = NED::q_user_link($user, $course);
            list($window_start_str, $window_end_str, $window_start, $window_end) =
                NED::dm_get_submission_window($cm, NED::PROXY, $user);
            $dmtimezone = TT::get_user_dm_timezone_ctx($cm->context);
            $user->completiondeadline = $window_start_str;
            $user->submissiondeadline = $window_end_str;

            // Countdown.
            $remainingtime = $window_end - time();
            $user->submissioncountdown = NED::time_counter_to_str_max($remainingtime, 2, true);

            // Submitted
            $submission = $this->assign->get_user_submission($user->id, false);
            if (isset($submission->timemodified) && $submission->status !== ASSIGN_SUBMISSION_STATUS_NEW){
                $user->submitted = userdate($submission->timemodified, TT::str('strtimeformat'), $dmtimezone);
            }

            // Files.
            $out = [];
            if ($submission){
                $fs = get_file_storage();
                $files = $fs->get_area_files($this->assign->get_context()->id, 'assignsubmission_proxy', 'submission_proxies', $submission->id);
                foreach ($files as $file){
                    if (!$file->is_directory()){
                        $out[] = NED::file_get_link($file, true, true);
                    }
                }
                if ($out){
                    $user->files = implode($br, $out);
                }
            }
            // Grade status.
            $gradestatus = $this->assign->get_grading_status($user->id);
            $user->gradestatus = get_string($gradestatus, 'assign');

            $submissionurl = new \moodle_url('/local/proxy/submit.php', [
                'id' => $cm->id,
                'groupid' => $this->groupid,
                'userid' => $user->id
            ]);

            $user->gradestatus .= helper::render_ngc_links($cm, $user->id);

            // Action.
            if ($gradestatus == ASSIGN_GRADING_STATUS_GRADED){
                $user->actiontext = "";
                $usergrade = $this->assign->get_user_grade($user->id, false);
                $item = $this->assign->get_grade_item();
                $user->gradestatus .= ' ' .  format_float($usergrade->grade, $item->get_decimals()) . '/' . format_float($item->grademax, $item->get_decimals());
                $user->actiontext = NED::str('viewgrade');
                $user->actionurl = (new \moodle_url('/mod/assign/view.php', [
                    'id' => $cm->id,
                    'action' => 'grading',
                    'prevpage' => 'proxy'
                ]))->out(false);

            } elseif (!empty($submission->status) && $submission->status === ASSIGN_SUBMISSION_STATUS_SUBMITTED){
                if ($gradestatus === ASSIGN_GRADING_STATUS_GRADED){
                    $user->actiontext = NED::str('view');
                } else {
                    $user->actiontext = NED::str('edit');
                }
                $user->actionurl = $submissionurl->out(false);
            } else {
                $user->actiontext = NED::str('submit');
                $user->actionurl = $submissionurl->out(false);
            }

            $users_processed[] = $user;
        }

        $data->students = $users_processed;
        $data->not_empty = !empty($data->students);
        return $data;
    }
}
