<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Proxy submission.
 *
 * @package   local_proxy
 * @copyright 2021 Michael Gardener <mgardener@cissq.com>
 * @license   http://www.gnu.org/copyleft/gpl.html gnu gpl v3 or later
 */

require_once(dirname(__FILE__, 3) . '/config.php');
require_once($CFG->dirroot . '/mod/assign/locallib.php');

use local_proxy\shared_lib as NED;

$id = required_param('id', PARAM_INT);
$userid = required_param('userid', PARAM_INT);
$groupid = optional_param('groupid', 0, PARAM_INT);

list($course, $cm) = get_course_and_cm_from_cmid($id, 'assign');

require_login($course, true, $cm);

$context = context_module::instance($cm->id);

require_capability('assignsubmission/proxy:submit', $context);

$user = $DB->get_record('user', array('id'=>$userid), '*', MUST_EXIST);
$assign = new assign($context, $cm, $course);

$submission = $assign->get_user_submission($user->id, false);

/** @var assign_submission_proxy $assignsubmissionproxy */
$assignsubmissionproxy  = $assign->get_submission_plugin_by_type('proxy');
$cfg = $assignsubmissionproxy->get_config();

if (empty($cfg->enabled)){
    NED::print_module_error('invalidmodule');
}

$url = new moodle_url('/local/proxy/submit.php', ['id' => $id, 'userid' => $userid, 'groupid' => $groupid]);
$redirecturl = new moodle_url('/local/proxy/submission.php', ['id' => $id, 'groupid' => $groupid]);
NED::page_set_title('proxysubmission', $url);

$mform = new \local_proxy\form\submission(null, [
    'assign' => $assign,
    'submission' => $submission,
    'assignsubmissionproxy' => $assignsubmissionproxy,
    'user' => $user
]);

$fileoptions = $mform->get_file_options($assignsubmissionproxy);

$submissionid = $submission ? $submission->id : 0;

$data = new stdClass();
$data->id = $id;
$data->userid = $user->id;
$data->groupid = $groupid;
$data = file_prepare_standard_filemanager(
    $data,
    'files',
    $fileoptions,
    $assign->get_context(),
    'assignsubmission_proxy',
    ASSIGNSUBMISSION_PROXY_FILEAREA,
    $submissionid
);
$mform->set_data($data);

if ($mform->is_cancelled()){
    redirect($redirecturl);
} elseif ($fromform = $mform->get_data()){
    $gradestatus = $assign->get_grading_status($user->id);
    if (!is_siteadmin()){
        if (!$assign->submissions_open($user->id) && $gradestatus == ASSIGN_GRADING_STATUS_GRADED){
            redirect($redirecturl, get_string('duedatereached', 'assign'));
            die;
        }
    }
    $instance = $assign->get_instance();
    $submission = $assign->get_user_submission($user->id, true);

    // Check that no one has modified the submission since we started looking at it.
    if (isset($fromform->lastmodified) && ($submission->timemodified > $fromform->lastmodified)){
        // Another user has submitted something. Notify the current user.
        if ($submission->status !== ASSIGN_SUBMISSION_STATUS_NEW){
            $msg = get_string('submissionmodified', 'mod_assign');
            redirect($redirecturl, $msg);
            die;
        }
    }

    $submission->status = ASSIGN_SUBMISSION_STATUS_SUBMITTED;
    $flags = $assign->get_user_flags($user->id, false);

    // Get the flags to check if it is locked.
    if ($flags && $flags->locked){
        NED::print_module_error('submissionslocked', 'assign');
    }

    $plugin  = $assign->get_submission_plugin_by_type('proxy');
    if ($plugin->is_enabled() && $plugin->is_visible()){
        if (!$plugin->save($submission, $fromform)){
            $msg = $plugin->get_error();
            redirect($redirecturl, $msg);
            die;
        }
    }

    // Update submision.
    $submission->timemodified = time();
    $DB->update_record('assign_submission', $submission);

    $users = [$userid];
    $complete = COMPLETION_INCOMPLETE;
    if ($submission->status == ASSIGN_SUBMISSION_STATUS_SUBMITTED){
        $complete = COMPLETION_COMPLETE;
    }

    $completion = new completion_info($assign->get_course());
    if ($completion->is_enabled($assign->get_course_module()) && $instance->completionsubmit){
        foreach ($users as $id){
            $completion->update_state($assign->get_course_module(), $complete, $id);
        }
    }


    redirect($redirecturl);
    exit;
}

$assign->update_effective_access($userid);
$viewfullnames = has_capability('moodle/site:viewfullnames', $assign->get_context());
$usersummary = new assign_user_summary($user,
    $assign->get_course()->id,
    $viewfullnames,
    $assign->is_blind_marking(),
    $assign->get_uniqueid_for_user($user->id),
    \core_user\fields::get_identity_fields($assign->get_context(), false),
    !$assign->is_active_user($userid));

echo $OUTPUT->header();
echo $assign->get_renderer()->render($usersummary);
$mform->display();
echo $OUTPUT->footer();