<?php
// This file is part of Moodle - https://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <https://www.gnu.org/licenses/>.

/**
 * Helper.
 *
 * @package     local_tem
 * @category    classes
 * @copyright   2023 Michael Gardener <mgardener@cissq.com>
 * @license     https://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace local_tem;

use Aws\Panorama\PanoramaClient;
use context_system;
use core_tag_tag;
use html_writer;
use local_proxy\shared_lib as NED;
use local_schoolmanager\school;
use local_schoolmanager\school_manager as SM;
use block_ned_teacher_tools\deadline_manager as DM;
use moodle_url;
use stdClass;

defined('MOODLE_INTERNAL') || die();

/** @var \stdClass $CFG */
require_once($CFG->dirroot . '/mod/quiz/attemptlib.php');

/**
 * Class helper
 */
class helper {

    const FILTER_ACTION_REQUIRED = 'actionrequired';
    const FILTER_COMPLETED = 'completed';
    const FILTER_NEXT_THIRTY = 'nextthirtydays';
    const FILTER_LAST_THIRTY = 'lastthirtydays';
    const FILTER_SHOW_ALL = 'showall';

    /**
     * @param $quizid
     * @return bool
     * @throws \coding_exception
     */
    public static function is_tem_enabled($quizid) {
        $cm = get_coursemodule_from_instance('quiz', $quizid);
        $tags = core_tag_tag::get_item_tags_array('core', 'course_modules', $cm->id);

        return ($tags && (in_array('Summative', $tags) || in_array('summative', $tags)));
    }


    /**
     * @param $attempt
     * @return void
     * @throws \coding_exception
     * @throws \dml_exception
     */
    public static function process_attempt_session($attempt) {
        global $DB;

        $quiz = $DB->get_record('quiz', ['id' => $attempt->quiz]);

        if (!self::is_tem_enabled($quiz->id)) {
            return;
        }

        $sql = "SELECT g.*
                  FROM {groups_members} gm 
                  JOIN {groups} g ON gm.groupid = g.id
                 WHERE g.courseid = ? 
                   AND gm.userid = ?
                   AND g.schedule = 2";

        if (!$groups = $DB->get_records_sql($sql, [$quiz->course, $attempt->userid])) {
            return;
        }

        foreach ($groups as $group) {
            $sql = "SELECT s.id, s.courseid, s.groupid, s.quizid, s.timestart
                      FROM {local_tem_session} s
                     WHERE s.courseid = ?
                       AND s.groupid = ?
                       AND s.quizid = ?
                  ORDER BY s.timestart DESC";

            $session = null;

            if ($sessions = $DB->get_records_sql($sql, [$quiz->course, $group->id, $quiz->id], 0, 1)) {
                $session = reset($sessions);
            }

            if (empty($session)) {
                $session = helper::add_session($quiz, $attempt, $group);
                helper::add_session_attempt($session, $attempt);
            } else {
                if (($attempt->timestart - $session->timestart) >= HOURSECS) {
                    $session = helper::add_session($quiz, $attempt, $group);
                    helper::add_session_attempt($session, $attempt);
                } else {
                    helper::add_session_attempt($session, $attempt);
                }
            }
        }
    }

    /**
     * @param $quiz
     * @param $attempt
     * @param $group
     * @return stdClass
     * @throws \dml_exception
     */
    public static function add_session($quiz, $attempt, $group): stdClass {
        global $DB;

        $cm = get_coursemodule_from_instance("quiz", $quiz->id, $quiz->course);
        $modinfo = get_fast_modinfo($quiz->course);
        $mod = $modinfo->cms[$cm->id];
        $module = new \block_ned_teacher_tools\mod\deadline_manager_quiz($mod);

        if (!$deadline = $module->get_group_override_date($group->id)) {
            $deadline = $module->get_default_deadline();
        }
        $coursecontext = \context_course::instance($quiz->course);
        $activestudents = get_enrolled_users($coursecontext, 'mod/assign:submit', $group->id, 'u.*', 'u.id', 0, 0, true);

        if ($cohort = DM::get_school_cohort($group)) {
            $schoolid = $cohort->id;
        } else {
            $schoolid = 0;
        }

        $session = new stdClass();
        $session->schoolid = $schoolid;
        $session->courseid = $quiz->course;
        $session->quizid = $quiz->id;
        $session->groupid = $group->id;
        $session->timestart = $attempt->timestart;
        $session->timescheduled = $deadline;
        $session->numberofstudents = count($activestudents);
        $session->id = $DB->insert_record('local_tem_session', $session);

        self::build_session_number($session->courseid, $session->quizid, $session->groupid);

        return $session;
    }

    /**
     * @param $courseid
     * @param $quizid
     * @param $groupid
     * @return void
     * @throws \dml_exception
     */
    public static function build_session_number($courseid, $quizid, $groupid): void {
        global $DB;

        $sql = "SELECT s.*
                  FROM {local_tem_session} s
                 WHERE s.courseid = ?
                   AND s.groupid = ?
                   AND s.quizid = ?
              ORDER BY s.timestart ASC";

        if ($sessions = $DB->get_records_sql($sql, [$courseid, $groupid, $quizid])) {
            $numberofsessions = count($sessions);
            $counter = 0;
            foreach ($sessions as $session) {
                $counter++;
                $rec = new stdClass();
                $rec->id = $session->id;
                $rec->sessionnum = $counter;
                $rec->totalsessionnum = $numberofsessions;
                $DB->update_record('local_tem_session', $rec);
            }
        }
    }

    /**
     * @param $session
     * @param $attempt
     * @return stdClass
     * @throws \dml_exception
     */
    public static function add_session_attempt($session, $attempt): stdClass {
        global $DB, $PAGE;

        $att = new stdClass();
        $att->sessionid = $session->id;
        $att->attemptid = $attempt->id;
        if (isset($attempt->ip)) {
            $att->ip = $attempt->ip;
        } else {
            $att->ip = $PAGE->requestip;
        }
        $att->id = $DB->insert_record('local_tem_attempt', $att);

        return $att;
    }

    /**
     * @param $data
     * @param $column
     * @param $counter
     * @param moodle_url $pageurl
     * @param null $pageparams
     * @param bool $export
     * @return false|float|string
     * @throws \coding_exception
     * @throws \dml_exception
     * @throws \moodle_exception
     */
    public static function session_data($data, $column, &$counter, $pageurl, $pageparams = null, $export = false) {
        global $DB, $OUTPUT;

        $var = '';
        $varclass = '_class_'.$column;

        $contextsystem = context_system::instance();

        switch ($column) {
            case 'rowcount':
                $var = ++$counter;
                break;
            case 'scheduled':
            case 'starttime':
                $var = '?';
                if ($data->$column > 0) {
                    $var = date("Y-m-d", $data->$column) . '<br>' . date("@ H:i", $data->$column);
                } else {
                    $data->{$varclass} = 'flagged';
                }
                break;
            case 'endtime':
                $var = '-';
                if ($data->$column > 0) {
                    $var = date("Y-m-d", $data->$column) . '<br>' . date("@ H:i", $data->$column);
                }
                break;
            case 'duration':
                if (!empty($data->endtime)) {
                    $var = format_time($data->endtime - $data->starttime);
                } else {
                    $var = '-';
                }
                break;
            case 'proctor':
                if ($proctors = self::get_school_proctors($data->groupid)) {
                    if (empty($data->proctor) || !array_key_exists($data->proctor, $proctors)) {
                        $data->proctor = 0;
                        $data->{$varclass} = 'flagged';
                    }
                    $tmpl = new \core\output\inplace_editable('local_tem', 'proctor', $data->id,
                        self::can_assign_proctor($data),
                        self::get_user_profile_url($data->proctor), $data->proctor
                    );

                    $tmpl->set_type_select($proctors);

                    $var = $OUTPUT->render($tmpl);
                } else {
                    $data->{$varclass} = 'flagged';
                    $var = '';
                }
                break;
            case 'proctormanager':
                $var = self::get_user_profile_url($data->proctormanager);
                break;
            case 'completed':
                $var = '-';
                $data->missed = 0;

                list($allusers, $availableusers, $completed) = self::get_session_users($data->quizid, $data->courseid, $data->groupid, $data->sessionnum);

                $totalnumber = count($availableusers);
                $numberofcompleted = count($completed);
                $var = "$numberofcompleted/$totalnumber";

                $data->missed = $totalnumber - $numberofcompleted;
                break;
            case 'video':
                $var = '-';
                if (!empty($data->linktovideorecording)) {
                    $var = \html_writer::link($data->linktovideorecording, get_string('openvideo', 'local_tem'));
                }
                break;
            case 'report':
                $actionlinks = '';

                // Report..
                if ((time() - $data->starttime) < HOURSECS) {
                    // No action
                } else  if (helper::can_submit_report($data)) {
                    $actionurl = new \moodle_url('/local/tem/report.php', ['sessionid' => $data->id]);
                    if ($submission =  $DB->get_record('local_tem_report', ['sessionid' => $data->id])) {
                        $actionicontext = get_string('view', 'local_tem');
                        $actionlinks .= html_writer::link($actionurl->out(false), $actionicontext,
                                ['class' => 'action-add btn btn-info btn-sm rounded', 'role' => 'button']).' ';
                    } else {
                        $actionicontext = get_string('submit', 'local_tem');
                        $actionlinks .= html_writer::link($actionurl->out(false), $actionicontext,
                                ['class' => 'action-add btn btn-danger btn-sm rounded', 'role' => 'button']).' ';
                    }
                }

                // Delete.
                if (is_siteadmin()) {
                    $actionurl = new moodle_url('/local/tem/session_delete.php', ['id' => $data->id]);
                    $actioniconurl = $OUTPUT->image_url('i/delete');
                    $actionicontext = get_string('delete');
                    $actionicon = html_writer::img($actioniconurl, $actionicontext,
                        ['class' => 'icon', 'title' => $actionicontext]);
                    $actionlinks .= html_writer::link($actionurl->out(false), $actionicon,
                        ['class' => 'action-delete', 'role' => 'button']).' ';
                }

                $var = $actionlinks;
                break;
            case 'course':
                $var = '';
                $url =  new \moodle_url('/course/view.php', ['id' => $data->courseid]);
                if ($data->irregulareventviolation) {
                    $var .= '<div class="flaggedsession red"><i class="fa fa-flag" aria-hidden="true"></i></div>';
                }
                $var .= \html_writer::link($url, $data->$column);
                break;
            case 'class':
                $url = new moodle_url('/blocks/ned_teacher_tools/deadline_manager.php', ['courseid' => $data->courseid, 'p' =>'group', 'group' => $data->groupid]);
                $var = \html_writer::link($url, $data->$column);
                break;
            case 'quiz':
                $quiz = $DB->get_record('quiz', ['id' => $data->quizid]);
                $cm = get_coursemodule_from_instance('quiz', $quiz->id);
                $url = new moodle_url('/mod/quiz/view.php', ['id' => $cm->id]);
                $var = \html_writer::link($url, $data->$column);
                break;
            case 'sessionnum':
                $sessionurl = clone $pageurl;
                if ($sessionurl->param('quizid')) {
                    $sessionurl->param('quizid', 0);
                    $sessionurl->param('groupid', 0);
                } else {
                    $sessionurl->param('quizid', $data->quizid);
                    $sessionurl->param('groupid', $data->groupid);
                }
                if ($data->totalsessionnum > 1) {
                    $var = \html_writer::link($sessionurl, $data->$column . '/' . $data->totalsessionnum);
                } else {
                    $var = $data->$column . '/' . $data->totalsessionnum;
                }
                break;
            default:
                $encoding = mb_detect_encoding($data->$column);
                $var = @iconv($encoding, "Windows-1252", $data->$column);
        }

        return $var;
    }

    public static function get_reort_file_links($sessionid) {
        global $DB;

        $out = [];
        $contextsystem = context_system::instance();

        if ($submission =  $DB->get_record('local_tem_report', ['sessionid' => $sessionid])) {
            $fs = get_file_storage();
            $files = $fs->get_area_files($contextsystem->id, 'local_tem', 'attachment', $sessionid);
            foreach ($files as $file) {
                if (!$file->is_directory()) {
                    $out[] = NED::file_get_link($file, true, true);
                }
            }

            return $out;
        }
    }

    /**
     * @param int $groupid
     * @return array
     * @throws \dml_exception
     */
    public static function get_school_proctors($groupid): array {
        global $DB;

        $proctorbadge = get_config('local_tem', 'proctorbadge');

        $proctors = [];

        $sql = "SELECT cm.cohortid 
                  FROM {groups_members} gm
                  JOIN {cohort_members} cm 
                    ON gm.userid = cm.userid
	              JOIN {user} u 
                    ON gm.userid = u.id
	              JOIN {cohort} c
                    ON cm.cohortid = c.id
                  JOIN {groups} g ON gm.groupid = g.id
                 WHERE u.deleted = 0
                   AND groupid = ?
                   AND c.name NOT LIKE '*%'
                   AND c.component = 'local_profilecohort'
                   AND c.idnumber = SUBSTR(g.name, 1, 4)
              GROUP BY cm.cohortid";

        if (!$schools = $DB->get_records_sql($sql, [$groupid])) {
            return $proctors;
        }

        list($insql, $params) = $DB->get_in_or_equal(array_keys($schools), SQL_PARAMS_NAMED);
        $params['badgeid'] = $proctorbadge;

        $sql = "SELECT u.* 
                  FROM {badge_issued} bi
                  JOIN {cohort_members} cm
                    ON bi.userid = cm.userid
                  JOIN {user} u 
                    ON bi.userid = u.id
                 WHERE badgeid = :badgeid 
                   AND cm.cohortid {$insql} 
                   AND u.deleted = 0";

        if (!$users = $DB->get_records_sql($sql, $params)) {
            return $proctors;
        }

        foreach ($users as $user) {
            $proctors[$user->id]  = fullname($user);
        }

        return [0 => get_string('choose') . '...'] + $proctors;
    }

    /**
     * @return bool
     */
    public static function can_assign_proctor($session): bool {
        return self::can_manage_session($session);
    }


    /**
     * @return bool
     * @throws \dml_exception
     */
    public static function can_view_sessions(): bool {
        $contextsystem = context_system::instance();

        if (has_any_capability([
            'local/tem:manageallsessions',
            'local/tem:viewallsessions',
            'local/tem:manageownsessions',
            'local/tem:viewownsessions',
            'local/tem:viewappointedsessions',
            'local/tem:viewownclasssessions',
            'local/tem:manageflaggedsessions',
        ], $contextsystem)) {
            return true;
        }

        return false;
    }

    /**
     * @param int $schoolid
     * @return array
     * @throws \coding_exception
     * @throws \dml_exception
     */
    public static function get_view_filter(int $schoolid = 0): array {
        global $USER, $DB;

        $contextsystem = context_system::instance();

        if (has_any_capability([
            'local/tem:manageallsessions',
            'local/tem:viewallsessions',
            'local/tem:manageownsessions',
            'local/tem:viewownsessions',
            'local/tem:viewappointedsessions',
            'local/tem:viewownclasssessions',
            'local/tem:manageflaggedsessions',
        ], $contextsystem)) {

            $filter = [];
            $params = [];

            // Can view all.
            if (has_any_capability(['local/tem:manageallsessions', 'local/tem:viewallsessions'], $contextsystem)) {
                self::add_school_filter($schoolid, $filter, $params);
                if (!$filter) {
                    return ['', []];
                }
            } else {
                // Can view own school.
                if (has_any_capability(['local/tem:manageownsessions', 'local/tem:viewownsessions'], $contextsystem)) {
                    if ($schoolid = NED::get_user_school($USER->id, true)) {
                        self::add_school_filter($schoolid, $filter, $params);
                    }
                }

                if (has_capability('local/tem:viewappointedsessions', $contextsystem)) {
                    $filter[] = 'p.proctor = :proctor1';
                    $params += ['proctor1' => $USER->id];
                }

                if (has_capability('local/tem:viewownclasssessions', $contextsystem)) {
                    $groups = [];
                    if ($courses = enrol_get_my_courses()) {
                        foreach ($courses as $course) {
                            if (has_capability('mod/assign:viewgrades', \context_course::instance($course->id))) {
                                $groups += NED::get_all_user_course_groups($course, $USER->id);
                            }
                        }
                    }
                    if ($groups) {
                        list($insql, $ownparams) = $DB->get_in_or_equal(array_keys($groups), SQL_PARAMS_NAMED);
                        $filter[] = "g.id {$insql}";
                        $params += $ownparams;
                    }
                }

                if (has_capability('local/tem:manageflaggedsessions', $contextsystem)) {
                    $filter[] = 'r.irregulareventviolation = 1';
                }
            }

            if ($filter) {
                $filter = ' AND (' . implode(' OR ', $filter) . ')';
                return [$filter, $params];
            }
        }

        return [' AND 0=1', []];
    }

    /**
     * @param $schoolid
     * @param $filter
     * @param $params
     * @return void
     * @throws \coding_exception
     * @throws \dml_exception
     */
    public static function add_school_filter($schoolid, &$filter, &$params): void {
        global $DB;

        if (!$schoolid) {
            return;
        }

        $sql = "SELECT DISTINCT gm.groupid
                  FROM {cohort} c
                  JOIN {cohort_members} m ON c.id = m.cohortid
                  JOIN {groups_members} gm ON m.userid = gm.userid
                  JOIN {groups} g ON gm.groupid = g.id
                 WHERE c.contextid = 1 
                   AND c.id = ?
                   AND c.idnumber = SUBSTR(g.name, 1, 4)";
        if ($groups = $DB->get_records_sql($sql, [$schoolid])) {
            list($insql, $ownparams) = $DB->get_in_or_equal(array_keys($groups), SQL_PARAMS_NAMED);
            $filter[] = "g.id {$insql}";
            $params += $ownparams;
        } else {
            $filter[] = "g.id = -1";
        }
    }

    /**
     * @return array
     * @throws \coding_exception
     * @throws \dml_exception
     */
    public static function get_manage_filter(): array {
        global $USER, $DB;

        $contextsystem = context_system::instance();

        if (has_any_capability([
            'local/tem:manageallsessions',
            'local/tem:manageownsessions',
            'local/tem:manageflaggedsessions',
        ], $contextsystem)) {
            // Can view all.
            if (has_capability('local/tem:manageallsessions', $contextsystem)) {
                return ['', []];
            }

            $filter = [];
            $params = [];

            // Can manage own school.
            if (has_capability('local/tem:manageownsessions', $contextsystem)) {
                if ($schoolid = NED::get_user_school($USER->id, true)) {
                    $sql = "SELECT DISTINCT gm.groupid
                              FROM {cohort} c
                              JOIN {cohort_members} m
                                ON c.id = m.cohortid
                              JOIN {groups_members} gm
                                ON m.userid = gm.userid
                              JOIN {groups} g
                                ON gm.groupid = g.id
                             WHERE c.contextid = 1 AND c.id = ?
                               AND c.idnumber = SUBSTR(g.name, 1, 4)";
                    if ($groups = $DB->get_records_sql($sql, [$schoolid])) {
                        list($insql, $ownparams) = $DB->get_in_or_equal(array_keys($groups), SQL_PARAMS_NAMED);
                        $filter[] = "g.id {$insql}";
                        $params += $ownparams;
                    }
                }
            }

            if (has_capability('local/tem:viewappointedsessions', $contextsystem)) {
                $filter[] = 'p.proctor = :proctor1';
                $params += ['proctor1' => $USER->id];

            }

            if (has_capability('local/tem:manageflaggedsessions', $contextsystem)) {
                $filter[] = 'r.irregulareventviolation = 1';
            }

            if ($filter) {
                $filter = ' AND (' . implode(' OR ', $filter) . ')';

                return [$filter, $params];
            }
        }

        return [' AND 0=1', []];
    }

    /**
     * @param $session
     * @return bool
     * @throws \coding_exception
     * @throws \dml_exception
     */
    public static function can_manage_session($session): bool {
        global $DB;

        if (!$enabletem = self::is_school_tem_enabled()) {
            return false;
        }

        list($filtersql, $params) = self::get_manage_filter();
        $params['sessionid'] = $session->id;

        $sql = "SELECT s.id                      
                  FROM {local_tem_session} AS s
             LEFT JOIN {local_tem_proctor} AS p ON s.id = p.sessionid
             LEFT JOIN {local_tem_report} AS r ON s.id = r.sessionid
                  JOIN {course} AS c ON s.courseid = c.id
                  JOIN {groups} AS g ON s.groupid = g.id
                  JOIN {quiz} AS q ON s.quizid = q.id
                  WHERE s.id = :sessionid
                        $filtersql";

        return $DB->record_exists_sql($sql, $params);
    }

    /**
     * @return bool
     * @throws \coding_exception
     * @throws \dml_exception
     */
    public static function can_manage_any_session(): bool {
        global $DB;
        list($filtersql, $params) = self::get_manage_filter();

        $sql = "SELECT s.id                      
                  FROM {local_tem_session} AS s
             LEFT JOIN {local_tem_proctor} AS p ON s.id = p.sessionid
             LEFT JOIN {local_tem_report} AS r ON s.id = r.sessionid
                  JOIN {course} AS c ON s.courseid = c.id
                  JOIN {groups} AS g ON s.groupid = g.id
                  JOIN {quiz} AS q ON s.quizid = q.id
                 WHERE 0 = 0
                       $filtersql";

        return $DB->record_exists_sql($sql, $params);
    }

    /**
     * @return bool
     * @throws \coding_exception
     * @throws \dml_exception
     */
    public static function manager_have_any_session_without_proctor_and_schedule($timeframe = 0): bool {
        global $DB;
        list($filtersql, $params) = self::get_manage_filter();

        if ($timeframe) {
            $timeframefilter = " AND (" . time() . " - (SELECT MAX(qa.timefinish) FROM {local_tem_attempt} a JOIN {quiz_attempts} qa ON a.attemptid = qa.id WHERE a.sessionid = s.id)) > " . $timeframe;
        } else {
            $timeframefilter = '';
        }

        $sql = "SELECT s.id, p.proctor, s.timescheduled                       
                  FROM {local_tem_session} AS s
             LEFT JOIN {local_tem_proctor} AS p ON s.id = p.sessionid
             LEFT JOIN {local_tem_report} AS r ON s.id = r.sessionid
                  JOIN {course} AS c ON s.courseid = c.id
                  JOIN {groups} AS g ON s.groupid = g.id
                  JOIN {quiz} AS q ON s.quizid = q.id
                 WHERE 0 = 0
                       $filtersql
                       $timeframefilter
                HAVING p.proctor IS NULL OR  p.proctor = 0 OR s.timescheduled IS NULL";

        return $DB->record_exists_sql($sql, $params);
    }

    /**
     * @param $session
     * @return bool
     * @throws \coding_exception
     * @throws \dml_exception
     */
    public static function can_delete_session($session): bool {
        return self::can_manage_session($session);
    }

    /**
     * @param $session
     * @return bool
     * @throws \coding_exception
     * @throws \dml_exception
     */
    public static function can_submit_report($session): bool {
        global $USER;

        if (!$enabletem = self::is_school_tem_enabled()) {
            return false;
        }

        if (self::can_manage_session($session)) {
            return true;
        }
        $context = context_system::instance();
        if ($USER->id == $session->proctor && has_capability('local/tem:viewappointedsessions', $context)) {
            return true;
        }
        return false;
    }

    /**
     * @param $session
     * @return bool
     * @throws \coding_exception
     * @throws \dml_exception
     */
    public static function can_manage_flagged($session): bool {
        global $USER;

        if (!$enabletem = self::is_school_tem_enabled()) {
            return false;
        }

        if (self::can_manage_session($session)) {
            return true;
        }
        $context = context_system::instance();
        if ($USER->id == $session->proctor && has_capability('local/tem:manageflaggedsessions', $context)) {
            return true;
        }
        return false;
    }

    /**
     * @param $sessionid
     * @return false|mixed
     * @throws \dml_exception
     */
    public static function get_session($sessionid) {
        global $DB;

        $sql = "SELECT s.id, 
                       s.courseid, 
                       c.shortname AS course, 
                       s.groupid,
                       g.name AS class, 
                       s.quizid, 
                       q.name AS quiz, 
                       q.timeopen,
                       s.timestart starttime,
                       s.sessionnum,
                       s.groupid,
                       p.proctor,
                       r.id reportid,
                       r.irregulareventviolation,
                       s.timescheduled AS scheduled,
                       s.numberofstudents
                  FROM {local_tem_session} AS s
             LEFT JOIN {local_tem_proctor} AS p ON s.id = p.sessionid
             LEFT JOIN {local_tem_report} AS r ON s.id = r.sessionid
                  JOIN {course} AS c ON s.courseid = c.id
                  JOIN {groups} AS g ON s.groupid = g.id
                  JOIN {quiz} AS q ON s.quizid = q.id
                  WHERE s.id = ?";
        return $DB->get_record_sql($sql, [$sessionid]);
    }

    /**
     * @param $userid
     * @return bool
     * @throws \dml_exception
     */
    public static function is_designated_proctor($userid): bool {
        global $DB;

        $context = context_system::instance();

        if (!has_capability('local/tem:viewappointedsessions', $context)) {
            return false;
        }

        $sql = "SELECT s.*
                  FROM {local_tem_session} s
	              JOIN {local_tem_proctor} p 
                    ON s.id = p.sessionid 
                   AND p.proctor = ?";

        return $DB->record_exists_sql($sql, [$userid]);
    }

    /**
     * @param $userid
     * @return bool
     * @throws \dml_exception
     */
    public static function is_proctor_manager($userid): bool {
        if ($schoolid = NED::get_user_school($userid, true)) {
            if ($school = new school($schoolid)) {
                $proctormanager = $school->get('proctormanager');
                if ($proctormanager) {
                    return $proctormanager == $userid;
                }
                $SM = SM::get_school_manager();
                if ($administrator = $SM->get_school_students($schoolid, true, $SM::SCHOOL_ADMINISTRATOR_ROLE, false)) {
                    $administrator = reset($administrator);
                    return $administrator->id == $userid;
                }
            }
        }
        return false;
    }

    /**
     * @param $userid
     * @return bool
     * @throws \dml_exception
     */
    public static function is_proctor_submission_required($userid, $timeframe = 0): bool {
        global $DB;

        if ($timeframe) {
            $timeframefilter = " AND (" . time() . " - (SELECT MAX(qa.timefinish) FROM {local_tem_attempt} a JOIN {quiz_attempts} qa ON a.attemptid = qa.id WHERE a.sessionid = s.id)) > " . $timeframe;
        } else {
            $timeframefilter = '';
        }

        $sql = "SELECT s.*, r.id reportid
                  FROM {local_tem_session} s
	              JOIN {local_tem_proctor} p 
                    ON s.id = p.sessionid
             LEFT JOIN {local_tem_report} r 
                    ON s.id = r.sessionid
                 WHERE p.proctor = ?
                       {$timeframefilter}
                HAVING reportid IS NULL";

        return $DB->record_exists_sql($sql, [$userid]);
    }

    /**
     * @param $userid
     * @return bool
     * @throws \dml_exception
     */
    public static function is_school_tem_enabled($userid = 0): bool {
        global $USER;

        if (is_siteadmin()) {
            return true;
        }

        if (!$userid) {
            $userid = $USER->id;
        }
        if ($schoolid = NED::get_user_school($userid, true)) {
            if ($school = new school($schoolid)) {
                if ($school->get('enabletem')) {
                    return true;
                }
            }
        }
        return false;
    }

    /**
     * @param $userid
     * @param $redirect
     * @return void
     * @throws \coding_exception
     * @throws \dml_exception
     * @throws \moodle_exception
     */
    public static function tem_submission_notify_or_redirect($userid = null, $redirect = false, $timeframe = 0) {
        global $USER, $SESSION;

        if (!$userid) {
            $userid = $USER->id;
        }

        $SESSION->temwarning = false;
        $url = new \moodle_url('/local/tem/sessions.php');

        $enabletem = self::is_school_tem_enabled($userid);

        if (self::is_designated_proctor($userid)) {
            if (self::is_proctor_submission_required($userid, $timeframe)) {
                $SESSION->temwarning = true;
                if ($redirect && $enabletem) {
                    redirect($url);
                }
            }
        } else if (self::is_proctor_manager($userid)) {
            if (self::manager_have_any_session_without_proctor_and_schedule($timeframe)) {
                $SESSION->temwarning = true;
                if ($redirect && $enabletem) {
                    redirect($url);
                }
            }
        }
    }

    /**
     * @return void
     * @throws \coding_exception
     * @throws \dml_exception
     * @throws \moodle_exception
     */
    public static function tem_submission_notify_or_redirect_check() {
        self::tem_submission_notify_or_redirect(null, true, DAYSECS);
    }

    /**
     * @param $userid
     * @return string
     * @throws \moodle_exception
     */
    public static function get_user_profile_url($userid): ?string {
        if ($user = \core_user::get_user($userid)) {
            $url = new moodle_url('/user/profile.php', ['id' => $user->id]);
            return html_writer::link($url, fullname($user), []);
        }
        return null;
    }

    /**
     * @param $quizid
     * @param $courseid
     * @param $groupid
     * @param $sessionnum
     * @return array|array[]
     * @throws \dml_exception
     */
    public static function get_session_users($quizid, $courseid, $groupid, $sessionnum) {
        global $DB;

        $coursecontext = \context_course::instance($courseid);

        if (!$activestudents = get_enrolled_users($coursecontext, 'mod/assign:submit', $groupid, 'u.*', 'u.lastname', 0, 0, false)) {
            return [[], [], []];
        }

        $allusers = array_keys($activestudents);
        $completed = [];

        $sql = "SELECT DISTINCT qa.userid
                  FROM {local_tem_attempt} a
                  JOIN {quiz_attempts} qa 
                    ON a.attemptid = qa.id
	              JOIN {local_tem_session} s
	                ON a.sessionid = s.id
                 WHERE s.courseid = ?
                   AND qa.quiz = ?
                   AND s.groupid = ?
                   AND s.sessionnum < ? 
                   AND qa.state = ?";

        if (!$prevattempts = $DB->get_records_sql($sql, [$courseid, $quizid, $groupid, $sessionnum, \quiz_attempt::FINISHED])) {
            $availableusers = $allusers;
        } else {
            foreach ($prevattempts as $userid => $attempt) {
                if (array_key_exists($userid, $activestudents)) {
                    unset($activestudents[$userid]);
                }
            }
            $availableusers = array_keys($activestudents);
        }

        $sql = "SELECT DISTINCT qa.userid
                  FROM {local_tem_attempt} a
                  JOIN {quiz_attempts} qa 
                    ON a.attemptid = qa.id
	              JOIN {local_tem_session} s
	                ON a.sessionid = s.id
                 WHERE s.courseid = ?
                   AND qa.quiz = ?
                   AND s.groupid = ?
                   AND s.sessionnum = ? 
                   AND qa.state = ?";
        if (!$attempts = $DB->get_records_sql($sql, [$courseid, $quizid, $groupid, $sessionnum, \quiz_attempt::FINISHED])) {
            return [$allusers, $availableusers, $completed];
        }

        foreach ($attempts as $userid => $attempt) {
            if (array_key_exists($userid, $activestudents)) {
                $completed[] = $userid;
            }
        }

        return [$allusers, $availableusers, $completed];
    }

    /**
     * @param $sessionid
     * @param $proctorid
     * @return void
     * @throws \dml_exception
     */
    public static function assign_proctor($sessionid, $proctorid) {
        global $DB;

        if ($record = $DB->get_record('local_tem_proctor', ['sessionid' => $sessionid])) {
            $DB->update_record('local_tem_proctor', ['id' => $record->id, 'proctor' => $proctorid]);
        } else {
            $DB->insert_record('local_tem_proctor', ['sessionid' => $sessionid, 'proctor' => $proctorid]);
        }
    }
    /**
     * @param $sessionid
     * @param $proctorid
     * @return void
     * @throws \dml_exception
     */
    public static function assign_default_proctor($sessionid) {
        global $DB;

        $SM = SM::get_school_manager();

        $session = $DB->get_record('local_tem_session', ['id' => $sessionid], '*', MUST_EXIST);

        if ($administrator = $SM->get_school_students($session->schoolid, true, $SM::SCHOOL_ADMINISTRATOR_ROLE, false)) {
            $administrator = reset($administrator);
            self::assign_proctor($session->id, $administrator->id);
        }
    }
}