<?php
// This file is part of Moodle - https://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <https://www.gnu.org/licenses/>.

/**
 * Report class.
 *
 * @package     local_tem
 * @category    classes
 * @copyright   2023 Michael Gardener <mgardener@cissq.com>
 * @license     https://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace local_tem;

use local_proxy\shared_lib as NED;
use local_schoolmanager\school;

defined('MOODLE_INTERNAL') || die();

/**
 * Class report
 */
class report {

    const STATUS_RESOLVED = 'resolved';
    const STATUS_FOLLOWUP = 'followup';
    const SEVERITY_LOW = 'low';
    const SEVERITY_HIGH = 'high';
    const SEVERITY_UNKNOWN = 'unknown';
    /**
     * @var int
     */
    private $sid;
    /**
     * @var false|mixed|\stdClass
     */
    public $session;
    /**
     * @var false|mixed|\stdClass
     */
    public $course;
    /**
     * @var false|mixed|\stdClass
     */
    public $group;
    /**
     * @var false|mixed|\stdClass
     */
    public $quiz;
    /**
     * @var false|mixed|\stdClass
     */
    public $instance;

    /**
     * @param $sid
     * @throws \dml_exception
     */
    public function __construct($sid) {
        $this->sid = $sid;
        $this->load_data();
    }

    /**
     * @return void
     * @throws \dml_exception
     */
    private function load_data(): void {
        global $DB;
        $this->instance = $DB->get_record('local_tem_report', ['sessionid' => $this->sid]);
        $this->session = $DB->get_record('local_tem_session', ['id' => $this->sid], '*', MUST_EXIST);
        $this->course = $DB->get_record('course', ['id' => $this->session->courseid], '*', MUST_EXIST);
        $this->group = $DB->get_record('groups', ['id' => $this->session->groupid], '*', MUST_EXIST);
        $this->quiz = $DB->get_record('quiz', ['id' => $this->session->quizid], '*', MUST_EXIST);
    }

    /**
     * @return array
     * @throws \dml_exception
     */
    public function get_attempts(): array {
        global $DB;

        $sql = "SELECT sa.id, 
                       sa.sessionid, 
                       sa.attemptid, 
                       sa.ip, 
                       qa.userid, 
                       qa.state, 
                       qa.timestart, 
                       qa.timefinish, 
                       u.firstname, 
                       u.lastname, 
                       u.deleted
                  FROM {local_tem_attempt} sa
                  JOIN {quiz_attempts} qa
                    ON sa.attemptid = qa.id
                  JOIN {user} u
                    ON qa.userid = u.id
                 WHERE sa.sessionid = ? AND u.deleted = 0
              ORDER BY qa.timestart ASC";

        return $DB->get_records_sql($sql, [$this->sid]);
    }

    /**
     * @return bool
     * @throws \dml_exception
     */
    public function user_has_any_attempt($userid): bool {
        global $DB;

        $sql = "SELECT sa.id, 
                       sa.sessionid, 
                       sa.attemptid, 
                       sa.ip, 
                       qa.quiz, 
                       qa.userid, 
                       qa.state, 
                       qa.timestart, 
                       qa.timefinish
                  FROM {local_tem_attempt} sa
                  JOIN {quiz_attempts} qa
                    ON sa.attemptid = qa.id
                 WHERE qa.quiz = ? AND qa.userid = ?";

        return $DB->record_exists_sql($sql, [$this->quiz->id, $userid]);
    }

    /**
     * @return form\report
     */
    public function get_form($edit): form\report {
        global $USER;

        $staticiptype = false;
        if ($schoolid = NED::get_user_school($USER->id, true)) {
            if ($school = new school($schoolid)) {
                $staticiptype = $school->iptype == school::IP_TYPE_STATIC;
            }
        }

        $mform = new \local_tem\form\report(null, [
            'sameip' => $this->all_attempts_has_same_ip(),
            'staticiptype' => $staticiptype,
            'hasuncompletedattempts' => $this->has_uncompleted_attempts(),
            'instance' => $this->instance,
            'edit' => $edit,
        ]);

        return $mform;
    }

    /**
     * @return bool
     */
    public function all_attempts_has_same_ip(): bool {
        $ips = [];
        $attempts = $this->get_attempts();
        foreach ($attempts as $attempt) {
            if (!empty($attempt->ip)) {
                $ips[$attempt->ip] = $attempt->ip;
            }
        }

        if (count($ips) == 1) {
            return true;
        }

        return false;
    }

    /**
     * @return bool
     * @throws \dml_exception
     */
    public function has_uncompleted_attempts(): bool {
        $attempts = $this->get_attempts();
        if ($this->session->numberofstudents > count($attempts)) {
            return true;
        }
        return false;
    }

    public static function resolution_status_options() {
        return [
            '' => get_string('choose'),
            self::STATUS_RESOLVED => get_string(self::STATUS_RESOLVED, 'local_tem'),
            self::STATUS_FOLLOWUP => get_string(self::STATUS_FOLLOWUP, 'local_tem'),
        ];
    }

    public static function resolution_severity_options() {
        return [
            '' => get_string('choose'),
            self::SEVERITY_LOW => get_string(self::SEVERITY_LOW, 'local_tem'),
            self::SEVERITY_HIGH => get_string(self::SEVERITY_HIGH, 'local_tem'),
            self::SEVERITY_UNKNOWN => get_string(self::SEVERITY_UNKNOWN, 'local_tem'),
        ];
    }
}
