<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Lib.
 *
 * @package     local_tem
 * @category    lib
 * @copyright   2023 Michael Gardener <mgardener@cissq.com>
 * @license     http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

use local_tem\helper;

defined('MOODLE_INTERNAL') || die();

/**
 * @param $itemtype
 * @param $itemid
 * @param $newvalue
 * @return \core\output\inplace_editable|void
 * @throws coding_exception
 * @throws dml_exception
 * @throws invalid_parameter_exception
 * @throws moodle_exception
 * @throws restricted_context_exception
 */
function local_tem_inplace_editable($itemtype, $itemid, $newvalue) {
    global $DB;

    if ($itemtype === 'proctor') {
        \external_api::validate_context(context_system::instance());

        $session = helper::get_session($itemid);

        if (!helper::can_assign_proctor($session)) {
            throw new moodle_exception('Permision');
        }

        // Clean input and update the record.
        $newvalue = clean_param($newvalue, PARAM_INT);

        helper::assign_proctor($itemid, $newvalue);

        $proctors = helper::get_school_proctors($session->groupid);

        $tmpl = new \core\output\inplace_editable('local_tem', 'proctor', $itemid, true, null, $newvalue);
        $tmpl->set_type_select($proctors);

        helper::tem_submission_notify_or_redirect();

        return $tmpl;
    }
}

/**
 * Map icons for font-awesome themes.
 */
function local_tem_controller_get_fontawesome_icon_map() {
    return [
        'local_tem:i/edit' => 'fa-external-link'
    ];
}

/**
 * Serves assignment submissions and other files.
 *
 * @param mixed $course course or id of the course
 * @param mixed $cm course module or id of the course module
 * @param context $context
 * @param string $filearea
 * @param array $args
 * @param bool $forcedownload
 * @param array $options - List of options affecting file serving.
 * @return bool false if file not found, does not return if found - just send the file
 */
function local_tem_pluginfile($course, $cm, context $context, $filearea, $args, $forcedownload, array $options=array()) {
    global $DB, $CFG;

    require_login();

    if ($context->contextlevel != CONTEXT_SYSTEM) {
        return false;
    }

    if (!has_any_capability([
        'local/tem:manageallsessions',
        'local/tem:viewallsessions',
        'local/tem:manageownsessions',
        'local/tem:viewownsessions',
        'local/tem:viewappointedsessions',
        'local/tem:viewownclasssessions',
        'local/tem:manageflaggedsessions',
    ], $context)) {
        return false;
    }

    $itemid = (int)array_shift($args);


    $record = $DB->get_record('local_tem_report',
        array('sessionid' => $itemid),
        '*',
        MUST_EXIST);

    $relativepath = implode('/', $args);

    $fullpath = "/{$context->id}/local_tem/$filearea/$itemid/$relativepath";

    $fs = get_file_storage();

    if (!($file = $fs->get_file_by_hash(sha1($fullpath))) || $file->is_directory()) {
        return false;
    }

    // Download MUST be forced - security!
    send_stored_file($file, 0, 0, true, $options);
}