<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Report.
 *
 * @package   local_tem
 * @copyright 2021 Michael Gardener <mgardener@cissq.com>
 * @license   http://www.gnu.org/copyleft/gpl.html gnu gpl v3 or later
 */

use local_tem\output\report_attempts;
use local_tem\output\report_header;
use local_tem\report;
use local_tem\helper;

require_once(dirname(__FILE__, 3) . '/config.php');

$sessionid = required_param('sessionid', PARAM_INT);
$returnurl = optional_param('returnurl', '', PARAM_LOCALURL);
$edit = optional_param('edit', 0, PARAM_INT);

require_login();

$context = context_system::instance();

$proctor = $DB->get_record('local_tem_proctor', ['sessionid' => $sessionid]);

$session = helper::get_session($sessionid);

if (!helper::can_submit_report($session) && !helper::can_manage_flagged($session)) {
    throw new moodle_exception('permission');
}

$thispageurl = new moodle_url('/local/tem/report.php', [
    'sessionid' => $sessionid, 'edit' => $edit, 'returnurl' => $returnurl
]);

if ($returnurl) {
    $redirecturl = new \moodle_url($returnurl);
} else {
    $redirecturl = new moodle_url('/local/tem/sessions.php', []);
}

$PAGE->set_url($thispageurl);
$PAGE->set_pagelayout('base');
$PAGE->set_context($context);
$PAGE->add_body_classes(['path-report-ghs', 'path-blocks-ned_teacher_tools']);

$reportrec = $DB->get_record('local_tem_report', ['sessionid' => $sessionid]);

$renderer = $PAGE->get_renderer('local_tem');

$report = new report($sessionid);
$reportheader = new report_header($report, $edit, $thispageurl);
$reportattempts = new report_attempts($report);

$mform = $report->get_form($edit);
$fileoptions = $mform->get_file_options();

if ($reportrec) {
    $data = clone $reportrec;
} else {
    $data = new stdClass();
    $data->sessionid = $sessionid;
}

$draftitemid = file_get_submitted_draft_itemid('files_filemanager');

file_prepare_draft_area(
    $draftitemid,
    $context->id,
    'local_tem',
    'attachment',
    $sessionid,
    $mform->get_file_options()
);

$data->files_filemanager = $draftitemid;
$data->signaturetext = $data->signature ?? '';
$data->returnurl = $returnurl;
$data->edit = $edit;

$mform->set_data($data);

if ($mform->is_cancelled()) {
    redirect($redirecturl);
} elseif ($fromform = $mform->get_data()) {
    $rec = new stdClass();
    $rec->sessionid = $fromform->sessionid;

    if (empty($reportrec) || (!empty($reportrec) && !empty($edit))) {
        $rec->numberofproctors = $fromform->numberofproctors;
        $rec->sameroom = $fromform->sameroom;

        if ($rec->sameroom) {
            $rec->sameroomreason = '';
            $rec->differentipreason = $fromform->differentipreason ?? '';
        } else {
            $rec->sameroomreason = $fromform->sameroomreason;
            $rec->differentipreason = '';
        }

        $rec->irregulareventviolation = $fromform->irregulareventviolation;
        $rec->irregulareventviolationreason = $fromform->irregulareventviolationreason ?? '';
        $rec->noattemptreason = $fromform->noattemptreason ?? '';
        $rec->linktovideorecording = $fromform->linktovideorecording;
        $rec->signature = $fromform->signature;

        file_save_draft_area_files(
            $fromform->files_filemanager,
            $context->id,
            'local_tem',
            'attachment',
            $rec->sessionid,
            $mform->get_file_options()
        );
    }

    if (has_capability('local/tem:manageflaggedsessions', $context)) {
        if (isset($fromform->resolutionstatus)) {
            $rec->resolutionstatus = $fromform->resolutionstatus;
        }
        if (isset($fromform->resolutionseverity)) {
            $rec->resolutionseverity = $fromform->resolutionseverity;
        }
        if (isset($fromform->resolutionnote)) {
            $rec->resolutionnote = $fromform->resolutionnote;
        }
    }

    if ($reportrec) {
        $rec->id = $reportrec->id;
        $rec->timemodified = time();
        $DB->update_record('local_tem_report', $rec);
    } else {
        $rec->timecreated = time();
        $DB->insert_record('local_tem_report', $rec);
    }

    $record = $DB->get_record('local_tem_proctor', ['sessionid' =>  $rec->sessionid]);
    if (empty($record->proctor)) {
        helper::assign_default_proctor($rec->sessionid);
    }

    helper::tem_submission_notify_or_redirect();
    redirect($redirecturl);
    exit;
}

echo $OUTPUT->header();
echo html_writer::start_div('session-report-wrapper');
echo $renderer->render($reportheader);
echo $renderer->render($reportattempts);
$mform->display();
echo html_writer::end_div();
echo $OUTPUT->footer();