<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Sessions
 *
 * @package     local_tem
 * @copyright   2023 Michael Gardener <mgardener@cissq.com>
 * @license     http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

require_once(dirname(dirname(dirname(__FILE__))) . '/config.php');
require_once("$CFG->libdir/phpspreadsheet/vendor/autoload.php");
require_once($CFG->libdir.'/adminlib.php');

use local_tem\helper;

$schoolid  = optional_param('schoolid', 0, PARAM_INT);
$quizid  = optional_param('quizid', 0, PARAM_INT);
$groupid  = optional_param('groupid', 0, PARAM_INT);
$page      = optional_param('page', 0, PARAM_INT);
$perpage   = optional_param('perpage', 50, PARAM_INT);
$sort      = optional_param('sort', 'id', PARAM_ALPHANUM);
$dir       = optional_param('dir', 'ASC', PARAM_ALPHA);
$filter    = optional_param('filter', helper::FILTER_ACTION_REQUIRED, PARAM_ALPHA);
$search    = optional_param('search', '', PARAM_TEXT);
$hidedisabledschools    = optional_param('hidedisabledschools', null, PARAM_BOOL);

require_login(null, false);
$contextsystem = context_system::instance();

// Permission.
if (!helper::can_view_sessions()) {
    throw new moodle_exception('permission');
}

if (!is_null($hidedisabledschools)) {
    set_user_preference('local_tem_hide_disabled_schools', $hidedisabledschools);
}

$search = trim($search);

$urlparams = [
    'schoolid' => $schoolid,
    'quizid' => $quizid,
    'groupid' => $groupid,
    'page' => $page,
    'perpage' => $perpage,
    'sort' => $sort,
    'dir' => $dir,
    'filter' => $filter,
    'search' => $search,
];
$thispageurl = new moodle_url('/local/tem/sessions.php', $urlparams);

$PAGE->set_url($thispageurl);
$PAGE->set_pagelayout('base');
$PAGE->set_context($contextsystem);
$PAGE->add_body_classes(['path-report-ghs', 'path-blocks-ned_teacher_tools']);

$handler = new \local_schoolmanager\school_handler();
$schoolfilteraction = '';
$hidedisabledschools = get_user_preferences('local_tem_hide_disabled_schools', true);
if ($schoolfilter =  $handler->get_control_form($schoolid, $thispageurl, $hidedisabledschools)) {
    $hideschoolurl = clone $thispageurl;
    $hideschoolurl->param('hidedisabledschools', !$hidedisabledschools);
    $schoolfilteraction = $OUTPUT->render_from_template('local_tem/school_action', [
        'url' => $hideschoolurl->out(false),
        'imageurl' => $OUTPUT->image_url(($hidedisabledschools) ? 'show_disabled' : 'hide_disabled', 'local_tem'),
        'hidedisabledschools' => $hidedisabledschools,
    ]);
} else {
    $schoolid = 0;
    $thispageurl->param('schoolid', $schoolid);
}

$name = get_string('pluginname', 'local_tem');
$title = get_string('pluginname', 'local_tem');
$heading = $SITE->fullname;

// Breadcrumb.

$PAGE->set_title($title);
$PAGE->set_heading($heading);

$datacolumns = array(
    'id' => 's.id',
    'course' => 'c.shortname',
    'class' => 'g.name',
    'quiz' => 'q.name',
    'sessionnum' => 's.sessionnum',
    'scheduled' => 's.timescheduled',
    'starttime' => 's.timestart',
    'proctor' => 'CONCAT(u.firstname, " ", u.lastname)',
    'groupid' => 's.groupid',
    'quizid' => 'q.id',
    'video' => 'r.linktovideorecording',
    'endtime' => "(SELECT MAX(qa.timefinish) FROM {local_tem_attempt} a JOIN {quiz_attempts} qa ON a.attemptid = qa.id WHERE a.sessionid = s.id)",
    'duration' => "((SELECT MAX(qa.timefinish) FROM {local_tem_attempt} a JOIN {quiz_attempts} qa ON a.attemptid = qa.id WHERE a.sessionid = s.id) - s.timestart)",
    'report' => 'r.id',
    'school' => 'sc.code',
    'proctormanager' => 'CONCAT(u2.firstname, " ", u2.lastname)',
);

$params = [];
$pageparams = [];

// Filter.
$where = '';
list($filtersql, $params) = helper::get_view_filter($schoolid);

if ($hidedisabledschools) {
    $filtersql .= ' AND sc.enabletem = 1';
}

if ($filter == helper::FILTER_ACTION_REQUIRED) {
    $filtersql .= ' AND (s.timescheduled  = 0 OR p.proctor IS NULL OR r.id IS NULL)';
} else if ($filter == helper::FILTER_COMPLETED) {
    $filtersql .= ' AND (s.timescheduled  != 0 AND p.proctor IS NOT NULL AND r.id IS NOT NULL)';
} else if ($filter == helper::FILTER_SHOW_ALL) {
    $filtersql .= '';
}

if ($groupid) {
    $where .= " AND {$datacolumns['groupid']} = :groupid";
    $params['groupid'] = $groupid;
}
if ($quizid) {
    $where .= " AND {$datacolumns['quizid']} = :quizid";
    $params['quizid'] = $quizid;
}

if ($search) {
    $where .= " AND (" . $DB->sql_like($datacolumns['course'], ':search1', false, false);
    $params['search1'] = '%'.$DB->sql_like_escape($search).'%';

    $where .= " OR " . $DB->sql_like($datacolumns['class'], ':search2', false, false);
    $params['search2'] = '%'.$DB->sql_like_escape($search).'%';

    $where .= " OR " . $DB->sql_like($datacolumns['quiz'], ':search3', false, false) . ')';
    $params['search3'] = '%'.$DB->sql_like_escape($search).'%';
}

// Sort.
$order = '';
if ($sort) {
    $order = " ORDER BY $datacolumns[$sort] $dir";
}

// Count records for paging.
$countsql = "SELECT COUNT(1)
               FROM {local_tem_session} AS s
          LEFT JOIN {local_tem_proctor} AS p ON s.id = p.sessionid
          LEFT JOIN {local_tem_report} AS r ON s.id = r.sessionid
          LEFT JOIN {user} AS u ON p.proctor = u.id
          LEFT JOIN {local_schoolmanager_school} AS sc ON s.schoolid = sc.id
               JOIN {course} AS c ON s.courseid = c.id
               JOIN {groups} AS g ON s.groupid = g.id
               JOIN {quiz} AS q ON s.quizid = q.id
              WHERE s.quizid NOT IN (SELECT cm.instance 
                                       FROM {tag_instance} ti
                                       JOIN {tag} t ON  ti.tagid = t.id
                                       JOIN {course_modules} cm ON ti.itemid = cm.id
                                       JOIN {modules} m ON cm.module= m.id
                                      WHERE ti.itemtype = 'course_modules' AND t.name = 'tem excluded' AND m.name = 'quiz')
                    $filtersql
                    $where";
$totalcount = $DB->count_records_sql($countsql, $params);

// Table columns.
if (!is_siteadmin()) {
    $columns = array(
        'course',
        'class',
        'quiz',
        'sessionnum',
        'scheduled',
        'starttime',
        'endtime',
        'duration',
        'proctor',
        'completed',
        'missed',
        'video',
        'report',
    );
} else {
    $columns = array(
        'school',
        'proctormanager',
        'course',
        'class',
        'quiz',
        'sessionnum',
        'scheduled',
        'starttime',
        'endtime',
        'duration',
        'proctor',
        'completed',
        'missed',
        'video',
        'report',
    );
}

$sql = "SELECT s.id, 
               s.courseid, 
               c.shortname AS course, 
               s.groupid,
               g.name AS class, 
               q.id AS quizid, 
               q.name AS quiz, 
               q.timeopen,
               s.timestart starttime,
               s.sessionnum,
               s.totalsessionnum,
               p.proctor,
               r.id reportid,
               r.irregulareventviolation,
               s.timescheduled AS scheduled,
               s.numberofstudents,
               r.linktovideorecording,
               sc.code school,
               sc.proctormanager,
               (SELECT MAX(qa.timefinish) FROM {local_tem_attempt} a JOIN {quiz_attempts} qa ON a.attemptid = qa.id WHERE a.sessionid = s.id) endtime
          FROM {local_tem_session} AS s
     LEFT JOIN {local_tem_proctor} AS p ON s.id = p.sessionid
     LEFT JOIN {local_tem_report} AS r ON s.id = r.sessionid
     LEFT JOIN {user} AS u ON p.proctor = u.id     
     LEFT JOIN {local_schoolmanager_school} AS sc ON s.schoolid = sc.id
     LEFT JOIN {user} AS u2 ON sc.proctormanager = u2.id
          JOIN {course} AS c ON s.courseid = c.id
          JOIN {groups} AS g ON s.groupid = g.id
          JOIN {quiz} AS q ON s.quizid = q.id
          WHERE s.quizid NOT IN (SELECT cm.instance 
                                   FROM {tag_instance} ti
                                   JOIN {tag} t ON  ti.tagid = t.id
                                   JOIN {course_modules} cm ON ti.itemid = cm.id
                                   JOIN {modules} m ON cm.module= m.id
                                  WHERE ti.itemtype = 'course_modules' AND t.name = 'tem excluded' AND m.name = 'quiz')
                $filtersql
                $where
                $order";

foreach ($columns as $column) {
    $string[$column] = get_string($column, 'local_tem');
    if ($sort != $column) {
        $columnicon = "";
        $columndir = "ASC";
    } else {
        $columndir = $dir == "ASC" ? "DESC" : "ASC";
        $columnicon = ($dir == "ASC") ? "sort_asc" : "sort_desc";
        $columnicon = $OUTPUT->pix_icon('t/'.$columnicon, '', 'moodle', array('class' => 'iconsort'));
    }

    if ($column == 'completed' || $column == 'missed') {
        $$column = $string[$column];
    } else {
        $sorturl = clone $thispageurl;
        $sorturl->param('perpage', $perpage);
        $sorturl->param('sort', $column);
        $sorturl->param('dir', $columndir);

        $$column = html_writer::link($sorturl->out(false), $string[$column]).' '.$columnicon;
    }
}

$table = new html_table();

$table->head = array();
$table->wrap = array();
$table->attributes = ['class' => 'nedtable fullwidth'];

foreach ($columns as $column) {
    $table->head[$column] = $$column;
    $table->wrap[$column] = '';
}

// Override cell wrap.
$table->wrap['report'] = 'nowrap';

$tablerows = $DB->get_records_sql($sql, $params, $page * $perpage, $perpage);

$counter = ($page * $perpage);

foreach ($tablerows as $tablerow) {
    $row = new html_table_row();

    foreach ($columns as $column) {
        $varname = 'cell'.$column;
        $varclass = '_class_'.$column;
        $$varname = new html_table_cell(helper::session_data($tablerow, $column, $counter, $thispageurl, $pageparams));
        if (isset($tablerow->{$varclass})) {
            $$varname->attributes = ['class' => $tablerow->{$varclass}];
        }
    }

    $row->cells = array();
    foreach ($columns as $column) {
        $varname = 'cell' . $column;
        $row->cells[$column] = $$varname;
    }
    $table->data[] = $row;

}

echo $OUTPUT->header();
echo html_writer::start_div('page-content-wrapper', array('id' => 'page-content'));
echo html_writer::tag('h1', $title, array('class' => 'page-title'));

$pagingurl = clone $thispageurl;
$pagingbar = new paging_bar($totalcount, $page, $perpage, $pagingurl, 'page');

// Action links.
$actionrequiredurl = clone $thispageurl;
$actionrequiredurl->remove_all_params();
$actionrequiredurl->param('schoolid', $schoolid);
$actionrequiredurl->param('filter', helper::FILTER_ACTION_REQUIRED);
$actionrequiredlink = html_writer::link($actionrequiredurl, get_string('actionrequired', 'local_tem'),
    (($filter == helper::FILTER_ACTION_REQUIRED) ? ['class' => 'yellow'] : []));

$completedurl = clone $thispageurl;
$completedurl->remove_all_params();
$completedurl->param('schoolid', $schoolid);
$completedurl->param('filter', helper::FILTER_COMPLETED);
$completedlink = html_writer::link($completedurl, get_string('completed', 'local_tem'),
    (($filter == helper::FILTER_COMPLETED) ? ['class' => 'yellow'] : []));

$showallurl = clone $thispageurl;
$showallurl->remove_all_params();
$showallurl->param('schoolid', $schoolid);
$showallurl->param('filter', helper::FILTER_SHOW_ALL);
$showalllink = html_writer::link($showallurl, get_string('showall', 'local_tem'),
    (($filter == helper::FILTER_SHOW_ALL) ? ['class' => 'yellow'] : []));

$actionlinks =  html_writer::div(
    $actionrequiredlink.'&nbsp;|&nbsp;' .
    $completedlink.'&nbsp;|&nbsp;' .
    $showalllink,
    'export-link-wrapper',
    array('style' => 'text-align:center;')
);

$hiddenfields = [];
foreach ($urlparams as $key => $pageparam) {
    if ($key == 'search') continue;
    $hiddenfields[] = ['name' => $key, 'value' => $pageparam];
}


echo html_writer::start_div('d-flex flex-row');
echo html_writer::start_div('p-2');
echo $schoolfilter;
echo html_writer::end_div();
echo $schoolfilteraction;
echo html_writer::start_div('p-2 d-flex align-items-end');
echo $OUTPUT->render_from_template('local_tem/session_filter', [
    'search' => $search,
    'hiddenfields' => $hiddenfields,
]);
echo html_writer::end_div();
echo html_writer::end_div();

if ($outputpagingbar = $OUTPUT->render($pagingbar)) {
    echo $outputpagingbar;
} else {
    echo html_writer::tag('div', '', ['class' => 'dummy-pagination']);
}
echo html_writer::div($actionlinks, 'export-link-wrapper-top');
echo html_writer::table($table);
echo $outputpagingbar;

echo html_writer::end_div(); // Main wrapper.
echo $OUTPUT->footer();
