<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Helper
 *
 * @package    local_worksheet
 * @copyright  2020 Michael Gardener <mgardener@cissq.com>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace local_worksheet;

defined('MOODLE_INTERNAL') || die();

/** @var \stdClass $CFG */
require_once($CFG->dirroot.'/lib/questionlib.php');
require_once($CFG->dirroot.'/mod/quiz/locallib.php');
require_once($CFG->dirroot.'/mod/quiz/attemptlib.php');

use quiz;

/**
 * Class helper
 * @package local_worksheet
 */
class helper {
    /**
     * @return array
     * @throws \coding_exception
     */
    public static function get_courses() {
        global $DB, $USER;

        if (is_siteadmin()) {
            $enrolledcourses = $DB->get_records('course');
        } else {
            $enrolledcourses = enrol_get_my_courses(null, 'fullname');
        }

        foreach ($enrolledcourses as $index => $enrolledcourse) {
            if (!has_capability('mod/quiz:grade', \context_course::instance($enrolledcourse->id))) {
                unset($enrolledcourses[$index]);
            }
        }
        return $enrolledcourses;
    }

    /**
     * @param int $courseid
     * @return array|null
     * @throws \dml_exception
     */
    public static function get_quizzes(int $courseid) {
        global $DB;

        $sql = "SELECT q.id, q.name,
                       (SELECT COUNT(1) FROM {quiz_slots} qs
                        JOIN {question_references} qre ON qre.itemid = qs.id
                        JOIN {question_bank_entries} qbe ON qbe.id = qre.questionbankentryid
                        JOIN {question_versions} qve ON qve.questionbankentryid = qbe.id
                        JOIN {question} qu ON qu.id = qve.questionid
                        WHERE qs.quizid = q.id AND qu.qtype = 'essay' AND qbe.idnumber LIKE 'W%') +
                       (SELECT COUNT(1) FROM {quiz_slots} qs
                        JOIN {question_set_references} qr ON qr.itemid = qs.id
                        JOIN {question_bank_entries} qbe ON qbe.questioncategoryid = TRIM(BOTH '\"' FROM json_extract(qr.filtercondition, '$.questioncategoryid'))
                        JOIN {question_versions} qv ON qv.questionbankentryid = qbe.id
                        JOIN {question} qu ON qu.id = qv.questionid
                        WHERE qs.quizid = q.id AND qu.qtype = 'essay' AND qbe.idnumber LIKE 'W%') numofwq
                FROM mdl_quiz q
                WHERE q.course = ?
                HAVING numofwq > 0";
        if ($quizzes = $DB->get_records_sql($sql, [$courseid])) {
            return $quizzes;
        }
        return null;
    }

    /**
     * @param int $quizid
     * @param string $search
     * @return array|null
     * @throws \dml_exception
     */
    public static function get_worksheet_questions(int $quizid, string $search) {
        global $DB, $USER, $PAGE;

        if (empty($quizid)) {
            return null;
        }

        $quiz = $DB->get_record('quiz', ['id' => $quizid], '*', MUST_EXIST);

        if (!$cm = get_coursemodule_from_instance('quiz', $quiz->id, $quiz->course)) {
            throw new \moodle_exception('invalidcoursemodule');
        }

        //$context = \context_module::instance($cm->id);

        $coursecontext = \context_course::instance($quiz->course);

        $params = [];
        $params['quizid'] = $quizid;
        $params['quizid2'] = $quizid;

        $filter = '';
        $filter2 = '';
        if ($search) {
            $filter = " AND " . $DB->sql_like('qbe.idnumber', ':search', false, false);
            $params['search'] = '%'.$DB->sql_like_escape($search).'%';

            $filter2 = " AND " . $DB->sql_like('qbe.idnumber', ':search2', false, false);
            $params['search2'] = '%'.$DB->sql_like_escape($search).'%';
        }

        $sql = "SELECT q.*
                  FROM
                        (SELECT qs.id, qu.name, qu.questiontext, qbe.idnumber, eo.graderinfo , qs.slot, qu.id questionid
                          FROM {quiz_slots} qs
                            JOIN {question_references} qre ON qre.itemid = qs.id
                            JOIN {question_bank_entries} qbe ON qbe.id = qre.questionbankentryid
                            JOIN {question_versions} qve ON qve.questionbankentryid = qbe.id
                            JOIN {question} qu ON qu.id = qve.questionid
                          JOIN {qtype_essay_options} eo
                            ON qve.questionid = eo.questionid
                          WHERE qs.quizid = :quizid
                            AND qu.qtype = 'essay'
                            AND qbe.idnumber LIKE 'W%'
                                {$filter}
                          
                         UNION
                         
                        SELECT qs.id, qu.name, qu.questiontext, qbe.idnumber, eo.graderinfo , qs.slot, qu.id questionid
                          FROM {quiz_slots} qs
                            JOIN {question_set_references} qr ON qr.itemid = qs.id
                            JOIN {question_bank_entries} qbe ON qbe.questioncategoryid = trim(BOTH '\"' FROM JSON_EXTRACT(qr.filtercondition, '$.questioncategoryid'))
                            JOIN {question_versions} qv ON qv.questionbankentryid = qbe.id
                            JOIN {question} qu ON qu.id = qv.questionid
                            JOIN {qtype_essay_options} eo ON qu.id = eo.questionid
                          WHERE qs.quizid = :quizid2 
                            AND qu.qtype = 'essay'
                            AND qbe.idnumber LIKE 'W%'
                               {$filter2}
                        ) q
              ORDER BY q.slot";
        $questions = [];
        if ($rs = $DB->get_recordset_sql($sql, $params)) {
            foreach ($rs as $question) {

                $question->questiontext = self::question_rewrite_question_urls(
                    $question->questiontext, $coursecontext->id, 'question', 'questiontext', [$question->slot], $question->questionid);
                $question->graderinfo = self::question_rewrite_question_urls(
                    $question->graderinfo, $coursecontext->id, 'qtype_essay', 'graderinfo', [$question->slot], $question->questionid);

                $question->questiontext = format_text($question->questiontext);
                $question->graderinfo = format_text($question->graderinfo);

                $questions[$question->questionid] = $question;
            }
            $rs->close();
            return $questions;
        }
        return null;
    }

    public static function question_rewrite_question_urls($text, $coursecontextid, $component, $filearea, $patcomponents, $itemid) {
        return question_rewrite_question_urls($text, 'local/worksheet/pluginfile.php',
            $coursecontextid, $component, $filearea,
            $patcomponents, $itemid);
    }
}