<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Overview
 *
 * @package    assignsubmission_ctsubmission
 * @copyright  2020 Michael Gardener <mgardener@cissq.com>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace assignsubmission_ctsubmission\output;

use assignsubmission_ctsubmission\shared_lib as NED;
use assignsubmission_ctsubmission\submission;
use assignsubmission_ctsubmission\helper;
use block_ned_teacher_tools\shared_lib as TT;

defined('MOODLE_INTERNAL') || die();

/** @var object $CFG */
require_once($CFG->dirroot . '/mod/assign/locallib.php');

/**
 * Class ct_overview
 * @package assignsubmission_ctsubmission\output
 */
class ct_overview implements \renderable, \templatable {
    /**
     * @var int
     */
    protected $courseid;
    /**
     * @var object[]
     */
    protected $courses;
    /**
     * @var int
     */
    protected $cmid;
    /**
     * @var int
     */
    protected $groupid;
    /**
     * @var array
     */
    protected $groups;
    /**
     * @var string
     */
    protected $submissionstatus;
    /**
     * @var array
     */
    protected $assignments;
    /**
     * @var string
     */
    protected $returnurl;
    protected $gt;

    /**
     * @param $courseid
     * @param $cmid
     * @param $groupid
     * @param $submissionstatus
     * @param $gt
     * @param $returnurl
     */
    public function __construct($courseid, $cmid, $groupid, $submissionstatus, $gt, $returnurl){
        $this->courseid = $courseid;
        $this->courses = helper::get_courses();
        if (empty($this->courseid) && !empty($this->courses)) {
            $this->courseid = reset($this->courses)->id;
        }
        $this->cmid = $cmid;
        $this->groupid = $groupid;
        $this->submissionstatus = $submissionstatus;
        $this->returnurl = $returnurl;
        $this->set_groups();
        $this->gt = $gt;
    }

    /**
     * @return void
     */
    public function set_groups(){
        foreach ($this->courses as $courseid => $course){
            $this->courses[$courseid] = static::set_course_groups($course);
        }
    }

    /**
     * @param object $course
     *
     * @return object
     */
    public static function set_course_groups($course){
        $groups = NED::get_all_user_course_groups($course->id);
        foreach ($groups as $index => $group){
            if (!$isallowed = \block_ned_teacher_tools\grading_tracker::is_groupid_allowed($group->id)){
                unset($groups[$index]);
            }
        }

        $course->groups = $groups;
        $course->assigns = helper::get_ctsubmission_assignments($course->id);

        return $course;
    }

    /**
     * Export page contents for template
     * @param \renderer_base $output
     *
     * @return \stdClass
     */
    public function export_for_template(\renderer_base $output){
        global $USER;

        $data = new \stdClass();
        $is_admin = is_siteadmin();

        // Filter.
        $data->formurl = (new \moodle_url('/mod/assign/submission/ctsubmission/overview.php'))->out();
        $data->courseid = $this->courseid;

        $strshowall = NED::str('showall');
        $strsubmitted = NED::str('submitted');
        $strmisseddeadline = NED::str('misseddeadline');
        $strapproved = NED::str('approved');
        $strunapprove = NED::str('unapprove');
        $strapprove = NED::str('approve');
        $strwaitingforsubmission = NED::str('waitingforsubmission');
        $strwaitingforapproval = NED::str('waitingforapproval');
        $strdeadlinerequired = NED::str('deadlinerequired');
        $strpending = NED::str('pending');
        $strrevisionrequired = NED::str('revisionrequired');

        $data->submissionstatuses = [
            [
                'id' => 'showall',
                'name' => $strshowall,
                'selected' => $this->submissionstatus === 'showall'
            ], [
                'id' => 'approved',
                'name' => $strapproved,
                'selected' => $this->submissionstatus === 'approved'
            ], [
                'id' => 'waitingforsubmission',
                'name' => $strwaitingforsubmission,
                'selected' => $this->submissionstatus === 'waitingforsubmission'
            ], [
                'id' => 'waitingforapproval',
                'name' => $strwaitingforapproval,
                'selected' => $this->submissionstatus === 'waitingforapproval'
            ], [
                'id' => 'misseddeadline',
                'name' => $strmisseddeadline,
                'selected' => $this->submissionstatus === 'misseddeadline'
            ], [
                'id' => 'deadlinerequired',
                'name' => $strdeadlinerequired,
                'selected' => $this->submissionstatus === 'deadlinerequired'
            ], [
                'id' => 'pending',
                'name' => $strpending,
                'selected' => $this->submissionstatus === 'pending'
            ]
        ];

        $data->courses = [];
        $data->courses[] = [
            'id' => 0,
            'name' => NED::str('allcourses'),
            'selected' => $this->courseid === 0
        ];

        foreach ($this->courses as $course){
            $data->courses[] = [
                'id' => $course->id,
                'name' => $course->shortname,
                'selected' => $this->courseid == $course->id
            ];
        }

        $data->activities = [];
        $data->activities[] = [
            'id' => 0,
            'name' => NED::str('allactivities'),
            'selected' => $this->cmid === 0
        ];

        $data->gts = [];

        $allgroups = [];
        $data->hasgroup = false;
        $classes = [];
        $time = time();
        $data->showactioncolumn = false;

        foreach ($this->courses as $course){
            if ($this->courseid && $this->courseid != $course->id){
                continue;
            }

            if (!$data->hasgroup){
                $data->hasgroup = !empty($course->groups);
                if (isset($course->groups[$this->groupid])){
                    $course->groups[$this->groupid]->selected = true;
                }
            }
            $allgroups += $course->groups;

            foreach ($course->assigns as $assign){
                $cm = NED::get_cm_by_params($course, NED::ASSIGN, $assign->assignment);
                if (!NED::check_activity_visible_by_cm($cm)) continue;

                if ($is_admin){
                    $hasapprovecap = $hasubmitcap = true;
                } else {
                    $hasapprovecap = NED::has_capability('approve', $cm->context);
                    $hasubmitcap = NED::has_capability('submit', $cm->context);
                }


                $assign = NED::ned_assign_by_cm($cm);
                $data->activities[] = [
                    'id' => $cm->id,
                    'name' => $cm->name,
                    'selected' => $this->cmid == $cm->id
                ];

                if (!empty($this->cmid) && $this->cmid != $cm->id){
                    continue;
                }

                /** @var \assign_submission_ctsubmission $assignsubmissionct */
                $assignsubmissionct = $assign->get_submission_plugin_by_type('ctsubmission');

                foreach ($course->groups as $group){
                    if (!empty($this->groupid) && $this->groupid != $group->id) continue;

                    $has_ct = NED::get_users_by_role(NED::ROLE_CT, $cm->course, $cm->id, $group->id, true);
                    if (!$has_ct) continue;

                    $approved = 0;
                    $submission = $assign->get_group_submission($USER->id, $group->id, true, -1);
                    $ctsubmission = submission::get_record(['assignment' => $assign->get_instance()->id, 'submission' => $submission->id]);
                    if ($ctsubmission){
                        $approved = $ctsubmission->get('approved');
                    }

                    $class = new \stdClass();
                    $class->coursename = NED::q_course_link($course);
                    $class->activity = $assign->get_instance()->name;
                    $class->activityurl = new \moodle_url("/mod/assign/view.php", ['id' => $cm->id]);
                    $class->class = $group->name;
                    $class->classurl = TT::url('~/deadline_manager.php', ['courseid' => $course->id, 'group' => $group->id])->out(false);

                    // Grading teachers.
                    $gts = NED::get_users_by_role(NED::ROLE_OT, $course->id, 0, $group->id, false, 'u.*');
                    $grading_teachers= [];
                    foreach ($gts as $gt){
                        $grading_teachers[] = NED::q_user_link($gt, $course->id);
                        $data->gts[$gt->id] = [
                            'id' => $gt->id,
                            'name' => fullname($gt),
                            'selected' => $this->gt == $gt->id
                        ];
                    }
                    $class->gradingteacher = implode(', ', $grading_teachers);

                    $class->classroomteacher = NED::get_role_users_fullnames(NED::ROLE_CT, $course->id, $group->id, 0, true, true);
                    $class->approvalcellcls = '';
                    list($window_start_str, $window_end_str, $window_start, $window_end) =
                        NED::dm_get_submission_window($cm, NED::CTSUBMISSION, null, $group, true);
                    $class->classdeadline = $window_end_str;
                    $class->classdeadlinesort = $window_end;

                    $count = $assignsubmissionct->count_files($submission->id, ASSIGNSUBMISSION_CTSUBMISSION_FILEAREA);
                    $class->submissionstatus = '';
                    $class->statusmenu = $this->status_menu($cm, $group->id, $this->returnurl, $hasapprovecap);

                    if ($count){
                        if (!$approved){
                            $class->submissionstatus = $strwaitingforapproval;
                        }
                        // Files.
                        if (($hasapprovecap || $hasubmitcap) && $submission){
                            $out = [];
                            $fs = get_file_storage();
                            $files = $fs->get_area_files($assign->get_context()->id, 'assignsubmission_ctsubmission',
                                ASSIGNSUBMISSION_CTSUBMISSION_FILEAREA, $submission->id);
                            foreach ($files as $file){
                                if (!$file->is_directory()){
                                    $out[] = NED::file_get_link($file, true, true, 16, true);
                                }
                            }
                            if ($out){
                                $class->files = implode('<br />', $out);
                            }
                        }
                    } else {
                        if ($window_start){
                            if ($window_start > $time){
                                $class->submissionstatus = $strpending;
                            } elseif ($window_end >= $time && $time >= $window_start){
                                $class->submissionstatus = $strwaitingforsubmission;
                            } elseif ($time > $window_end){
                                $class->submissionstatus = $strmisseddeadline;
                            }
                        } else {
                            $class->submissionstatus = $strdeadlinerequired;
                        }
                    }

                    if ($approved == ASSIGNSUBMISSION_CTSUBMISSION_APPROVED){
                        $class->submissionstatus = $strapproved;
                    } elseif ($approved == ASSIGNSUBMISSION_CTSUBMISSION_REVISIONREQUESTED){
                        $class->submissionstatus = $strrevisionrequired;
                    }

                    // Filter table.
                    if ($this->submissionstatus !== 'showall'){
                        $var = "str{$this->submissionstatus}";
                        if (isset($$var) && $class->submissionstatus != $$var){
                            continue;
                        }
                    }
                    if ($this->gt){
                        if (!$gts || !array_key_exists($this->gt, $gts)){
                            continue;
                        }
                    }

                    if ($this->submissionstatus === 'approved'){
                        if (empty($class->submissionstatus) || $class->submissionstatus !== $strapproved){
                            continue;
                        }
                    }

                    switch ($class->submissionstatus){
                        case $strwaitingforsubmission:
                            if ($hasubmitcap){
                                $class->submissionstatuscellcls = 'bg-red';
                            }
                            break;
                        case $strwaitingforapproval:
                            if ($hasapprovecap){
                                $class->submissionstatuscellcls = 'bg-green';
                            }
                            break;
                        case $strmisseddeadline:
                        case $strdeadlinerequired:
                        case $strrevisionrequired:
                            if ($hasubmitcap || $hasapprovecap){
                                $class->submissionstatuscellcls = 'bg-red';
                            }
                            break;
                        default;
                            $class->submissionstatuscellcls = '';
                            break;
                    }

                    if ($class->submissionstatus == $strwaitingforapproval){
                        $class->submissionstatuscellcls = 'bg-green';
                    }

                    // Action.
                    if ($hasubmitcap){
                        $class->actiontext = ($count) ? TT::str('edit') : TT::str('submit');
                        $class->actionurl = (new \moodle_url('/mod/assign/submission/ctsubmission/submit.php',
                            ['id' => $cm->id, 'groupid' => $group->id, 'returnurl' => $this->returnurl]))->out(false);
                        $data->showactioncolumn = true;
                    }
                    $classes[] = $class;
                }
            }

            $data->classes = $classes;
        }

        $data->groups = array_values($allgroups);
        $data->gts = array_values($data->gts);

        return $data;
    }

    /**
     * @param $cm
     * @param $groupid
     * @param $returnurl
     * @param $hasapprovecap
     *
     * @return mixed
     */
    public function status_menu($cm, $groupid, $returnurl, $hasapprovecap){
        global $OUTPUT;

        if (!$hasapprovecap){
            return '';
        }

        $menu = new \action_menu();
        $menu->set_menu_trigger(\html_writer::tag('i', '', ['class' => 'icon fa fa-pencil']));
        $menu->attributes['class'] .= ' item-actions';
        $menu->add(new \action_menu_link(
            new \moodle_url('/mod/assign/submission/ctsubmission/submit.php',
                ['id' => $cm->id, 'groupid' => $groupid, 'action' => 'approve', 'returnurl' => $returnurl]),
            null,
            NED::str('approve'),
            false,
            array('data-action' => 'approve', 'class' => 'action-'.'approve')
        ));
        $menu->add(new \action_menu_link(
            new \moodle_url('/mod/assign/submission/ctsubmission/submit.php',
                ['id' => $cm->id, 'groupid' => $groupid, 'action' => 'unapprove', 'returnurl' => $returnurl]),
            null,
            NED::str('removeapproval'),
            false,
            array('data-action' => 'approve', 'class' => 'action-'.'approve')
        ));
        $menu->add(new \action_menu_link(
            new \moodle_url('/mod/assign/submission/ctsubmission/submit.php',
                ['id' => $cm->id, 'groupid' => $groupid, 'action' => 'revision', 'returnurl' => $returnurl]),
            null,
            NED::str('requestrevision'),
            false,
            array('data-action' => 'approve', 'class' => 'action-'.'approve')
        ));

        return $OUTPUT->render($menu);
    }
}