<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Plugin strings are defined here.
 *
 * @package     mod_nedactivitycluster
 * @copyright   2018 Michael Gardener <mgardener@cissq.com>
 * @license     http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
namespace mod_nedactivitycluster\output;

defined('MOODLE_INTERNAL') || die();

global $CFG;
require_once($CFG->libdir . '/completionlib.php');

use coding_exception;
use mod_nedactivitycluster\helper;
use moodle_url;
use renderable;
use renderer_base;
use stdClass;
use templatable;
use single_select;
use html_writer;

/**
 * Class containing data for users that need to be given with 360 feedback.
 *
 * @copyright  2015 Jun Pataleta
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
class view implements renderable, templatable {

    /** @var stdClass The 360 instance.  */
    protected $nedactivitycluster;

    /** @var int The user ID of the respondent. */
    protected $userid;

    /** @var int The user ID of the respondent. */
    protected $courseid;
    /** @var int The user ID of the respondent. */
    protected $cmid;

    /** @var bool Whether the user has the capability to view reports. */
    protected $canview = false;

    /** @var bool Whether the user has the capability to view reports. */
    protected $canedit = false;

    /** @var bool Whether the user has the capability to view reports. */
    protected $canviewstudentprogress = false;

    /** @var bool Whether the user has the capability to view reports. */
    protected $params;

    /**
     * list_participants constructor.
     * @param stdClass $nedactivitycluster The 360 instance.
     * @param int $userid The respondent's user ID.
     * @param array $participants The array of participants for the 360 feedback, excluding the respondent.
     * @param bool $canviewreports Whether the user has the capability to view reports.
     */
    public function __construct($nedactivitycluster, $userid, $cmid, $params) {
        $this->nedactivitycluster = $nedactivitycluster;
        $this->courseid = $nedactivitycluster->course;
        $this->userid = $userid;
        $this->cmid = $cmid;
        $this->canview = $params->canview;
        $this->canedit = $params->canedit;
        $this->canviewstudentprogress = $params->viewstudentprogress;
        $this->params = $params;
    }

    /**
     * Function to export the renderer data in a format that is suitable for a
     * mustache template. This means:
     * 1. No compx types - only stdClass, array, int, string, float, bool
     * 2. Any additional info that is required for the template is pre-calculated (e.g. capability checks).
     *
     * @param renderer_base $output Used to do a final render of any components that need to be rendered for export.
     * @return stdClass|array
     * @throws coding_exception
     */
    public function export_for_template(renderer_base $output) {
        global $DB, $OUTPUT, $USER;

        $data = new stdClass();
        // Permissions.
        $data->canview = $this->canview;
        $data->canedit = $this->canedit;
        $data->showactivitytypecolumn = $this->nedactivitycluster->showactivitytypecolumn;
        $data->showgrade = $this->nedactivitycluster->showgrade;
        $data->showstudentmenu = $this->nedactivitycluster->showstudentmenu;
        $data->showtitleheader = $this->nedactivitycluster->showtitleheader;
        $data->showpassfailbackgroundcolors = $this->nedactivitycluster->showpassfailbackgroundcolors;
        // Icons.
        $data->editicon = $OUTPUT->image_url('i/settings');
        $data->dragdropicon = $OUTPUT->image_url('i/dragdrop');
        $data->deleteicon = $OUTPUT->image_url('i/delete');

        $addclusterurl = new moodle_url('/mod/nedactivitycluster/cluster_edit.php', array('cmid' => $this->cmid));
        $data->addclusterurl = $addclusterurl->out();

        $data->name = $this->nedactivitycluster->name;
        $data->intro = $this->nedactivitycluster->intro;

        $clusters = $DB->get_records('nedactivitycluster_clusters',
            array('nedactivitycluster' => $this->nedactivitycluster->id), 'section');

        $modinfo = get_fast_modinfo($this->courseid, $this->userid );
        $courseactivities = $modinfo->get_cms();
        $excludedmodules = nedactivitycluster_excluded_modules();
        $course = $DB->get_record('course', array('id' => $this->courseid));

        foreach ($clusters as $index => $cluster) {
            $activities = [];
            $totaltimespent = 0;
            $totaltarget = 0;
            $cluster->totaltimespent = 0;
            if ($cluster->sequence){
                $mods = explode(',', $cluster->sequence);

                $pulldatafromquestionnaire = $this->nedactivitycluster->pulldatafromquestionnaire && helper::has_questionnaire($courseactivities, $mods);

                foreach ($mods as $cmid) {
                    if (!isset($courseactivities[$cmid])) {
                        continue;
                    }
                    $activity = new stdClass();
                    $mod = $courseactivities[$cmid];
                    if ($excludedmodules && in_array($mod->modname, $excludedmodules)) {
                        continue;
                    }
                    list($gradepercentage, $gradeformatted, $contributiontocoursetotal, $status) = nedactivitycluster_get_formatted_grade($mod, $this->userid );

                    $activity->gradepercentage = $gradepercentage;
                    $activity->gradeformatted = $gradeformatted;
                    $activity->contributiontocoursetotal = $contributiontocoursetotal;
                    $activity->status = $status;
                    $activity->icon = $mod->get_icon_url();
                    $activity->activityname = $mod->name;
                    $activity->modname = get_string('pluginname', $mod->modname);
                    $modurl = new moodle_url('/mod/'.$mod->modname.'/view.php', array('id' => $mod->id));
                    $activity->modurl = $modurl->out();

                    $activity->completionstatus = helper::get_completion_status($mod);

                    if ($mod->modname == 'questionnaire') {
                        $activity->timespent  = helper::get_survey_data($mod->instance, $this->nedactivitycluster->questionnairequestionname);
                        if ($this->nedactivitycluster->calculatetotal) {
                            $totaltimespent += $activity->timespent;
                        }
                        if ($this->nedactivitycluster->calculatetargettotal) {
                            preg_match_all('/(?<=Target:).*?(?=hour)/mi', $activity->activityname, $matches, PREG_SET_ORDER, 0);
                            if (isset($matches[0][0])) {
                                $totaltarget += (float)trim($matches[0][0]);
                            }
                        }
                    }

                    $activities[] = $activity;
                }
            }
            $editurl = new moodle_url('/mod/nedactivitycluster/cluster_edit.php', array('id' => $cluster->id, 'cmid' => $this->cmid));
            $cluster->editurl = $editurl->out(false);

            $deleteurl = new moodle_url('/mod/nedactivitycluster/cluster_delete.php', array('id' => $cluster->id, 'cmid' => $this->cmid));
            $cluster->deleteurl = $deleteurl->out(false);

            $tmpl = new \core\output\inplace_editable('nedactivitycluster', 'clustername', $cluster->id,
                $data->canedit,
                format_string($cluster->name, true, $course->id),
                $cluster->name,
                get_string('editclustername', 'nedactivitycluster'),
                get_string('newvaluefor', 'nedactivitycluster', format_string($cluster->name, true, $course->id))
            );
            $cluster->name =  $OUTPUT->render($tmpl);

            if (!empty($pulldatafromquestionnaire)){
                $cluster->pulldatafromquestionnaire = $pulldatafromquestionnaire;
                $cluster->questionnametoshow = $this->nedactivitycluster->questionnametoshow;
                $cluster->calculatetotal = $this->nedactivitycluster->calculatetotal;
                $cluster->calculatetargettotal = $this->nedactivitycluster->calculatetargettotal;
                $cluster->displayfooter = ($cluster->calculatetotal || $cluster->calculatetargettotal);
                $cluster->totaltimespent = $totaltimespent;
                $cluster->totaltarget = $totaltarget;
            }

            if ($activities) {
                $activities[(count($activities) - 1)]->lastactivity = true;
            }

            $cluster->activities = $activities;

            $clusters[$index] = $cluster;
        }

        // Student progress view.
        if ($this->canviewstudentprogress) {
            $groupurl = new moodle_url(
                '/mod/nedactivitycluster/view.php',
                array(
                    'id' => $this->cmid
                )
            );

            $data->studentselector = nedactivitycluster_groups_print_course_menu($course, $groupurl->out(), true, $this->params->group);

            $groupmembers = groups_get_members_by_role($this->params->group, $course->id);
            $studentoptions = array('0' => get_string('selectstudent', 'nedactivitycluster'));
            if (isset($groupmembers[5]->users)) {
                foreach ($groupmembers[5]->users as $groupmember) {
                    $studentoptions[$groupmember->id] = fullname($groupmember);
                }
            }
            $studenturl = new moodle_url(
                '/mod/nedactivitycluster/view.php',
                array(
                    'id' => $this->cmid,
                    'group' => $this->params->group,
                )
            );
            $studentselect = new single_select($studenturl, 'u', $studentoptions, $this->params->u, '');
            $studentselect->formid = 'fn_participants';
            $studentselect->label = html_writer::img($OUTPUT->image_url('i/user'), '');
            $data->studentselector .= html_writer::div($OUTPUT->render($studentselect), 'groupselector student-selector');
        }

        $data->clusters = array_values($clusters);

        return $data;
    }
}
