<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Library of interface functions and constants.
 *
 * @package     mod_nedactivitycluster
 * @copyright   2018 Michael Gardener <mgardener@cissq.com>
 * @license     http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

defined('MOODLE_INTERNAL') || die();

define('NEDACTIVITYCLUSTER_ACTIVITYSETTING', '1');
define('NEDACTIVITYCLUSTER_PASSGRADEPERCENT', '2');

define('NEDACTIVITYCLUSTER_SHOW_ACTIVITIES', '1');
define('NEDACTIVITYCLUSTER_SHOW_RESOURCES', '2');
define('NEDACTIVITYCLUSTER_SHOW_BOTH', '3');

require_once($CFG->dirroot.'/lib/grade/constants.php');

/**
 * Return if the plugin supports $feature.
 *
 * @param string $feature Constant representing the feature.
 * @return true | null True if the feature is supported, null otherwise.
 */
function nedactivitycluster_supports($feature) {
    switch($feature) {
        case FEATURE_MOD_INTRO:
        case FEATURE_COMPLETION_TRACKS_VIEWS:
        case FEATURE_COMPLETION_HAS_RULES:
        case FEATURE_BACKUP_MOODLE2:
        case FEATURE_SHOW_DESCRIPTION:
            return true;
        default:
            return null;
    }
}

/**
 * Saves a new instance of the mod_nedactivitycluster into the database.
 *
 * Given an object containing all the necessary data, (defined by the form
 * in mod_form.php) this function will create a new instance and return the id
 * number of the instance.
 *
 * @param object $moduleinstance An object from the form.
 * @param nedactivitycluster_mod_form $mform The form.
 * @return int The id of the newly inserted record.
 */
function nedactivitycluster_add_instance($moduleinstance, $mform = null) {
    global $DB;

    $moduleinstance->timecreated = time();

    $id = $DB->insert_record('nedactivitycluster', $moduleinstance);

    return $id;
}

/**
 * Updates an instance of the mod_nedactivitycluster in the database.
 *
 * Given an object containing all the necessary data (defined in mod_form.php),
 * this function will update an existing instance with new data.
 *
 * @param object $moduleinstance An object from the form in mod_form.php.
 * @param nedactivitycluster_mod_form $mform The form.
 * @return bool True if successful, false otherwise.
 */
function nedactivitycluster_update_instance($moduleinstance, $mform = null) {
    global $DB;

    $moduleinstance->timemodified = time();
    $moduleinstance->id = $moduleinstance->instance;

    return $DB->update_record('nedactivitycluster', $moduleinstance);
}

/**
 * Removes an instance of the mod_activitycluster from the database.
 *
 * @param int $id Id of the module instance.
 * @return bool True if successful, false on failure.
 */
function nedactivitycluster_delete_instance($id) {
    global $DB;

    $exists = $DB->get_record('nedactivitycluster', array('id' => $id));
    if (!$exists) {
        return false;
    }

    $DB->delete_records('nedactivitycluster', array('id' => $id));

    return true;
}

/**
 * Returns the lists of all browsable file areas within the given module context.
 *
 * The file area 'intro' for the activity introduction field is added automatically
 * by {@link file_browser::get_file_info_context_module()}.
 *
 * @package     mod_nedactivitycluster
 * @category    files
 *
 * @param stdClass $course.
 * @param stdClass $cm.
 * @param stdClass $context.
 * @return string[].
 */
function nedactivitycluster_get_file_areas($course, $cm, $context) {
    return array();
}

/**
 * File browsing support for mod_nedactivitycluster file areas.
 *
 * @package     mod_nedactivitycluster
 * @category    files
 *
 * @param file_browser $browser.
 * @param array $areas.
 * @param stdClass $course.
 * @param stdClass $cm.
 * @param stdClass $context.
 * @param string $filearea.
 * @param int $itemid.
 * @param string $filepath.
 * @param string $filename.
 * @return file_info Instance or null if not found.
 */
function nedactivitycluster_get_file_info($browser, $areas, $course, $cm, $context, $filearea, $itemid, $filepath, $filename) {
    return null;
}

/**
 * Serves the files from the mod_nedactivitycluster file areas.
 *
 * @package     mod_nedactivitycluster
 * @category    files
 *
 * @param stdClass $course The course object.
 * @param stdClass $cm The course module object.
 * @param stdClass $context The mod_nedactivitycluster's context.
 * @param string $filearea The name of the file area.
 * @param array $args Extra arguments (itemid, path).
 * @param bool $forcedownload Whether or not force download.
 * @param array $options Additional options affecting the file serving.
 */
function nedactivitycluster_pluginfile($course, $cm, $context, $filearea, $args, $forcedownload, $options = array()) {
    global $DB, $CFG;

    if ($context->contextlevel != CONTEXT_MODULE) {
        send_file_not_found();
    }

    require_login($course, true, $cm);
    send_file_not_found();
}

/**
 * Extends the global navigation tree by adding mod_nedactivitycluster nodes if there is a relevant content.
 *
 * This can be called by an AJAX request so do not rely on $PAGE as it might not be set up properly.
 *
 * @param navigation_node $nedactivityclusternode An object representing the navigation tree node.
 * @param stdClass $course.
 * @param stdClass $module.
 * @param cm_info $cm.
 */
function nedactivitycluster_extend_navigation($nedactivityclusternode, $course, $module, $cm) {
}

/**
 * Extends the settings navigation with the mod_nedactivitycluster settings.
 *
 * This function is called when the context for the page is a mod_nedactivitycluster module.
 * This is not called by AJAX so it is safe to rely on the $PAGE.
 *
 * @param settings_navigation $settingsnav {@link settings_navigation}
 * @param navigation_node $nedactivityclusternode {@link navigation_node}
 */
function nedactivitycluster_extend_settings_navigation($settingsnav, $nedactivityclusternode = null) {
}

function nedactivitycluster_get_sequence_from_form($fromform) {
    $sequence = [];
    foreach ($fromform as $key => $value) {
        if (strpos($key, 'activity') === 0) {
            if ((int)$value > 0) {
                $sequence[$value] = $value;
            }
        }
    }
    return implode(',', $sequence);
}
function nedactivitycluster_get_next_section($nedactivitycluster) {
    global $DB;
    if (!$nedactivitycluster) {
        return 0;
    }
    $sql = "SELECT MAX(c.section) maxnum FROM {nedactivitycluster_clusters} c WHERE c.nedactivitycluster = ?";

    $cluster = $DB->get_record_sql($sql, array($nedactivitycluster));
    return $cluster->maxnum + 1;
}
function nedactivitycluster_get_formatted_grade($mod, $userid=0) {
    global $DB, $USER;

    if (!$userid) {
        $userid = $USER->id;
    }
    $gradepercentage = 0;
    $gradeformatted = '-';
    $contributiontocoursetotal = '-';
    $status = 'notgraded';

    $config = get_config('nedactivitycluster');

    if ($gradeitem = $DB->get_record('grade_items',
        array('itemtype' => 'mod', 'itemmodule' => $mod->modname, 'iteminstance' => $mod->instance))
    ) {
        if ($gradeitem->gradetype == GRADE_TYPE_SCALE) {
            $gradeformatted =  '-';
        } else if ($gradeitem->gradetype == GRADE_TYPE_VALUE) {
            $gradeformatted =  '/' . round($gradeitem->grademax);
        }

        if ($grade = $DB->get_record('grade_grades', array('itemid' => $gradeitem->id, 'userid' => $userid))) {
            if (!is_null($grade->finalgrade)) {
                if ($gradeitem->gradetype == GRADE_TYPE_SCALE) {
                    if (!empty($gradeitem->scaleid)) {
                        $scale = $DB->get_record('scale', array('id' => $gradeitem->scaleid));
                        $gradeval = (int)$grade->finalgrade; // scales use only integers
                        $scales = explode(",", $scale->scale);
                        $gradeformatted = $scales[($gradeval-1)];
                    }
                } else if ($gradeitem->gradetype == GRADE_TYPE_VALUE) {
                    $gradepercentage = round(($grade->finalgrade / $grade->rawgrademax) * 100);
                    $gradeformatted = round($grade->finalgrade) . '/' . round($grade->rawgrademax) . ' (' . $gradepercentage . '%)';

                    if ($config->passgrade == NEDACTIVITYCLUSTER_ACTIVITYSETTING) {
                        if ($gradeitem->gradepass > 0) {
                            if ($gradeitem->gradepass <= $grade->finalgrade) {
                                $status = 'passing';
                            } else {
                                $status = 'failing';
                            }
                        };
                    } else if ($config->passgrade == NEDACTIVITYCLUSTER_PASSGRADEPERCENT) {
                        if ($config->passgradepercent <= $gradepercentage) {
                            $status = 'passing';
                        } else {
                            $status = 'failing';
                        }
                    }
                }
            }
        }
    }
    if ($contributiontocoursetotal == '0.00 %') {
        $contributiontocoursetotal = '-';
    }
    return array($gradepercentage, $gradeformatted, $contributiontocoursetotal, $status);
}

/**
 * Sets the automatic completion state for this database item based on the
 * count of on its entries.
 * @since Moodle 3.3
 * @param object $data The data object for this activity
 * @param object $course Course
 * @param object $cm course-module
 */
function nedactivitycluster_update_completion_state($data, $course, $cm) {
    // If completion option is enabled, evaluate it and return true/false.
    $completion = new completion_info($course);
    if ($data->completionentries && $completion->is_enabled($cm)) {
        $numentries = data_numentries($data);
        // Check the number of entries required against the number of entries already made.
        if ($numentries >= $data->completionentries) {
            $completion->update_state($cm, COMPLETION_COMPLETE);
        } else {
            $completion->update_state($cm, COMPLETION_INCOMPLETE);
        }
    }
}

/**
 * Obtains the automatic completion state for this forum based on any conditions
 * in forum settings.
 *
 * @param object $course Course
 * @param object $cm Course-module
 * @param int $userid User ID
 * @param bool $type Type of comparison (or/and; can be used as return value if no conditions)
 * @return bool True if completed, false if not, $type if conditions not set.
 */
function nedactivitycluster_get_completion_state($course, $cm, $userid, $type) {
    global $DB;
    $nedactivitycluster = $DB->get_record('nedactivitycluster', array('id' => $cm->instance), '*', MUST_EXIST);

    // If completion option is enabled, evaluate it and return true/false
    if ($nedactivitycluster->allactivitiesmustbecompleted) {
        $cmids = [];
        $completion = new completion_info($course);
        $activities = $completion->get_activities();
        $excludedmodules = nedactivitycluster_excluded_modules();

        if ($clusters = $DB->get_records('nedactivitycluster_clusters', array('nedactivitycluster' => $cm->instance))) {
            foreach ($clusters as $cluster) {
                if ($sectionmods = explode(",", $cluster->sequence)) {
                    foreach ($sectionmods as $sectionmod) {
                        $cmids[$sectionmod] = $sectionmod;
                    }
                }
            }
        }
        $var = false;
        if (!empty($cmids)) {
            foreach ($cmids as $cmid) {
                if (!isset($activities[$cmid])) {
                    continue;
                }
                $mod = $activities[$cmid];
                if ($excludedmodules && in_array($mod->modname, $excludedmodules)) {
                    continue;
                }
                if ($completion->is_enabled($mod)) {
                    $sql = "SELECT cmc.*
                              FROM {course_modules} cm
                              JOIN {course_modules_completion} cmc
                                ON cm.id = cmc.coursemoduleid
                             WHERE cm.course = ?
                               AND cmc.userid = ?
                               AND cmc.coursemoduleid = ?";
                    if ($progress = $DB->get_record_sql($sql, array($course->id, $userid, $cmid))) {
                        $state = $progress->completionstate;
                        switch ($state) {
                            case COMPLETION_INCOMPLETE    :
                                return false;
                            case COMPLETION_COMPLETE      :
                            case COMPLETION_COMPLETE_PASS :
                            case COMPLETION_COMPLETE_FAIL :
                                $var = true;
                                break;
                        }
                    } else {
                        return false;
                    }
                }
            }
        }
        return $var;
    } else {
        return $type;
    }
}

/**
 * Move save the items of the given $nedactivitycluster in the order of $itemlist.
 * @param array $itemlist list with item ids
 * @param stdClass $nedactivitycluster
 * @return bool true if success
 */
function nedactivitycluster_ajax_saveitemorder($itemlist, $nedactivitycluster) {
    global $DB;

    $result = true;
    $section = 0;
    foreach ($itemlist as $itemid) {
        $section++;
        $result = $result && $DB->set_field('nedactivitycluster_clusters',
                'section',
                $section,
                array('id' => $itemid, 'nedactivitycluster' => $nedactivitycluster->id));
    }
    return $result;
}

function nedactivitycluster_excluded_modules() {
    $excludedmods = array('label');

    $showmod  = get_config('nedactivitycluster', 'showmod');
    if ($showmod == NEDACTIVITYCLUSTER_SHOW_BOTH) {
        return $excludedmods;
    }

    $modnames = get_module_types_names();

    unset($modnames['label']);

    foreach ($modnames as $modname => $modulename) {
        $archetype = plugin_supports('mod', $modname, FEATURE_MOD_ARCHETYPE, MOD_ARCHETYPE_OTHER);
        if ($showmod == NEDACTIVITYCLUSTER_SHOW_ACTIVITIES && $archetype) {
            $excludedmods[] = $modname;
        } else if ($showmod == NEDACTIVITYCLUSTER_SHOW_RESOURCES && !$archetype) {
            $excludedmods[] = $modname;
        }
    }

    return $excludedmods;
}

/**
 * Remove Font Awesome shortcode from text
 *
 * @param string $text to be processed by the text
 * @return string text after processing
 */
function  nedactivitycluster_remove_fontawesome_code($text) {
    $search = "(\[(fa-.*?)\])is";
    $text = preg_replace($search, '', $text);
    return $text;
}


/**
 * Formatted activity name.
 *
 * @param string $name activity name
 * @param string $modname module name
 * @return string text after processing
 */
function  nedactivitycluster_activity_name_formatter($name, $modname) {
    return '[<span class="red">'.get_string('modulename', $modname).'</span>] '.
        nedactivitycluster_remove_fontawesome_code($name);
}

function nedactivitycluster_groups_print_course_menu($course, $urlroot, $return=false, $activegroup=false) {
    global $USER, $OUTPUT;

    if (!$groupmode = $course->groupmode) {
        if ($return) {
            return '';
        } else {
            return;
        }
    }

    $context = context_course::instance($course->id);
    $aag = has_capability('moodle/site:accessallgroups', $context);

    $usergroups = array();
    if ($groupmode == VISIBLEGROUPS or $aag) {
        $allowedgroups = groups_get_all_groups($course->id, 0, $course->defaultgroupingid);
        // Get user's own groups and put to the top.
        $usergroups = groups_get_all_groups($course->id, $USER->id, $course->defaultgroupingid);
    } else {
        $allowedgroups = groups_get_all_groups($course->id, $USER->id, $course->defaultgroupingid);
    }

    if ($activegroup === false) {
        $activegroup = groups_get_course_group($course, true, $allowedgroups);
    }

    $groupsmenu = array();
    $groupsmenuoptions = groups_sort_menu_options($allowedgroups, $usergroups);

    if ((!$allowedgroups or $groupmode == VISIBLEGROUPS or $groupmode == SEPARATEGROUPS or $aag) && (count($groupsmenuoptions) > 1)) {
        $groupsmenu[0] = get_string('allgroups', 'block_ned_teacher_tools');
    }

    $groupsmenu += groups_sort_menu_options($allowedgroups, $usergroups);

    if ($groupmode == VISIBLEGROUPS) {
        $grouplabel = get_string('groupsvisible');
    } else {
        $grouplabel = get_string('groupsseparate');
    }

    if ($aag and $course->defaultgroupingid) {
        if ($grouping = groups_get_grouping($course->defaultgroupingid)) {
            $grouplabel = $grouplabel . ' (' . format_string($grouping->name) . ')';
        }
    }

    if (count($groupsmenu) == 1) {
        $groupname = reset($groupsmenu);
        $output = html_writer::img($OUTPUT->image_url('i/users'), '').' '.$groupname;
    } else {
        $select = new single_select(new moodle_url($urlroot), 'group', $groupsmenu, $activegroup, null, 'selectgroup');
        $select->label = html_writer::img($OUTPUT->image_url('i/users'), '');
        $output = $OUTPUT->render($select);
    }

    $output = '<div class="groupselector">'.$output.'</div>';

    if ($return) {
        return $output;
    } else {
        echo $output;
    }
}
function nedactivitycluster_inplace_editable($itemtype, $itemid, $newvalue) {
    if ($itemtype === 'clustername') {
        global $DB;
        $record = $DB->get_record('nedactivitycluster_clusters', array('id' => $itemid), '*', MUST_EXIST);
        $cm = get_coursemodule_from_instance('nedactivitycluster', $record->nedactivitycluster, 0, false, MUST_EXIST);

        $modulecontext = context_module::instance($cm->id);

        \external_api::validate_context($modulecontext);

        // Check permission of the user to update this item.
        require_capability('mod/nedactivitycluster:addinstance', $modulecontext);

        // Clean input and update the record.
        $newvalue = clean_param($newvalue, PARAM_TEXT);
        $DB->update_record('nedactivitycluster_clusters', array('id' => $itemid, 'name' => $newvalue));

        // Prepare the element for the output:
        $record->name = $newvalue;

        return new \core\output\inplace_editable('nedactivitycluster', 'clustername', $record->id, true,
            format_string($record->name, true, $cm->course),
            $record->name,
            get_string('editclustername', 'nedactivitycluster'),
            get_string('newvaluefor', 'nedactivitycluster', format_string($record->name, true, $cm->course))
        );
    }
}