<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Importer.
 *
 * @package     report_ghs
 * @copyright   2020 Michael Gardener <mgardener@cissq.com>
 * @license     http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace report_ghs;

defined('MOODLE_INTERNAL') || die();

global $CFG;
require_once($CFG->dirroot . '/group/lib.php');

use grade_scale;
use report_ghs\shared_lib as NED;
use stdClass;
use context_system;
use csv_import_reader;
use block_ned_teacher_tools\deadline_manager;

/**
 * Import framework form.
 *
 * @package     report_ghs
 * @copyright   2020 Michael Gardener <mgardener@cissq.com>
 * @license     http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
class english_proficiency_importer {

    /** @var string $error The errors message from reading the xml */
    var $error = '';

    /** @var array $flat The flat competencies tree */
    var $flat = array();
    /** @var array $framework The framework info */
    var $framework = array();
    var $mappings = array();
    var $importid = 0;
    var $importer = null;
    var $foundheaders = array();

    public function fail($msg) {
        $this->error = $msg;
        return false;
    }

    public function get_importid() {
        return $this->importid;
    }

    public static function list_required_headers() {
        return array(
            0 => 'School',
            1 => 'ESL',
            2 => 'OEN',
            3 => 'FirstName',
            4 => 'LastName',
            5 => 'OSSLT_Status',
            6 => 'OSSLT_Score',
            7 => 'ESLDO',
            8 => 'ESLEO',
            9 => 'ENG3U',
            10 => 'ENG4U',
            11 => 'OPT',
            12 => 'OLC4O',
            13 => 'Graduation',
            14 => 'OSSLT_Year',
            15 => 'Program',
            16 => 'Gender',
            17 => 'First_Language',
            18 => 'Grade',
        );
    }

    public function list_found_headers() {
        return $this->foundheaders;
    }

    private function read_mapping_data($data) {
        if ($data) {
            return array(
                'school' => $data->header0,
                'esl' => $data->header1,
                'oen' => $data->header2,
                'firstname' => $data->header3,
                'lastname' => $data->header4,
                'ossltstatus' => $data->header5,
                'ossltscore' => $data->header6,
                'esldo' => $data->header7,
                'esleo' => $data->header8,
                'eng3u' => $data->header9,
                'eng4u' => $data->header10,
                'opt' => $data->header11,
                'olc4o' => $data->header12,
                'graduation' => $data->header13,
                'ossltyear' => $data->header14,
                'program' => $data->header15,
                'gender' => $data->header16,
                'firstlanguage' => $data->header17,
                'grade' => $data->header18,

            );
        } else {
            return array(
                'school' => 0,
                'esl' => 1,
                'oen' => 2,
                'firstname' => 3,
                'lastname' => 4,
                'ossltstatus' => 5,
                'ossltscore' => 6,
                'esldo' => 7,
                'esleo' => 8,
                'eng3u' => 9,
                'eng4u' => 10,
                'opt' => 11,
                'olc4o' => 12,
                'graduation' => 13,
                'ossltyear' => 14,
                'program' => 15,
                'gender' => 16,
                'firstlanguage' => 17,
                'grade' => 18,
            );
        }
    }

    private function get_row_data($row, $index) {
        if ($index < 0) {
            return '';
        }
        return isset($row[$index]) ? trim($row[$index]) : '';
    }

    /**
     * Constructor - parses the raw text for sanity.
     */
    public function __construct($text = null, $encoding = null, $delimiter = null, $importid = 0, $mappingdata = null) {
        global $CFG;

        // The idnumber is concatenated with the category names.
        require_once($CFG->libdir . '/csvlib.class.php');

        $type = 'competency_framework';

        if (!$importid) {
            if ($text === null) {
                return;
            }
            $this->importid = csv_import_reader::get_new_iid($type);

            $this->importer = new csv_import_reader($this->importid, $type);

            if (!$this->importer->load_csv_content($text, $encoding, $delimiter)) {
                $this->fail(get_string('invalidimportfile', 'report_ghs'));
                $this->importer->cleanup();
                return;
            }

        } else {
            $this->importid = $importid;

            $this->importer = new csv_import_reader($this->importid, $type);
        }


        if (!$this->importer->init()) {
            $this->fail(get_string('invalidimportfile', 'report_ghs'));
            $this->importer->cleanup();
            return;
        }

        $this->foundheaders = $this->importer->get_columns();

        $flat = array();
        $csvline = null;

        while ($row = $this->importer->next()) {
            $mapping = $this->read_mapping_data($mappingdata);

            $csvline = new stdClass();
            $csvline->school = $this->get_row_data($row, $mapping['school']);
            $csvline->esl = $this->get_row_data($row, $mapping['esl']);
            $csvline->oen = $this->get_row_data($row, $mapping['oen']);
            $csvline->firstname = $this->get_row_data($row, $mapping['firstname']);
            $csvline->lastname = $this->get_row_data($row, $mapping['lastname']);
            $csvline->ossltstatus = $this->get_row_data($row, $mapping['ossltstatus']);
            $csvline->ossltscore = $this->get_row_data($row, $mapping['ossltscore']);
            $csvline->esldo = $this->get_row_data($row, $mapping['esldo']);
            $csvline->esleo = $this->get_row_data($row, $mapping['esleo']);
            $csvline->eng3u = $this->get_row_data($row, $mapping['eng3u']);
            $csvline->eng4u = $this->get_row_data($row, $mapping['eng4u']);
            $csvline->opt = $this->get_row_data($row, $mapping['opt']);
            $csvline->olc4o = $this->get_row_data($row, $mapping['olc4o']);
            $csvline->graduation = $this->get_row_data($row, $mapping['graduation']);
            $csvline->ossltyear = $this->get_row_data($row, $mapping['ossltyear']);
            $csvline->program = $this->get_row_data($row, $mapping['program']);
            $csvline->defaultrole = NED::ENGLISH_PROFICIENCY_DEFAUL_ROLE;
            $csvline->gender = $this->get_row_data($row, $mapping['gender']);
            $csvline->firstlanguage = $this->get_row_data($row, $mapping['firstlanguage']);
            $csvline->grade = $this->get_row_data($row, $mapping['grade']);

            $flat[] = $csvline;
        }

        $this->framework = $flat;

        $this->importer->close();
        if ($this->framework == null) {
            $this->fail(get_string('invalidimportfile', 'report_ghs'));
            return;
        }
    }

    /**
     * @return array of errors from parsing the xml.
     */
    public function get_error() {
        return $this->error;
    }

    /**
     * Do the job.
     */
    public function import() {
        global $DB;

        $records = $this->framework;

        $fields = NED::ENGLISH_PROFICIENCY_FIELDS;
        $keys = array_keys($fields);
        $oenfield = reset($keys);
        unset($fields[$oenfield]);

        foreach ($records as $record) {
            $record->oen = trim(str_replace('-', '', $record->oen));
            if (empty($record->oen)) {
                continue;
            }
            $len = strlen($record->oen);
            if ($len < 9) {
                $record->oen = str_pad($record->oen, 9, '0', STR_PAD_LEFT);
            }
            $record->oen = substr($record->oen, 0, 3) . '-' . substr($record->oen, 3, 3) . '-' . substr($record->oen, 6, 3);

            if ($user = $DB->get_record_sql("SELECT d.userid FROM {user_info_data} d WHERE d.fieldid = 11 AND d.data = ?", [$record->oen])) {
                $record->userid = $user->userid;
                foreach ($fields as $fieldid => $fieldname) {
                    if (isset($record->$fieldname)) {
                        if ($data = $DB->get_record_sql("SELECT d.* FROM {user_info_data} d WHERE d.userid = ? AND d.fieldid = ?", [$user->userid, $fieldid])) {
                            if (!empty($record->$fieldname) && $data->data != $record->$fieldname) {
                                $DB->set_field('user_info_data', 'data', $record->$fieldname, ['id' => $data->id]);
                            }
                        } else {
                            $rec = new stdClass();
                            $rec->userid = $user->userid;
                            $rec->fieldid = $fieldid;
                            $rec->data = $record->$fieldname;
                            $DB->insert_record('user_info_data', $rec);
                        }
                    }
                }

                if (!$proficiency = $DB->get_record('report_ghs_english_proficien', ['userid' => $user->userid])) {
                    $proficiency = $DB->get_record('report_ghs_english_proficien', ['oen' => $record->oen]);
                }

                if ($proficiency) {
                    $record->id = $proficiency->id;
                    $DB->update_record('report_ghs_english_proficien', $record);
                } else {
                    $DB->insert_record('report_ghs_english_proficien', $record);
                }
            } else {
                if ($proficiency = $DB->get_record('report_ghs_english_proficien', ['oen' => $record->oen])) {
                    $record->id = $proficiency->id;
                    $DB->update_record('report_ghs_english_proficien', $record);
                } else {
                    $DB->insert_record('report_ghs_english_proficien', $record);
                }
            }
        }

        $this->importer->cleanup();

        return true;
    }
}
