<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * ghs_english_proficiency_summary class.
 *
 * @package    report_ghs
 * @copyright  2023 Michael Gardener <mgardener@cissq.com>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
namespace report_ghs\output;

use renderable;
use renderer_base;
use templatable;
use stdClass;

/**
 * ghs_english_proficiency_summary class.
 *
 * @package    report_ghs
 * @copyright  2023 Michael Gardener <mgardener@cissq.com>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
class ghs_english_proficiency_summary implements renderable, templatable {
    /**
     * @var int
     */
    private $totalstudents;
    private $selectedstudents;
    private $sql;
    private $params;

    /**
     * Construct.
     */
    public function __construct($selectedstudents, $sql, $params) {
        $this->totalstudents = $this->get_number_of_student();
        $this->selectedstudents = $selectedstudents;
        $this->sql = $sql;
        $this->params = $params;
    }
    /**
     * Export the data.
     *
     * @param renderer_base $output
     * @return stdClass
     */
    public function export_for_template(renderer_base $output) {
        global $DB;

        $data = new stdClass();
        $data->selectedstudents = $this->selectedstudents;
        $data->in1Yearprogram = 0;
        $data->in2Yearprogram = 0;
        $data->in25Yearprogram = 0;
        $data->in3Yearprogram = 0;
        $data->in4Yearprogram = 0;
        $data->inotherprogram = 0;
        $data->inundeclaredprogram = 0;
        $data->inindependentprogram = 0;
        $data->innoncredit = 0;
        $data->withc2opt = 0;
        $data->withc1opt = 0;
        $data->withb2opt = 0;
        $data->withb1opt = 0;
        $data->witha2opt = 0;
        $data->witha1opt = 0;
        $data->withoptnotreported = 0;
        $data->withoptnotrequired = 0;
        $data->completedosslt = 0;
        $data->passedosslt = 0;
        $data->failedosslt = 0;
        $data->failedosslteng3uover = 0;
        $data->failedosslteng4uover = 0;
        $data->failedosslteng3ueng4uover = 0;

        $rs = $DB->get_recordset_sql($this->sql, $this->params);

        foreach ($rs as $rec) {
            if ($rec->program == '1 Year') {
                $data->in1Yearprogram++;
            }
            if ($rec->program == '2 Year') {
                $data->in2Yearprogram++;
            }
            if ($rec->program == '2.5 Year') {
                $data->in25Yearprogram++;
            }
            if ($rec->program == '3 Year') {
                $data->in3Yearprogram++;
            }
            if ($rec->program == '4 Year') {
                $data->in4Yearprogram++;
            }
            if ($rec->program == 'Other') {
                $data->inotherprogram++;
            }
            if (empty($rec->program)) {
                $data->inundeclaredprogram++;
            }
            if (in_array($rec->program, ['Independent', 'Independent OSSD'])) {
                $data->inindependentprogram++;
            }
            if ($rec->program == 'Non-credit') {
                $data->innoncredit++;
            }
            if ($rec->opt == get_string('c2', 'report_ghs')) {
                $data->withc2opt++;
            }
            if ($rec->opt == get_string('c1', 'report_ghs')) {
                $data->withc1opt++;
            }
            if ($rec->opt == get_string('b2', 'report_ghs')) {
                $data->withb2opt++;
            }
            if ($rec->opt == get_string('b1', 'report_ghs')) {
                $data->withb1opt++;
            }
            if ($rec->opt == get_string('a2', 'report_ghs')) {
                $data->witha2opt++;
            }
            if ($rec->opt == get_string('a1', 'report_ghs')) {
                $data->witha1opt++;
            }
            if ($rec->opt == get_string('notreported', 'report_ghs') || empty($rec->opt)) {
                $data->withoptnotreported++;
            }
            if ($rec->opt == get_string('notrequired', 'report_ghs')) {
                $data->withoptnotrequired++;
            }
            if (in_array($rec->ossltstatus, ['Pass', 'Fail'])) {
                $data->completedosslt++;
            }
            if ($rec->ossltstatus === 'Pass') {
                $data->passedosslt++;
            }
            if ($rec->ossltstatus === 'Fail') {
                $data->failedosslt++;
            }
            if ($rec->ossltstatus === 'Fail' && ($rec->_eng3u >= 75 || $rec->eng3u_grade >= 75)) {
                $data->failedosslteng3uover++;
            }
            if ($rec->ossltstatus === 'Fail' && ($rec->_eng4u >= 75 || $rec->eng4u_grade >= 75)) {
                $data->failedosslteng4uover++;
            }
            if ($rec->ossltstatus === 'Fail' && ($rec->_eng3u >= 75 || $rec->eng3u_grade >= 75 || $rec->_eng4u >= 75 || $rec->eng4u_grade >= 75)) {
                $data->failedosslteng3ueng4uover++;
            }
        }
        $rs->close();
        $data->selectedstudentpercentage = $this->percentage($this->selectedstudents, $this->selectedstudents);
        $data->in1Yearprogrampercent = $this->percentage($data->in1Yearprogram, $this->selectedstudents);
        $data->in2Yearprogrampercent = $this->percentage($data->in2Yearprogram, $this->selectedstudents);
        $data->in25Yearprogrampercent = $this->percentage($data->in25Yearprogram, $this->selectedstudents);
        $data->in3Yearprogrampercent = $this->percentage($data->in3Yearprogram, $this->selectedstudents);
        $data->in4Yearprogrampercent = $this->percentage($data->in4Yearprogram, $this->selectedstudents);
        $data->inotherprogrampercent = $this->percentage($data->inotherprogram, $this->selectedstudents);
        $data->inundeclaredprogrampercent = $this->percentage($data->inundeclaredprogram, $this->selectedstudents);
        $data->inindependentprogrampercent = $this->percentage($data->inindependentprogram, $this->selectedstudents);
        $data->innoncreditpercent = $this->percentage($data->innoncredit, $this->selectedstudents);
        $data->withc2optpercent = $this->percentage($data->withc2opt, $this->selectedstudents);
        $data->withc1optpercent = $this->percentage($data->withc1opt, $this->selectedstudents);
        $data->withb2optpercent = $this->percentage($data->withb2opt, $this->selectedstudents);
        $data->withb1optpercent = $this->percentage($data->withb1opt, $this->selectedstudents);
        $data->witha2optpercent = $this->percentage($data->witha2opt, $this->selectedstudents);
        $data->witha1optpercent = $this->percentage($data->witha1opt, $this->selectedstudents);
        $data->withoptnotreportedpercent = $this->percentage($data->withoptnotreported, $this->selectedstudents);
        $data->withoptnotrequiredpercent = $this->percentage($data->withoptnotrequired, $this->selectedstudents);
        $data->completedossltpercent = $this->percentage($data->completedosslt, $this->selectedstudents);

        $data->completedossltpercent2 = $this->percentage($data->completedosslt, $data->completedosslt);
        $data->passedossltpercent = $this->percentage($data->passedosslt, $data->completedosslt);
        $data->failedossltpercent = $this->percentage($data->failedosslt, $data->completedosslt);
        $data->failedosslteng3uoverpercent = $this->percentage($data->failedosslteng3uover, $data->completedosslt);
        $data->failedosslteng4uoverpercent = $this->percentage($data->failedosslteng4uover, $data->completedosslt);
        $data->failedosslteng3ueng4uoverpercent = $this->percentage($data->failedosslteng3ueng4uover, $data->completedosslt);

        return $data;
    }

    private function get_number_of_student() {
        global $DB;

        return $DB->count_records_sql("SELECT COUNT(1) FROM {report_ghs_english_proficien} p WHERE p.defaultrole = 'Student'");
    }

    private function percentage($number, $total) {
        if (empty($total)) {
            return 0;
        }
        return round((($number / $total) * 100), 0);
    }
}
