<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Adhoc task for KICA report
 *
 * @package     report_ghs
 * @copyright   2019 Michael Gardener <mgardener@cissq.com>
 * @license     http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace report_ghs\task;

defined('MOODLE_INTERNAL') || die();

use core_tag_tag;
use local_kica as kica;
use local_kica\output\menu_bar as MB;
use local_ned_controller\shared\kica_util;
use report_ghs\helper;
use report_ghs\shared_lib as NED;

/** @var \stdClass $CFG */
require_once($CFG->dirroot . '/report/ghs/lib.php');
require_once($CFG->dirroot.'/local/kica/lib.php');
require_once($CFG->dirroot.'/grade/grading/lib.php');
require_once($CFG->dirroot.'/lib/gradelib.php');


/**
 * Class adhoc_ghs_sla_report
 * @package report_ghs\task
 */
class adhoc_ghs_activity_report extends base_ghs_adhoc_task {

    /**
     * @return bool
     */
    public function execute() {
        if (!static::can_execute()){
            return true;
        }

        $cd = $this->get_custom_data();
        self::update_report($cd->userid ?? null);

        return true;
    }

    /**
     * @return mixed
     * @throws \dml_exception
     */
    public static function get_task() {
        global $DB;

        $classname = self::class;
        if ($task = $DB->get_record('task_adhoc', ['classname' => '\\'.$classname])) {
            $task->customdata = json_decode($task->customdata);
        }

        return $task;
    }

    /**
     * @param null $userid
     * @return bool
     * @throws \dml_exception
     * @throws \moodle_exception
     */
    public static function update_report($userid=null) {
        global $DB, $USER;

        $USER = get_admin();

        $DB->execute("TRUNCATE TABLE {report_ghs_activity}");

        $contextsystem = \context_system::instance();

        $courselist = helper::get_report_courses();

        list($insql, $params) = $DB->get_in_or_equal($courselist);

        // Courses.
        $sql = "SELECT c.*
                 FROM {course} c
                WHERE c.id {$insql}
             ORDER BY c.shortname ASC";

        $sqlsubmassign = "SELECT s.timemodified FROM {assign_submission} s WHERE s.assignment = ? AND s.userid = ? AND s.status = 'submitted' AND s.latest = 1";
        $sqlsubmquiz = "SELECT qa.timefinish FROM {quiz_attempts} qa WHERE qa.quiz = ? AND qa.userid = ? AND qa.state = 'finished' ORDER BY qa.timefinish DESC LIMIT 1";
        $sqlsubm = ['assign' => $sqlsubmassign, 'quiz' => $sqlsubmquiz];

        $sqlactivitygrade = "SELECT gg.id, gg.finalgrade, gg.timemodified, gi.gradetype, gi.grademax, gg.excluded
                               FROM {grade_items} gi
                               JOIN {grade_grades} AS gg ON gi.id = gg.itemid
                              WHERE gi.itemtype = 'mod' AND gi.itemmodule = ? AND gi.iteminstance = ? AND gg.userid = ?";

        $sqlgroup = "SELECT g.* FROM {groups} g JOIN {groups_members} m ON g.id = m.groupid WHERE g.courseid = ? AND m.userid = ?";

        $sqlcompletion = "SELECT cmc.* FROM {course_modules_completion} cmc INNER JOIN {course_modules} cm ON cmc.coursemoduleid = cm.id
                           WHERE cmc.coursemoduleid = ? AND cmc.userid = ? AND cm.deletioninprogress = 0";

        $sqlcohort = "SELECT c.* FROM {cohort} c JOIN {cohort_members} m ON c.id = m.cohortid WHERE c.contextid = ? AND m.userid = ? AND c.name NOT LIKE '*%'";

        $rs = $DB->get_recordset_sql($sql, $params);

        $stractive = get_string('active', 'report_ghs');
        $strsuspendedcourse = get_string('suspendedcourse', 'report_ghs');
        $strsuspendedsite = get_string('suspendedsite', 'report_ghs');
        $strcompleted = get_string('completed', 'report_ghs');
        $otrole = $DB->get_record('role', array('shortname' => 'online-teacher'));
        $ctrole = $DB->get_record('role', array('shortname' => 'classroom-teacher'));
        $DM = NED::get_DM();

        foreach ($rs as $course) {
            $coursecontext = \context_course::instance($course->id);
            $graders = [];
            if (!$users = get_enrolled_users($coursecontext, 'mod/assign:submit', 0, 'u.*', 'u.id', 0, 0, false)) {
                continue;
            }

            $courseid = $course->id;
            $kica = NED::get_kica_enabled($courseid);
            if ($kica){
                NED::ki_get_all_by_course($courseid, null, $users, true);
                NED::kg_get_grades_by_course($courseid, array_keys($users), false, true);
            } else {
                continue;
            }

            foreach ($users as $user) {
                $group = null;
                if ($groups = $DB->get_records_sql($sqlgroup, [$course->id, $user->id])) {
                    $group = reset($groups);
                }

                $cohort = $DB->get_record_sql($sqlcohort, [$contextsystem->id, $user->id], IGNORE_MULTIPLE);

                if (helper::exclude_user_from_kica_report($user)) {
                    continue;
                }

                $data = new \stdClass();
                $data->userid = $user->id;
                $data->courseid = $course->id;
                $data->groupid = $group->id ?? null;
                $data->lastactivity = helper::last_activity_completed($user->id, $course->id);

                if ($cohort) {
                    $data->cohortid =  $cohort->id;
                }

                $ROUND = 2;
                $kica = \local_kica\shared_lib::get_kica($courseid);
                $calc_natural = $kica->calculation == kica\kica_grade::CALCULATION_NATURAL;

                $kicagroups = NED::get_kica_groups($courseid);

                // Grade tables
                $kica_grades = NED::kg_get_grades_by_course($courseid, $user->id, true)[$user->id] ?? [];

                foreach ($kicagroups as $kicagroupvalue => $kicagroup_title) {
                    $kica_activities = NED::ki_get_all_by_course($courseid, $kicagroupvalue, $user->id);

                    foreach ($kica_activities as $kicaitem) {
                        // Reset values.
                        $data->finalgrade = null;
                        foreach (NED::KICA_KEYS as $kica_key) {
                            $data->{$kica_key} = null;
                        }

                        $kicagrade = $kica_grades[$kicaitem->id] ?? null;
                        if (empty($kicagrade) || $kicagrade->flag) {
                            continue;
                        }
                        if (!$kicagrade->is_graded()) {
                            continue;
                        }

                        $data->cmid = $kicaitem->cm->id;
                        $data->module = $kicaitem->cm->modname;
                        $data->instance = $kicaitem->cm->instance;
                        $data->activityname = $kicaitem->cm->name;
                        $data->timecreated = time();

                        $grademax = $kicaitem->knowledge + $kicaitem->inquiry + $kicaitem->communication + $kicaitem->application;
                        $finalgrade = $kicagrade->get_finalgrade();

                        $isnull = true;

                        foreach (NED::KICA_KEYS as $kica_key){
                            if (!empty($kicaitem->{$kica_key}) && $kicaitem->{$kica_key} > 0 && !is_null($finalgrade)) {
                                if (!is_null($kicagrade->{$kica_key})) {
                                    $isnull = false;
                                    $data->{$kica_key} = ($kicagrade->{$kica_key} / $kicaitem->{$kica_key}) * 100;
                                }
                            }
                        }

                        if (!$isnull && !empty($grademax) && $grademax > 0 && !is_null($finalgrade)) {
                            $data->finalgrade =  ($finalgrade / $grademax) * 100;
                        }

                        $DB->insert_record('report_ghs_activity', $data);
                    }
                }
            }
        }

        $rs->close();

        if (empty($CFG->noemailever) && $userid) {
            $site = get_site();
            $from = \core_user::get_support_user();
            $from->firstname = $site->fullname;
            $from->lastname = "";

            $to = $DB->get_record('user', array('id' => $userid));
            $to->mailformat = 1;

            email_to_user($to, $from, get_string('ghsactivity', 'report_ghs'), '', "TASK HAS BEEN EXECUTED!");
        }

        return true;
    }

}