<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Adhoc task for GRoup Enrollment Report
 *
 * @package     report_ghs
 * @copyright   2024 Michael Gardener <mgardener@cissq.com>
 * @license     http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace report_ghs\task;

use block_ned_teacher_tools\deadline_manager;
use report_ghs\helper;
use report_ghs\shared_lib as NED;
use stdClass;

defined('MOODLE_INTERNAL') || die();

global $CFG;
require_once($CFG->dirroot . '/report/ghs/lib.php');
require_once($CFG->dirroot.'/local/kica/lib.php');
require_once($CFG->dirroot.'/grade/grading/lib.php');
require_once($CFG->dirroot.'/lib/gradelib.php');


/**
 * Class adhoc_ghs_group_enrollment_report
 * @package report_ghs\task
 */
class adhoc_ghs_class_deadlines_report extends base_ghs_adhoc_task {

    /**
     * @return bool
     */
    public function execute() {
        global $CFG, $DB;

        $cd = $this->get_custom_data();

        self::update_report($cd->userid ?? null);

        return true;
    }

    /**
     * @return mixed
     * @throws \dml_exception
     */
    public static function get_task() {
        global $DB;

        $classname = self::class;
        if ($task = $DB->get_record('task_adhoc', ['classname' => '\\'.$classname])) {
            $task->customdata = json_decode($task->customdata);
        }

        return $task;
    }

    /**
     * @param null $userid
     * @return bool
     * @throws \dml_exception
     * @throws \moodle_exception
     */
    public static function update_report($userid=null) {
        global $CFG, $DB;

        $time = time();

        $cfg = get_config('report_ghs');

        $DB->delete_records('report_ghs_class_deadlines');

        $contextsystem = \context_system::instance();

        $courselist = helper::get_report_courses();

        list($insql, $params) = $DB->get_in_or_equal($courselist);

        $sql = "SELECT cm.id cmid, 
                       cm.instance,
                       c.id courseid,
                       c.shortname course,
                       m.name module,
                       CASE
                           WHEN m.name = 'assign' THEN (SELECT q.name FROM {assign} q WHERE q.id = cm.instance)
                           WHEN m.name = 'quiz' THEN (SELECT q.name FROM {quiz} q WHERE q.id = cm.instance)    
                           ELSE ''
                       END as 'activity',
                       g.id classid,
                       g.name class,
                       g.enddate,
                       s.id schoolid,
                       s.code school
                  FROM {course_modules} cm
                  JOIN {modules} m ON cm.module = m.id
                  JOIN {groups} g ON cm.course = g.courseid
                  JOIN {course} c ON cm.course = c.id
             LEFT JOIN {local_schoolmanager_school} s ON s.code = SUBSTRING(g.name, 1, 4)
                 WHERE m.name IN ('assign','quiz')
                   AND cm.deletioninprogress = 0
                   AND g.schedule = 2 
                   AND cm.id IN (SELECT ti2.itemid FROM {tag_instance} ti2 JOIN {tag} t2 ON ti2.tagid = t2.id WHERE ti2.itemtype = 'course_modules' AND t2.name = 'summative')
                   AND cm.course {$insql} 
                   AND cm.course NOT IN (SELECT fd.instanceid FROM {customfield_data} fd JOIN {customfield_field} f ON  fd.fieldid = f.id WHERE f.shortname = 'ghs_report' AND fd.intvalue = 1)
              ORDER BY g.id";


        $rs = $DB->get_recordset_sql($sql, $params);

        $userscache = [];

        foreach ($rs as $rec) {
            $coursecontext = \context_course::instance($rec->courseid);

            if (!isset($userscache[$rec->classid])) {
                $activestudents = get_enrolled_users($coursecontext, 'mod/assign:submit', $rec->classid, 'u.*', 'u.id', 0, 0, true);

                foreach ($activestudents as $index => $activestudent) {
                    if ($activestudent->suspended == 1) {
                        unset($activestudents[$index]);
                    }
                }

                $userscache = [$rec->classid => $activestudents];
            }

            if (empty($userscache[$rec->classid])) {
                continue;
            }

            $data = new \stdClass();
            $data->cohortid = $rec->schoolid ?? 0;
            $data->courseid = $rec->courseid;
            $data->groupid = $rec->classid;
            $data->cmid = $rec->cmid;
            $data->instance = $rec->instance;
            $data->module = $rec->module;
            $data->totalstudents = count($activestudents ?? []);

            $classdeadline = 0;

            if ($data->module === 'assign') {
                $sql = "SELECT MAX(o.duedate) FROM {assign_overrides} o WHERE o.assignid = ? AND o.groupid = ?";
                if ($override = $DB->get_field_sql($sql, [$data->instance, $rec->classid])) {
                    $classdeadline = $override;
                }
            } else if ($data->module === 'quiz') {
                $sql = "SELECT MAX(o.timeclose) FROM {quiz_overrides} o WHERE o.quiz = ? AND o.groupid = ?";
                if ($override = $DB->get_field_sql($sql, [$data->instance, $rec->classid])) {
                    $classdeadline = $override;
                }
            }

            foreach ($userscache[$rec->classid] as $activestudent) {
                $data->deadline = 0;
                $data->deadlinetype = '';
                $data->userid = $activestudent->id;

                if ($data->module === 'assign') {
                    $sql = "SELECT MAX(o.duedate) FROM {assign_overrides} o WHERE o.assignid = ? AND o.userid = ?";
                    if ($override = $DB->get_field_sql($sql, [$data->instance, $activestudent->id])) {
                        $data->deadline = $override;
                        $data->deadlinetype = NED::DEADLINE_TYPE_EXTENSION;
                    }
                } else if ($data->module === 'quiz') {
                    $sql = "SELECT MAX(o.timeclose) FROM {quiz_overrides} o WHERE o.quiz = ? AND o.userid = ?";
                    if ($override = $DB->get_field_sql($sql, [$data->instance, $activestudent->id])) {
                        $data->deadline = $override;
                        $data->deadlinetype = NED::DEADLINE_TYPE_EXTENSION;
                    }
                }

                if (empty($data->deadlinetype) && !empty($classdeadline)) {
                    $data->deadline = $classdeadline;
                    $data->deadlinetype = NED::DEADLINE_TYPE_CLASS;
                }

                $data->timecreated = time();
                $DB->insert_record('report_ghs_class_deadlines', $data);
            }
        }

        $rs->close();

        set_config('classdeadlineslastupdate', time(), 'report_ghs');

        if (empty($CFG->noemailever) && $userid) {
            $site = get_site();
            $from = \core_user::get_support_user();
            $from->firstname = $site->fullname;
            $from->lastname = "";

            $to = $DB->get_record('user', array('id' => $userid));
            $to->mailformat = 1;

            email_to_user($to, $from, get_string('ghsclassdeadlines', 'report_ghs'), '', "TASK HAS BEEN EXECUTED!");
        }

        return true;
    }
}