<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Adhoc task for Enrollment Report
 *
 * @package     report_ghs
 * @copyright   2019 Michael Gardener <mgardener@cissq.com>
 * @license     http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace report_ghs\task;

use report_ghs\helper;
use stdClass;

defined('MOODLE_INTERNAL') || die();

global $CFG;
require_once($CFG->dirroot . '/report/ghs/lib.php');
require_once($CFG->dirroot.'/local/kica/lib.php');
require_once($CFG->dirroot.'/grade/grading/lib.php');
require_once($CFG->dirroot.'/lib/gradelib.php');


/**
 * Class adhoc_ghs_enrollment_report
 * @package report_ghs\task
 */
class adhoc_ghs_enrollment_report extends base_ghs_adhoc_task {

    /**
     * @return bool
     */
    public function execute() {
        if (!static::can_execute()){
            return true;
        }

        $cd = $this->get_custom_data();
        self::update_report($cd->userid ?? null);

        return true;
    }

    /**
     * @return mixed
     * @throws \dml_exception
     */
    public static function get_task() {
        global $DB;

        $classname = self::class;
        if ($task = $DB->get_record('task_adhoc', ['classname' => '\\'.$classname])) {
            $task->customdata = json_decode($task->customdata);
        }

        return $task;
    }

    /**
     * @param null $userid
     * @return bool
     * @throws \dml_exception
     * @throws \moodle_exception
     */
    public static function update_report($userid=null) {
        global $DB;

        $cfg = get_config('report_ghs');

        $DB->execute("TRUNCATE TABLE {report_ghs_enrollment}");

        $contextsystem = \context_system::instance();

        // $courses81 = \core_course_category::get(81, MUST_EXIST, true)->get_courses(['recursive' => 1]);
        // $courses82 = \core_course_category::get(82, MUST_EXIST, true)->get_courses(['recursive' => 1]);

        $courselist = helper::get_report_courses();

        list($insql, $params) = $DB->get_in_or_equal($courselist);

        // Courses.
        $sql = "SELECT c.* FROM {course} c WHERE c.id {$insql} ORDER BY c.shortname ASC";

        $sqlgroup = "SELECT g.* FROM {groups} g JOIN {groups_members} m ON g.id = m.groupid WHERE g.courseid = ? AND m.userid = ?";
        $sqlcohort = "SELECT c.* FROM {cohort} c JOIN {cohort_members} m ON c.id = m.cohortid WHERE c.contextid = ? AND m.userid = ? AND c.name NOT LIKE '*%'";

        $sqlenrol = "SELECT MIN(ue.timestart) timestart, MAX(ue.timeend) timeend
                       FROM {user_enrolments} ue
                       JOIN {enrol} e ON ue.enrolid = e.id
                      WHERE ue.userid = ? AND e.courseid = ?";

        $sqlcourserole = "SELECT r.id FROM {role_assignments} ra JOIN {context} cx ON ra.contextid = cx.id JOIN {role} r ON ra.roleid = r.id 
                          WHERE ra.userid = ? AND cx.contextlevel = ".CONTEXT_COURSE." AND cx.instanceid = ?";

        $sqlcustomfield = "SELECT d.data FROM {user_info_field} f JOIN {user_info_data} d ON f.id = d.fieldid WHERE f.shortname = ? AND d.userid = ?";

        $rs = $DB->get_recordset_sql($sql, $params);

        foreach ($rs as $course) {
            $coursecontext = \context_course::instance($course->id);

            if (!$users = get_enrolled_users($coursecontext)) {
                continue;
            }

            foreach ($users as $user) {
                $data = new \stdClass();
                $data->userid = $user->id;
                $data->courseid = $course->id;

                if ($cohort = $DB->get_record_sql($sqlcohort, [$contextsystem->id, $user->id], IGNORE_MULTIPLE)) {
                    $data->cohortid = $cohort->id;
                }

                foreach (['OEN', 'default_role', 'graduation', 'program', 'registered'] as $item) {
                    if ($customfield = $DB->get_record_sql($sqlcustomfield, [$item, $user->id])) {
                        $item = strtolower(str_replace('_', '', $item));
                        $data->{$item} = $customfield->data;
                    }
                }

                $data->frate = helper::get_rate('formative', $user->id, $course->id);
                $data->srate = helper::get_rate('summative', $user->id, $course->id);

                $userenrolment = $DB->get_record_sql($sqlenrol, [$user->id, $course->id]);
                $data->timestart = $userenrolment->timestart ?? null;
                $data->timeend = $userenrolment->timeend ?? null;

                $groups = $DB->get_records_sql($sqlgroup, [$course->id, $user->id]);
                $roles = $DB->get_records_sql($sqlcourserole, [$user->id, $course->id]);

                $data->timecreated = time();

                if ($groups && $roles) {
                    foreach ($groups as $group) {
                        $data->groupid = $group->id;
                        foreach ($roles as $role) {
                            $data->courserole = $role->id;
                            $DB->insert_record('report_ghs_enrollment', $data);
                        }
                    }
                } else if ($groups) {
                    foreach ($groups as $group) {
                        $data->groupid = $group->id;
                        $DB->insert_record('report_ghs_enrollment', $data);
                    }
                } else if ($roles) {
                    foreach ($roles as $role) {
                        $data->courserole = $role->id;
                        $DB->insert_record('report_ghs_enrollment', $data);
                    }
                } else {
                    $DB->insert_record('report_ghs_enrollment', $data);
                }
            }
        }

        $rs->close();

        set_config('enrollmentlastupdate', time(), 'report_ghs');

        if (empty($CFG->noemailever) && $userid) {
            $site = get_site();
            $from = \core_user::get_support_user();
            $from->firstname = $site->fullname;
            $from->lastname = "";

            $to = $DB->get_record('user', array('id' => $userid));
            $to->mailformat = 1;

            email_to_user($to, $from, get_string('ghsenrollment', 'report_ghs'), '', "TASK HAS BEEN EXECUTED!");
        }

        return true;
    }
}