<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Adhoc task for KICA report
 *
 * @package     report_ghs
 * @copyright   2019 Michael Gardener <mgardener@cissq.com>
 * @license     http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace report_ghs\task;

use local_kica as kica;
use local_kica\helper as kica_helper;
use local_kica\kica_grade;
use local_kica\shared_lib;
use report_ghs\helper;
use report_ghs\shared_lib as NED;

defined('MOODLE_INTERNAL') || die();

/** @var \stdClass $CFG */
require_once($CFG->dirroot . '/report/ghs/lib.php');
require_once($CFG->dirroot.'/local/kica/lib.php');
require_once($CFG->dirroot.'/grade/grading/lib.php');
require_once($CFG->dirroot.'/lib/gradelib.php');


/**
 * Class adhoc_ghs_grade_comparisons_report
 * @package report_ghs\task
 */
class adhoc_ghs_grade_comparisons_report extends base_ghs_adhoc_task {

    /**
     * @return bool
     */
    public function execute() {
        if (!static::can_execute()){
            return true;
        }

        $cd = $this->get_custom_data();

        self::update_report($cd->userid ?? null);
        return true;
    }

    /**
     * @return mixed
     * @throws \dml_exception
     */
    public static function get_task() {
        global $DB;

        $classname = self::class;
        if ($task = $DB->get_record('task_adhoc', ['classname' => '\\'.$classname])) {
            $task->customdata = json_decode($task->customdata);
        }

        return $task;
    }

    /**
     * @param null $userid
     * @return bool
     * @throws \dml_exception
     * @throws \moodle_exception
     */
    public static function update_report($userid=null) {
        global $DB;

        $admin = get_admin();

        // Do everything as admin by default.
        \core\session\manager::set_user($admin);

        $cfg = get_config('report_ghs');

        $DB->execute("TRUNCATE TABLE {report_ghs_grade_comparisons}");

        $contextsystem = \context_system::instance();
        $ctrole = $DB->get_record('role', array('shortname' => 'classroom-teacher'));

        $courselist = helper::get_report_courses();

        list($insql, $params) = $DB->get_in_or_equal($courselist);

        // Courses.
        $sql = "SELECT c.* 
                  FROM {local_kica} k 
                  JOIN {course} c 
                    ON k.courseid = c.id
                 WHERE k.enabled = 1
                   AND c.id {$insql}
              ORDER BY c.shortname ASC";

        $sqlcourseavg = "SELECT gg.id, gg.finalgrade
             FROM {grade_items} gi
             JOIN {grade_grades} AS gg
               ON gi.id = gg.itemid
            WHERE gi.itemtype = 'course' AND gi.courseid = ? AND gg.userid = ?";

        $sqlcohort = "SELECT c.* FROM {cohort} c JOIN {cohort_members} m ON c.id = m.cohortid WHERE c.contextid = ? AND m.userid = ?";

        $sqlgroup = "SELECT g.* FROM {groups} g JOIN {groups_members} m ON g.id = m.groupid WHERE g.courseid = ? AND m.userid = ?";

        $sqlenrol = "SELECT MIN(ue.timestart) FROM {user_enrolments} ue JOIN {enrol} e ON ue.enrolid = e.id WHERE ue.timestart > 0 AND e.courseid = ? AND ue.userid = ?";

        $rs = $DB->get_recordset_sql($sql, $params);

        foreach ($rs as $course) {
            $kica = NED::get_kica($course->id);
            if (!$kica){
                continue;
            }

            $coursecontext = \context_course::instance($course->id);

            if (!$users = get_enrolled_users($coursecontext, 'mod/assign:submit', 0, 'u.*', 'u.id', 0, 0, false)) {
                continue;
            }

            $graders = [];
            $kicagroups = NED::get_kica_groups($course->id, true);
            NED::kg_get_grades_by_course($course->id, array_keys($users), false, true);

            foreach ($users as $index => $user) {
                $data = new \stdClass();
                $data->courseid = $course->id;
                $data->userid = $user->id;
                $data->gradingerror = 0;

                // Grading error check.
                foreach ($kicagroups as $kicagroupvalue) {
                    $kica_items = NED::ki_get_all_by_course($course->id, $kicagroupvalue);
                    foreach ($kica_items as $kicaitem) {
                        $kicagrade = kica\kica_grade::get_by_userid_itemid($user->id, $kicaitem->id);
                        if (empty($kicagrade->id) || $kicagrade->flag) {
                            continue;
                        }

                        $needsupdate = kica_helper::is_gradingform_needupdate($kicaitem, $user->id) || $kicagrade->grade_mismatch();
                        if ($needsupdate && $kicagrade->is_gradable() && !$kicagrade->is_graded()){
                            $data->gradingerror++;
                        }
                    }
                }

                if ($groups = $DB->get_records_sql($sqlgroup, [$course->id, $user->id])) {
                    $data->groupid = (reset($groups))->id;
                    if (!isset($graders[$data->groupid])) {
                        if ($users = get_enrolled_users($coursecontext, 'mod/assign:grade', $data->groupid)) {
                            $graders[$data->groupid] = (reset($users))->id;
                            $data->gtid = $graders[$data->groupid];
                        }
                    } else {
                        $data->gtid = $graders[$data->groupid];
                    }
                    if ($ctusers = helper::get_role_users($ctrole->id, $coursecontext, $data->groupid)) {
                        $data->ctid = (count($ctusers) > 1) ? -1 : $ctusers[0]->id;
                    }
                }

                $kica_average_data = shared_lib::kg_get_kica_average($course->id, $user->id);
                $data->grade = shared_lib::kg_get_course_average($course->id, $user->id, shared_lib::FINALGRADE, false, $kica_average_data);

                $assign = kica_grade::get_kica_average($course->id, $user->id, null, null, false, null, 'assign');
                $assignment_average_grades = shared_lib::kg_get_course_average($course->id, $user->id, null, false, $assign);
                $data->assigns = shared_lib::grade_val($assignment_average_grades[shared_lib::FINALGRADE] ?? null, false, 5, null);

                $quiz = kica_grade::get_kica_average($course->id, $user->id, null, null, false, null, 'quiz');
                $test_average_grades = shared_lib::kg_get_course_average($course->id, $user->id, null, false, $quiz);
                $data->tests = shared_lib::grade_val($test_average_grades[shared_lib::FINALGRADE] ?? null, false, 5, null);

                $assigncompleted = $assign['stat']['assigncompleted'] ?? 0;
                $testcompleted = $quiz['stat']['testcompleted'] ?? 0;

                if ($assigncompleted < 2 || $testcompleted < 2) {
                    continue;
                }

                if (abs($data->assigns - $data->tests) < 20) {
                    continue;
                }

                $data->timecreated = time();

                $DB->insert_record('report_ghs_grade_comparisons', $data);
            }

            NED::purge_course_depended_caches();
        }

        $rs->close();

        if (empty($CFG->noemailever) && $userid) {
            $site = get_site();
            $from = \core_user::get_support_user();
            $from->firstname = $site->fullname;
            $from->lastname = "";

            $to = $DB->get_record('user', array('id' => $userid));
            $to->mailformat = 1;

            email_to_user($to, $from, get_string('ghsgrades', 'report_ghs'), '', "TASK HAS BEEN EXECUTED!");
        }
        set_config('kicalastupdate', time(), 'report_ghs');
        return true;
    }
}